/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include "j-cf.h"

#include "j-chkmail.h"

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
char           *conf_file = J_CONF_FILE;

char            domain[128];

int             log_level = 10;
int             log_facility = LOG_LOCAL5;

unsigned int    statistics_interval = 300;

char            sm_sock[512];

#define      STRING(a)     #a

#define      CF_NONE    0

#define      J_STR      1
#define      J_ENUM     2
#define      J_INT      3

#define      DIM_CF     256

static char    *ENUM_ACTION[] = { "OK", "REJECT", "WARN", "DISCARD", NULL };
static char    *ENUM_REJECT[] = { "OK", "REJECT", "TEMPFAIL", NULL };
static char    *ENUM_NO_YES[] = { "NO", "YES", NULL };
static char    *ENUM_SENDER[] = { "SENDER", "OTHER", NULL };
static char    *ENUM_SUBJECT[] = { "SUBJECT", "OTHER", NULL };
static char    *ENUM_HOSTNAME[] = { "SYSTEM", "SENDMAIL", "OTHER", NULL };
static char    *ENUM_PRESENCE[] = { "SHOW", "HIDE", NULL };
static char    *ENUM_AV_TYPE[] = { "NONE", "USER", "UVSCAN",
  "SOPHOS", "TREND", NULL
};
static char    *ENUM_TEXT_DB[] = { "TEXT", "DB", NULL };
static char    *ENUM_SCOPE[] = { "ALL", "ATTACHMENTS", NULL };
static char    *ENUM_ORIGIN[] = { "ALL", "UNKNOWN", NULL };

struct jcfrec {
  int             id;
  char            name[64];
  int             kind;
  int             slen;
  long            iv;
  char           *sv;
  char          **opt;
} jcfrec;

static struct jcfrec cf[DIM_CF];

int             cf_init ();

void            cf_clear_values ();

void            cf_load_default ();

int             cf_add_id_enum (int id, char *name, char **opt, char *val);
int             cf_add_id_str (int id, char *name, int len, char *val);
int             cf_add_id_int (int id, char *name, char *val);

int             cf_set_str_val (int id, char *val);

int             cf_set_int_val (int id, int val);
int             cf_get_int_val (int id);

int             cf_set_val (int id, char *name);

static void     cf_define ();

int             cf_get_id (char *);

static int      get_enum_index (char *, char **);

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static int
cf_compare (a, b)
     const struct jcfrec *a;
     const struct jcfrec *b;
{
  if (a->id == 0 || b->id == 0)
    return 0;
  if (a->id == 0)
    return 1;
  if (b->id == 0)
    return -1;
  return (a->id - b->id);
}

static void
cf_sort ()
{
  qsort (cf, DIM_CF, sizeof (struct jcfrec), cf_compare);
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_init ()
{
  memset (cf, 0, sizeof (cf));

  cf_define ();

  return 0;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static void
cf_define ()
{
  cf_add_id_str (CF_VERSION, STRING (CF_VERSION), 256, "1.3");

  /* n */
  cf_add_id_str (CF_DOMAIN, STRING (DOMAIN), 256, "");
  cf_add_id_enum (CF_J_HOSTNAME, STRING (J_HOSTNAME), ENUM_HOSTNAME, "SYSTEM");
  cf_add_id_enum (CF_PRESENCE, STRING (PRESENCE), ENUM_PRESENCE, "SHOW");
  cf_add_id_str (CF_USER, STRING (USER), 256, RUN_AS_USER);
  cf_add_id_str (CF_GROUP, STRING (GROUP), 256, RUN_AS_GROUP);


  cf_add_id_str (CF_FILE_DESCRIPTORS, STRING (FILE_DESCRIPTORS), 32, "MAX");
  cf_add_id_int (CF_FD_FREE_SOFT, STRING (FD_FREE_SOFT), "100");
  cf_add_id_int (CF_FD_FREE_HARD, STRING (FD_FREE_HARD), "50");
  cf_add_id_enum (CF_USE_SELECT_LIMIT, STRING (USE_SELECT_LIMIT), ENUM_NO_YES, "YES");

  cf_add_id_int (CF_FILTER_THROTTLE, STRING (FILTER_THROTTLE), "1000");

  /* 1n */
  cf_add_id_str (CF_SOCKET, STRING (SOCKET), 256, "");
  cf_add_id_int (CF_INET_SOCK, STRING (INET_SOCK), "2000");
  cf_add_id_str (CF_UNIX_SOCK, STRING (UNIX_SOCK), 256, "");

  /* 2n */
  cf_add_id_enum (CF_WARN_SENDER, STRING (WARN_SENDER), ENUM_NO_YES, "YES");
  cf_add_id_enum (CF_WARN_RCPT, STRING (WARN_RCPT), ENUM_NO_YES, "YES");
  cf_add_id_enum (CF_J_SENDER, STRING (J_SENDER), ENUM_SENDER, "SENDER");
  cf_add_id_enum (CF_J_SUBJECT, STRING (J_SUBJECT), ENUM_SUBJECT, "SUBJECT");


  /* 3n */
  cf_add_id_str (CF_LOG_FACILITY, STRING (LOG_FACILITY), 256, "local5");
  cf_add_id_int (CF_LOG_LEVEL, STRING (LOG_LEVEL), "10");
  cf_add_id_enum (CF_LOG_ATTACHMENTS, STRING (LOG_ATTACHMENTS), ENUM_NO_YES, "NO");
  cf_add_id_enum (CF_LOG_COUNTERS, STRING (LOG_COUNTERS), ENUM_NO_YES, "YES");
  cf_add_id_enum (CF_DUMP_COUNTERS, STRING (DUMP_COUNTERS), ENUM_NO_YES, "YES");

  cf_add_id_int (CF_STATS_INTERVAL, STRING (STATS_INTERVAL), "300");
  cf_add_id_int (CF_CLEANUP_INTERVAL, STRING (CLEANUP_INTERVAL), "21600");
  cf_add_id_int (CF_QUARANTINE_MAX_AGE, STRING (QUARANTINE_MAX_AGE), "86400");

  /* 4n */
  cf_add_id_str (CF_ERROR_MSG_FILE, STRING (ERROR_MSG_FILE), 256, J_ERROR_MSG_FILE);
  cf_add_id_str (CF_CW_FILE, STRING (CW_FILE), 256, J_CW_FILE);
  cf_add_id_str (CF_NETS_FILE, STRING (NETS_FILE), 256, J_NETS_FILE);
  cf_add_id_str (CF_USERS_FILE, STRING (USERS_FILE), 256, J_USERS_FILE);
  cf_add_id_str (CF_HOST_ACCESS_FILE, STRING (HOST_ACCESS_FILE), 256,
                 J_HOST_ACCESS_FILE);
  cf_add_id_str (CF_USER_ACCESS_FILE, STRING (USER_ACCESS_FILE), 256,
                 J_USER_ACCESS_FILE);
  cf_add_id_str (CF_REGEX_FILE, STRING (REGEX_FILE), 256, J_REGEX_FILE);

  cf_add_id_int (CF_AUTO_RELOAD_CONF, STRING (AUTO_RELOAD_CONF), "0");
  cf_add_id_int (CF_AUTO_RELOAD_TABLES, STRING (AUTO_RELOAD_TABLES), "0");

  cf_add_id_enum (CF_CONF_MODE, STRING (CONF_MODE), ENUM_TEXT_DB, "TEXT");

  /* 5n */
  cf_add_id_str (CF_WORKDIR, STRING (WORKDIR), 256, J_WORKDIR);
  cf_add_id_str (CF_SPOOLDIR, STRING (SPOOLDIR), 256, J_SPOOLDIR);
  cf_add_id_str (CF_PID_FILE, STRING (PID_FILE), 256, J_PID_FILE);
  cf_add_id_str (CF_STATE_FILE, STRING (STATE_FILE), 256, J_STATE_FILE);
  cf_add_id_str (CF_STATS_FILE, STRING (STATS_FILE), 256, J_STATS_FILE);
  cf_add_id_str (CF_XFILES_LOG_FILE, STRING (XFILES_LOG_FILE), 256, J_XFILES_LOG);
  cf_add_id_str (CF_REGEX_LOG_FILE, STRING (REGEX_LOG_FILE), 256, J_REGEX_LOG);

  /* 6n */
  cf_add_id_enum (CF_DELETE_EXEC_FILES, STRING (DELETE_EXEC_FILES), ENUM_NO_YES, "NO");
  cf_add_id_enum (CF_XFILES, STRING (XFILES), ENUM_ACTION, "OK");
  cf_add_id_str (CF_FILE_EXT, STRING (FILE_EXT), 2048, "");
  cf_add_id_str (CF_FILE_REGEX, STRING (FILE_REGEX), 2048, "");
  cf_add_id_enum (CF_XFILE_SAVE_MSG, STRING (XFILE_SAVE_MSG), ENUM_NO_YES, "NO");

  cf_add_id_int (CF_HISTORY_ENTRIES, STRING (HISTORY_ENTRIES), "32");

  /* 7n */
  cf_add_id_enum (CF_AV_ACTION, STRING (AV_ACTION), ENUM_ACTION, "OK");
  cf_add_id_int (CF_AV_PORT, STRING (AV_PORT), "2001");
  cf_add_id_int (CF_AV_NB_SERVERS, STRING (AV_NB_SERVERS), "5");
  cf_add_id_enum (CF_AV_TYPE, STRING (AV_TYPE), ENUM_AV_TYPE, "USER");
  cf_add_id_str (CF_AV_PATH, STRING (AV_PATH), 256, "/usr/local/bin/userfilter");
  cf_add_id_str (CF_AV_ARGS, STRING (AV_ARGS), 256, "");
  cf_add_id_enum (CF_AV_SCOPE, STRING (AV_SCOPE), ENUM_SCOPE, "ALL");
  cf_add_id_enum (CF_AV_SAVE_MSG, STRING (AV_SAVE_MSG), ENUM_NO_YES, "NO");
  cf_add_id_int (CF_AV_TIMEOUT, STRING (AV_TIMEOUT), "30");

  /* 8n */
  cf_add_id_enum (CF_CHECK_HEADERS_CONTENT, STRING (CHECK_HEADERS_CONTENT),
                  ENUM_NO_YES, "NO");
  cf_add_id_enum (CF_CHECK_SUBJECT_CONTENT, STRING (CHECK_SUBJECT_CONTENT),
                  ENUM_NO_YES, "NO");
  cf_add_id_enum (CF_CHECK_HELO_CONTENT, STRING (CHECK_HELO_CONTENT),
                  ENUM_NO_YES, "NO");

  cf_add_id_enum (CF_CHECK_BODY_CONTENT, STRING (CHECK_BODY_CONTENT),
                  ENUM_NO_YES, "NO");
  cf_add_id_int (CF_CONTENT_CHECK_SCORE, STRING (CONTENT_CHECK_SCORE), "10");
  cf_add_id_int (CF_CONTENT_CHECK_SIZE, STRING (CONTENT_CHECK_SIZE), "100000");
  cf_add_id_enum (CF_CONTENT_CHECK_ORIGIN, STRING (CONTENT_CHECK_ORIGIN),
                  ENUM_ORIGIN, "UNKNOWN");
  cf_add_id_enum (CF_DUMP_FOUND_REGEX, STRING (DUMP_FOUND_REGEX), ENUM_NO_YES, "YES");

  cf_add_id_enum (CF_ENCODING_BINARY, STRING (ENCODING_BINARY), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_ENCODING_BASE64, STRING (ENCODING_BASE64), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_ENCODING_QUOTED_PRINTABLE,
                  STRING (ENCODING_QUOTED_PRINTABLE), ENUM_REJECT, "OK");

  cf_add_id_enum (CF_CHECK_BODY_LINE_LENGTH, STRING (CHECK_BODY_LINE_LENGTH), ENUM_NO_YES, "YES");

  /* 9n */
  cf_add_id_enum (CF_CHECK_NB_RCPT, STRING (CHECK_NB_RCPT), ENUM_NO_YES, "NO");

  cf_add_id_int (CF_MAX_RCPT_FROM_DOMAIN, STRING (MAX_RCPT_FROM_DOMAIN), "300");
  cf_add_id_int (CF_MAX_RCPT_FROM_LOCAL, STRING (MAX_RCPT_FROM_LOCAL), "1000");
  cf_add_id_int (CF_MAX_RCPT_FROM_FRIEND, STRING (MAX_RCPT_FROM_FRIEND), "200");
  cf_add_id_int (CF_MAX_RCPT_FROM_OUTSIDE, STRING (MAX_RCPT_FROM_OUTSIDE), "25");

  cf_add_id_enum (CF_CHECK_THROTTLE, STRING (CHECK_THROTTLE), ENUM_NO_YES, "NO");
  cf_add_id_enum (CF_CHECK_THROTTLE_CONN, STRING (CHECK_THROTTLE_CONN), ENUM_NO_YES,
                  "NO");
  cf_add_id_enum (CF_CHECK_THROTTLE_RCPT, STRING (CHECK_THROTTLE_RCPT), ENUM_NO_YES,
                  "NO");

  cf_add_id_enum (CF_RES_THROTTLE_FROM_DOMAIN,
                  STRING (RES_THROTTLE_FROM_DOMAIN), ENUM_REJECT, "TEMPFAIL");
  cf_add_id_enum (CF_RES_THROTTLE_FROM_LOCAL,
                  STRING (RES_THROTTLE_FROM_LOCAL), ENUM_REJECT, "TEMPFAIL");
  cf_add_id_enum (CF_RES_THROTTLE_FROM_FRIEND,
                  STRING (RES_THROTTLE_FROM_FRIEND), ENUM_REJECT, "TEMPFAIL");
  cf_add_id_enum (CF_RES_THROTTLE_FROM_OUTSIDE,
                  STRING (RES_THROTTLE_FROM_OUTSIDE), ENUM_REJECT, "REJECT");

  cf_add_id_int (CF_RCPT_THROTTLE_FROM_DOMAIN,
                 STRING (RCPT_THROTTLE_FROM_DOMAIN), "200");
  cf_add_id_int (CF_RCPT_THROTTLE_FROM_LOCAL, STRING (RCPT_THROTTLE_FROM_LOCAL), "300");
  cf_add_id_int (CF_RCPT_THROTTLE_FROM_FRIEND,
                 STRING (RCPT_THROTTLE_FROM_FRIEND), "100");
  cf_add_id_int (CF_RCPT_THROTTLE_FROM_OUTSIDE,
                 STRING (RCPT_THROTTLE_FROM_OUTSIDE), "25");

  cf_add_id_int (CF_CONN_THROTTLE_FROM_DOMAIN,
                 STRING (CONN_THROTTLE_FROM_DOMAIN), "200");
  cf_add_id_int (CF_CONN_THROTTLE_FROM_LOCAL, STRING (CONN_THROTTLE_FROM_LOCAL), "300");
  cf_add_id_int (CF_CONN_THROTTLE_FROM_FRIEND,
                 STRING (CONN_THROTTLE_FROM_FRIEND), "100");
  cf_add_id_int (CF_CONN_THROTTLE_FROM_OUTSIDE,
                 STRING (CONN_THROTTLE_FROM_OUTSIDE), "10");

  /* 11n */
  cf_add_id_enum (CF_RESOLVE_FAIL, STRING (RESOLVE_FAIL), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_RESOLVE_FORGED, STRING (RESOLVE_FORGED), ENUM_REJECT, "OK");

  cf_add_id_int (CF_RESOLVE_ACCEPT_06H, STRING (RESOLVE_ACCEPT_06H), "3");
  cf_add_id_int (CF_RESOLVE_ACCEPT_12H, STRING (RESOLVE_ACCEPT_12H), "4");
  cf_add_id_int (CF_RESOLVE_ACCEPT_18H, STRING (RESOLVE_ACCEPT_18H), "5");
  cf_add_id_int (CF_RESOLVE_ACCEPT_24H, STRING (RESOLVE_ACCEPT_24H), "6");

  /* 12n */
  cf_add_id_enum (CF_NO_TO_HEADERS, STRING (NO_TO_HEADERS), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_NO_FROM_HEADERS, STRING (NO_FROM_HEADERS), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_NO_SUBJECT_HEADER, STRING (NO_SUBJECT_HEADER), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_NO_HEADERS, STRING (NO_HEADERS), ENUM_REJECT, "OK");

  cf_add_id_enum (CF_CHECK_LOCAL_USERS, STRING (CHECK_LOCAL_USERS), ENUM_NO_YES, "NO");
  cf_add_id_enum (CF_CHECK_USER_ACCESS, STRING (CHECK_USER_ACCESS), ENUM_NO_YES, "NO");
  cf_add_id_enum (CF_DEFAULT_USER_ACCESS, STRING (DEFAULT_USER_ACCESS),
                  ENUM_REJECT, "OK");

  /* 13n */
  cf_add_id_enum (CF_CHECK_DOMAIN_ADDRESS, STRING (CHECK_DOMAIN_ADDRESS),
                  ENUM_NO_YES, "NO");

  /* 12n */
  cf_add_id_enum (CF_OUTLOOK, STRING (OUTLOOK), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_OUTLOOK_LOCAL, STRING (OUTLOOK_LOCAL), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_OUTLOOK_DOMAIN, STRING (OUTLOOK_DOMAIN), ENUM_REJECT, "OK");
  cf_add_id_enum (CF_OUTLOOK_FRIEND, STRING (OUTLOOK_FRIEND), ENUM_REJECT, "OK");

  cf_sort ();
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
void
cf_defaults ()
{

  /* n */
  cf_set_val (CF_DOMAIN, "");
  cf_set_val (CF_J_HOSTNAME, "SYSTEM");
  cf_set_val (CF_PRESENCE, "SHOW");
  cf_set_val (CF_USER, RUN_AS_USER);
  cf_set_val (CF_GROUP, RUN_AS_GROUP);

  /* 1n */
  cf_set_val (CF_SOCKET, "");
  cf_set_val (CF_INET_SOCK, "2000");
  cf_set_val (CF_UNIX_SOCK, "");

  /* 2n */
  cf_set_val (CF_WARN_SENDER, "YES");
  cf_set_val (CF_WARN_RCPT, "YES");
  cf_set_val (CF_J_SENDER, "SENDER");
  cf_set_val (CF_J_SUBJECT, "SUBJECT");

  /* 3n */
  cf_set_val (CF_LOG_FACILITY, "local5");
  cf_set_val (CF_LOG_LEVEL, "10");
  cf_set_val (CF_STATS_INTERVAL, "300");
  cf_set_val (CF_LOG_ATTACHMENTS, "NO");

  /* 4n */
  cf_set_val (CF_ERROR_MSG_FILE, J_ERROR_MSG_FILE);
  cf_set_val (CF_CW_FILE, J_CW_FILE);
  cf_set_val (CF_NETS_FILE, J_NETS_FILE);
  cf_set_val (CF_USERS_FILE, J_USERS_FILE);
  cf_set_val (CF_HOST_ACCESS_FILE, J_HOST_ACCESS_FILE);
  cf_set_val (CF_USER_ACCESS_FILE, J_USER_ACCESS_FILE);
  cf_set_val (CF_REGEX_FILE, J_REGEX_FILE);
  cf_set_val (CF_CONF_MODE, "TEXT");

  /* 5n */
  cf_set_val (CF_WORKDIR, J_WORKDIR);
  cf_set_val (CF_SPOOLDIR, J_SPOOLDIR);
  cf_set_val (CF_STATE_FILE, J_STATE_FILE);
  cf_set_val (CF_STATS_FILE, J_STATS_FILE);
  cf_set_val (CF_PID_FILE, J_PID_FILE);
  cf_set_val (CF_XFILES_LOG_FILE, J_XFILES_LOG);
  cf_set_val (CF_REGEX_LOG_FILE, J_XFILES_LOG);

  /* 6n */
  cf_set_val (CF_DELETE_EXEC_FILES, "NO");
  cf_set_val (CF_XFILES, "OK");
  cf_set_val (CF_FILE_EXT, "");
  cf_set_val (CF_FILE_REGEX, "");
  cf_set_val (CF_XFILE_SAVE_MSG, "NO");

  /* 7N */
  cf_set_val (CF_AV_ACTION, "OK");
  cf_set_val (CF_AV_PORT, "2001");
  cf_set_val (CF_AV_NB_SERVERS, "5");
  cf_set_val (CF_AV_TYPE, "USER");
  cf_set_val (CF_AV_PATH, "/usr/local/bin/userfilter");
  cf_set_val (CF_AV_ARGS, "");
  cf_set_val (CF_AV_SCOPE, "ALL");
  cf_set_val (CF_AV_SAVE_MSG, "NO");

  /* 8n */
  cf_set_val (CF_CHECK_LOCAL_USERS, "NO");

  cf_set_val (CF_CHECK_NB_RCPT, "NO");

  cf_set_val (CF_MAX_RCPT_FROM_DOMAIN, "200");
  cf_set_val (CF_MAX_RCPT_FROM_LOCAL, "1000");
  cf_set_val (CF_MAX_RCPT_FROM_FRIEND, "200");
  cf_set_val (CF_MAX_RCPT_FROM_OUTSIDE, "25");

  cf_set_val (CF_CHECK_THROTTLE, "NO");

  cf_set_val (CF_RCPT_THROTTLE_FROM_DOMAIN, "200");
  cf_set_val (CF_RCPT_THROTTLE_FROM_LOCAL, "300");
  cf_set_val (CF_RCPT_THROTTLE_FROM_FRIEND, "100");
  cf_set_val (CF_RCPT_THROTTLE_FROM_OUTSIDE, "10");

  cf_set_val (CF_CONN_THROTTLE_FROM_DOMAIN, "200");
  cf_set_val (CF_CONN_THROTTLE_FROM_LOCAL, "300");
  cf_set_val (CF_CONN_THROTTLE_FROM_FRIEND, "100");
  cf_set_val (CF_CONN_THROTTLE_FROM_OUTSIDE, "100");

  /* 9n */
  cf_set_val (CF_NO_TO_HEADERS, "OK");
  cf_set_val (CF_NO_FROM_HEADERS, "OK");
  cf_set_val (CF_NO_SUBJECT_HEADER, "OK");
  cf_set_val (CF_NO_HEADERS, "OK");

  cf_set_val (CF_CHECK_DOMAIN_ADDRESS, "NO");

  cf_set_val (CF_RESOLVE_FAIL, "OK");
  cf_set_val (CF_RESOLVE_FORGED, "OK");

  cf_set_val (CF_RESOLVE_ACCEPT_06H, "3");
  cf_set_val (CF_RESOLVE_ACCEPT_12H, "4");
  cf_set_val (CF_RESOLVE_ACCEPT_18H, "5");
  cf_set_val (CF_RESOLVE_ACCEPT_24H, "6");


  cf_set_val (CF_CHECK_USER_ACCESS, "NO");
  cf_set_val (CF_DEFAULT_USER_ACCESS, "NO");

  /* 10n */

  cf_set_val (CF_CHECK_DOMAIN_ADDRESS, "NO");

  /* 11n */
  cf_set_val (CF_CHECK_BODY_CONTENT, "NO");
  cf_set_val (CF_CHECK_HEADERS_CONTENT, "NO");
  cf_set_val (CF_CHECK_SUBJECT_CONTENT, "NO");
  cf_set_val (CF_CONTENT_CHECK_SCORE, "1");

  cf_set_val (CF_ENCODING_BINARY, "OK");
  cf_set_val (CF_ENCODING_BASE64, "OK");
  cf_set_val (CF_ENCODING_QUOTED_PRINTABLE, "OK");

  /* 12n */
  cf_set_val (CF_OUTLOOK, "OK");
  cf_set_val (CF_OUTLOOK_LOCAL, "OK");
  cf_set_val (CF_OUTLOOK_DOMAIN, "OK");
  cf_set_val (CF_OUTLOOK_FRIEND, "OK");

}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

void
cf_dump (full)
     int             full;
{
  int             i, j;
  char           *p;

  for (i = 0; i < DIM_CF; i++) {
    if (cf[i].id != 0) {
      switch (cf[i].kind) {
        case J_INT:
          if (full) {
            printf (" %4d %-26s %-4s %7ld %6d %s\n",
                    cf[i].id,
                    cf[i].name,
                    "INT", cf[i].iv, cf[i].slen, cf[i].sv ? cf[i].sv : "(null)");
          } else {
            printf (" %4d %-26s %7ld\n", cf[i].id, cf[i].name, cf[i].iv);
          }
          break;
        case J_ENUM:
          p = cf[i].sv;
          if (p == NULL || strlen (p) == 0) {
            for (j = 0; cf[i].opt[j] != NULL; j++) {
              if (j == cf[i].iv) {
                p = cf[i].opt[j];
                break;
              }
            }
          }
          if (full) {
            printf (" %4d %-26s %-4s %7ld %6d %s\n",
                    cf[i].id,
                    cf[i].name, "ENUM", cf[i].iv, cf[i].slen, p ? p : "(null)");
          } else {
            printf (" %4d %-26s %7ld %s\n",
                    cf[i].id, cf[i].name, cf[i].iv, p ? p : "(null)");
          }
          break;
        case J_STR:
          if (cf[i].sv != NULL) {
            int             n = 0;

            p = cf[i].sv;
            while (*p) {
              int             nc = strcspn (p, "\n");

              if (nc > 0) {
                char            out[256];

                strncpy (out, p, nc);
                out[nc] = '\0';
                if (full) {
                  printf (" %4d %-26s %-4s %7ld %6d %s\n",
                          cf[i].id, cf[i].name, "STR", cf[i].iv, cf[i].slen, out);
                } else {
                  printf (" %4d %-26s         %s\n", cf[i].id, cf[i].name, out);
                }
                p += nc;
                n++;
              }
              p += strspn (p, "\n");
            }
            if (n == 0) {
              if (full) {
                printf (" %4d %-26s %-4s %7ld %6d %s\n",
                        cf[i].id, cf[i].name, "STR", cf[i].iv, cf[i].slen, "");
              } else {
                printf (" %4d %-26s         %s\n", cf[i].id, cf[i].name, "(null)");
              }
            }
          }
          break;
        default:
          printf (" %4d %-26s %-4s %7ld %6d %s\n",
                  cf[i].id,
                  cf[i].name,
                  "????", cf[i].iv, cf[i].slen, cf[i].sv ? cf[i].sv : "(null)");
          break;
      }
    }
  }
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
void
cf_clear_values ()
{
  int             i;

  for (i = 0; i < DIM_CF; i++) {
    switch (cf[i].kind) {
      case J_STR:
        if (cf[i].sv)
          *cf[i].sv = '\0';
        cf[i].iv = 0;
        break;
      case J_INT:
        cf[i].iv = 0;
        break;
      case J_ENUM:
        cf[i].iv = 0;
        if (cf[i].sv)
          *cf[i].sv = '\0';
        break;
      default:
        cf[i].iv = 0;
        break;
    }
  }
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_add_id_enum (id, name, opt, val)
     int             id;
     char           *name;
     char          **opt;
     char           *val;
{
  int             i = 0;
  int             len = 128;

  while (i < DIM_CF && cf[i].id != CF_NONE) {
    if (cf[i].id == id)
      return -1;
    i++;
  }
  if (i >= DIM_CF)
    return -2;

  if ((cf[i].sv = malloc (len + 1)) == NULL) {
    syslog (LOG_WARNING, "j-cf - ERROR malloc %s", strerror (errno));
    return -3;
  }

  snprintf (cf[i].name, sizeof (cf[i].name), "%s", name);
  memset (cf[i].sv, 0, len + 1);
  cf[i].iv = 0;
  cf[i].id = id;
  cf[i].kind = J_ENUM;
  cf[i].slen = len;
  cf[i].opt = opt;

  cf_set_val (id, val);

  return id;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_add_id_str (id, name, len, val)
     int             id;
     char           *name;
     int             len;
     char           *val;
{
  int             i = 0;

  while (i < DIM_CF && cf[i].id != CF_NONE) {
    if (cf[i].id == id)
      return -1;
    i++;
  }
  if (i >= DIM_CF)
    return -2;

  if ((cf[i].sv = malloc (len + 1)) == NULL) {
    syslog (LOG_WARNING, "j-cf - ERROR malloc %s", strerror (errno));
    return -3;
  }

  snprintf (cf[i].name, sizeof (cf[i].name), "%s", name);
  memset (cf[i].sv, 0, len + 1);
  cf[i].iv = 0;
  cf[i].id = id;
  cf[i].kind = J_STR;
  cf[i].slen = len;

  cf_set_val (id, val);

  return id;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_add_id_int (id, name, val)
     int             id;
     char           *name;
     char           *val;
{
  int             i = 0;

  while (i < DIM_CF && cf[i].id != CF_NONE) {
    if (cf[i].id == id)
      return -1;
    i++;
  }
  if (i >= DIM_CF)
    return -2;

  snprintf (cf[i].name, sizeof (cf[i].name), "%s", name);
  cf[i].sv = NULL;
  cf[i].iv = 0;
  cf[i].id = id;
  cf[i].kind = J_INT;
  cf[i].slen = 0;

  cf_set_val (id, val);

  return id;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_get_id (tag)
     char           *tag;
{
  int             i = 0;

  for (i = 0; i < DIM_CF; i++) {
    if (strcmp (tag, cf[i].name) == 0)
      return cf[i].id;
  }
  return CF_NONE;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_get_kind (id)
     int             id;
{
  int             i = 0;

  for (i = 0; i < DIM_CF; i++) {
    if (id == cf[i].id)
      return cf[i].kind;
  }
  return -1;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_set_val (id, value)
     int             id;
     char           *value;
{
  int             i = 0;
  long            val;

  if (value == NULL || strlen (value) == 0)
    return CF_NONE;

  for (i = 0; i < DIM_CF; i++) {
    if (id == cf[i].id)
      break;
  }
  if (i == DIM_CF)
    return CF_NONE;

  switch (cf[i].kind) {
    case J_STR:
      snprintf (cf[i].sv, cf[i].slen, "%s", value);
      break;
    case J_INT:
      val = strtol (value, (char **) NULL, 10);
      if (errno == ERANGE)
        return CF_NONE;
      cf[i].iv = val;
      break;
    case J_ENUM:
      val = get_enum_index (value, cf[i].opt);
      if (val < 0)
        return CF_NONE;
      cf[i].iv = val;
      if (strcasecmp (cf[i].opt[val], "OTHER") == 0)
        snprintf (cf[i].sv, cf[i].slen, "%s", value);
      else
        snprintf (cf[i].sv, cf[i].slen, "%s", cf[i].opt[val]);
      break;
    default:
      return CF_NONE;
  }
  return id;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_set_str_val (id, value)
     int             id;
     char           *value;
{
  int             i = 0;

  if (value == NULL || strlen (value) == 0)
    return CF_NONE;

  for (i = 0; i < DIM_CF; i++) {
    if (id == cf[i].id)
      break;
  }
  if (i == DIM_CF)
    return CF_NONE;

  if (cf[i].kind == J_STR) {
    snprintf (cf[i].sv, cf[i].slen, "%s", value);
  }
  return id;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_append_str_val (id, value)
     int             id;
     char           *value;
{
  int             i = 0;

  if (value == NULL || strlen (value) == 0)
    return CF_NONE;

  for (i = 0; i < DIM_CF; i++) {
    if (id == cf[i].id)
      break;
  }
  if (i == DIM_CF)
    return CF_NONE;

  if (cf[i].kind == J_STR) {
    if (strlen (cf[i].sv) > 0) {
#if 0
      snprintf (cf[i].sv, cf[i].slen, "%s\n%s", cf[i].sv, value);
#else
      if ((strlen (cf[i].sv) + strlen (value) + strlen("\n")) < cf[i].slen) {
        strcat (cf[i].sv, "\n");
        strcat (cf[i].sv, value);
      } else
        syslog (LOG_WARNING, "%s : Error appending string", J_FUNCTION);
#endif
    } else
      snprintf (cf[i].sv, cf[i].slen, "%s", value);
  }
  return id;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_get_int_val (id)
     int             id;
{
  int             i = 0;

  if (id <= 0)
    return -1;

  for (i = 0; i < DIM_CF; i++) {
    if (id == cf[i].id)
      return cf[i].iv;
  }
  return -1;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_get_int (id)
     int             id;
{
  int             i = 0;

  if (id <= 0)
    return -1;

  for (i = 0; i < DIM_CF; i++) {
    if (id == cf[i].id)
      return cf[i].iv;
  }
  return -1;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
char           *
cf_get_str (id)
     int             id;
{
  int             i = 0, j;

  if (id <= 0)
    return NULL;

  for (i = 0; i < DIM_CF; i++) {
    if (id == cf[i].id) {
      switch (cf[i].kind) {
        case J_STR:
          return cf[i].sv;
          break;
        case J_INT:
          return NULL;
          break;
        case J_ENUM:
          if (cf[i].opt != NULL) {
            for (j = 0; cf[i].opt[j] != NULL; j++) {
              if (j == cf[i].iv) {
                if (strcasecmp (cf[i].opt[j], "OTHER") != 0)
                  return cf[i].opt[j];
                else
                  return cf[i].sv;
              }
            }
          }
          return NULL;
          break;
        default:
          return NULL;
          break;
      }
    }
  }
  return NULL;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
cf_get_enum_val (id, value, len)
     int             id;
     char           *value;
     int             len;
{
  int             i = 0;

  if (id <= 0)
    return -1;

  for (i = 0; i < DIM_CF; i++) {
    if (id == cf[i].id) {
      if (value) {
        if (cf[i].opt[cf[i].iv] == NULL) {
          strcpy (value, "");
          return -1;
        }
        if (strcasecmp ("OTHER", cf[i].opt[cf[i].iv]) == 0)
          snprintf (value, len, "%s", cf[i].sv);
        else
          snprintf (value, len, "%s", cf[i].opt[cf[i].iv]);
      }
      return cf[i].iv;
    }
  }
  return -1;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static int
get_enum_index (val, opt)
     char           *val;
     char          **opt;
{
  int             i;

  if (opt == NULL)
    return -1;
  for (i = 0; opt[i] != NULL; i++) {
    if (strcasecmp (opt[i], val) == 0 || strcasecmp (opt[i], "OTHER") == 0)
      return i;
  }
  return -1;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

#define TOKEN_TAG     "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_0123456789@"

int
cf_read_file (fname)
     char           *fname;
{
  FILE           *fin;
  char            line[512];
  int             lineno = 0;
  char            logbuf[1024];

  free_fregex ();
  free_fext ();

  if (fname == NULL || strlen (fname) == 0)
    return -1;

  if ((fin = fopen (fname, "r")) == NULL) {
    snprintf (logbuf, sizeof (logbuf), "configure : Error opening file : %s", fname);
    j_print_msg_sys (j_output, LOG_ERR, logbuf);
    j_print_msg (j_output, LOG_ERR, "Exiting ...");
    exit (1);
  }

  while (fgets (line, sizeof (line), fin) == line) {
    char           *p, token[256], value[256];
    int             da;
    int             id;

    lineno++;

    if ((p = strpbrk (line, "\n\r")) != NULL)
      *p = '\0';

    if (strlen (line) == 0)
      continue;

    p = line;
    p += strspn (p, " \t");
    if (*p == '#' || strlen (p) == 0)
      continue;

    da = strspn (p, TOKEN_TAG);

    strncpy (token, p, da);
    token[da] = '\0';

    p += da;
    p += strspn (p, " \t");
    snprintf (value, sizeof (value), "%s", p);
    if (strlen (value) > 0) {
      p = value + strlen (value) - 1;
      while (p > value && (*p == ' ' || *p == '\t'))
        *p-- = '\0';
    } else
      continue;

    if ((id = cf_get_id (token)) > 0) {
      int             res;

#if DEBUG == 1
      printf ("TOKEN : %-32s (%3d) --%s--\n", token, id, value);
#endif
      switch (id) {
        case CF_FILE_EXT:
          res = cf_append_str_val (id, value);
          break;
        case CF_FILE_REGEX:
          res = cf_append_str_val (id, value);
          if (add_fregex (value) == FALSE) {
            snprintf (logbuf, sizeof (logbuf),
                      "n read_j_conf_file : %d - add_fregex(%s) error", lineno, value);
            j_print_msg (j_output, LOG_WARNING, logbuf);
          }
          break;
        default:
          res = cf_set_val (id, value);
          break;
      }
      if (res <= 0) {
        snprintf (logbuf, sizeof (logbuf),
                  " read_cf_file : Error setting parameter value : %d\n", lineno);
        j_print_msg (j_output, LOG_WARNING, logbuf);
      }
    } else
#if DEBUG == 1
      printf ("# %4d : UNKNOWN TOKEN : %s \n", lineno, token);
#else
      ;
#endif

  }
  fclose (fin);

  return lineno;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
configure (fname, only_cf)
     char           *fname;
     int             only_cf;
{
  static int      init_ok = 0;
  int             n;
  char            out[1024];
  char           *p;

#if DEBUG == 1
  printf ("CF_FILE = %s\n", fname);
#endif

  if (init_ok == 0) {
    if (cf_init () != 0) {
      fprintf (stderr, "cf_init : Can't initialise : %s\n", strerror (errno));
      exit (1);
    }
    init_ok = 1;
  }
#if 0
  cf_defaults ();
#endif

  cf_read_file (fname);

  /* Log level */
  log_level = cf_get_int_val (CF_LOG_LEVEL);

  /* Log facility */
  if ((p = cf_get_str (CF_LOG_FACILITY)) != NULL) {
    n = facility_value (p);
    if (n != -1 && n != log_facility) {
      set_log_facility (p);
      closelog ();
      openlog ("j-chkmail", LOG_PID | LOG_NOWAIT | LOG_NDELAY, log_facility);
      if (log_level > 10) {
        sprintf (out, "NEW FACILITY : %d - %s", log_facility,
                 facility_name (log_facility));
        if (j_output == J_SYSLOG || j_output == J_OUT_ALL)
          j_print_msg (j_output, LOG_INFO, out);
      }
    }
  }


  if (cf_get_int (CF_DELETE_EXEC_FILES) == OPT_YES && cf_get_int (CF_XFILES) == OPT_OK)
    cf_set_val (CF_XFILES, "WARN");

  p = cf_get_str (CF_FILE_EXT);

  while (*p) {
    char            ext[64];
    int             nc;

    nc = strcspn (p, " \t\n");
    if (nc > 0) {
      strncpy (ext, p, nc);
      ext[nc] = '\0';
      if (add_fext (ext) == FALSE) {
        snprintf (out, sizeof (out), "add_fext(%s) error", ext);
        j_print_msg (j_output, LOG_WARNING, out);
      }
      p += nc;
    }
    p += strspn (p, " \t\n");
  }

  init_file_extension_regex ();

  statistics_interval = cf_get_int (CF_STATS_INTERVAL);

  if (only_cf)
    return 0;

  /* Load j-host-access file */
  {
    char           *fname;

    fname = cf_get_str (CF_HOST_ACCESS_FILE);

    if (log_level >= 20)
      syslog (LOG_DEBUG, "j-host-access...");
    load_host_access_table (fname);
    if (log_level >= 20)
      syslog (LOG_DEBUG, " ...OK");
  }

  /* Load j-user-access file */
  {
    char           *fname;

    fname = cf_get_str (CF_USER_ACCESS_FILE);

    if (log_level >= 20)
      syslog (LOG_DEBUG, "j-user-access...");
    load_user_access_table (fname);
    if (log_level >= 20)
      syslog (LOG_DEBUG, " ...OK");
  }

  /* Load j-local_users file */
  {
    char           *fname;

    fname = cf_get_str (CF_USERS_FILE);

    if (log_level >= 20)
      syslog (LOG_DEBUG, "j-local-users...");
    load_local_users_table (fname);
    if (log_level >= 20)
      syslog (LOG_DEBUG, " ...OK");
  }

  /* Load local-host-names file */
  {
    char           *fname;

    fname = cf_get_str (CF_CW_FILE);

    if (log_level >= 20)
      syslog (LOG_DEBUG, "local-host-names...");
    load_classw_table (fname);
    if (log_level >= 20)
      syslog (LOG_DEBUG, " ...OK");
  }

  /* Load j-regex file */
  {
    char           *fname;

    fname = cf_get_str (CF_REGEX_FILE);

    if (log_level >= 20)
      syslog (LOG_DEBUG, "j-regex...");
    load_regex_table (fname);
    if (log_level >= 20)
      syslog (LOG_DEBUG, " ...OK");
  }

  /* Load j-nets file */
  {
    char           *fname;

    fname = cf_get_str (CF_NETS_FILE);

    if (log_level >= 20)
      syslog (LOG_DEBUG, "j-nets...");
    load_networks_table (fname);
    if (log_level >= 20)
      syslog (LOG_DEBUG, " ...OK");
  }

  {
    int             port;
    char           *u_sock;
    char           *g_sock;

    port = cf_get_int (CF_INET_SOCK);
    snprintf (sm_sock, sizeof (sm_sock), "inet:%d@localhost", port);

    u_sock = cf_get_str (CF_UNIX_SOCK);
    if (u_sock != NULL && strlen (u_sock) > 0)
      snprintf (sm_sock, sizeof (sm_sock), "local:%s", u_sock);

    g_sock = cf_get_str (CF_SOCKET);
    if (g_sock != NULL && strlen (g_sock) > 0)
      snprintf (sm_sock, sizeof (sm_sock), "%s", g_sock);

    if (opt->arg_i)
      snprintf (sm_sock, sizeof (sm_sock), "%s", opt->arg_i);

    if (opt->arg_u)
      snprintf (sm_sock, sizeof (sm_sock), "%s", opt->arg_u);

    if (opt->arg_p)
      snprintf (sm_sock, sizeof (sm_sock), "%s", opt->arg_p);

#if 1
    if (log_level > 10)
      printf ("SM_SOCK = %s\n", sm_sock);
#endif
  }

  return 0;
}
