/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <j-sys.h>

#include <libmilter/mfapi.h>

#include <j-avserver.h>
#include <j-chkmail.h>


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
#define DO_CHECK_AV               1

#define RELAUNCH_DIED_CHILDREN    1

#define PROCESS_SIG_ALARM         0

static int      launch_av_pf_server ();

static RETSIGTYPE j_pfserv_sig_handler (int);

#define SZ_QUEUE   32

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

char           *av_args[256];

int             av_port = 2001;
int             av_type = AV_NONE;

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */


#define MAX_CHILDREN    50

static pid_t    pids[MAX_CHILDREN];

static int      nchildren = 10;

static void     j_pf_kill_all_pids ();

static int      glob_sd = 0;


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

int
j_avserver ()
{
  char           *p;
  int             i;
  char           *dir;

  dir = cf_get_str (CF_SPOOLDIR);
  if (strlen (dir) > 0) {
    if (*dir != '/')
      syslog (LOG_WARNING,
              "%s : warning : SPOOLDIR doesn't begins with a / : %s", J_FUNCTION, dir);
    if (chdir (dir) != 0)
      syslog (LOG_ERR,
              "%s : error changing to dir %s : %s", J_FUNCTION, dir, strerror (errno));
  }

  if ((av_port = cf_get_int (CF_AV_PORT)) <= 0) {
    syslog (LOG_ERR, "%s : Error : invalid av_port number (%d)", J_FUNCTION, av_port);
    exit (1);
  }

  memset (av_args, 0, sizeof (av_args));
  p = cf_get_str (CF_AV_PATH);
  if (p == NULL || strlen (p) == 0) {
    syslog (LOG_ERR, "%s : ERROR : av_path undefined or NULL pointer", J_FUNCTION);
    return 1;
  } else
    av_args[0] = p;

  p = cf_get_str (CF_AV_ARGS);
  if (p != NULL && strlen (p) > 0)
    split_args (p, av_args + 1);

  i = cf_get_int (CF_AV_TYPE);
  if (i < 0) {
    syslog (LOG_ERR, "%s : ERROR : av_type not valid", J_FUNCTION);
    return 1;
  } else
    av_type = i;

  if (log_level >= 20) {
    int             i;

    for (i = 0; av_args[i] != NULL; i++)
      syslog (LOG_DEBUG, "%s : AV_ARGS[%3d] = %s", J_FUNCTION, i, av_args[i]);
  }

  signal (SIGTERM, SIG_DFL);
  signal (SIGHUP, SIG_DFL);
  signal (SIGUSR1, SIG_DFL);
  signal (SIGUSR2, SIG_DFL);
  signal (SIGALRM, SIG_DFL);
  signal (SIGCHLD, SIG_DFL);
  alarm (2 * DT_SIGALRM);

  i = cf_get_int (CF_AV_NB_SERVERS);
  if (i > MAX_CHILDREN)
    i = MAX_CHILDREN;
  if (i < 2)
    i = 2;
  nchildren = i;

#if DO_CHECK_AV == 1
  launch_av_pf_server ();
#else
  while (1) {
    sleep (DT_SIGALRM);
  }
#endif
  return 0;
}



/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static unsigned int t_stats = 0;

unsigned int    msg_ok = 0;

static          RETSIGTYPE
j_pfserv_sig_handler (signo)
     int             signo;
{
  time_t          now = time (NULL);

#if PROCESS_SIG_ALARM
  int             msg;
#endif
  pid_t           pid;
  int             i;

  signal (signo, j_pfserv_sig_handler);

  if (log_level >= 20)
    syslog (LOG_DEBUG, "*** Received SIGNAL %d : %s", signo, ctime (&now));

  switch (signo) {
    case SIGUSR1:
      break;
    case SIGCHLD:
      while ((pid = waitpid (-1, NULL, WNOHANG)) > 0) {
        for (i = 0; i < nchildren; i++) {
          if (pid == pids[i]) {
            pids[i] = 0;
            break;
          }
        }
        continue;
      }
      break;
    case SIGINT:
      j_pf_kill_all_pids ();
      if (errno != ECHILD) {
        syslog (LOG_ERR, "j_pf_sig_int %s", strerror (errno));
        exit (1);
      }
      while (wait (NULL) > 0);
      exit (0);
      break;

    case SIGALRM:
#if PROCESS_SIG_ALARM
      if (!recv_msg_channel (pipe_avserv, &msg, CHAN_CHILD)) {
        msg_ok = 0;
        if (log_level >= 20)
          syslog (LOG_DEBUG, "%s: AVSERV - MSG RECV : %d", J_FUNCTION, msg);
        switch (msg) {
          case MSG_OK:
            break;
          case MSG_TERM:
            j_pf_kill_all_pids ();
            exit (0);
            break;
          case MSG_CONF:
            j_pf_kill_all_pids ();
            exit (0);
            break;
          default:
            ;
        }
      } else {
        msg_ok++;
        if (log_level >= 15)
          syslog (LOG_DEBUG, "%s : AVSERV - MSG RECV : NULL : %d", J_FUNCTION, msg_ok);
      }

      if (msg_ok > 3 || getppid () == 1) {
        syslog (LOG_WARNING, "%s : AVSERV - FATHER DIED ???", J_FUNCTION);
        j_pf_kill_all_pids ();
        exit (0);
      }

      if (statistics_interval > 0) {
        t_stats += DT_SIGALRM;
        if (t_stats >= statistics_interval) {
          t_stats = 0;
        }
      }

      alarm (DT_SIGALRM);
#endif
      break;
    default:
      syslog (LOG_WARNING, "%s : Undefined behavior for signal %d !",
              J_FUNCTION, signo);
      break;
  }

}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static void
j_pf_kill_all_pids ()
{
  int             i;
  pid_t           pid;

  for (i = 0; i < nchildren; i++)
    kill (pids[i], SIGTERM);
  while ((pid = wait (NULL)) > 0) {
    for (i = 0; i < nchildren; i++) {
      if (pid == pids[i]) {
        pids[i] = 0;
        break;
      }
    }
  }
}



/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static int
launch_av_pf_server ()
{
  struct sockaddr_in recv_sock;
  int             sopt;
  int             i;
  int             msg_ok = 0;
  time_t          t_now, t_last;

  if (log_level >= 20)
    syslog (LOG_DEBUG, "AV_SERVER RUNNING\n");

  /* Let's create reception socket... */
  if ((glob_sd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
    syslog (LOG_ERR, "%s : Error creating socket : %s", J_FUNCTION, strerror (errno));
    exit (1);
  }

  /* ... and give him a name */
  memset (&recv_sock, 0, sizeof (recv_sock));
  recv_sock.sin_family = AF_INET;
  if (1) {
    if (j_inet_pton ("127.0.0.1", &recv_sock.sin_addr.s_addr) < 0)
      recv_sock.sin_addr.s_addr = INADDR_ANY;
  } else
    recv_sock.sin_addr.s_addr = INADDR_ANY;

  recv_sock.sin_port = ntohs (av_port);


  sopt = 1;
  /* Tell the system to allow local addresses to be reused. */
  if (setsockopt (glob_sd, SOL_SOCKET, SO_REUSEADDR, (void *) &sopt, sizeof (sopt)) < 0) {
    syslog (LOG_ERR, "%s : Error setsockopt(SO_REUSEADDR : %s", J_FUNCTION,
            strerror (errno));
    (void) close (glob_sd);
    exit (1);
  }

  if (bind (glob_sd, (struct sockaddr *) &recv_sock, sizeof (recv_sock)) < 0) {
    syslog (LOG_ERR, "%s : Error binding socket : %s", J_FUNCTION, strerror (errno));
    exit (1);
  }

  if (listen (glob_sd, SZ_QUEUE) < 0) {
    syslog (LOG_ERR, "%s : Error creating socket listening queue : %s",
            J_FUNCTION, strerror (errno));
    exit (1);
  }

  memset (pids, 0, sizeof (pids));

  j_pf_lock_init ("/tmp/j-chkmail.lock");

  for (i = 0; i < nchildren; i++)
    pids[i] = j_pf_child_make (i, glob_sd);

  signal (SIGINT, j_pfserv_sig_handler);
  signal (SIGUSR1, j_pfserv_sig_handler);
  signal (SIGUSR2, j_pfserv_sig_handler);
  signal (SIGALRM, j_pfserv_sig_handler);
  signal (SIGCHLD, j_pfserv_sig_handler);

  t_now = time (NULL);
  for (;;) {
    t_last = t_now;
    sleep (DT_SIGALRM);
    t_now = time (NULL);

#if PROCESS_SIG_ALARM == 0
    if ((t_last + DT_SIGALRM) >= t_now) {
      int             nb = 0;
      int             msg;

      nb = 0;

      while (!recv_msg_channel (pipe_avserv, &msg, CHAN_CHILD)) {
        nb++;
        msg_ok = 0;
        if (log_level >= 20)
          syslog (LOG_DEBUG, "%s: AVSERV - MSG RECV : %d", J_FUNCTION, msg);
        switch (msg) {
          case MSG_OK:
            break;
          case MSG_TERM:
            j_pf_kill_all_pids ();
            exit (0);
            break;
          case MSG_CONF:
            j_pf_kill_all_pids ();
            exit (0);
            break;
          default:
            ;
        }
      }
      if (nb == 0) {
        msg_ok++;
        if (log_level >= 15)
          syslog (LOG_DEBUG, "%s : AVSERV - MSG RECV : NULL : %d", J_FUNCTION, msg_ok);
      }

      if (msg_ok > 3 || getppid () == 1) {
        syslog (LOG_WARNING, "%s : AVSERV - FATHER DIED ???", J_FUNCTION);
        j_pf_kill_all_pids ();
        exit (0);
      }

      if (statistics_interval > 0) {
        t_stats += DT_SIGALRM;
        if (t_stats >= statistics_interval) {
          t_stats = 0;
        }
      }
    }
#endif

    /*
     * Check to see if all children are alive
     * If not, relaunch them
     */
#if RELAUNCH_DIED_CHILDREN == 1
    for (i = 0; i < nchildren; i++) {
      if (pids[i] == 0) {
        pids[i] = j_pf_child_make (i, glob_sd);
      }
    }
#endif
  }

  close (glob_sd);
  return 0;
}
