/*-
 * Copyright (c) 2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: sdb_st_mod.c,v 1.12 2012/07/10 20:35:58 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/stat.h>

#include <netinet/in.h>

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <ipa_mod.h>

#include "ipa_sdb.h"

#include "queue.h"

#if IPA_ST_MOD_API_VERSION != 1 && IPA_ST_MOD_API_VERSION != 2
# error "This module supports only ipa_st_mod API versions 1 and 2"
#endif

#if IPA_MEMFUNC_API_VERSION != 1
# error "This module supports only ipa_memfunc API version 1"
#endif

#ifdef WITH_ST_PTHREAD
# define MOD_FLAGS IPA_MOD_FLAG_PTHREAD_SAFE
#else
# define MOD_FLAGS 0
#endif

#define MEMFLAG_OPTIMIZE IPA_MEMFUNC_FLAG_OPTIMIZE

#define MOD_NAME	"ipa_st_sdb"
#define MOD_ST_NAME	"sdb"
#define MOD_CONF_PREFIX	"sdb"

#define MARRAY_NAME(x)	MOD_NAME ":" #x
#define MZONE_NAME(x)	MOD_NAME ":" #x
#define MTYPE_NAME(x)	MOD_NAME ":m_" #x

/* Is defined at the end of this file. */
extern struct ipa_st_mod IPA_ST_MOD_ENTRY(ipa_st_sdb);

static const ipa_suppfunc *suppfunc;	/* Exported support functions. */

static const ipa_memfunc *memfunc;	/* Exported memory functions. */

static ipa_mem_type *m_parser;		/* Memory from parser. */
static ipa_mem_type *m_anon;		/* Anonymous memory allocations. */
static ipa_mem_type *m_tmp;		/* Temporary memory allocations. */

static void	*(*mem_malloc)(size_t, ipa_mem_type *);
static void	*(*mem_realloc)(void *, size_t, ipa_mem_type *);
static char	*(*mem_strdup)(const char *, ipa_mem_type *);
static void	(*mem_free)(void *, ipa_mem_type *);

static ipa_marray *(*marray_init)(const char *, const char *, unsigned int,
		    void **, size_t, unsigned int, unsigned int);
static void	(*marray_deinit)(ipa_marray *);
static int	(*marray_alloc)(ipa_marray *, unsigned int *, int);
static void	(*marray_free)(ipa_marray *, unsigned int);
static void	(*marray_minimize)(ipa_marray *);
static int	(*marray_check_index)(ipa_marray *, unsigned int);

static ipa_mzone *(*mzone_init)(const char *, const char *, unsigned int,
		    size_t, unsigned int, unsigned int);
static void	(*mzone_deinit)(ipa_mzone *);
static void	*(*mzone_alloc)(ipa_mzone *);
static void	(*mzone_free)(ipa_mzone *, void *);
static unsigned int (*mzone_nused)(ipa_mzone *);

static bool	config_section;		/* Set if there was sdb: {} section. */

static signed char quiet;		/* sdb: { quiet } */
static signed char allow_symlinks;	/* sdb: { allow_symlinks } */
static signed char check_version;	/* sdb: { check_version } */

static char	*global_db_dir;		/* global { sdb:db_dir } */
static bool	global_db_dir_checked;	/* Set if global_db_dir was checked. */

/* Default database directory. */
static char	*db_dir_default = IPA_SDB_DB_DIR;

#define DESC_LIST_NALLOC 100
#define RULE_STAT_NALLOC 200

#define RULE_NREC_READ						\
	(sizeof(struct ipa_sdb_rule_record) < 1024 ?		\
	 (1024 / sizeof(struct ipa_sdb_rule_record)) : 1)

#define RULE_NSIZE	30
#define RULE_NALLOC	20

#ifdef WITH_LIMITS
/*
 * Module's data for limit{}.
 */
struct limit {
	char		*db_dir;	/* Database directory for limit. */
};

/* Pointer to limit by number. */
#define LIMIT(r, x) ((r)->limits[(x)])

#define LIMIT_NREC_READ						\
	(sizeof(struct ipa_sdb_limit_record) < 1024 ?		\
	 (1024 / sizeof(struct ipa_sdb_limit_record)) : 1)

#define LIMIT_STAT_NALLOC 50

#define LIMIT_NSIZE	RULE_NSIZE
#define LIMIT_NALLOC	RULE_NALLOC

static ipa_mzone *limit_mzone;		/* Mzone for all limits. */

#endif /* WITH_LIMITS*/

#ifdef WITH_THRESHOLDS
/*
 * Module's data for threshold{}.
 */
struct threshold {
	char		*db_dir;	/* Database directory for threshold. */
};

/* Pointer to threshold by number. */
#define THRESHOLD(r, x) ((r)->thresholds[(x)])

#define THRESHOLD_NSIZE	RULE_NSIZE
#define THRESHOLD_NALLOC RULE_NALLOC

static ipa_mzone *threshold_mzone;	/* Mzone for all thresholds. */

#endif /* WITH_THRESHOLDS */

#define RULE_INITED	0x01	/* Rule was initialized in init_rule(). */

/*
 * Module's data for rule{}.
 */
struct rule {
	unsigned int	no;		/* Rule ordinal number. */
	const char	*name;		/* Name of this rule. */
	char		*db_dir;	/* rule { sdb:db_dir } */
#ifdef WITH_LIMITS
	struct limit	**limits;	/* Array of pointers to limits. */
	char		*limits_dir;	/* Directory for limits. */
	unsigned int	nlimits;	/* Number of limits. */
#endif
#ifdef WITH_THRESHOLDS
	struct threshold **thresholds;	/* Array of pointers to thresholds. */
	char		*thresholds_dir;/* Directory for thresholds. */
	unsigned int	nthresholds;	/* Number of thresholds. */
#endif
	char		*rule_db_dir;	/* Database directory for rule. */
	unsigned int	ref_count;	/* Reference counter. */
	unsigned char	flags;		/* RULE_xxx flags. */
	TAILQ_ENTRY(rule) link;		/* For list of all rules. */
};

static struct rule **rules_ptr;		/* Array of pointers to rules. */
static unsigned int nstatrules;		/* Number of static rules. */
static ipa_mzone *rule_mzone;		/* Mzone for all rules. */
static ipa_marray *rules_ptr_marray;	/* Marray for array of ptrs to rules. */

#define RULE(x)	rules_ptr[(x)]		/* Pointer to rule by number. */

static TAILQ_HEAD(EMPTY, rule) rules_list; /* List of all rules. */

static struct rule *currule;		/* Current rule. */

static bool	in_rule;		/* Set if we are in rule{} section. */

/* stat() or lstat(), depending on allow_symlinks. */
static int	(*stat_func)(const char *, struct stat *);

/* Name of stat_func: "stat" or "lstat". */
static const char *stat_func_name;

#define cnt_from_base(cnt, c_high, c_low) do {		\
	(cnt) = ntohl(c_high);				\
	(cnt) <<= 32;					\
	(cnt) += ntohl(c_low);				\
} while (/* CONSTCOND */ 0)

#define MONTHES_IN_YEAR		12
#define HOURS_IN_DAY		24

#define SECONDS_IN_MINUTE	60
#define MINUTES_IN_HOUR		60
#define SECONDS_IN_HOUR		(MINUTES_IN_HOUR * SECONDS_IN_MINUTE)

static void	(*print_param_end)(void);

static void	logmsg(int, const char *, ...) ATTR_FORMAT(printf, 2, 3);
static void	logmsgx(int, const char *, ...) ATTR_FORMAT(printf, 2, 3);
static void	logconfx(const char *, ...) ATTR_FORMAT(printf, 1, 2);

static int	mem_asprintf(ipa_mem_type *, char **, const char *, ...)
		    ATTR_FORMAT(printf, 3, 4);

/*
 * A wrapper for imported logmsg() function.
 */
static void
logmsg(int priority, const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	suppfunc->logmsg(MOD_NAME, priority, errno, format, ap);
	va_end(ap);
}

/*
 * A wrapper for imported logmsg() function,
 * but error code is always zero here.
 */
static void
logmsgx(int priority, const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	suppfunc->logmsg(MOD_NAME, priority, 0, format, ap);
	va_end(ap);
}

static void
logbt(const char *msg)
{
	logmsgx(IPA_LOG_ERR, " `- %s", msg);
}

/*
 * A wrapper for imported logconferr() function,
 * but error code is always zero here.
 */
static void
logconfx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	suppfunc->logconferr(MOD_NAME, 0, format, ap);
	va_end(ap);
}

/*
 * A wrapper for imported print_param_name() function.
 */
static void
print_param_name(const char *name)
{
	suppfunc->print_param_name(MOD_CONF_PREFIX, name);
}

static int
mem_asprintf(ipa_mem_type *mem_type, char **bufp, const char *format, ...)
{
	va_list ap;
	int rv;

	va_start(ap, format);
	rv = memfunc->mem_vasprintf(mem_type, bufp, format, ap);
	va_end(ap);
	return (rv);
}

/*
 * A wrapper for read(2) syscall.
 */
static ssize_t
readn(int fd, const char *path, void *vptr, size_t n)
{
	char *ptr;
	size_t nleft;
	ssize_t nread;

	ptr = vptr;
	nleft = n;
	for (;;) {
		nread = read(fd, ptr, nleft);
		if (nread < 0) {
			if (errno == EINTR) {
				logmsg(IPA_LOG_WARNING, "readn: read(%s)",
				    path);
				nread = 0;
			} else {
				logmsg(IPA_LOG_ERR, "readn: read(%s)", path);
				return (-1);
			}
		} else if (nread == 0) {
			/* EOF */
			break;
		}
		nleft -= nread;
		if (nleft == 0)
			break;
		ptr += nread;
		logmsgx(IPA_LOG_WARNING, "readn: read: short read from %s",
		    path);
	}
	return (n - nleft);
}

static struct rule *
alloc_rule(unsigned int ruleno)
{
	struct rule *rule;

	rule = mzone_alloc(rule_mzone);
	if (rule == NULL)
		return (NULL);
	if (marray_alloc(rules_ptr_marray, &ruleno, 1) < 0) {
		mzone_free(rule_mzone, rule);
		return (NULL);
	}

	RULE(ruleno) = rule;
	TAILQ_INSERT_TAIL(&rules_list, rule, link);

	rule->no = ruleno;

#ifdef WITH_LIMITS
	rule->limits = NULL;
	rule->limits_dir = NULL;
	rule->nlimits = 0;
#endif

#ifdef WITH_THRESHOLDS
	rule->thresholds = NULL;
	rule->thresholds_dir = NULL;
	rule->nthresholds = 0;
#endif

	rule->db_dir = rule->rule_db_dir = NULL;
	rule->ref_count = 1;
	rule->flags = 0;

	return (rule);
}

static struct rule *
alloc_currule(void)
{
	if (currule == NULL) {
		currule = alloc_rule(nstatrules - 1);
		if (currule == NULL)
			logconfx("alloc_rule failed");
	}
	return (currule);
}

/*
 * Parse "sdb:" section.
 */
/* ARGSUSED */
static int
parse_config_section(void *arg ATTR_UNUSED)
{
	config_section = true;
	return (0);
}

/*
 * Parse "db_dir" parameter.
 */
static int
parse_db_dir(void *arg)
{
	char **dstp, *str;

	str = *(char **)arg;
	if (*str == '\0') {
		logconfx("argument should be a non-empty string");
		return (-1);
	}
	if (*str != '/') {
		logconfx("directory should be given with absolute path");
		return (-1);
	}

	if (in_rule) {
		struct rule *rule;

		rule = alloc_currule();
		if (rule == NULL)
			return (-1);
		dstp = &rule->db_dir;
	} else
		dstp = &global_db_dir;

	mem_free(*dstp, m_parser);
	*dstp = str;

	return (0);
}

/*
 * Parse "quiet" parameter.
 */
static int
parse_quiet(void *arg)
{
	quiet = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse "allow_symlinks" parameter.
 */
static int
parse_allow_symlinks(void *arg)
{
	allow_symlinks = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse "check_version" parameter.
 */
static int
parse_check_version(void *arg)
{
	check_version = (signed char)*(int *)arg;
	return (0);
}

static int
conf_event(unsigned int event, unsigned int no, const void *arg ATTR_UNUSED)
{
	switch (event) {
	case IPA_CONF_EVENT_RULE_BEGIN:
		nstatrules = no + 1;
		in_rule = true;
		currule = NULL;
		break;
	case IPA_CONF_EVENT_RULE_END:
		in_rule = false;
		break;
	}
	return (0);
}

static void
conf_real(void)
{
	if (global_db_dir == NULL)
		global_db_dir = db_dir_default;
	if (quiet < 0)
		quiet = 1;
	if (allow_symlinks < 0)
		allow_symlinks = 0;
	if (check_version < 0)
		check_version = 1;
}

static int
conf_mimic_real(void)
{
	struct rule *rule;

	conf_real();

	config_section = true;

	TAILQ_FOREACH(rule, &rules_list, link)
		if (rule->db_dir == NULL)
			rule->db_dir = global_db_dir;

	return (0);
}

static int
conf_init(void)
{
	suppfunc = IPA_ST_MOD_ENTRY(ipa_st_sdb).suppfunc;

	print_param_end = suppfunc->print_param_end;

	memfunc = IPA_ST_MOD_ENTRY(ipa_st_sdb).memfunc;
	if (memfunc->api_ver != IPA_MEMFUNC_API_VERSION) {
		logconfx("module understands memfunc API version %u, "
		    "exported memfunc API version is %u",
		    IPA_MEMFUNC_API_VERSION, memfunc->api_ver);
		return (-1);
	}

	m_parser = memfunc->m_parser;

	m_anon = memfunc->mem_type_new(MTYPE_NAME(anon),
	    "Anonymous memory", 0);
	m_tmp = memfunc->mem_type_new(MTYPE_NAME(tmp),
	    "Temporal memory", 0);
	if (m_anon == NULL || m_tmp == NULL) {
		logconfx("mem_type_new failed");
		return (-1);
	}

	mem_malloc = memfunc->mem_malloc;
	mem_realloc = memfunc->mem_realloc;
	mem_strdup = memfunc->mem_strdup;
	mem_free = memfunc->mem_free;

	marray_init = memfunc->marray_init;
	marray_deinit = memfunc->marray_deinit;
	marray_alloc = memfunc->marray_alloc;
	marray_free = memfunc->marray_free;
	marray_minimize = memfunc->marray_minimize;
	marray_check_index = memfunc->marray_check_index;

	mzone_init = memfunc->mzone_init;
	mzone_deinit = memfunc->mzone_deinit;
	mzone_alloc = memfunc->mzone_alloc;
	mzone_free = memfunc->mzone_free;
	mzone_nused = memfunc->mzone_nused;

	rules_ptr_marray = marray_init(MARRAY_NAME(rules_ptr),
	    "Pointers to rules", 0, (void *)&rules_ptr, sizeof(struct rule *),
	    RULE_NSIZE, RULE_NALLOC);
	if (rules_ptr_marray == NULL) {
		logconfx("marray_init failed");
		return (-1);
	}

	rule_mzone = mzone_init(MZONE_NAME(rule), "Rules", MEMFLAG_OPTIMIZE,
	    sizeof(struct rule), RULE_NSIZE, RULE_NALLOC);
	if (rule_mzone == NULL) {
		logconfx("mzone_init failed");
		return (-1);
	}
	nstatrules = 0;
	in_rule = false;

	global_db_dir = NULL;
	quiet = allow_symlinks = check_version = -1;
	config_section = false;

	TAILQ_INIT(&rules_list);

	return (0);
}

static int
conf_deinit(void)
{
	return (0);
}

static void
show_db_dir(const char *db_dir)
{
	if (db_dir != NULL) {
		print_param_name("db_dir");
		suppfunc->print_string(db_dir);
		print_param_end();
	}
}

static void
show_param_boolean0(const char *name, int boolean)
{
	if (boolean >= 0) {
		suppfunc->print_param_name((char *)NULL, name);
		suppfunc->print_boolean(boolean);
		print_param_end();
	}
}

static void
conf_show(unsigned int sect_id, unsigned int no)
{
	switch (sect_id) {
	case IPA_CONF_SECT_ROOT:
		if (config_section) {
			suppfunc->print_sect_name(MOD_ST_NAME, "");
			suppfunc->print_sect_begin();
#if IPA_ST_MOD_API_VERSION == 1
			suppfunc->set_indent(1);
#endif
			show_param_boolean0("quiet", quiet);
			show_param_boolean0("allow_symlinks", allow_symlinks);
			show_param_boolean0("check_version", check_version);
#if IPA_ST_MOD_API_VERSION == 1
			suppfunc->set_indent(0);
#endif
			suppfunc->print_sect_end();
		}
		break;
	case IPA_CONF_SECT_GLOBAL:
		show_db_dir(global_db_dir);
		break;
	case IPA_CONF_SECT_RULE:
		if (marray_check_index(rules_ptr_marray, no))
			show_db_dir(RULE(no)->db_dir);
		break;
	}
}

static int
st_pre_init(void)
{
	if (!quiet)
		logmsgx(IPA_LOG_INFO, "module version "PACKAGE_VERSION);

	conf_real();

	if (allow_symlinks > 0) {
		stat_func = stat;
		stat_func_name = "stat";
	} else {
		stat_func = lstat;
		stat_func_name = "lstat";
	}

	/* Try to minimize rules_ptr_marray. */
	if (nstatrules != 0)
		marray_minimize(rules_ptr_marray);

	global_db_dir_checked = false;

#ifdef WITH_LIMITS
	limit_mzone = NULL;
#endif
#ifdef WITH_THRESHOLDS
	threshold_mzone = NULL;
#endif

	return (0);
}

static int
st_init(void)
{
	return (0);
}

static int
check_file(const char *path, const struct stat *statbuf)
{
	if (!S_ISREG(statbuf->st_mode)) {
		logmsgx(IPA_LOG_ERR, "check_file: %s: is expected to be "
		    "a regular file", path);
		return (-1);
	}
	return (0);
}

static int
check_dir(const char *path, const struct stat *statbuf)
{
	if (!S_ISDIR(statbuf->st_mode)) {
		logmsgx(IPA_LOG_ERR, "check_dir: %s: is expected to be "
		    "a directory", path);
		return (-1);
	}
	return (0);
}

static int
check_db_dir(const char *db_dir)
{
	struct stat statbuf;
	FILE *fp;
	char *path;

	if (db_dir == global_db_dir && global_db_dir_checked)
		return (0);

	/* Check main database directory. */
	if (stat_func(db_dir, &statbuf) < 0) {
		logmsg(IPA_LOG_ERR, "check_db_dir: %s(%s)", stat_func_name,
		    db_dir);
		return (-1);
	}
	if (check_dir(db_dir, &statbuf) < 0) {
		logbt("check_db_dir");
		return (-1);
	}

	if (check_version) {
		/* Check database format version file. */
		if (mem_asprintf(m_tmp, &path, "%s/"IPA_SDB_VERSION_FILE,
		    db_dir) < 0) {
			logmsgx(IPA_LOG_ERR, "check_db_dir: "
			    "mem_asprintf failed");
			return (-1);
		}
		if (stat_func(path, &statbuf) < 0) {
			if (errno != ENOENT) {
				logmsg(IPA_LOG_ERR, "check_db_dir: %s(%s)",
				    stat_func_name, path);
				goto failed;
			}
			logmsgx(IPA_LOG_WARNING, "check_db_dir: file %s does "
			    "not exist, cannot check version of format of %s "
			    "database", path, db_dir);
		} else {
			unsigned int version;

			fp = fopen(path, "r");
			if (fp == NULL) {
				logmsg(IPA_LOG_ERR, "check_db_dir: "
				    "fopen(%s, \"r\")", path);
				goto failed;
			}
			if (fscanf(fp, "%u", &version) != 1) {
				logmsg(IPA_LOG_ERR, "check_db_dir: "
				    "fscanf(%s, \"%%u\") failed", path);
				goto failed_close;
			}
			if (version != IPA_SDB_FORMAT_VERSION) {
				logmsgx(IPA_LOG_ERR, "check_db_dir: format "
				    "version of %s database is %u, my format "
				    "version is %u", db_dir, version,
				    IPA_SDB_FORMAT_VERSION);
				goto failed_close;
			}
			if (fclose(fp) != 0) {
				logmsg(IPA_LOG_ERR, "check_db_dir: "
				    "fclose(%s)", path);
				goto failed;
			}
		}
		mem_free(path, m_tmp);
	}

	if (db_dir == global_db_dir)
		global_db_dir_checked = true;
	return (0);

failed_close:
	if (fclose(fp) != 0)
		logmsg(IPA_LOG_ERR, "check_db_dir: fclose(%s)", path);
failed:
	mem_free(path, m_tmp);
	return (-1);
}

static void
free_rule(struct rule *rule)
{
	rule->ref_count--;
	if (rule->ref_count == 0) {
		TAILQ_REMOVE(&rules_list, rule, link);
		if (rule->db_dir != global_db_dir)
			mem_free(rule->db_dir, m_parser);
#ifdef WITH_LIMITS
		mem_free(rule->limits, m_anon);
		mem_free(rule->limits_dir, m_anon);
#endif
#ifdef WITH_THRESHOLDS
		mem_free(rule->thresholds, m_anon);
		mem_free(rule->thresholds_dir, m_anon);
#endif
		mem_free(rule->rule_db_dir, m_anon);
		marray_free(rules_ptr_marray, rule->no);
		mzone_free(rule_mzone, rule);
	}
}

/*
 * The same as free_rule(), but free only those memory, which
 * could be allocated during configuration phase.
 */
static void
free_rule_fast(struct rule *rule)
{
	rule->ref_count--;
	if (rule->ref_count == 0) {
		TAILQ_REMOVE(&rules_list, rule, link);
		if (rule->db_dir != global_db_dir)
			mem_free(rule->db_dir, m_parser);
		marray_free(rules_ptr_marray, rule->no);
		mzone_free(rule_mzone, rule);
	}
}

static int
st_init_rule(unsigned int ruleno, const char *rule_name)
{
	struct stat statbuf;
	struct rule *rule;

	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		rule = alloc_rule(ruleno);
		if (rule == NULL) {
			logmsgx(IPA_LOG_ERR, "st_init_rule: alloc_rule failed");
			return (-1);
		}
	} else
		rule = RULE(ruleno);

	/* Inherit settings from global{}. */
	if (rule->db_dir == NULL)
		rule->db_dir = global_db_dir;

	rule->name = rule_name;

	if (check_db_dir(rule->db_dir) < 0) {
		logbt("st_init_rule");
		goto failed;
	}

	if (mem_asprintf(m_anon, &rule->rule_db_dir, "%s/%s",
	    rule->db_dir, rule_name) < 0) {
		logmsgx(IPA_LOG_ERR, "st_init_rule: mem_asprintf failed");
		goto failed;
	}

	if (stat_func(rule->rule_db_dir, &statbuf) < 0) {
		if (errno == ENOENT)
			logmsgx(IPA_LOG_ERR, "st_init_rule: unknown rule %s",
			    rule_name);
		logmsg(IPA_LOG_ERR, "st_init_rule: %s(%s)",
		    stat_func_name, rule->rule_db_dir);
		goto failed;
	}

	if (check_dir(rule->rule_db_dir, &statbuf) < 0) {
		logbt("st_init_rule");
		goto failed;
	}

	rule->flags |= RULE_INITED;
	return (0);

failed:
	free_rule(rule);
	return (-1);
}

#ifdef WITH_LIMITS
static void
free_limit(struct rule *rule, struct limit *limit, unsigned int limitno)
{
	mem_free(limit->db_dir, m_anon);
	LIMIT(rule, limitno) = NULL;
	mzone_free(limit_mzone, limit);
	free_rule(rule);
}

static int
st_init_limit(unsigned int ruleno, const char *rule_name, unsigned int limitno,
    const char *limit_name)
{
	struct stat statbuf;
	struct rule *rule;
	struct limit *limit, **limits;

	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		if (st_init_rule(ruleno, rule_name) < 0) {
			logbt("st_init_limit");
			return (-1);
		}
		rule = RULE(ruleno);
	} else {
		rule = RULE(ruleno);
		rule->ref_count++;
	}

	if (rule->limits_dir == NULL)
		if (mem_asprintf(m_anon, &rule->limits_dir,
		    "%s/"IPA_SDB_LIMITS_DIR, rule->rule_db_dir) < 0) {
			logmsgx(IPA_LOG_ERR, "st_init_limit: "
			    "mem_asprintf failed");
			goto failed;
		}

	if (limit_mzone == NULL) {
		limit_mzone = mzone_init(MZONE_NAME(limit), "Limits", 0,
		    sizeof(struct limit), LIMIT_NSIZE, LIMIT_NALLOC);
		if (limit_mzone == NULL) {
			logmsgx(IPA_LOG_ERR, "st_init_limit: "
			    "mzone_init failed");
			goto failed;
		}
	}

	if (rule->nlimits >= limitno + 1) {
		logmsgx(IPA_LOG_ERR, "st_init_limit: unexpected ordinal "
		    "number %u of limit", limitno);
		goto failed;
	}
	limits = mem_realloc(rule->limits, (limitno + 1) *
	    sizeof(*rule->limits), m_anon);
	if (limits == NULL) {
		logmsgx(IPA_LOG_ERR, "st_init_limit: mem_realloc failed");
		goto failed;
	}
	rule->limits = limits;
	rule->nlimits = limitno + 1;

	limit = mzone_alloc(limit_mzone);
	if (limit == NULL) {
		logmsgx(IPA_LOG_ERR, "st_init_limit: mzone_alloc failed");
		LIMIT(rule, limitno) = NULL;
		goto failed;
	}
	LIMIT(rule, limitno) = limit;

	if (mem_asprintf(m_anon, &limit->db_dir, "%s/%s",
	    rule->limits_dir, limit_name) < 0) {
		logmsgx(IPA_LOG_ERR, "st_init_limit: mem_asprintf failed");
		goto limit_failed;
	}

	if (stat_func(limit->db_dir, &statbuf) < 0) {
		if (errno == ENOENT)
			logmsgx(IPA_LOG_ERR, "st_init_limit: unknown rule %s, "
			    "limit %s", rule->name, limit_name);
		logmsg(IPA_LOG_ERR, "st_init_limit: %s(%s)",
		    stat_func_name, limit->db_dir);
		goto limit_failed;
	}

	if (check_dir(limit->db_dir, &statbuf) < 0) {
		logbt("st_init_limit");
		goto limit_failed;
	}
	return (0);

limit_failed:
	free_limit(rule, limit, limitno);
	return (-1);

failed:
	free_rule(rule);
	return (-1);
}

static int
st_deinit_limit(unsigned int ruleno, unsigned int limitno)
{
	if (marray_check_index(rules_ptr_marray, ruleno)) {
		struct rule *rule;

		/* Such rule was registered. */
		rule = RULE(ruleno);
		if (limitno < rule->nlimits) {
			struct limit *limit;

			/*
			 * Such limit was registered, but free_limit()
			 * could already be called for it.
			 */
			limit = LIMIT(rule, limitno);
			if (limit != NULL)
				free_limit(rule, limit, limitno);
		}
	}
	return (0);
}
#else
# define st_init_limit NULL
# define st_deinit_limit NULL
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
static void
free_threshold(struct rule *rule, struct threshold *threshold,
    unsigned int thresholdno)
{
	mem_free(threshold->db_dir, m_anon);
	THRESHOLD(rule, thresholdno) = NULL;
	mzone_free(threshold_mzone, threshold);
	free_rule(rule);
}

static int
st_init_threshold(unsigned int ruleno, const char *rule_name,
    unsigned int thresholdno, const char *threshold_name)
{
	struct stat statbuf;
	struct rule *rule;
	struct threshold *threshold, **thresholds;

	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		if (st_init_rule(ruleno, rule_name) < 0) {
			logbt("st_init_threshold");
			return (-1);
		}
		rule = RULE(ruleno);
	} else {
		rule = RULE(ruleno);
		rule->ref_count++;
	}

	if (rule->thresholds_dir == NULL)
		if (mem_asprintf(m_anon, &rule->thresholds_dir,
		    "%s/"IPA_SDB_THRESHOLDS_DIR, rule->rule_db_dir) < 0) {
			logmsgx(IPA_LOG_ERR, "st_init_threshold: "
			    "mem_asprintf failed");
			goto failed;
		}

	if (threshold_mzone == NULL) {
		threshold_mzone = mzone_init(MZONE_NAME(threshold),
		    "Thresholds", 0, sizeof(struct threshold),
		    THRESHOLD_NSIZE, THRESHOLD_NALLOC);
		if (threshold_mzone == NULL) {
			logmsgx(IPA_LOG_ERR, "st_init_threshold: "
			    "mzone_init failed");
			goto failed;
		}
	}

	if (rule->nthresholds >= thresholdno + 1) {
		logmsgx(IPA_LOG_ERR, "st_init_threshold: unexpected ordinal "
		    "number %u of threshold", thresholdno);
		goto failed;
	}
	thresholds = mem_realloc(rule->thresholds, (thresholdno + 1) *
	    sizeof(*rule->thresholds), m_anon);
	if (thresholds == NULL) {
		logmsgx(IPA_LOG_ERR, "st_init_threshold: mem_realloc failed");
		goto failed;
	}
	rule->thresholds = thresholds;
	rule->nthresholds = thresholdno + 1;

	threshold = mzone_alloc(threshold_mzone);
	if (threshold == NULL) {
		logmsgx(IPA_LOG_ERR, "st_init_threshold: mzone_alloc failed");
		THRESHOLD(rule, thresholdno) = NULL;
		goto failed;
	}
	THRESHOLD(rule, thresholdno) = threshold;

	if (mem_asprintf(m_anon, &threshold->db_dir, "%s/%s",
	    rule->thresholds_dir, threshold_name) < 0) {
		logmsgx(IPA_LOG_ERR, "st_init_threshold: mem_asprintf failed");
		goto threshold_failed;
	}

	if (stat_func(threshold->db_dir, &statbuf) < 0) {
		if (errno == ENOENT)
			logmsgx(IPA_LOG_ERR, "st_init_threshold: unknown "
			    "rule %s, threshold %s", rule->name,
			    threshold_name);
		logmsg(IPA_LOG_ERR, "st_init_threshold: %s(%s)",
		    stat_func_name, threshold->db_dir);
		goto threshold_failed;
	}

	if (check_dir(threshold->db_dir, &statbuf) < 0) {
		logbt("st_init_threshold");
		goto threshold_failed;
	}
	return (0);

threshold_failed:
	free_threshold(rule, threshold, thresholdno);
	return (-1);

failed:
	free_rule(rule);
	return (-1);
}

static int
st_deinit_threshold(unsigned int ruleno, unsigned int thresholdno)
{
	if (marray_check_index(rules_ptr_marray, ruleno)) {
		struct rule *rule;

		/* Such rule was registered. */
		rule = RULE(ruleno);
		if (thresholdno < rule->nthresholds) {
			struct threshold *threshold;

			/*
			 * Such threshold was registered, but free_threshold()
			 * could already be called for it,
			 */
			threshold = THRESHOLD(rule, thresholdno);
			if (threshold != NULL)
				free_threshold(rule, threshold, thresholdno);
		}
	}
	return (0);
}
#else
# define st_init_threshold NULL
# define st_deinit_threshold NULL
#endif /* WITH_THRESHOLDS */

static int
st_deinit_rule(unsigned int ruleno)
{
	if (marray_check_index(rules_ptr_marray, ruleno)) {
		/* Such rule was registered. */
		free_rule(RULE(ruleno));
	}
	return (0);
}

static int
st_deinit(void)
{
	struct rule *rule;
	unsigned int n;

	if (global_db_dir != db_dir_default)
		mem_free(global_db_dir, m_parser);

	while ((rule = TAILQ_FIRST(&rules_list)) != NULL) {
		if (rule->flags & RULE_INITED) {
			logmsgx(IPA_LOG_ERR, "internal error: st_deinit: "
			    "rule %s: rule was initialized, but was not "
			    "deinitialized (ref_count %u)", rule->name,
			    rule->ref_count);
			return (-1);
		}
		free_rule_fast(rule);
	}

	n = memfunc->marray_nused(rules_ptr_marray);
	if (n != 0) {
		logmsgx(IPA_LOG_ERR, "internal error: st_deinit: "
		    "rules_ptr_marray is not empty: %u", n);
		return (-1);
	}
	marray_deinit(rules_ptr_marray);

	n = mzone_nused(rule_mzone);
	if (n != 0) {
		logmsgx(IPA_LOG_ERR, "internal error: st_deinit: rule_mzone "
		    "is not empty: %u", n);
		return (-1);
	}
	mzone_deinit(rule_mzone);

#ifdef WITH_LIMITS
	if (limit_mzone != NULL) {
		n = mzone_nused(limit_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: st_deinit: "
			    "limit_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(limit_mzone);
	}
#endif

#ifdef WITH_THRESHOLDS
	if (threshold_mzone != NULL) {
		n = mzone_nused(threshold_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: st_deinit: "
			    "threshold_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(threshold_mzone);
	}
#endif

	return (0);
}

static int
read_info(ipa_mem_type *mem_type, const char *path, char **infop)
{
	struct stat statbuf;
	char *info;
	ssize_t size;
	int fd;

	if (stat_func(path, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "read_info: %s(%s)",
			    stat_func_name, path);
			return (-1);
		}
		*infop = NULL;
		return (0);
	}

	fd = open(path, O_RDONLY);
	if (fd < 0) {
		logmsg(IPA_LOG_ERR, "read_info: open(%s, O_RDONLY)", path);
		return (-1);
	}

	info = NULL;

	size = statbuf.st_size;
	info = mem_malloc(size + 1, mem_type);
	if (info == NULL) {
		logmsgx(IPA_LOG_ERR, "read_info: mem_malloc failed");
		goto failed_close;
	}

	size = readn(fd, path, info, size);
	if (size < 0) {
		logbt("read_info");
		goto failed_close;
	}
	info[size] = '\0';

	if (close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "read_info: close(%s)", path);
		goto failed;
	}

	*infop = info;
	return (0);

failed_close:
	if (close(fd) < 0)
		logmsg(IPA_LOG_ERR, "read_info: close(%s)", path);
failed:
	mem_free(info, mem_type);
	return (-1);
}

static int
st_get_rule_info(unsigned int ruleno, ipa_mem_type *mem_type, char **infop)
{
	const struct rule *rule;
	char *path;

	rule = RULE(ruleno);

	if (mem_asprintf(m_tmp, &path, "%s/"IPA_SDB_INFO_FILE,
	    rule->rule_db_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "st_get_rule_info: mem_asprintf failed");
		return (-1);
	}

	if (read_info(mem_type, path, infop) < 0) {
		logbt("st_get_rule_info");
		mem_free(path, m_tmp);
		return (-1);
	}

	mem_free(path, m_tmp);
	return (0);
}

/*
 * Generic function for getting list of rule, limits and thresholds.
 */
static int
get_desc_list(const char *dir_path, const regex_t *regexp,
    const char *ignore_name, ipa_mem_type *mem_type, unsigned int *n,
    struct ipa_entity_desc **listp)
{
#ifdef WITH_ST_PTHREAD
	struct dirent de;
#endif
	struct ipa_entity_desc *desc, *desc_list;
	struct dirent *dp;
	char *info_path, *name, *info;
	DIR *dirp;
	unsigned int i, nalloc;

	dirp = opendir(dir_path);
	if (dirp == NULL) {
		logmsg(IPA_LOG_ERR, "get_desc_list: opendir(%s)", dir_path);
		return (-1);
	}

	i = nalloc = 0;
	desc_list = NULL;

	for (;;) {
#ifdef WITH_ST_PTHREAD
		errno = readdir_r(dirp, &de, &dp);
		if (errno != 0) {
			logmsg(IPA_LOG_ERR, "get_desc_list: "
			    "readdir_r(%s)", dir_path);
			goto failed1;
		}
		if (dp == NULL)
			break;
#else
		errno = 0;
		dp = readdir(dirp);
		if (dp == NULL) {
			if (errno != 0) {
				logmsg(IPA_LOG_ERR, "get_desc_list: "
				    "readdir(%s)", dir_path);
				goto failed1;
			}
			break;
		}
#endif /* WITH_ST_PTHREAD */

		/* Ignore "." and ".." directories. */
		if (dp->d_name[0] == '.')
			switch (dp->d_name[1]) {
			case '\0':
				continue;
			case '.':
				if (dp->d_name[2] == '\0')
					continue;
			}

		if (ignore_name != NULL &&
		    strcmp(dp->d_name, ignore_name) == 0)
			continue;
		if (regexp != NULL &&
		    regexec(regexp, dp->d_name, 0, (regmatch_t *)NULL, 0) != 0)
			continue;

		info = info_path = NULL;
		name = mem_strdup(dp->d_name, mem_type);
		if (name == NULL) {
			logmsgx(IPA_LOG_ERR, "get_desc_list: "
			    "mem_strdup failed");
			goto failed;
		}

		if (mem_asprintf(m_tmp, &info_path, "%s/%s/"IPA_SDB_INFO_FILE,
		    dir_path, name) < 0) {
			logmsgx(IPA_LOG_ERR, "get_desc_list: "
			    "mem_asprintf failed");
			goto failed;
		}
		if (read_info(mem_type, info_path, &info) < 0) {
			logbt("get_desc_list");
			goto failed;
		}
		mem_free(info_path, m_tmp);
		info_path = NULL;

		if (nalloc == 0) {
			desc = mem_realloc(desc_list, (i + DESC_LIST_NALLOC) *
			    sizeof(*desc_list), mem_type);
			if (desc == NULL) {
				logmsgx(IPA_LOG_ERR, "get_desc_list: "
				    "mem_realloc failed");
				goto failed;
			}
			desc_list = desc;
			nalloc = DESC_LIST_NALLOC;
		}
		desc = desc_list + i;
		desc->name = name;
		desc->info = info;
		++i;
		--nalloc;
	}

	if (closedir(dirp) < 0) {
		logmsg(IPA_LOG_ERR, "get_desc_list: closedir(%s)", dir_path);
		goto failed2;
	}

	*n = i;
	*listp = desc_list;
	return (0);

failed:
	mem_free(name, mem_type);
	mem_free(info, mem_type);
	mem_free(info_path, m_tmp);
failed1:
	if (closedir(dirp) < 0)
		logmsg(IPA_LOG_ERR, "get_desc_list: closedir(%s)", dir_path);
failed2:
	if (desc_list != NULL) {
		for (desc = desc_list; desc < desc_list + i; ++desc) {
			mem_free(desc->name, mem_type);
			mem_free(desc->info, mem_type);
		}
		mem_free(desc_list, mem_type);
	}
	return (-1);
}

static int
st_get_rules_list(const char *pat ATTR_UNUSED, const regex_t *regexp,
    ipa_mem_type *mem_type, unsigned int *n, struct ipa_entity_desc **listp)
{
	if (check_db_dir(global_db_dir) < 0)
		goto failed;

	if (get_desc_list(global_db_dir, regexp, IPA_SDB_VERSION_FILE,
	    mem_type, n, listp) < 0)
		goto failed;

	return (0);

failed:
	logbt("st_get_rules_list");
	return (-1);
}

static int
st_get_rule_stat(unsigned int ruleno, const ipa_tm *tm1, const ipa_tm *tm2,
    int exact, ipa_mem_type *mem_type, unsigned int *n,
    struct ipa_rule_stat **bufp)
{
	struct stat statbuf;
	const struct rule *rule;
	struct ipa_rule_stat *rule_stat_arr, *rule_stat;
	struct ipa_sdb_rule_record *recs, *rec;
	char *path;
	size_t src_size, recs_size;
	ssize_t nread;
	unsigned int i, nalloc, nrecs;
	int fd;
	int year, year1, year2, mon, mon1, mon2, mday, mday1, mday2;
	int sec1, sec2, rsec1, rsec2;

	rule = RULE(ruleno);

	i = nalloc = 0;
	rule_stat_arr = NULL;

	recs_size = RULE_NREC_READ * sizeof(*recs);
	recs = mem_malloc(recs_size, m_tmp);
	if (recs == NULL) {
		logmsgx(IPA_LOG_ERR, "st_get_rule_stat: mem_malloc failed");
		return (-1);
	}

	year1 = tm1->tm_year;
	year2 = tm2->tm_year;
	for (year = year1; year <= year2; ++year) {
		mon1 = year == year1 ? tm1->tm_mon : 1;
		mon2 = year == year2 ? tm2->tm_mon : MONTHES_IN_YEAR;
		for (mon = mon1; mon <= mon2; ++mon) {
			if (mem_asprintf(m_tmp, &path, "%s/%d%02d",
			    rule->rule_db_dir, year, mon) < 0) {
				logmsgx(IPA_LOG_ERR, "st_get_rule_stat: "
				    "mem_asprintf failed");
				goto failed;
			}
			if (stat_func(path, &statbuf) < 0) {
				if (errno != ENOENT) {
					logmsg(IPA_LOG_ERR,
					    "st_get_rule_stat: %s(%s)",
					    stat_func_name, path);
					goto failed;
				}
				mem_free(path, m_tmp);
				continue;
			}
			if (check_file(path, &statbuf) < 0) {
				logbt("st_get_rule_stat");
				goto failed;
			}
			src_size = statbuf.st_size;
			if (src_size == 0) {
				mem_free(path, m_tmp);
				continue;
			}
			if (src_size % sizeof(*recs)) {
				logmsg(IPA_LOG_ERR, "st_get_rule_stat: "
				    "wrong size of %s", path);
				goto failed;
			}
			fd = open(path, O_RDONLY);
			if (fd < 0) {
				logmsg(IPA_LOG_ERR, "st_get_rule_stat: "
				    "open(%s, O_RDONLY)", path);
				goto failed;
			}

			if (year == year1 && mon == mon1) {
				mday1 = tm1->tm_mday;
				sec1 = tm1->tm_hour * SECONDS_IN_HOUR +
				    tm1->tm_min * SECONDS_IN_MINUTE +
				    tm1->tm_sec;
			} else {
				mday1 = 1;
				sec1 = 0;
			}

			if (year == year2 && mon == mon2) {
				mday2 = tm2->tm_mday;
				sec2 = tm2->tm_hour * SECONDS_IN_HOUR +
				    tm2->tm_min * SECONDS_IN_MINUTE +
				    tm2->tm_sec;
			} else {
				mday2 = 31;
				sec2 = HOURS_IN_DAY * SECONDS_IN_HOUR;
			}

			for (;;) {
				nread = src_size < recs_size ?
				    src_size : recs_size;
				nread = readn(fd, path, recs, nread);
				if (nread < 0) {
					logbt("st_get_rule_stat");
					goto failed_close;
				}
				if (nread == 0) {
					logmsgx(IPA_LOG_WARNING,
					    "st_get_rule_stat: read from "
					    "%s returned 0", path);
					break;
				}
				if (nread % sizeof(*recs) != 0) {
					logmsgx(IPA_LOG_ERR,
					    "st_get_rule_stat: read: "
					    "wrong size of %s", path);
					goto failed_close;
				}
				rec = recs;
				nrecs = nread / sizeof(*recs);
				for (; nrecs > 0; ++rec, --nrecs) {
					mday = rec->mday;
					if (mday < mday1 || mday > mday2)
						continue;
					/* In days range. */
					if (mday == mday1 || mday == mday2) {
						/* Check boundaries. */
						rsec1 =
						    rec->h1 * SECONDS_IN_HOUR +
						    rec->m1 * SECONDS_IN_MINUTE
						    + rec->s1;
						rsec2 =
						    rec->h2 * SECONDS_IN_HOUR +
						    rec->m2 * SECONDS_IN_MINUTE
						    + rec->s2;
						if (exact) {
							if (mday == mday1 &&
							    rsec1 < sec1)
								continue;
							if (mday == mday2 &&
							    rsec2 > sec2)
								continue;
						} else {
							if (mday == mday1 &&
							    rsec1 < sec1 &&
							    rsec2 < sec1)
								continue;
							if (mday == mday2 &&
							    rsec1 > sec2 &&
							    rsec2 > sec2)
								continue;
						}
					}
					if (nalloc == 0) {
						rule_stat =
						    mem_realloc(rule_stat_arr,
						    (i + RULE_STAT_NALLOC) *
						    sizeof(*rule_stat_arr),
						    mem_type);
						if (rule_stat == NULL) {
							logmsgx(IPA_LOG_ERR,
							    "st_get_rule_stat: "
							    "mem_realloc failed"
							    );
							goto failed_close;
						}
						rule_stat_arr = rule_stat;
						nalloc = RULE_STAT_NALLOC;
					}
					rule_stat = rule_stat_arr + i;
					rule_stat->year = year;
					rule_stat->mon = mon;
					rule_stat->mday = mday;
					rule_stat->h1 = rec->h1;
					rule_stat->m1 = rec->m1;
					rule_stat->s1 = rec->s1;
					rule_stat->h2 = rec->h2;
					rule_stat->m2 = rec->m2;
					rule_stat->s2 = rec->s2;
					cnt_from_base(rule_stat->cnt,
					    rec->c_high, rec->c_low);
					++i;
					--nalloc;
				}
				src_size -= nread;
				if (src_size == 0)
					break;
			}
			if (close(fd) < 0) {
				logmsg(IPA_LOG_ERR, "st_get_rule_stat: "
				    "close(%s)", path);
				goto failed;
			}
			mem_free(path, m_tmp);
		}
	}

	mem_free(recs, m_tmp);

	*n = i;
	*bufp = rule_stat_arr;
	return (0);

failed_close:
	if (close(fd) < 0)
		logmsg(IPA_LOG_ERR, "st_get_rule_stat: close(%s)", path);
failed:
	mem_free(path, m_tmp);
	mem_free(recs, m_tmp);
	mem_free(rule_stat_arr, mem_type);
	return (-1);
}

#ifdef WITH_ANY_LIMITS
/*
 * Convert machine independent ipa_sdb_date{} values to ipa_tm{}.
 */
static void
copy_db_date_to_tm(const ipa_sdb_date *date, ipa_tm *tm)
{
	tm->tm_year = ntohs(date->year);
	tm->tm_mon  = date->mon;
	tm->tm_mday = date->mday;
	tm->tm_hour = date->hour;
	tm->tm_min  = date->min;
	tm->tm_sec  = date->sec;
}
#endif /* WITH_ANY_LIMITS */

#ifdef WITH_LIMITS
static int
st_get_limit_info(unsigned int ruleno, unsigned int limitno,
    ipa_mem_type *mem_type, char **infop)
{
	const struct limit *limit;
	char *path;

	limit = LIMIT(RULE(ruleno), limitno);

	if (mem_asprintf(m_tmp, &path, "%s/"IPA_SDB_INFO_FILE,
	    limit->db_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "st_get_limit_info: mem_asprintf failed");
		return (-1);
	}

	if (read_info(mem_type, path, infop) < 0) {
		logbt("st_get_limit_info");
		mem_free(path, m_tmp);
		return (-1);
	}

	mem_free(path, m_tmp);
	return (0);
}

static int
st_get_limits_list(unsigned int ruleno, const char *pat ATTR_UNUSED,
    const regex_t *regexp, ipa_mem_type *mem_type, unsigned int *n,
    struct ipa_entity_desc **listp)
{
	struct stat statbuf;
	struct rule *rule;

	rule = RULE(ruleno);

	if (rule->limits_dir == NULL)
		if (mem_asprintf(m_anon, &rule->limits_dir,
		    "%s/"IPA_SDB_LIMITS_DIR, rule->rule_db_dir) < 0) {
			logmsgx(IPA_LOG_ERR, "st_get_limits_list: "
			    "mem_asprintf failed");
			return (-1);
		}

	if (stat_func(rule->limits_dir, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsgx(IPA_LOG_ERR, "st_get_limits_list: %s(%s)",
			    stat_func_name, rule->limits_dir);
			return (-1);
		}
		*n = 0;
		*listp = NULL;
		return (0);
	}

	if (check_dir(rule->limits_dir, &statbuf) < 0)
		goto failed;

	if (get_desc_list(rule->limits_dir, regexp, (const char *)NULL,
	    mem_type, n, listp) < 0)
		goto failed;

	return (0);

failed:
	logbt("st_get_limits_list");
	return (-1);
}

static void
copy_sdb_limit_record(const struct ipa_sdb_limit_record *rec,
    struct ipa_limit_state *state)
{
	state->event_date_set = 0;
	if (rec->set & IPA_SDB_LIMIT_EVENT_START_SET) {
		state->event_date_set = IPA_LIMIT_EVENT_START_SET;
		copy_db_date_to_tm(&rec->start,
		    &state->event_date[IPA_LIMIT_EVENT_START]);
	}
	if (rec->set & IPA_SDB_LIMIT_EVENT_RESTART_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_RESTART_SET;
		copy_db_date_to_tm(&rec->restart,
		    &state->event_date[IPA_LIMIT_EVENT_RESTART]);
	}
	if (rec->set & IPA_SDB_LIMIT_EVENT_RESTART_EXEC_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_RESTART_EXEC_SET;
		copy_db_date_to_tm(&rec->restart_exec,
		    &state->event_date[IPA_LIMIT_EVENT_RESTART_EXEC]);
	}
	if (rec->set & IPA_SDB_LIMIT_EVENT_REACH_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_REACH_SET;
		copy_db_date_to_tm(&rec->reach,
		    &state->event_date[IPA_LIMIT_EVENT_REACH]);
	}
	if (rec->set & IPA_SDB_LIMIT_EVENT_REACH_EXEC_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_REACH_EXEC_SET;
		copy_db_date_to_tm(&rec->reach_exec,
		    &state->event_date[IPA_LIMIT_EVENT_REACH_EXEC]);
	}
	if (rec->set & IPA_SDB_LIMIT_EVENT_EXPIRE_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_EXPIRE_SET;
		copy_db_date_to_tm(&rec->expire,
		    &state->event_date[IPA_LIMIT_EVENT_EXPIRE]);
	}
	if (rec->set & IPA_SDB_LIMIT_EVENT_EXPIRE_EXEC_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_EXPIRE_EXEC_SET;
		copy_db_date_to_tm(&rec->expire_exec,
		    &state->event_date[IPA_LIMIT_EVENT_EXPIRE_EXEC]);
	}
	if (rec->set & IPA_SDB_LIMIT_EVENT_UPDATED_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_UPDATED_SET;
		copy_db_date_to_tm(&rec->updated,
		    &state->event_date[IPA_LIMIT_EVENT_UPDATED]);
	}
	cnt_from_base(state->lim, rec->l_high, rec->l_low);
	cnt_from_base(state->cnt, rec->c_high, rec->c_low);
}

static int
get_limit_cur_stat(const struct limit *limit, ipa_mem_type *mem_type,
    unsigned int *n, struct ipa_limit_state **bufp)
{
	struct ipa_sdb_limit_record rec;
	struct stat statbuf;
	struct ipa_limit_state *state;
	char *path;
	size_t src_size;
	int fd;

	if (mem_asprintf(m_tmp, &path, "%s/"IPA_SDB_LIMIT_CURRENT,
	    limit->db_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "get_limit_cur_stat: mem_asprintf failed");
		return (-1);
	}
	if (stat_func(path, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "get_limit_cur_stat: %s(%s)",
			    stat_func_name, path);
			goto failed;
		}
		src_size = 0;
	} else {
		if (check_file(path, &statbuf) < 0) {
			logbt("get_limit_cur_stat");
			goto failed;
		}
		src_size = statbuf.st_size;
		if (src_size % sizeof(rec) != 0) {
			logmsgx(IPA_LOG_ERR, "get_limit_cur_stat: "
			    "wrong size of %s", path);
			goto failed;
		}
	}

	if (src_size == 0) {
		*n = 0;
		*bufp = NULL;
	} else {
		state = mem_malloc(sizeof(*state), mem_type);
		if (state == NULL) {
			logmsgx(IPA_LOG_ERR, "get_limit_cur_stat: "
			    "mem_malloc failed");
			goto failed;
		}

		fd = open(path, O_RDONLY);
		if (fd < 0) {
			logmsg(IPA_LOG_ERR, "get_limit_cur_stat: "
			    "open(%s, O_RDONLY)", path);
			goto failed;
		}
		if (lseek(fd, -(off_t)sizeof(rec), SEEK_END) == (off_t)-1) {
			logmsg(IPA_LOG_ERR, "get_limit_cur_stat: "
			    "lseek(%s, -%lu, SEEK_END", path,
			    (unsigned long)sizeof(rec));
			goto failed_close;
		}
		if (readn(fd, path, &rec, sizeof(rec)) != sizeof(rec)) {
			logbt("get_limit_cur_stat");
			goto failed_close;
		}
		if (close(fd) < 0) {
			logmsg(IPA_LOG_ERR, "get_limit_cur_stat: "
			    "close(%s)", path);
			goto failed;
		}
		copy_sdb_limit_record(&rec, state);

		*n = 1;
		*bufp = state;
	}

	mem_free(path, m_tmp);
	return (0);

failed_close:
	if (close(fd) < 0)
		logmsg(IPA_LOG_ERR, "get_limit_cur_stat: close(%s)", path);
failed:
	mem_free(path, m_tmp);
	return (-1);

}

static int
get_limit_int_stat(const struct limit *limit, const ipa_tm *tm1,
    const ipa_tm *tm2, ipa_mem_type *mem_type, unsigned int *n,
    struct ipa_limit_state **bufp)
{
	struct stat statbuf;
	const struct ipa_sdb_limit_record *rec;
	const ipa_sdb_date *sd;
	struct ipa_sdb_limit_record *recs;
	struct ipa_limit_state *state_arr, *state;
	char *path;
	size_t src_size, recs_size;
	ssize_t nread;
	unsigned int i, nalloc, nrecs;
	int fd;
	int year, year1, year2, mon, mon1, mon2, mday, mday1, mday2;
	int sec1, sec2, rsec;

	recs = NULL;
	state_arr = NULL;

	recs_size = LIMIT_NREC_READ * sizeof(*recs);
	recs = mem_malloc(recs_size, m_tmp);
	if (recs == NULL) {
		logmsgx(IPA_LOG_ERR, "get_limit_int_stat: mem_malloc failed");
		return (-1);
	}

	i = nalloc = 0;

	year1 = tm1->tm_year;
	year2 = tm2->tm_year;
	for (year = year1; year <= year2; ++year) {
		mon1 = year == year1 ? tm1->tm_mon : 1;
		mon2 = year == year2 ? tm2->tm_mon : MONTHES_IN_YEAR;
		for (mon = mon1; mon <= mon2; ++mon) {
			if (mem_asprintf(m_tmp, &path, "%s/%d%02d",
			    limit->db_dir, year, mon) < 0) {
				logmsgx(IPA_LOG_ERR, "get_limit_int_stat: "
				    "mem_asprintf failed");
				goto failed;
			}
			if (stat_func(path, &statbuf) < 0) {
				if (errno != ENOENT) {
					logmsgx(IPA_LOG_ERR,
					    "get_limit_int_stat: %s(%s)",
					    stat_func_name, path);
					goto failed;
				}
				mem_free(path, m_tmp);
				continue;
			}
			if (check_file(path, &statbuf) < 0) {
				logbt("get_limit_int_stat");
				goto failed;
			}
			src_size = statbuf.st_size;
			if (src_size == 0) {
				mem_free(path, m_tmp);
				continue;
			}
			fd = open(path, O_RDONLY);
			if (fd < 0) {
				logmsgx(IPA_LOG_ERR,"get_limit_int_stat: "
				    "open(%s, O_RDONLY)", path);
				goto failed;
			}

			if (year == year1 && mon == mon1) {
				mday1 = tm1->tm_mday;
				sec1 = tm1->tm_hour * SECONDS_IN_HOUR +
				    tm1->tm_min * SECONDS_IN_MINUTE +
				    tm1->tm_sec;
			} else {
				mday1 = 1;
				sec1 = 0;
			}

			if (year == year2 && mon == mon2) {
				mday2 = tm2->tm_mday;
				sec2 = tm2->tm_hour * SECONDS_IN_HOUR +
				    tm2->tm_min * SECONDS_IN_MINUTE +
				    tm2->tm_sec;
			} else {
				mday2 = 31;
				sec2 = HOURS_IN_DAY * SECONDS_IN_HOUR;
			}

			for (;;) {
				nread = src_size < recs_size ?
				    src_size : recs_size;
				nread = readn(fd, path, recs, nread);
				if (nread < 0) {
					logbt("get_limit_int_stat");
					goto failed_close;
				}
				if (nread == 0) {
					logmsgx(IPA_LOG_WARNING,
					    "get_limit_int_stat: read from %s "
					    "returned 0", path);
					break;
				}
				if (nread % sizeof(*recs) != 0) {
					logmsgx(IPA_LOG_ERR,
					    "get_limit_int_stat: read: "
					    "wrong size of %s", path);
					goto failed_close;
				}
				rec = recs;
				nrecs = nread / sizeof(*recs);
				for (; nrecs > 0; ++rec, --nrecs) {
					sd = &rec->start;
					mday = sd->mday;
					if (mday < mday1 || mday > mday2)
						continue;
					/* In days range. */
					if (mday == mday1 || mday == mday2) {
						/* Check boundaries. */
						rsec =
						    sd->hour * SECONDS_IN_HOUR +
						    sd->min * SECONDS_IN_MINUTE
						    + sd->sec;
						if (mday == mday1 &&
						    rsec < sec1)
							continue;
						if (mday == mday2 &&
						    rsec > sec2)
							continue;
					}
					if (nalloc == 0) {
						state = mem_realloc(state_arr,
						    (i + LIMIT_STAT_NALLOC) *
						    sizeof(*state), mem_type);
						if (state == NULL) {
							logmsgx(IPA_LOG_ERR,
							    "get_limit_int_stat"
							    ": mem_realloc "
							    "failed");
							goto failed_close;
						}
						state_arr = state;
						nalloc = LIMIT_STAT_NALLOC;
					}
					state = state_arr + i;
					copy_sdb_limit_record(rec, state);
					++i;
					--nalloc;
				}
				src_size -= nread;
				if (src_size == 0)
					break;
			}
			if (close(fd) < 0) {
				logmsgx(IPA_LOG_ERR, "get_limit_int_stat: "
				    "close(%s)", path);
				goto failed;
			}
			mem_free(path, m_tmp);
		}
	}

	mem_free(recs, m_tmp);

	*n = i;
	*bufp = state_arr;
	return (0);

failed_close:
	if (close(fd) < 0)
		logmsg(IPA_LOG_ERR, "get_limit_int_stat: close(%s)", path);
failed:
	mem_free(path, m_tmp);
	mem_free(recs, m_tmp);
	mem_free(state_arr, mem_type);
	return (-1);
}

static int
st_get_limit_stat(unsigned int ruleno, unsigned int limitno,
    const ipa_tm *tm1, const ipa_tm *tm2, ipa_mem_type *mem_type,
    unsigned int *n, struct ipa_limit_state **bufp)
{
	const struct limit *limit;
	int rv;

	limit = LIMIT(RULE(ruleno), limitno);
	rv = tm1 == NULL ?
	    get_limit_cur_stat(limit, mem_type, n, bufp) :
	    get_limit_int_stat(limit, tm1, tm2, mem_type, n, bufp);
	if (rv < 0) {
		logbt("st_get_limit_stat");
		return (-1);
	}
	return (0);
}
#else
# define st_get_limit_info NULL
# define st_get_limits_list NULL
# define st_get_limit_stat NULL
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
static int
st_get_threshold_info(unsigned int ruleno, unsigned int thresholdno,
    ipa_mem_type *mem_type, char **infop)
{
	const struct threshold *threshold;
	char *path;

	threshold = THRESHOLD(RULE(ruleno), thresholdno);

	if (mem_asprintf(m_tmp, &path, "%s/"IPA_SDB_INFO_FILE,
	    threshold->db_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "st_get_threshold_info: "
		    "mem_asprintf failed");
		return (-1);
	}

	if (read_info(mem_type, path, infop) < 0) {
		logbt("st_get_threshold_info");
		mem_free(path, m_tmp);
		return (-1);
	}

	mem_free(path, m_tmp);
	return (0);
}

static int
st_get_thresholds_list(unsigned int ruleno, const char *pat ATTR_UNUSED,
    const regex_t *regexp, ipa_mem_type *mem_type, unsigned int *n,
    struct ipa_entity_desc **listp)
{
	struct stat statbuf;
	struct rule *rule;

	rule = RULE(ruleno);

	if (rule->thresholds_dir == NULL)
		if (mem_asprintf(m_anon, &rule->thresholds_dir,
		    "%s/"IPA_SDB_THRESHOLDS_DIR, rule->rule_db_dir) < 0) {
			logmsgx(IPA_LOG_ERR, "st_get_thresholds_list: "
			    "mem_asprintf failed");
			return (-1);
		}

	if (stat_func(rule->thresholds_dir, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsgx(IPA_LOG_ERR, "st_get_thresholds_list: %s(%s)",
			    stat_func_name, rule->thresholds_dir);
			return (-1);
		}
		*n = 0;
		*listp = NULL;
		return (0);
	}

	if (check_dir(rule->thresholds_dir, &statbuf) < 0)
		goto failed;

	if (get_desc_list(rule->thresholds_dir, regexp, (const char *)NULL,
	    mem_type, n, listp) < 0)
		goto failed;

	return (0);

failed:
	logbt("st_get_thresholds_list");
	return (-1);
}

static int
st_get_threshold_stat(unsigned int ruleno, unsigned int thresholdno,
    struct ipa_threshold_state *buf)
{
	struct ipa_sdb_threshold_record rec;
	struct stat statbuf;
	const struct threshold *threshold;
	char *path;
	int fd, rv;

	threshold = THRESHOLD(RULE(ruleno), thresholdno);

	if (mem_asprintf(m_tmp, &path, "%s/"IPA_SDB_THRESHOLD_STATE,
	    threshold->db_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "st_get_threshold_stat: "
		    "mem_asprintf failed");
		return (-1);
	}

	if (stat_func(path, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "st_get_threshold_stat: %s(%s)",
			    stat_func_name, path);
			goto failed;
		}
		rv = 0;
	} else {
		if (check_file(path, &statbuf) < 0) {
			logbt("st_get_threshold");
			goto failed;
		}
		if (statbuf.st_size == 0)
			rv = 0;
		else {
			if (statbuf.st_size != sizeof(rec)) {
				logmsgx(IPA_LOG_ERR, "st_get_threshold_stat: "
				    "wrong size of %s", path);
				goto failed;
			}

			fd = open(path, O_RDONLY);
			if (fd < 0) {
				logmsg(IPA_LOG_ERR, "st_get_threshold_stat: "
				    "open(%s, O_RDONLY)", path);
				goto failed;
			}

			if (readn(fd, path, &rec, sizeof(rec)) != sizeof(rec)) {
				logbt("st_get_threshold_stat");
				goto failed_close;
			}

			cnt_from_base(buf->thr, rec.t_high, rec.t_low);
			cnt_from_base(buf->cnt, rec.c_high, rec.c_low);
			copy_db_date_to_tm(&rec.tm_started, &buf->tm_from);
			copy_db_date_to_tm(&rec.tm_updated, &buf->tm_updated);

			if (close(fd) < 0) {
				logmsg(IPA_LOG_ERR, "st_get_threshold_stat: "
				    "close(%s)", path);
				goto failed;
			}
			rv = 1;
		}
	}

	mem_free(path, m_tmp);
	return (rv);

failed_close:
	if (close(fd) < 0)
		logmsg(IPA_LOG_ERR, "st_get_threshold_stat: close(%s)", path);
failed:
	mem_free(path, m_tmp);
	return (-1);
}
#else
# define st_get_threshold_info NULL
# define st_get_thresholds_list NULL
# define st_get_threshold_stat NULL
#endif /* WITH_THRESHOLDS */

#define SDB_CONF_ROOT_SECT (IPA_CONF_SECT_CUSTOM_OFFSET + 1)

static const unsigned int sect_root[] = { IPA_CONF_SECT_ROOT, 0 };
static const unsigned int sect_cust_root[] = { SDB_CONF_ROOT_SECT, 0 };
static const unsigned int sect_any_rule[] = { IPA_CONF_SECT_GLOBAL,
	IPA_CONF_SECT_RULE, 0
};

static ipa_conf_sect conf_sect_tbl[] = {
	{ "", SDB_CONF_ROOT_SECT, 0, NULL, NULL, IPA_CONF_TYPE_MISC,
	  sect_root, parse_config_section
	},
	{ NULL, 0, 0, NULL, NULL, IPA_CONF_TYPE_MISC,
	  NULL, NULL
	}
};

static ipa_conf_param conf_param_tbl[] = {
	{ "db_dir", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_any_rule, parse_db_dir
	},
	{ "quiet", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_cust_root, parse_quiet
	},
	{ "allow_symlinks", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_cust_root, parse_allow_symlinks
	},
	{ "check_version", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_cust_root, parse_check_version
	},
	{ NULL, 0, NULL, NULL, IPA_CONF_TYPE_MISC,
	  NULL, NULL
	}
};

struct ipa_st_mod IPA_ST_MOD_ENTRY(ipa_st_sdb) = {
	IPA_ST_MOD_API_VERSION,	/* api_ver			*/
	MOD_FLAGS,		/* mod_flags			*/
	MOD_ST_NAME,		/* st_name			*/
	NULL,			/* suppfunc			*/
	NULL,			/* memfunc			*/
	MOD_CONF_PREFIX,	/* conf_prefix			*/
	conf_sect_tbl,		/* conf_sect_tbl		*/
	conf_param_tbl,		/* conf_param_tbl		*/
	conf_init,		/* conf_init			*/
	conf_deinit,		/* conf_deinit			*/
	conf_event,		/* conf_event			*/
	conf_mimic_real,	/* conf_mimic_real		*/
	NULL,			/* conf_inherit			*/
	conf_show,		/* conf_show			*/
	st_pre_init,		/* st_pre_init			*/
	st_init_rule,		/* st_init_rule			*/
	st_init_limit,		/* st_init_limit		*/
	st_init_threshold,	/* st_init_threshold		*/
	st_init,		/* st_init			*/
	st_deinit_threshold,	/* st_deinit_threshold		*/
	st_deinit_limit,	/* st_deinit_limit		*/
	st_deinit_rule,		/* st_deinit_rule		*/
	st_deinit,		/* st_deinit			*/
	st_get_rule_info,	/* st_get_rule_info		*/
	st_get_limit_info,	/* st_get_limit_info		*/
	st_get_threshold_info,	/* st_get_threshold_info	*/
	st_get_rules_list,	/* st_get_rules_list		*/
	st_get_limits_list,	/* st_get_limits_list		*/
	st_get_thresholds_list,	/* st_get_thresholds_list	*/
	st_get_rule_stat,	/* st_get_rule_stat		*/
	st_get_limit_stat,	/* st_get_limit_stat		*/
	st_get_threshold_stat	/* st_get_threshold_stat	*/
};
