/*-
 * Copyright (c) 2003-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: sdb_db_mod.c,v 1.11 2012/07/10 20:35:57 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/stat.h>

#include <netinet/in.h>

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <limits.h>
#include <regex.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <ipa_mod.h>

#include "ipa_sdb.h"

#include "queue.h"

#if IPA_DB_MOD_API_VERSION != 1 && IPA_DB_MOD_API_VERSION != 2
# error "This module supports only ipa_db_mod API versions 1 and 2"
#endif

#if IPA_MEMFUNC_API_VERSION != 1
# error "This module supports only ipa_memfunc API version 1"
#endif

#ifdef WITH_DB_PTHREAD
# define MOD_FLAGS IPA_MOD_FLAG_PTHREAD_SAFE
#else
# define MOD_FLAGS 0
#endif

#define MEMFLAG_OPTIMIZE IPA_MEMFUNC_FLAG_OPTIMIZE

#define MOD_NAME	"ipa_db_sdb"
#define MOD_DB_NAME	"sdb"
#define MOD_CONF_PREFIX	"sdb"

#define MARRAY_NAME(x)	MOD_NAME ":" #x
#define MZONE_NAME(x)	MOD_NAME ":" #x
#define MTYPE_NAME(x)	MOD_NAME ":m_" #x

/* Is defined at the end of this file. */
extern struct ipa_db_mod IPA_DB_MOD_ENTRY(ipa_db_sdb);

static const ipa_suppfunc *suppfunc;	/* Exported support functions. */

static const ipa_memfunc *memfunc;	/* Exported memory functions. */

static ipa_mem_type *m_parser;		/* Memory from parser. */
static ipa_mem_type *m_anon;		/* Anonymous memory allocations. */
static ipa_mem_type *m_tmp;		/* Temporary memory allocations. */

static void	*(*mem_malloc)(size_t, ipa_mem_type *);
static void	*(*mem_realloc)(void *, size_t, ipa_mem_type *);
static void	(*mem_free)(void *, ipa_mem_type *);

static ipa_marray *(*marray_init)(const char *, const char *, unsigned int,
		    void **, size_t, unsigned int, unsigned int);
static void	(*marray_deinit)(ipa_marray *);
static int	(*marray_alloc)(ipa_marray *, unsigned int *, int);
static void	(*marray_free)(ipa_marray *, unsigned int);
static void	(*marray_minimize)(ipa_marray *);
static int	(*marray_check_index)(ipa_marray *, unsigned int);

static ipa_mzone *(*mzone_init)(const char *, const char *, unsigned int,
		    size_t, unsigned int, unsigned int);
static void	(*mzone_deinit)(ipa_mzone *);
static void	*(*mzone_alloc)(ipa_mzone *);
static void	(*mzone_free)(ipa_mzone *, void *);
static unsigned int (*mzone_nused)(ipa_mzone *);

static bool	config_section;		/* Set if there was sdb: {} section. */

static signed char allow_symlinks;	/* sdb: { allow_symlinks } */
static signed char check_version;	/* sdb: { check_version } */

#define DB_RULE_DIR_PERM_UG (S_IRWXU|S_IRGRP|S_IXGRP)		/* 0750 */
#define DB_RULE_DIR_PERM_U  S_IRWXU				/* 0700 */

#define DB_DIR_PERM  (S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH)	/* 0755 */
#define DB_FILE_PERM (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH)		/* 0644 */

#define DB_MAIN_DIR_PERM DB_DIR_PERM				/* 0755 */

#define PERM_MASK(x) ((x) & (S_IRWXU|S_IRWXG|S_IRWXO))		/* 0777 & x */

/*
 * One "db_group" parameter.
 */
struct db_group {
	gid_t		gid;		/* GID of database files. */
	mode_t		perm;		/* Mode of the database directory. */
	char		set;		/* 0 not set, 1 set, 2 set & named. */
};

static char	*global_db_dir;		/* global { sdb:db_dir } */
static signed char global_close_fd;	/* global { sdb:close_fd } */
static struct db_group global_db_group;	/* global { sdb:db_group } */
static bool	global_db_dir_checked;	/* Set if global_db_dir was checked. */

/* Default database directory. */
static char	*db_dir_default = IPA_SDB_DB_DIR;

static uid_t	proc_uid;		/* Current process UID. */
static gid_t	proc_gid;		/* Current process GID. */

static int	bt_indent;		/* Indent for logbt(). */

#define RULE_NSIZE	30
#define RULE_NALLOC	20

#ifdef WITH_LIMITS
/*
 * Module's data for limit{}.
 * If fd < 0, then limit is not initialized or is inactive.
 */
struct limit {
	int		fd;		/* Database file descriptor. */
	int		start_year;	/* Year when limit was started. */
	int		start_mon;	/* Month when limit was started. */
	struct ipa_sdb_limit_record rec;/* Current database record. */
	char		*db_file;	/* Database file for limit. */
	char		*db_dir;	/* Database directory for limit. */
};

static ipa_mzone *limit_mzone;		/* Mzone for all limits. */

#define LIMIT_NREC_READ						\
	(sizeof(struct ipa_sdb_limit_record) < 1024 ?		\
	 (1024 / sizeof(struct ipa_sdb_limit_record)) : 1)

#define LIMIT_NSIZE	RULE_NSIZE
#define LIMIT_NALLOC	RULE_NALLOC

/* Pointer to limit by number. */
#define LIMIT(r, x) ((r)->limits[(x)])

/* Set if configuration file has limit. */
static bool	conf_has_limit;

#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
/*
 * Module's data for threshold{}.
 * If fd < 0, then threshold is not initialized or is inactive.
 */
struct threshold {
	int		fd;		/* Database file descriptor. */
	char		*db_file;	/* Database file for threshold. */
};

static ipa_mzone *threshold_mzone;	/* Mzone for all thresholds. */

#define THRESHOLD_NSIZE	RULE_NSIZE
#define THRESHOLD_NALLOC RULE_NALLOC

/* Pointer to threshold by number. */
#define THRESHOLD(r, x) ((r)->thresholds[(x)])

/* Set if configuration file has threshold. */
static bool	conf_has_threshold;

#endif /* WITH_THRESHOLDS */

#define RULE_FREE_DB_DIR 0x01	/* Need to free db_dir in struct rule{}. */
#define RULE_INITED	 0x02	/* Rule was initialized in init_rule(). */

/*
 * Module's data for rule{}.
 * If fd < 0, then rule is not initialized or is inactive.
 */
struct rule {
	const char	*name;		/* Name of this rule. */
	unsigned int	no;		/* Rule ordinal number. */
#ifdef WITH_LIMITS
	struct limit	**limits;	/* Array of pointers to limits. */
	char		*limits_dir;	/* Database directory for limits. */
	unsigned int	nlimits;	/* Number of limits. */
#endif
#ifdef WITH_THRESHOLDS
	struct threshold **thresholds;	/* Array of pointers to thresholds. */
	char		*thresholds_dir;/* Database directory for thresholds. */
	unsigned int	nthresholds;	/* Number of thresholds. */
#endif
	int		fd;		/* Database file descriptor. */
	int		cur_year;	/* Current year. */
	int		cur_mon;	/* Current month. */
	struct db_group	db_group;	/* rule { sdb:db_group } */
	char		*db_dir;	/* rule { sdb:db_dir } */
	char		*db_file;	/* Database file for rule. */
	unsigned int	ref_count;	/* Reference counter. */
	unsigned char	flags;		/* RULE_xxx flags. */
	signed char	close_fd;	/* rule { sdb:close_fd } */
	TAILQ_ENTRY(rule) link;		/* For list of all rules. */
};

static struct rule **rules_ptr;		/* Array of pointers to rules. */
static unsigned int nstatrules;		/* Number of static rules. */
static unsigned int nrulesalloced;	/* Number of allocated rules. */
static unsigned int nrulesinited;	/* Number of initialized rules. */
static ipa_mzone *rule_mzone;		/* Mzone for all rules. */
static ipa_marray *rules_ptr_marray;	/* Marray for array of ptrs to rules. */

#define RULE(x) rules_ptr[(x)]		/* Pointer to rule by number. */

static TAILQ_HEAD(EMPTY, rule) rules_list; /* List of all rules. */

#ifdef WITH_AUTORULES
/*
 * Module's data for each autorule{}.
 */
struct autorule {
	char		*db_dir;	/* autorule { sdb:db_dir } */
	struct db_group	db_group;	/* autorule { sdb:db_group } */
	signed char	close_fd;	/* autorule { sdb:close_fd } */
};

static struct autorule *curautorule;	/* Current autorule. */
static struct autorule *autorules;	/* Array of autorules. */
static unsigned int nautorules;		/* Number of autorules. */
static ipa_marray *autorules_marray;	/* Marray of autorules. */

#define AUTORULE_NSIZE	10
#define AUTORULE_NALLOC	10

#define AUTORULE(x) &autorules[(x)]	/* Pointer to autorule by number. */

#endif /* WITH_AUTORULES */

#ifdef WITH_RULES
static bool	in_rule;		/* Set if we are in rule{} section. */
static struct rule *currule;		/* Current rule. */
#endif

/* stat() or lstat(), depending on allow_symlinks. */
static int	(*stat_func)(const char *, struct stat *);

/* Name of stat_func: "stat" or "lstat". */
static const char *stat_func_name;

#define cnt_from_base(cnt, c_high, c_low) do {		\
	(cnt) = ntohl(c_high);				\
	(cnt) <<= 32;					\
	(cnt) += ntohl(c_low);				\
} while (/* CONSTCOND */ 0)

#define cnt_to_base(cnt, c_high, c_low) do {		\
	(c_high) = htonl((uint32_t)((cnt) >> 32));	\
	(c_low)  = htonl((uint32_t)(cnt));		\
} while (/* CONSTCOND */ 0)

static void	(*print_param_end)(void);
static void	print_param_args(const char *, ...) ATTR_FORMAT(printf, 1, 2);

static void	logmsg(int, const char *, ...) ATTR_FORMAT(printf, 2, 3);
static void	logmsgx(int, const char *, ...) ATTR_FORMAT(printf, 2, 3);
static void	logconf(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	logconfx(const char *, ...) ATTR_FORMAT(printf, 1, 2);

static int	mem_asprintf(ipa_mem_type *, char **, const char *, ...)
		    ATTR_FORMAT(printf, 3, 4);

/*
 * A wrapper for imported logmsg() function.
 */
static void
logmsg(int priority, const char *format, ...)
{
	va_list ap;

	bt_indent = 0;
	va_start(ap, format);
	suppfunc->logmsg(MOD_NAME, priority, errno, format, ap);
	va_end(ap);
}

/*
 * A wrapper for imported logmsg() function,
 * but error code is always zero here.
 */
static void
logmsgx(int priority, const char *format, ...)
{
	va_list ap;

	bt_indent = 0;
	va_start(ap, format);
	suppfunc->logmsg(MOD_NAME, priority, 0, format, ap);
	va_end(ap);
}

static void
logbt(const char *msg)
{
	int indent;

	indent = bt_indent;
	logmsgx(IPA_LOG_ERR, " %*s`- %s", indent, "", msg);
	bt_indent = indent + 2;
}

/*
 * A wrapper for imported logconferr() function.
 */
static void
logconf(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	suppfunc->logconferr(MOD_NAME, errno, format, ap);
	va_end(ap);
}

/*
 * A wrapper for imported logconferr() function,
 * but error code is always zero here.
 */
static void
logconfx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	suppfunc->logconferr(MOD_NAME, 0, format, ap);
	va_end(ap);
}

/*
 * A wrapper for imported print_param_args() function.
 */
static void
print_param_args(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	suppfunc->print_param_args(format, ap);
	va_end(ap);
}

/*
 * A wrapper for imported print_param_name() function.
 */
static void
print_param_name(const char *name)
{
	suppfunc->print_param_name(MOD_CONF_PREFIX, name);
}

static int
mem_asprintf(ipa_mem_type *mem_type, char **bufp, const char *format, ...)
{
	va_list ap;
	int rv;

	va_start(ap, format);
	rv = memfunc->mem_vasprintf(mem_type, bufp, format, ap);
	va_end(ap);
	return (rv);
}

#ifdef WITH_ANY_LIMITS
/*
 * A wrapper for read(2) syscall.
 */
static ssize_t
readn(int fd, const char *path, void *vptr, size_t n)
{
	char *ptr;
	size_t nleft;
	ssize_t nread;

	ptr = vptr;
	nleft = n;
	for (;;) {
		nread = read(fd, ptr, nleft);
		if (nread < 0) {
			if (errno == EINTR) {
				logmsg(IPA_LOG_WARNING, "readn: read(%s)",
				    path);
				continue;
			} else {
				logmsg(IPA_LOG_ERR, "readn: read(%s)", path);
				return (-1);
			}
		} else if (nread == 0) {
			/* EOF */
			break;
		}
		nleft -= nread;
		if (nleft == 0)
			break;
		ptr += nread;
		logmsgx(IPA_LOG_WARNING, "readn: read: short read from %s",
		    path);
	}
	return (n - nleft);
}
#endif /* WITH_ANY_LIMITS */

/*
 * A wrapper for write(2) system call.
 */
static ssize_t
writen(int fd, const char *path, const void *vptr, size_t n)
{
	const char *ptr;
	size_t nleft;
	ssize_t nwritten;

	ptr = vptr;
	nleft = n;
	for (;;) {
		nwritten = write(fd, ptr, nleft);
		if (nwritten < 0) {
			if (errno == EINTR) {
				logmsg(IPA_LOG_WARNING, "writen: write(%s)",
				    path);
				continue;
			} else {
				logmsg(IPA_LOG_ERR, "writen: write(%s)", path);
				return (-1);
			}
		}
		nleft -= nwritten;
		if (nleft == 0)
			break;
		ptr += nwritten;
		logmsgx(IPA_LOG_WARNING, "writen: write: short write to %s",
		    path);
	}
	return (n - nleft);
}

static struct rule *
alloc_rule(unsigned int ruleno)
{
	struct rule *rule;

	rule = mzone_alloc(rule_mzone);
	if (rule == NULL)
		return (NULL);
	if (marray_alloc(rules_ptr_marray, &ruleno, 1) < 0) {
		mzone_free(rule_mzone, rule);
		return (NULL);
	}

	RULE(ruleno) = rule;
	TAILQ_INSERT_TAIL(&rules_list, rule, link);

	rule->no = ruleno;

#ifdef WITH_LIMITS
	rule->limits = NULL;
	rule->limits_dir = NULL;
	rule->nlimits = 0;
#endif

#ifdef WITH_THRESHOLDS
	rule->thresholds = NULL;
	rule->thresholds_dir = NULL;
	rule->nthresholds = 0;
#endif

	rule->fd = -1;

	/* Means that rule->db_file was not built. */
	rule->cur_year = 0;

	rule->db_group.set = 0;
	rule->close_fd = -1;
	rule->db_dir = rule->db_file = NULL;
	rule->ref_count = 1;
	rule->flags = 0;

	++nrulesalloced;

	return (rule);
}

#ifdef WITH_RULES
static struct rule *
alloc_currule(void)
{
	if (currule == NULL) {
		currule = alloc_rule(nstatrules - 1);
		if (currule == NULL)
			logconfx("alloc_rule failed");
	}
	return (currule);
}
#endif

#ifdef WITH_AUTORULES
static struct autorule *
alloc_autorule(unsigned int autoruleno)
{
	struct autorule *autorule;

	if (marray_alloc(autorules_marray, &autoruleno, 1) < 0) {
		logconfx("alloc_autorule: marray_alloc failed");
		return (NULL);
	}
	autorule = AUTORULE(autoruleno);
	autorule->db_dir = NULL;
	autorule->db_group.set = 0;
	autorule->close_fd = -1;

	return (autorule);
}
#endif

/*
 * Parse "allow_symlinks" parameter.
 */
static int
parse_allow_symlinks(void *arg)
{
	allow_symlinks = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse "check_version" parameter.
 */
static int
parse_check_version(void *arg)
{
	check_version = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse "sdb:" section.
 */
/* ARGSUSED */
static int
parse_config_section(void *arg ATTR_UNUSED)
{
	config_section = true;
	return (0);
}

/*
 * Parse "db_dir" parameter.
 */
static int
parse_db_dir(void *arg)
{
	char **dstp, *str;

	str = *(char **)arg;
	if (*str == '\0') {
		logconfx("argument should be a non-empty string");
		return (-1);
	}
	if (*str != '/') {
		logconfx("directory should be given with absolute path");
		return (-1);
	}

#ifdef WITH_RULES
	if (in_rule) {
		struct rule *rule;

		rule = alloc_currule();
		if (rule == NULL)
			return (-1);
		dstp = &rule->db_dir;
		rule->flags |= RULE_FREE_DB_DIR;
	} else
#endif
#ifdef WITH_AUTORULES
	if (curautorule != NULL)
		dstp = &curautorule->db_dir;
	else
#endif
		dstp = &global_db_dir;

	mem_free(*dstp, m_parser);
	*dstp = str;

	return (0);
}

/*
 * Convert string to 'unsigned int'.
 * The first character in a string should be checked
 * for '-' before calling this function (remember that
 * strtoul() and strtoull() work with negative values).
 */
static int
strto_u_int(unsigned int *result, const char *nptr, char **endptr)
{
	unsigned long val;

	errno = 0;
	val = strtoul(nptr, endptr, 10);
	if (errno != 0) {
		logconf("strtoul");
		return (-1);
	}
#if ULONG_MAX > UINT_MAX
	if (val > UINT_MAX) {
		logconfx("too big value for 'unsigned int'");
		return (-1);
	}
#endif
	*result = (unsigned int)val;
	return (0);
}

/*
 * Parse "db_group" parameter.
 */
static int
parse_db_group(void *arg)
{
	const struct group *grp;
	struct db_group	*db_group;
	const char *group;
	char *endptr;
	unsigned int gid;
	char set;

	group = *(char **)arg;

	errno = 0;
	grp = getgrnam(group);
	if (grp == NULL) {
		if (errno != 0) {
			logconf("getgrnam(%s)", group);
			return (-1);
		}
		if (isdigit((unsigned char)*group) == 0) {
			logconfx("cannot find group \"%s\"", group);
			return (-1);
		}
		if (strto_u_int(&gid, group, &endptr) < 0)
			return (-1);
		if (*endptr != '\0') {
			logconfx("cannot find group \"%s\"", group);
			return (-1);
		}
		set = 1;
	} else {
		gid = grp->gr_gid;
		set = 2;
	}
	endgrent();

#ifdef WITH_RULES
	if (in_rule) {
		struct rule *rule;

		rule = alloc_currule();
		if (rule == NULL)
			return (-1);
		db_group = &rule->db_group;
	} else
#endif
#ifdef WITH_AUTORULES
	if (curautorule != NULL)
		db_group = &curautorule->db_group;
	else
#endif
		db_group = &global_db_group;

	db_group->gid = (uid_t)gid;
	db_group->set = set;
	db_group->perm = DB_RULE_DIR_PERM_UG;

	return (0);
}

/*
 * Parse "close_fd" parameter.
 */
static int
parse_close_fd(void *arg)
{
#ifdef WITH_RULES
	if (in_rule) {
		struct rule *rule;

		rule = alloc_currule();
		if (rule == NULL)
			return (-1);
		rule->close_fd = *(int *)arg;
	} else
#endif
#ifdef WITH_AUTORULES
	if (curautorule != NULL)
		curautorule->close_fd = *(int *)arg;
	else
#endif
		global_close_fd = *(int *)arg;

	return (0);
}

static void
show_param_boolean(const char *name, int boolean)
{
	if (boolean >= 0) {
		print_param_name(name);
		suppfunc->print_boolean(boolean);
		print_param_end();
	}
}

static void
show_param_boolean0(const char *name, int boolean)
{
	if (boolean >= 0) {
		suppfunc->print_param_name((char *)NULL, name);
		suppfunc->print_boolean(boolean);
		print_param_end();
	}
}

static void
show_db_dir(const char *db_dir)
{
	if (db_dir != NULL) {
		print_param_name("db_dir");
		suppfunc->print_string(db_dir);
		print_param_end();
	}
}

static void
show_close_fd(int val)
{
	show_param_boolean("close_fd", val);
}

static void
show_db_group(const struct db_group *db_group)
{
	if (db_group->set) {
		const struct group *grp;

		print_param_name("db_group");
		if (db_group->set == 2) {
			grp = getgrgid(db_group->gid);
			if (grp == NULL)
				print_param_args("%lu",
				    (unsigned long)db_group->gid);
			else
				print_param_args("%s", grp->gr_name);
		} else
			print_param_args("%lu", (unsigned long)db_group->gid);
		print_param_end();
	}
}

static int
conf_init(void)
{
	suppfunc = IPA_DB_MOD_ENTRY(ipa_db_sdb).suppfunc;

	print_param_end = suppfunc->print_param_end;

	memfunc = IPA_DB_MOD_ENTRY(ipa_db_sdb).memfunc;
	if (memfunc->api_ver != IPA_MEMFUNC_API_VERSION) {
		logconfx("module understands memfunc API version %u, "
		    "exported memfunc API version is %u",
		    IPA_MEMFUNC_API_VERSION, memfunc->api_ver);
		return (-1);
	}

	m_parser = memfunc->m_parser;

	m_anon = memfunc->mem_type_new(MTYPE_NAME(anon),
	    "Anonymous memory", 0);
	m_tmp = memfunc->mem_type_new(MTYPE_NAME(tmp),
	    "Temporal memory", 0);
	if (m_anon == NULL || m_tmp == NULL) {
		logconfx("mem_type_new failed");
		return (-1);
	}

	mem_malloc = memfunc->mem_malloc;
	mem_realloc = memfunc->mem_realloc;
	mem_free = memfunc->mem_free;

	marray_init = memfunc->marray_init;
	marray_deinit = memfunc->marray_deinit;
	marray_alloc = memfunc->marray_alloc;
	marray_free = memfunc->marray_free;
	marray_minimize = memfunc->marray_minimize;
	marray_check_index = memfunc->marray_check_index;

	mzone_init = memfunc->mzone_init;
	mzone_deinit = memfunc->mzone_deinit;
	mzone_alloc = memfunc->mzone_alloc;
	mzone_free = memfunc->mzone_free;
	mzone_nused = memfunc->mzone_nused;

	rules_ptr_marray = marray_init(MARRAY_NAME(rules_ptr),
	    "Pointers to rules", 0, (void *)&rules_ptr, sizeof(struct rule *),
	    RULE_NSIZE, RULE_NALLOC);
	if (rules_ptr_marray == NULL) {
		logconfx("marray_init failed");
		return (-1);
	}

#ifdef WITH_AUTORULES
	autorules_marray = marray_init(MARRAY_NAME(autorules),
	    "Autorules", 0, (void *)&autorules, sizeof(struct autorule),
	    AUTORULE_NSIZE, AUTORULE_NALLOC);
	if (autorules_marray == NULL) {
		logconfx("marray_init failed");
		return (-1);
	}
#endif

	rule_mzone = mzone_init(MZONE_NAME(rule),
	    "Rules", MEMFLAG_OPTIMIZE, sizeof(struct rule),
	    RULE_NSIZE, RULE_NALLOC);
	if (rule_mzone == NULL) {
		logconfx("mzone_init failed");
		return (-1);
	}

	nstatrules = nrulesalloced = nrulesinited = 0;

#ifdef WITH_RULES
	in_rule = false;
#endif

#ifdef WITH_AUTORULES
	nautorules = 0;
	curautorule = NULL;
#endif

	global_db_dir = NULL;
	global_close_fd = allow_symlinks = check_version = -1;
	config_section = false;
	global_db_group.set = 0;

#ifdef WITH_LIMITS
	conf_has_limit = false;
#endif
#ifdef WITH_THRESHOLDS
	conf_has_threshold = false;
#endif

	TAILQ_INIT(&rules_list);

	return (0);
}

static int
conf_deinit(void)
{
	return (0);
}

static void
conf_real(void)
{
#ifdef WITH_AUTORULES
	struct autorule *autorule;
	unsigned int i;
#endif

	if (global_db_dir == NULL)
		global_db_dir = db_dir_default;
	if (global_close_fd < 0)
		global_close_fd = 0;
	if (allow_symlinks < 0)
		allow_symlinks = 0;
	if (check_version < 0)
		check_version = 1;

#ifdef WITH_AUTORULES
	for (i = 0, autorule = autorules; i < nautorules; ++autorule, ++i) {
		if (autorule->db_dir == NULL)
			autorule->db_dir = global_db_dir;
		if (!autorule->db_group.set)
			autorule->db_group = global_db_group;
		if (autorule->close_fd < 0)
			autorule->close_fd = global_close_fd;
	}
#endif
}

static int
conf_mimic_real(void)
{
#ifdef WITH_RULES
	struct rule *rule;
#endif

	conf_real();

	config_section = true;

#ifdef WITH_RULES
	TAILQ_FOREACH(rule, &rules_list, link) {
		if (rule->db_dir == NULL)
			rule->db_dir = global_db_dir;
		if (!rule->db_group.set)
			rule->db_group = global_db_group;
		if (rule->close_fd < 0)
			rule->close_fd = global_close_fd;
	}
#endif

	return (0);
}

static void
conf_show(unsigned int sect_id, unsigned int no)
{
	switch (sect_id) {
	case IPA_CONF_SECT_ROOT:
		if (config_section) {
			suppfunc->print_sect_name(MOD_DB_NAME, "");
			suppfunc->print_sect_begin();
#if IPA_DB_MOD_API_VERSION == 1
			suppfunc->set_indent(1);
#endif
			show_param_boolean0("allow_symlinks", allow_symlinks);
			show_param_boolean0("check_version", check_version);
#if IPA_DB_MOD_API_VERSION == 1
			suppfunc->set_indent(0);
#endif
			suppfunc->print_sect_end();
		}
		break;
	case IPA_CONF_SECT_GLOBAL:
		show_db_dir(global_db_dir);
		show_db_group(&global_db_group);
		show_close_fd(global_close_fd);
		break;
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		if (marray_check_index(rules_ptr_marray, no)) {
			const struct rule *rule;

			rule = RULE(no);
			show_db_dir(rule->db_dir);
			show_db_group(&rule->db_group);
			show_close_fd(rule->close_fd);
		}
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_SECT_AUTORULE:
		{
			const struct autorule *autorule;

			autorule = AUTORULE(no);
			show_db_dir(autorule->db_dir);
			show_db_group(&autorule->db_group);
			show_close_fd(autorule->close_fd);
		}
		break;
#endif
	}
}

static int
conf_event(unsigned int event, unsigned int no, const void *arg ATTR_UNUSED)
{
	switch (event) {
#ifdef WITH_RULES
	case IPA_CONF_EVENT_RULE_BEGIN:
		nstatrules = no + 1;
		in_rule = true;
		currule = NULL;
		break;
	case IPA_CONF_EVENT_RULE_END:
		in_rule = false;
		break;
#endif
#ifdef WITH_AUTORULES
	case IPA_CONF_EVENT_AUTORULE_BEGIN:
		curautorule = alloc_autorule(no);
		if (curautorule == NULL)
			return (-1);
		nautorules = no + 1;
		break;
	case IPA_CONF_EVENT_AUTORULE_END:
		curautorule = NULL;
		break;
#endif
#ifdef WITH_LIMITS
	case IPA_CONF_EVENT_LIMIT_BEGIN:
		conf_has_limit = true;
		break;
#endif
#ifdef WITH_THRESHOLDS
	case IPA_CONF_EVENT_THRESHOLD_BEGIN:
		conf_has_threshold = true;
		break;
#endif
	}
	return (0);
}

static int
create_db_file(const char *path)
{
	int fd;

	fd = open(path, O_RDWR|O_CREAT, DB_FILE_PERM);
	if (fd < 0) {
		logmsg(IPA_LOG_ERR, "create_db_file: "
		    "open(%s, O_RDWR|O_CREAT, 0%03o)", path, DB_FILE_PERM);
		return (-1);
	}
	if (fchown(fd, proc_uid, proc_gid) < 0) {
		logmsg(IPA_LOG_ERR, "create_db_file: fchown(%s, %lu, %lu)",
		    path, (unsigned long)proc_uid, (unsigned long)proc_gid);
		return (-1);
	}
	return (fd);
}

/*
 * Create or unlink info file in the database.
 */
static int
init_info_file(const char *db_dir, const char *info)
{
	char *path;
	size_t len;
	int fd;

	if (mem_asprintf(m_tmp, &path, "%s/"IPA_SDB_INFO_FILE, db_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "init_info_file: mem_asprintf failed");
		return (-1);
	}

	if (info != NULL) {
		fd = create_db_file(path);
		if (fd < 0) {
			logbt("init_info_file");
			goto failed;
		}
		len = strlen(info);
		if (writen(fd, path, info, len) != len) {
			logbt("init_info_file");
			goto failed_close;
		}
		if (ftruncate(fd, (off_t)len) < 0) {
			logmsg(IPA_LOG_ERR, "init_info_file: "
			    "ftruncate(%s, %lu)", path, (unsigned long)len);
			goto failed_close;
		}
		if (close(fd) < 0) {
			logmsg(IPA_LOG_ERR, "init_info_file: close(%s)", path);
			goto failed;
		}
	} else {
		if (unlink(path) < 0)
			if (errno != ENOENT) {
				logmsg(IPA_LOG_ERR, "init_info_file: "
				    "unlink(%s)", path);
				goto failed;
			}
	}

	mem_free(path, m_tmp);
	return (0);

failed_close:
	if (close(fd) < 0)
		logmsg(IPA_LOG_ERR, "init_info_file: close(%s)", path);
failed:
	mem_free(path, m_tmp);
	return (-1);
}

#ifdef WITH_ANY_LIMITS
static int
create_db_dir(const char *db_dir)
{
	if (mkdir(db_dir, DB_DIR_PERM) < 0) {
		logmsg(IPA_LOG_ERR, "create_db_dir: mkdir(%s, 0%03o)",
		    db_dir, DB_DIR_PERM);
		return (-1);
	}
	if (chown(db_dir, proc_uid, proc_gid) < 0) {
		logmsg(IPA_LOG_ERR, "create_db_dir: chown(%s, %lu, %lu)",
		    db_dir, (unsigned long)proc_uid, (unsigned long)proc_gid);
		return (-1);
	}
	return (0);
}
#endif /* WITH_ANY_LIMITS */

static int
check_dir(const char *path, const struct stat *statbuf)
{
	if (!S_ISDIR(statbuf->st_mode)) {
		logmsgx(IPA_LOG_ERR, "check_dir: %s: is expected to be "
		    "a directory", path);
		return (-1);
	}
	return (0);
}

static int
check_file(const char *path, const struct stat *statbuf)
{
	if (!S_ISREG(statbuf->st_mode)) {
		logmsgx(IPA_LOG_ERR, "check_file: %s: is expected to be "
		    "a regular file", path);
		return (-1);
	}
	return (0);
}

static int
check_db_file(const char *db_file, const struct stat *statbuf)
{
	if (check_file(db_file, statbuf) < 0) {
		logbt("check_db_file");
		return (-1);
	}
	if (statbuf->st_uid != proc_uid) {
		logmsgx(IPA_LOG_ERR, "check_db_file: file %s: file's UID %lu, "
		    "process's UID %lu: wrong owner of the file", db_file,
		    (unsigned long)statbuf->st_uid, (unsigned long)proc_uid);
		return (-1);
	}
	if (statbuf->st_mode & (S_IWGRP|S_IWOTH)) {
		logmsgx(IPA_LOG_ERR, "check_db_file: file %s: should not be "
		    "writable by group or other users", db_file);
		return (-1);
	}
	return (0);
}

static int
check_db_dir(const char *db_dir)
{
	struct stat statbuf;
	FILE *fp;
	char *path;
	bool newbase;

	if (db_dir == global_db_dir && global_db_dir_checked)
		return (0);

	/* Create and check main database directory. */
	if (stat_func(db_dir, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "check_db_dir: %s(%s)",
			    stat_func_name, db_dir);
			return (-1);
		}
		if (mkdir(db_dir, DB_MAIN_DIR_PERM) < 0) {
			logmsg(IPA_LOG_ERR, "check_db_dir: mkdir(%s, 0%03o)",
			    db_dir, DB_MAIN_DIR_PERM);
			return (-1);
		}
		logmsgx(IPA_LOG_INFO, "database directory %s was created",
		    db_dir);
		newbase = true;
	} else {
		if (check_dir(db_dir, &statbuf) < 0) {
			logbt("check_db_dir");
			return (-1);
		}
		if (statbuf.st_uid != proc_uid) {
			logmsgx(IPA_LOG_ERR, "check_db_dir: directory %s: "
			    "directory's UID %lu, process UID %lu: wrong "
			    "owner of the directory", db_dir,
			    (unsigned long)statbuf.st_uid,
			    (unsigned long)proc_uid);
			return (-1);
		}
		if (statbuf.st_mode & (S_IWGRP|S_IWOTH)) {
			logmsgx(IPA_LOG_ERR, "check_db_dir: directory %s: "
			    "should not be writable by group and other users",
			    db_dir);
			return (-1);
		}
		newbase = false;
	}

	if (check_version) {
		/* Create or check database format version file. */
		if (mem_asprintf(m_tmp, &path, "%s/"IPA_SDB_VERSION_FILE,
		    db_dir) < 0) {
			logmsgx(IPA_LOG_ERR, "check_db_dir: "
			    "mem_asprintf failed");
			return (-1);
		}
		if (stat_func(path, &statbuf) < 0) {
			int nwritten;

			if (errno != ENOENT) {
				logmsg(IPA_LOG_ERR, "check_db_dir: %s(%s)",
				    stat_func_name, path);
				goto failed;
			}
			if (!newbase)
				logmsgx(IPA_LOG_WARNING, "check_db_dir: "
				    "file %s does not exist, cannot check "
				    "format version of %s database",
				    path, db_dir);
			fp = fopen(path, "w");
			if (fp == NULL) {
				logmsg(IPA_LOG_ERR, "check_db_dir: "
				    "fopen(%s, \"w\")", path);
				goto failed;
			}
			nwritten = fprintf(fp, "%u", IPA_SDB_FORMAT_VERSION);
			if (nwritten < 0) {
				logmsg(IPA_LOG_ERR, "check_db_dir: "
				    "fprintf(%s)", path);
				goto failed_close;
			}
			if (nwritten != IPA_SDB_FORMAT_VERSION_LEN) {
				logmsgx(IPA_LOG_ERR, "check_db_dir: "
				    "fprintf(%s): short write, written %d "
				    "of %u bytes", path, nwritten,
				    IPA_SDB_FORMAT_VERSION_LEN);
				goto failed_close;
			}
		} else {
			unsigned int version;

			if (check_db_file(path, &statbuf) < 0) {
				logbt("check_db_dir");
				goto failed;
			}
			fp = fopen(path, "r");
			if (fp == NULL) {
				logmsg(IPA_LOG_ERR, "check_db_dir: "
				    "fopen(%s, \"r\")", path);
				goto failed;
			}
			if (fscanf(fp, "%u", &version) != 1) {
				logmsg(IPA_LOG_ERR, "check_db_dir: "
				    "fscanf(%s, \"%%u\") failed", path);
				goto failed_close;
			}
			if (version != IPA_SDB_FORMAT_VERSION) {
				logmsgx(IPA_LOG_ERR, "check_db_dir: formar "
				    "version of %s database is %u, my format "
				    "version is %u", db_dir, version,
				    IPA_SDB_FORMAT_VERSION);
				goto failed_close;
			}
		}
		if (fclose(fp) != 0) {
			logmsg(IPA_LOG_ERR, "check_db_dir: fclose(%s)", path);
			goto failed;
		}
		mem_free(path, m_tmp);
	}
	if (db_dir == global_db_dir)
		global_db_dir_checked = true;
	return (0);

failed_close:
	if (fclose(fp) != 0)
		logmsg(IPA_LOG_ERR, "check_db_dir: fclose(%s)", path);
failed:
	mem_free(path, m_tmp);
	return (-1);
}

/*
 * General initialization of the database.
 */
static int
db_pre_init(void)
{
#ifdef WITH_AUTORULES
	const struct autorule *autorule;
	unsigned int i;
#endif

	logmsgx(IPA_LOG_INFO, "module version "PACKAGE_VERSION);

	proc_uid = getuid();
	proc_gid = getgid();

	if (!global_db_group.set) {
		global_db_group.gid = proc_gid;
		global_db_group.perm = DB_RULE_DIR_PERM_U;
	}

	conf_real();

	if (allow_symlinks > 0) {
		stat_func = stat;
		stat_func_name = "stat";
	} else {
		stat_func = lstat;
		stat_func_name = "lstat";
	}

	global_db_dir_checked = false;
	if (check_db_dir(global_db_dir) < 0) {
		logbt("db_pre_init");
		return (-1);
	}

#ifdef WITH_AUTORULES
	for (i = 0, autorule = autorules; i < nautorules; ++autorule, ++i)
		if (check_db_dir(autorule->db_dir) < 0) {
			logbt("db_pre_init");
			return (-1);
		}

	/* Try to minimize autorules_marray. */
	if (nautorules != 0)
		marray_minimize(autorules_marray);
	else {
		marray_deinit(autorules_marray);
		autorules_marray = NULL;
	}
#endif

	/* Try to minimize rules_ptr_marray. */
	if (nstatrules != 0)
		marray_minimize(rules_ptr_marray);

	/*
	 * Heuristic decision, if configuration file has limits or
	 * thresholds, then it is very likely that this database will
	 * be used for them, another variant is to check limit_mzone
	 * and threshold_mzone in each db_init_*() call.
	 */
#ifdef WITH_LIMITS
	if (conf_has_limit) {
		limit_mzone = mzone_init(MZONE_NAME(limit), "Limits",
		    MEMFLAG_OPTIMIZE, sizeof(struct limit), LIMIT_NSIZE,
		    LIMIT_NALLOC);
		if (limit_mzone == NULL) {
			logmsgx(IPA_LOG_ERR, "db_pre_init: mzone_init failed");
			return (-1);
		}
	} else
		limit_mzone = NULL;
#endif

#ifdef WITH_THRESHOLDS
	if (conf_has_threshold) {
		threshold_mzone = mzone_init(MZONE_NAME(threshold),
		    "Thresholds", MEMFLAG_OPTIMIZE, sizeof(struct threshold),
		    THRESHOLD_NSIZE, THRESHOLD_NALLOC);
		if (threshold_mzone == NULL) {
			logmsgx(IPA_LOG_ERR, "db_pre_init: mzone_init failed");
			return (-1);
		}
	} else
		threshold_mzone = NULL;
#endif

	return (0);
}

static int
db_init(void)
{
#ifdef WITH_RULES
	if (nrulesalloced != nrulesinited) {
		const struct rule *rule;

		TAILQ_FOREACH(rule, &rules_list, link)
			if (!(rule->flags & RULE_INITED))
				logmsgx(IPA_LOG_WARNING, "rule %s: this rule "
				    "has some module's parameter, but it does "
				    "not use my database", rule->name);
	}
#endif
	return (0);
}

static void
free_rule(struct rule *rule)
{
	rule->ref_count--;
	if (rule->ref_count == 0) {
		TAILQ_REMOVE(&rules_list, rule, link);
		if (rule->flags & RULE_FREE_DB_DIR)
			mem_free(rule->db_dir, m_parser);
#ifdef WITH_LIMITS
		mem_free(rule->limits, m_anon);
		mem_free(rule->limits_dir, m_anon);
#endif
#ifdef WITH_THRESHOLDS
		mem_free(rule->thresholds, m_anon);
		mem_free(rule->thresholds_dir, m_anon);
#endif
		mem_free(rule->db_file, m_anon);
		marray_free(rules_ptr_marray, rule->no);
		mzone_free(rule_mzone, rule);
		--nrulesalloced;
		--nrulesinited;
	}
}

#ifdef WITH_RULES
/*
 * The same as free_rule(), but free only those memory which
 * could be allocated during configuration phase.
 */
static void
free_rule_fast(struct rule *rule)
{
	rule->ref_count--;
	if (rule->ref_count == 0) {
		if (rule->flags & RULE_FREE_DB_DIR)
			mem_free(rule->db_dir, m_parser);
		marray_free(rules_ptr_marray, rule->no);
		mzone_free(rule_mzone, rule);
		--nrulesalloced;
	}
}
#endif

static int
init_rule(unsigned int ruleno, const char *rule_name, const char *rule_info)
{
	struct stat statbuf;
	struct rule *rule;
	char *rule_dir;
	mode_t perm;

	rule = RULE(ruleno);
	rule->name = rule_name;
	rule_dir = NULL;

	if (mem_asprintf(m_anon, &rule_dir, "%s/%s", rule->db_dir,
	    rule_name) < 0) {
		logmsgx(IPA_LOG_ERR, "init_rule: mem_asprintf failed");
		goto failed;
	}

	if (stat_func(rule_dir, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "init_rule: %s(%s)",
			    stat_func_name, rule_dir);
			goto failed;
		}
		if (mkdir(rule_dir, rule->db_group.perm) < 0) {
			logmsg(IPA_LOG_ERR, "init_rule: mkdir(%s, 0%03o)",
			    rule_dir, (unsigned int)rule->db_group.perm);
			goto failed;
		}
		if (chown(rule_dir, proc_uid, rule->db_group.gid) < 0) {
			logmsg(IPA_LOG_ERR, "init_rule: chown(%s, %lu, %lu)",
			    rule_dir, (unsigned long)proc_uid,
			    (unsigned long)rule->db_group.gid);
			goto failed;
		}
		logmsgx(IPA_LOG_INFO, "rule database directory %s was created",
		    rule_dir);
	} else {
		if (check_dir(rule_dir, &statbuf) < 0)
			goto failed;
		if (statbuf.st_uid != proc_uid) {
			logmsgx(IPA_LOG_ERR, "init_rule: directory %s: "
			    "directory's UID %lu, process's UID %lu: wrong "
			    "owner of the directory", rule_dir,
			    (unsigned long)statbuf.st_uid,
			    (unsigned long)proc_uid);
			goto failed;
		}
		perm = PERM_MASK(statbuf.st_mode);
		if (perm & (S_IWGRP|S_IWOTH)) {
			logmsgx(IPA_LOG_ERR, "init_rule: directory %s: should "
			    "not be writable by group and other users",
			    rule_dir);
			goto failed;
		}
		if (statbuf.st_gid != rule->db_group.gid) {
			logmsgx(IPA_LOG_WARNING, "init_rule: directory %s has "
			    "incorrect GID %lu, changing GID to %lu",
			    rule_dir, (unsigned long)statbuf.st_gid,
			    (unsigned long)rule->db_group.gid);
			if (chown(rule_dir, proc_uid, rule->db_group.gid) < 0) {
				logmsg(IPA_LOG_ERR, "init_rule: "
				    "chown(%s, %lu, %lu)", rule_dir,
				    (unsigned long)proc_uid,
				    (unsigned long)rule->db_group.gid);
				goto failed;
			}
		}
		if (perm != rule->db_group.perm) {
			logmsgx(IPA_LOG_WARNING, "init_rule: directory %s has "
			    "wrong mode 0%03o, changing mode to 0%03o",
			    rule_dir, (unsigned int)perm,
			    (unsigned int)rule->db_group.perm);
			if (chmod(rule_dir, rule->db_group.perm) < 0) {
				logmsg(IPA_LOG_ERR, "init_rule: "
				    "chmod(%s, 0%03o)", rule_dir,
				    (unsigned int)rule->db_group.perm);
				goto failed;
			}
		}
	}

	if (init_info_file(rule_dir, rule_info) < 0) {
		logbt("init_rule");
		goto failed;
	}

#ifdef WITH_LIMITS
	if (mem_asprintf(m_anon, &rule->limits_dir,
	    "%s/"IPA_SDB_LIMITS_DIR, rule_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "init_rule: mem_asprintf failed");
		goto failed;
	}
#endif

#ifdef WITH_THRESHOLDS
	if (mem_asprintf(m_anon, &rule->thresholds_dir,
	    "%s/"IPA_SDB_THRESHOLDS_DIR, rule_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "init_rule: mem_asprintf failed");
		goto failed;
	}
#endif

	rule->flags |= RULE_INITED;
	++nrulesinited;
	mem_free(rule_dir, m_anon);
	return (0);

failed:
	free_rule(rule);
	mem_free(rule_dir, m_anon);
	return (-1);
}

#ifdef WITH_RULES
/*
 * Initialize the database for a static rule.
 */
static int
db_init_statrule(unsigned int ruleno, const char *rule_name,
    const char *rule_info)
{
	struct rule *rule;

	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		rule = alloc_rule(ruleno);
		if (rule == NULL) {
			logmsgx(IPA_LOG_ERR, "db_init_statrule: "
			    "alloc_rule failed");
			return (-1);
		}
	} else
		rule = RULE(ruleno);

	/* Inherit settings from global{}. */
	if (rule->db_dir == NULL) {
		rule->db_dir = global_db_dir;
		rule->flags &= ~RULE_FREE_DB_DIR;
	} else if (check_db_dir(rule->db_dir) < 0)
		goto failed;

	if (!rule->db_group.set)
		rule->db_group = global_db_group;
	if (rule->close_fd < 0)
		rule->close_fd = global_close_fd;

	if (init_rule(ruleno, rule_name, rule_info) < 0)
		goto failed;
	return (0);

failed:
	logbt("db_init_statrule");
	return (-1);
}
#else
# define db_init_statrule NULL
#endif /* WITH_RULES */

#ifdef WITH_AUTORULES
/*
 * Initialize the database for a dynamic rule.
 */
static int
db_init_dynrule(unsigned int autoruleno, unsigned int ruleno,
    const char *rule_name, const char *rule_info)
{
	const struct autorule *autorule;
	struct rule *rule;

	rule = alloc_rule(ruleno);
	if (rule == NULL) {
		logmsgx(IPA_LOG_ERR, "db_init_dynrule: alloc_rule failed");
		return (-1);
	}

	autorule = AUTORULE(autoruleno);

	/* Inherit settings from autorule. */
	rule->db_dir = autorule->db_dir;
	rule->flags &= ~RULE_FREE_DB_DIR;
	rule->db_group = autorule->db_group;
	rule->close_fd = autorule->close_fd;

	if (init_rule(ruleno, rule_name, rule_info) < 0) {
		logbt("db_init_dynrule");
		return (-1);
	}
	return (0);
}
#else
# define db_init_dynrule NULL
#endif /* WITH_AUTORULES */

#ifdef WITH_LIMITS
static void
free_limit(struct rule *rule, struct limit *limit, unsigned int limitno)
{
	mem_free(limit->db_file, m_anon);
	mem_free(limit->db_dir, m_anon);
	LIMIT(rule, limitno) = NULL;
	mzone_free(limit_mzone, limit);
	free_rule(rule);
}

static int
init_limit(struct rule *rule, unsigned int limitno,
    const char *limit_name, const char *limit_info)
{
	struct stat statbuf;
	struct limit *limit, **limits;

	if (rule->nlimits == 0) {
		/* Check the directory for limits. */
		if (stat_func(rule->limits_dir, &statbuf) < 0) {
			if (errno != ENOENT) {
				logmsg(IPA_LOG_ERR, "init_limit: %s(%s)",
				    stat_func_name, rule->limits_dir);
				goto failed;
			}
			if (create_db_dir(rule->limits_dir) < 0) {
				logbt("init_limit");
				goto failed;
			}
		} else if (check_dir(rule->limits_dir, &statbuf) < 0) {
			logbt("init_limit");
			goto failed;
		}
	}

	if (rule->nlimits >= limitno + 1) {
		logmsgx(IPA_LOG_ERR, "init_limit: unexpected ordinal "
		    "number %u of limit", limitno);
		goto failed;
	}

	limits = mem_realloc(rule->limits, (limitno + 1) *
	    sizeof(*rule->limits), m_anon);
	if (limits == NULL) {
		logmsgx(IPA_LOG_ERR, "init_limit: mem_realloc failed");
		goto failed;
	}
	rule->limits = limits;
	rule->nlimits = limitno + 1;

	limit = mzone_alloc(limit_mzone);
	if (limit == NULL) {
		logmsgx(IPA_LOG_ERR, "init_limit: mzone_alloc failed");
		LIMIT(rule, limitno) = NULL;
		goto failed;
	}
	LIMIT(rule, limitno) = limit;
	limit->db_file = limit->db_dir = NULL;

	if (mem_asprintf(m_anon, &limit->db_dir, "%s/%s", rule->limits_dir,
	    limit_name) < 0) {
		logmsgx(IPA_LOG_ERR, "init_limit: mem_asprintf failed");
		goto limit_failed;
	}

	if (stat_func(limit->db_dir, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "init_limit: %s(%s)",
			    stat_func_name, limit->db_dir);
			goto limit_failed;
		}
		if (create_db_dir(limit->db_dir) < 0)
			goto limit_failed_bt;
	} else if (check_dir(limit->db_dir, &statbuf) < 0)
		goto limit_failed_bt;

	if (init_info_file(limit->db_dir, limit_info) < 0)
		goto limit_failed_bt;

	if (mem_asprintf(m_anon, &limit->db_file, "%s/"IPA_SDB_LIMIT_CURRENT,
	    limit->db_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "init_limit: mem_asprintf failed");
		goto limit_failed;
	}

	limit->fd = -1;

	/* Means "current" was not linked to database file. */
	limit->start_year = 0;

	return (0);

limit_failed_bt:
	logbt("init_limit");
limit_failed:
	free_limit(rule, limit, limitno);
	return (-1);

failed:
	free_rule(rule);
	return (-1);
}
#endif /* WITH_LIMITS */

#if defined(WITH_LIMITS) && defined(WITH_RULES)
/*
 * Initialize the database for a static limit.
 */
static int
db_init_statlimit(unsigned int ruleno, const char *rule_name,
    const char *rule_info, unsigned int limitno, const char *limit_name,
    const char *limit_info)
{
	struct rule *rule;

	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		if (db_init_statrule(ruleno, rule_name, rule_info) < 0)
			goto failed;
		rule = RULE(ruleno);
	} else {
		rule = RULE(ruleno);
		rule->ref_count++;
	}

	if (init_limit(rule, limitno, limit_name, limit_info) < 0)
		goto failed;

	return (0);

failed:
	logbt("db_init_statlimit");
	return (-1);
}
#else
# define db_init_statlimit NULL
#endif /* WITH_LIMITS && WITH_RULES */

#if defined(WITH_LIMITS) && defined(WITH_AUTORULES)
/*
 * Initialize the database for a dynamic limit.
 */
static int
db_init_dynlimit(unsigned int autoruleno, unsigned int ruleno,
    const char *rule_name, const char *rule_info, unsigned int limitno,
    const char *limit_name, const char *limit_info)
{
	struct rule *rule;

	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		if (db_init_dynrule(autoruleno, ruleno, rule_name,
		    rule_info) < 0)
			goto failed;
		rule = RULE(ruleno);
	} else {
		rule = RULE(ruleno);
		rule->ref_count++;
	}

	if (init_limit(rule, limitno, limit_name, limit_info) < 0)
		goto failed;
	return (0);

failed:
	logbt("db_init_dynlimit");
	return (-1);
}
#else
# define db_init_dynlimit NULL
#endif /* WITH_LIMITS && WITH_AUTORULES */

#ifdef WITH_THRESHOLDS
static void
free_threshold(struct rule *rule, struct threshold *threshold,
    unsigned int thresholdno)
{
	mem_free(threshold->db_file, m_anon);
	THRESHOLD(rule, thresholdno) = NULL;
	mzone_free(threshold_mzone, threshold);
	free_rule(rule);
}

static int
init_threshold(struct rule *rule, unsigned int thresholdno,
    const char *threshold_name, const char *threshold_info)
{
	struct stat statbuf;
	const char *db_file;
	struct threshold *threshold, **thresholds;
	char *db_dir;

	db_dir = NULL;
	if (rule->nthresholds == 0) {
		/* Check the directory for thresholds. */
		if (stat_func(rule->thresholds_dir, &statbuf) < 0) {
			if (errno != ENOENT) {
				logmsg(IPA_LOG_ERR, "init_threshold: %s(%s)",
				    stat_func_name, rule->thresholds_dir);
				goto failed;
			}
			if (create_db_dir(rule->thresholds_dir) < 0) {
				logbt("init_threshold");
				goto failed;
			}
		} else if (check_dir(rule->thresholds_dir, &statbuf) < 0) {
			logbt("init_threshold");
			goto failed;
		}
	}

	if (rule->nthresholds >= thresholdno + 1) {
		logmsgx(IPA_LOG_ERR, "init_threshold: unexpected ordinal "
		    "number %u of threshold", thresholdno);
		goto failed;
	}
	thresholds = mem_realloc(rule->thresholds, (thresholdno + 1) *
	    sizeof(*rule->thresholds), m_anon);
	if (thresholds == NULL) {
		logmsgx(IPA_LOG_ERR, "init_threshold: mem_realloc failed");
		goto failed;
	}
	rule->thresholds = thresholds;
	rule->nthresholds = thresholdno + 1;

	threshold = mzone_alloc(threshold_mzone);
	if (threshold == NULL) {
		logmsgx(IPA_LOG_ERR, "init_threshold: mzone_alloc failed");
		THRESHOLD(rule, thresholdno) = NULL;
		goto failed;
	}
	THRESHOLD(rule, thresholdno) = threshold;
	threshold->db_file = NULL;

	if (mem_asprintf(m_anon, &db_dir, "%s/%s", rule->thresholds_dir,
	    threshold_name) < 0) {
		logmsgx(IPA_LOG_ERR, "init_threshold: mem_asprintf failed");
		goto threshold_failed;
	}

	if (stat_func(db_dir, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "init_threshold: %s(%s)",
			    stat_func_name, db_dir);
			goto threshold_failed;
		}
		if (create_db_dir(db_dir) < 0)
			goto threshold_failed_bt;
	} else if (check_dir(db_dir, &statbuf) < 0)
		goto threshold_failed_bt;

	if (init_info_file(db_dir, threshold_info) < 0)
		goto threshold_failed_bt;

	if (mem_asprintf(m_anon, &threshold->db_file,
	    "%s/"IPA_SDB_THRESHOLD_STATE, db_dir) < 0) {
		logmsgx(IPA_LOG_ERR, "init_threshold: mem_asprintf failed");
		goto threshold_failed;
	}
	db_file = threshold->db_file;

	threshold->fd = create_db_file(db_file);
	if (threshold->fd < 0)
		goto threshold_failed_bt;

	if (rule->close_fd && close(threshold->fd) < 0) {
		logmsg(IPA_LOG_INFO, "init_threshold: close(%s)", db_file);
		goto failed;
	}
	mem_free(db_dir, m_anon);

	return (0);

threshold_failed_bt:
	logbt("init_threshold");
threshold_failed:
	free_threshold(rule, threshold, thresholdno);
	mem_free(db_dir, m_anon);
	return (-1);

failed:
	free_rule(rule);
	return (-1);
}
#endif /* WITH_THRESHOLDS */

#if defined(WITH_THRESHOLDS) && defined(WITH_RULES)
/*
 * Initialize the database for a static threshold.
 */
static int
db_init_statthreshold(unsigned int ruleno, const char *rule_name,
    const char *rule_info, unsigned int thresholdno,
    const char *threshold_name, const char *threshold_info)
{
	struct rule *rule;

	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		if (db_init_statrule(ruleno, rule_name, rule_info) < 0)
			goto failed;
		rule = RULE(ruleno);
	} else {
		rule = RULE(ruleno);
		rule->ref_count++;
	}

	if (init_threshold(rule, thresholdno, threshold_name,
	    threshold_info) < 0)
		goto failed;
	return (0);

failed:
	logbt("db_init_statthreshold");
	return (-1);
}
#else
# define db_init_statthreshold NULL
#endif /* WITH_THRESHOLDS && WITH_RULES */

#if defined(WITH_THRESHOLDS) && defined(WITH_AUTORULES)
/*
 * Initialize the database for a dynamic threshold.
 */
static int
db_init_dynthreshold(unsigned int autoruleno, unsigned int ruleno,
    const char *rule_name, const char *rule_info, unsigned int thresholdno,
    const char *threshold_name, const char *threshold_info)
{
	struct rule *rule;

	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		if (db_init_dynrule(autoruleno, ruleno, rule_name,
		    rule_info) < 0)
			goto failed;
		rule = RULE(ruleno);
	} else {
		rule = RULE(ruleno);
		rule->ref_count++;
	}

	if (init_threshold(rule, thresholdno, threshold_name,
	    threshold_info) < 0)
		goto failed;
	return (0);

failed:
	logbt("db_init_dynthreshold");
	return (-1);
}
#else
# define db_init_dynthreshold NULL
#endif /* WITH_THRESHOLDS && WITH_AUTORULES */

#ifdef WITH_ANY_LIMITS
/*
 * Convert machine independent ipa_sdb_date{} values to ipa_tm{}.
 */
static void
copy_db_date_to_tm(const ipa_sdb_date *date, ipa_tm *tm)
{
	tm->tm_year = ntohs(date->year);
	tm->tm_mon  = date->mon;
	tm->tm_mday = date->mday;
	tm->tm_hour = date->hour;
	tm->tm_min  = date->min;
	tm->tm_sec  = date->sec;
}

/* Set if cannot save year in database format. */
static char	year_too_big = false;

/*
 * Convert ipa_tm{} values to machine independent ipa_sdb_date{}.
 */
static int
copy_tm_to_db_date(const ipa_tm *tm, ipa_sdb_date *date)
{
	if (tm->tm_year > UINT16_MAX) {
		year_too_big = true;
		logmsgx(IPA_LOG_ERR, "copy_tm_to_db_date: too big value %d "
		    "for year in date", tm->tm_year);
		return (-1);
	}
	date->year = htons((uint16_t)tm->tm_year);
	date->mon  = (uint8_t)tm->tm_mon;
	date->mday = (uint8_t)tm->tm_mday;
	date->hour = (uint8_t)tm->tm_hour;
	date->min  = (uint8_t)tm->tm_min;
	date->sec  = (uint8_t)tm->tm_sec;
	return (0);
}
#endif /* WITH_ANY_LIMITS */

#ifdef WITH_LIMITS
/*
 * Return limit state.
 */
static int
db_get_limit_state(unsigned int ruleno, unsigned int limitno,
    struct ipa_limit_state *state)
{
	struct ipa_sdb_limit_record rec;
	struct stat statbuf;
	const struct rule *rule;
	const char *db_file;
	struct limit *limit;
	int fd;

	rule = RULE(ruleno);
	limit = LIMIT(rule, limitno);
	db_file = limit->db_file;

	if (stat_func(db_file, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "db_get_limit_state: %s(%s)",
			    stat_func_name, db_file);
			return (-1);
		}
#if IPA_DB_MOD_API_VERSION == 1
		state->lim = 0;
#endif
		return (0);
	}

	if (check_file(db_file, &statbuf) < 0) {
		logbt("db_get_limit_state");
		return (-1);
	}
	if (statbuf.st_size == 0) {
#if IPA_DB_MOD_API_VERSION == 1
		state->lim = 0;
#endif
		return (0);
	}
	if (statbuf.st_size % sizeof(rec) != 0) {
		logmsgx(IPA_LOG_ERR, "db_get_limit_state: wrong size of %s",
		    db_file);
		return (-1);
	}

	if (rule->close_fd || limit->fd < 0) {
		limit->fd = open(db_file, O_RDWR);
		if (limit->fd < 0) {
			logmsg(IPA_LOG_ERR, "db_get_limit_state: "
			    "open(%s, O_RDWR)", db_file);
			return (-1);
		}
	}
	fd = limit->fd;
	if (lseek(fd, -(off_t)sizeof(rec), SEEK_END) == (off_t)-1) {
		logmsg(IPA_LOG_ERR, "db_get_limit_state: lseek(%s, -%lu, "
		    "SEEK_END)", db_file, (unsigned long)sizeof(rec));
		goto failed;
	}
	if (readn(fd, db_file, &rec, sizeof(rec)) != sizeof(rec)) {
		logbt("db_get_limit_state");
		goto failed;
	}
	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_get_limit_state: close(%s)", db_file);
		return (-1);
	}

	cnt_from_base(state->lim, rec.l_high, rec.l_low);
	cnt_from_base(state->cnt, rec.c_high, rec.c_low);
	state->event_date_set = 0;
	if (rec.set & IPA_SDB_LIMIT_EVENT_START_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_START_SET;
		copy_db_date_to_tm(&rec.start,
		    &state->event_date[IPA_LIMIT_EVENT_START]);
	}
	if (rec.set & IPA_SDB_LIMIT_EVENT_RESTART_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_RESTART_SET;
		copy_db_date_to_tm(&rec.restart,
		    &state->event_date[IPA_LIMIT_EVENT_RESTART]);
	}
	if (rec.set & IPA_SDB_LIMIT_EVENT_RESTART_EXEC_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_RESTART_EXEC_SET;
		copy_db_date_to_tm(&rec.restart_exec,
		    &state->event_date[IPA_LIMIT_EVENT_RESTART_EXEC]);
	}
	if (rec.set & IPA_SDB_LIMIT_EVENT_REACH_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_REACH_SET;
		copy_db_date_to_tm(&rec.reach,
		    &state->event_date[IPA_LIMIT_EVENT_REACH]);
	}
	if (rec.set & IPA_SDB_LIMIT_EVENT_REACH_EXEC_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_REACH_EXEC_SET;
		copy_db_date_to_tm(&rec.reach_exec,
		    &state->event_date[IPA_LIMIT_EVENT_REACH_EXEC]);
	}
	if (rec.set & IPA_SDB_LIMIT_EVENT_EXPIRE_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_EXPIRE_SET;
		copy_db_date_to_tm(&rec.expire,
		    &state->event_date[IPA_LIMIT_EVENT_EXPIRE]);
	}
	if (rec.set & IPA_SDB_LIMIT_EVENT_EXPIRE_EXEC_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_EXPIRE_EXEC_SET;
		copy_db_date_to_tm(&rec.expire_exec,
		    &state->event_date[IPA_LIMIT_EVENT_EXPIRE_EXEC]);
	}
	if (rec.set & IPA_SDB_LIMIT_EVENT_UPDATED_SET) {
		state->event_date_set |= IPA_LIMIT_EVENT_UPDATED_SET;
		copy_db_date_to_tm(&rec.updated,
		    &state->event_date[IPA_LIMIT_EVENT_UPDATED]);
	}
#if IPA_DB_MOD_API_VERSION == 1
	return (0);
#else
	return (1);
#endif

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_get_limit_state: close(%s)", db_file);
	return (-1);
}

/*
 * Move statistics from <limit>/current to <limit>/yyyymm file.
 */
static int
archive_limit(const struct limit *limit)
{
	struct stat statbuf;
	struct ipa_sdb_limit_record rec1;
	ipa_tm start_tm;
	struct ipa_sdb_limit_record *recs;
	const char *db_file;
	char *path;
	size_t src_size, recs_size;
	ssize_t	nread;
	int fdsrc, fddst;

	db_file = limit->db_file;
	fdsrc = open(db_file, O_RDONLY);
	if (fdsrc < 0) {
		logmsg(IPA_LOG_ERR, "archive_limit: open(%s, O_RDONLY)",
		    db_file);
		return (-1);
	}
	if (readn(fdsrc, db_file, &rec1, sizeof(rec1)) != sizeof(rec1)) {
		logbt("archive_limit");
		return (-1);
	}
	if (close(fdsrc) < 0) {
		logmsg(IPA_LOG_ERR, "archive_limit: close(%s)", db_file);
		return (-1);
	}

	copy_db_date_to_tm(&rec1.start, &start_tm);

	/* Build the name for the year/month file. */
	if (mem_asprintf(m_tmp, &path, "%s/%d%02d", limit->db_dir,
	    start_tm.tm_year, start_tm.tm_mon) < 0) {
		logmsgx(IPA_LOG_ERR, "archive_limit: mem_asprintf failed");
		return (-1);
	}

	recs = NULL;
	fdsrc = fddst = -1;

	if (stat_func(path, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "archive_limit: %s(%s)",
			    stat_func_name, path);
			goto failed;
		}
		logmsgx(IPA_LOG_INFO, "archive_limit: rename file %s to %s",
		    db_file, path);
		if (rename(db_file, path) < 0) {
			logmsg(IPA_LOG_ERR, "archive_limit: rename(%s, %s)",
			    db_file, path);
			goto failed;
		}
	} else {
		if (check_file(path, &statbuf) < 0) {
			logbt("archive_limit");
			goto failed;
		}
		logmsgx(IPA_LOG_WARNING, "archive_limit: file %s already "
		    "exists, merging current statistics to it", path);
		if (stat_func(db_file, &statbuf) < 0) {
			logmsg(IPA_LOG_ERR, "archive_limit: %s(%s)",
			    stat_func_name, db_file);
			goto failed;
		}
		src_size = statbuf.st_size;
		if (src_size % sizeof(*recs) != 0) {
			logmsgx(IPA_LOG_ERR, "archive_limit: wrong size of %s",
			    db_file);
			goto failed;
		}
		recs_size = src_size < (LIMIT_NREC_READ * sizeof(*recs)) ?
		    src_size : (LIMIT_NREC_READ * sizeof(*recs));
		recs = mem_malloc(recs_size, m_tmp);
		if (recs == NULL) {
			logmsgx(IPA_LOG_ERR, "archive_limit: "
			    "mem_malloc failed");
			goto failed;
		}
		fdsrc = open(db_file, O_RDONLY);
		if (fdsrc < 0) {
			logmsg(IPA_LOG_ERR, "archive_limit: "
			    "open(%s, O_RDONLY)", db_file);
			goto failed;
		}
		fddst = open(path, O_WRONLY);
		if (fddst < 0) {
			logmsg(IPA_LOG_ERR, "archive_limit: "
			    "open(%s, O_WRONLY)", path);
			goto failed;
		}
		if (lseek(fddst, (off_t)0, SEEK_END) == (off_t)-1) {
			logmsg(IPA_LOG_ERR, "archive_limit: "
			    "lseek(%s, 0, SEEK_END)", path);
			goto failed;
		}
		nread = recs_size;
		while ((nread = readn(fdsrc, db_file, recs, nread)) > 0) {
			if (nread % sizeof(*recs) != 0) {
				logmsgx(IPA_LOG_ERR, "archive_limit: "
				    "read: wrong size of %s", db_file);
				goto failed;
			}
			if (writen(fddst, path, recs, nread) != nread) {
				logbt("archive_limit");
				goto failed;
			}
			src_size -= nread;
			if (src_size == 0)
				break;
			nread = src_size < recs_size ? src_size : recs_size;
		}
		switch (nread) {
		case 0:
			logmsgx(IPA_LOG_WARNING, "archive_limit: read from %s "
			    "returned nothing", db_file);
			break;
		case -1:
			logmsgx(IPA_LOG_ERR, "archive_limit: read from %s "
			    "failed", db_file);
			goto failed;
		}
		if (close(fdsrc) < 0) {
			logmsg(IPA_LOG_ERR, "archive_limit: close(%s)",
			    db_file);
			fdsrc = -1;
			goto failed;
		}
		fdsrc = -1;
		if (close(fddst) < 0) {
			logmsg(IPA_LOG_ERR, "archive_limit: close(%s)",
			    path);
			fddst = -1;
			goto failed;
		}
		fddst = -1;
		if (unlink(db_file) < 0) {
			logmsg(IPA_LOG_ERR, "archive_limit: unlink(%s)",
			    db_file);
			goto failed;
		}
	}

	mem_free(path, m_tmp);
	mem_free(recs, m_tmp);
	return (0);

failed:
	if (fdsrc >= 0 && close(fdsrc) < 0)
		logmsg(IPA_LOG_ERR, "archive_limit: close(%s)", db_file);
	if (fddst >= 0 && close(fddst) < 0)
		logmsg(IPA_LOG_ERR, "archive_limit: close(%s)", path);
	mem_free(path, m_tmp);
	mem_free(recs, m_tmp);
	return (-1);
}

static int
limit_new_file(const struct rule *rule, struct limit *limit,
    const ipa_tm *start_tm, bool *file_is_empty)
{
	struct stat statbuf1, statbuf2;
	const char *db_file;
	char *path;
	int fd;

	db_file = limit->db_file;

	/* Build the name for the year/month file. */
	if (mem_asprintf(m_tmp, &path, "%s/%d%02d", limit->db_dir,
	    start_tm->tm_year, start_tm->tm_mon) < 0) {
		logmsgx(IPA_LOG_ERR, "limit_new_file: mem_asprintf failed");
		return (-1);
	}

	/* Open or create a file for year/month. */
	fd = -1;
	if (stat_func(path, &statbuf1) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "limit_new_file: %s(%s)",
			    stat_func_name, path);
			goto failed;
		}
		/* There is no file for year/month, create it. */
		fd = create_db_file(path);
		if (fd < 0) {
			logbt("limit_new_file");
			goto failed;
		}
		if (stat_func(path, &statbuf1) < 0) {
			logmsg(IPA_LOG_ERR, "limit_new_file: %s(%s)",
			    stat_func_name, path);
			goto failed;
		}
		*file_is_empty = true;
	} else {
		/* There is a file for year/month. */
		if (check_file(path, &statbuf1) < 0) {
			logbt("limit_new_file");
			goto failed;
		}
		if (statbuf1.st_size != 0) {
			if (statbuf1.st_size %
			    sizeof(struct ipa_sdb_limit_record) != 0) {
				logmsgx(IPA_LOG_ERR, "limit_new_file: "
				    "wrong size of file %s", path);
				goto failed;
			}
		} else
			*file_is_empty = true;
		fd = open(path, O_RDWR);
		if (fd < 0) {
			logmsg(IPA_LOG_ERR, "limit_new_file: open(%s, O_RDWR)",
			    path);
			goto failed;
		}
	}

	if (stat_func(db_file, &statbuf2) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "limit_new_file: %s(%s)",
			    stat_func_name, db_file);
			goto failed;
		}
		/* There is no "current" file, link it to year/month file. */
		if (link(path, db_file) < 0) {
			logmsg(IPA_LOG_ERR, "limit_new_file: link(%s, %s)",
			    path, db_file);
			goto failed;
		}
	} else {
		/* There is "current" file. */
		if (statbuf2.st_nlink == 1) {
			logmsgx(IPA_LOG_WARNING, "limit_new_file: file %s has "
			    "only one hard link, fixing...", db_file);
			if (statbuf2.st_size == 0) {
				logmsgx(IPA_LOG_WARNING, "limit_new_state: "
				    "file %s is empty, unlinking it", db_file);
				if (unlink(db_file) < 0) {
					logmsg(IPA_LOG_ERR, "limit_new_file: "
					    "unlink(%s)", db_file);
					goto failed;
				}
			} else {
				if (statbuf2.st_size %
				    sizeof(struct ipa_sdb_limit_record) != 0) {
					logmsgx(IPA_LOG_ERR, "limit_new_file: "
					    "wrong size of %s", db_file);
					goto failed;
				}
				if (archive_limit(limit) < 0) {
					logbt("limit_new_file");
					goto failed;
				}
			}
			/* Link "current" file to year/month file. */
			if (link(path, db_file) < 0) {
				logmsg(IPA_LOG_ERR, "limit_new_file: "
				    "link(%s, %s)", path, db_file);
				goto failed;
			}
		} else if (statbuf1.st_dev != statbuf2.st_dev ||
		    statbuf1.st_ino != statbuf2.st_ino) {
			if (unlink(db_file) < 0) {
				logmsg(IPA_LOG_ERR, "limit_new_file: "
				    "unlink(%s)", db_file);
				goto failed;
			}
			/* Link "current" file to year/month file. */
			if (link(path, db_file) < 0) {
				logmsg(IPA_LOG_ERR, "limit_new_file: "
				    "link(%s, %s)", path, db_file);
				goto failed;
			}
		}
	}

	mem_free(path, m_tmp);
	limit->fd = fd;
	limit->start_year = start_tm->tm_year;
	limit->start_mon = start_tm->tm_mon;
	return (0);

failed:
	mem_free(path, m_tmp);
	if (fd >= 0 && rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "limit_new_file: close(%s)", db_file);
	return (-1);
}

/*
 * Set limit state.
 */
static int
db_set_limit_state(unsigned int ruleno, unsigned int limitno,
    const struct ipa_limit_state *state, int new_state)
{
	const struct rule *rule;
	const char *db_file;
	struct ipa_sdb_limit_record *rec;
	const ipa_tm *start_tm;
	struct limit *limit;
	unsigned int event_date_set;
	int fd;
	bool file_is_empty;

	event_date_set = state->event_date_set;
	if (!(event_date_set & IPA_LIMIT_EVENT_START_SET)) {
		logmsgx(IPA_LOG_ERR, "db_set_limit_state: "
		    "EVENT_START is not set");
		return (-1);
	}

	rule = RULE(ruleno);
	limit = LIMIT(rule, limitno);
	rec = &limit->rec;

	start_tm = &state->event_date[IPA_LIMIT_EVENT_START];

	rec->set = IPA_SDB_LIMIT_EVENT_START_SET;
	(void)copy_tm_to_db_date(start_tm, &rec->start);

	if (event_date_set & IPA_LIMIT_EVENT_RESTART_SET) {
		rec->set |= IPA_SDB_LIMIT_EVENT_RESTART_SET;
		(void)copy_tm_to_db_date(
		    &state->event_date[IPA_LIMIT_EVENT_RESTART],
		    &rec->restart);
	}
	if (event_date_set & IPA_LIMIT_EVENT_RESTART_EXEC_SET) {
		rec->set |= IPA_SDB_LIMIT_EVENT_RESTART_EXEC_SET;
		(void)copy_tm_to_db_date(
		    &state->event_date[IPA_LIMIT_EVENT_RESTART_EXEC],
		    &rec->restart_exec);
	}
	if (event_date_set & IPA_LIMIT_EVENT_REACH_SET) {
		rec->set |= IPA_SDB_LIMIT_EVENT_REACH_SET;
		(void)copy_tm_to_db_date(
		    &state->event_date[IPA_LIMIT_EVENT_REACH],
		    &rec->reach);
	}
	if (event_date_set & IPA_LIMIT_EVENT_REACH_EXEC_SET) {
		rec->set |= IPA_SDB_LIMIT_EVENT_REACH_EXEC_SET;
		(void)copy_tm_to_db_date(
		    &state->event_date[IPA_LIMIT_EVENT_REACH_EXEC],
		    &rec->reach_exec);
	}
	if (event_date_set & IPA_LIMIT_EVENT_EXPIRE_SET) {
		rec->set |= IPA_SDB_LIMIT_EVENT_EXPIRE_SET;
		(void)copy_tm_to_db_date(
		    &state->event_date[IPA_LIMIT_EVENT_EXPIRE],
		    &rec->expire);
	}
	if (event_date_set & IPA_LIMIT_EVENT_EXPIRE_EXEC_SET) {
		rec->set |= IPA_SDB_LIMIT_EVENT_EXPIRE_EXEC_SET;
		(void)copy_tm_to_db_date(
		    &state->event_date[IPA_LIMIT_EVENT_EXPIRE_EXEC],
		    &rec->expire_exec);
	}
	if (event_date_set & IPA_LIMIT_EVENT_UPDATED_SET) {
		rec->set |= IPA_SDB_LIMIT_EVENT_UPDATED_SET;
		(void)copy_tm_to_db_date(
		    &state->event_date[IPA_LIMIT_EVENT_UPDATED],
		    &rec->updated);
	}

	if (year_too_big) {
		year_too_big = false;
		logbt("db_set_limit_state");
		return (-1);
	}

	cnt_to_base(state->cnt, rec->c_high, rec->c_low);
	cnt_to_base(state->lim, rec->l_high, rec->l_low);

	db_file = limit->db_file;
	file_is_empty = false;
	if (start_tm->tm_year == limit->start_year &&
	    start_tm->tm_mon == limit->start_mon) {
		/* Open file for new limit state. */
		if (rule->close_fd) {
			limit->fd = open(db_file, O_RDWR);
			if (limit->fd < 0) {
				logmsg(IPA_LOG_ERR, "db_set_limit_state: "
				    "fopen(%s, O_RDWR)", db_file);
				return (-1);
			}
		}
	} else {
		/* Create file for new limit state. */
		if (!rule->close_fd && limit->fd >= 0) {
			if (close(limit->fd) < 0) {
				logmsg(IPA_LOG_ERR, "db_set_limit_state: "
				    "close(%s)", db_file);
				return (-1);
			}
			limit->fd = -1;
		}
		if (limit_new_file(rule, limit, start_tm, &file_is_empty) < 0) {
			logbt("db_set_limit_state");
			return (-1);
		}
	}

	/* Set the new limit state. */
	fd = limit->fd;
	if (new_state) {
		if (lseek(fd, (off_t)0, SEEK_END) == (off_t)-1) {
			logmsg(IPA_LOG_ERR, "db_set_limit_state: "
			    "lseek(%s, 0, SEEK_END)", db_file);
			goto failed;
		}
	} else if (!file_is_empty) {
		if (lseek(fd, -(off_t)sizeof(*rec), SEEK_END) == (off_t)-1) {
			logmsg(IPA_LOG_ERR, "db_set_limit_state: "
			    "lseek(%s, -%lu, SEEK_END)", db_file,
			    (unsigned long)sizeof(*rec));
			goto failed;
		}
	}
	if (writen(fd, db_file, rec, sizeof(*rec)) != sizeof(*rec)) {
		logbt("db_set_limit_state");
		goto failed;
	}
	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_set_limit_state: close(%s)", db_file);
		return (-1);
	}
	return (0);

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_set_limit_state: close(%s)", db_file);
	return (-1);
}

/*
 * Deinitialize a limit.
 */
static int
db_deinit_limit(unsigned int ruleno, unsigned int limitno)
{
	struct rule *rule;
	struct limit *limit;
	int rv;

	rv = 0;
	if (marray_check_index(rules_ptr_marray, ruleno)) {
		/* Such rule was registered. */
		rule = RULE(ruleno);
		if (limitno < rule->nlimits) {
			/*
			 * Such limit was registered, but free_limit()
			 * could already be called for it.
			 */
			limit = LIMIT(rule, limitno);
			if (limit != NULL) {
				if (!rule->close_fd && limit->fd >= 0 &&
				    close(limit->fd) < 0) {
					logmsg(IPA_LOG_ERR, "db_deinit_limit: "
					    "close(%s)", limit->db_file);
					rv = -1;
				}
				free_limit(rule, limit, limitno);
			}
		}
	}
	return (rv);
}

/*
 * Update statistics for a limit.
 */
static int
db_update_limit(unsigned int ruleno, unsigned int limitno,
    const uint64_t *cnt, const ipa_tm *ctm)
{
#define UPDATE_SIZE (sizeof(struct ipa_sdb_limit_record) - \
	offsetof(struct ipa_sdb_limit_record, updated))

	const char *db_file;
	const struct rule *rule;
	struct ipa_sdb_limit_record *rec;
	struct limit *limit;
	int fd;

	rule = RULE(ruleno);
	limit = LIMIT(rule, limitno);
	rec = &limit->rec;
	db_file = limit->db_file;

	if (copy_tm_to_db_date(ctm, &rec->updated) < 0) {
		logbt("db_update_limit");
		return (-1);
	}
	cnt_to_base(*cnt, rec->c_high, rec->c_low);

	if (rule->close_fd) {
		fd = open(db_file, O_WRONLY);
		if (fd < 0) {
			logmsg(IPA_LOG_ERR, "db_update_limit: "
			    "fopen(%s, O_WRONLY)", db_file);
			return (-1);
		}
	} else
		fd = limit->fd;

	if (lseek(fd, -(off_t)UPDATE_SIZE, SEEK_END) == (off_t)-1) {
		logmsg(IPA_LOG_ERR, "db_update_limit: lseek(%s, -%lu, "
		    "SEEK_END)", db_file, (unsigned long)UPDATE_SIZE);
		goto failed;
	}
	if (writen(fd, db_file, &rec->updated, UPDATE_SIZE) != UPDATE_SIZE) {
		logbt("db_update_limit");
		goto failed;
	}
	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_update_limit: close(%s)", db_file);
		return (-1);
	}
	return (0);

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_update_limit: close(%s)", db_file);
	return (-1);
#undef UPDATE_SIZE
}

/*
 * Register a limit's event.
 */
static int
db_limit_event(unsigned int ruleno, unsigned int limitno,
    unsigned int event, const ipa_tm *etm, const ipa_tm *ctm)
{
	const char *db_file;
	const struct rule *rule;
	struct limit *limit;
	struct ipa_sdb_limit_record *rec;
	int fd;

	rule = RULE(ruleno);
	limit = LIMIT(rule, limitno);
	rec = &limit->rec;
	db_file = limit->db_file;

	switch (event) {
	case IPA_LIMIT_EVENT_RESTART_EXEC:
		(void)copy_tm_to_db_date(etm, &rec->restart_exec);
		rec->set |= IPA_LIMIT_EVENT_RESTART_EXEC_SET;
		break;
	case IPA_LIMIT_EVENT_REACH:
		(void)copy_tm_to_db_date(etm, &rec->reach);
		rec->set |= IPA_LIMIT_EVENT_REACH_SET;
		break;
	case IPA_LIMIT_EVENT_REACH_EXEC:
		(void)copy_tm_to_db_date(etm, &rec->reach_exec);
		rec->set |= IPA_LIMIT_EVENT_REACH_EXEC_SET;
		break;
	case IPA_LIMIT_EVENT_EXPIRE:
		(void)copy_tm_to_db_date(etm, &rec->expire);
		rec->set |= IPA_LIMIT_EVENT_EXPIRE_SET;
		break;
	case IPA_LIMIT_EVENT_EXPIRE_EXEC:
		(void)copy_tm_to_db_date(etm, &rec->expire_exec);
		rec->set |= IPA_LIMIT_EVENT_EXPIRE_EXEC_SET;
		break;
	default:
		logmsgx(IPA_LOG_ERR, "db_limit_event: unexpected limit "
		    "event %u", event);
		return (-1);
	}
	(void)copy_tm_to_db_date(ctm, &rec->updated);

	if (year_too_big) {
		year_too_big = false;
		logbt("db_limit_event");
		return (-1);
	}

	if (rule->close_fd) {
		fd = open(db_file, O_WRONLY);
		if (fd < 0) {
			logmsg(IPA_LOG_ERR, "db_limit_event: "
			    "open(%s, O_WRONLY)", db_file);
			return (-1);
		}
	} else
		fd = limit->fd;

	if (lseek(fd, -(off_t)sizeof(*rec), SEEK_END) == (off_t)-1) {
		logmsg(IPA_LOG_ERR, "db_limit_event: lseek(%s, -%lu, SEEK_END)",
		    db_file, (unsigned long)sizeof(*rec));
		goto failed;
	}

	if (writen(fd, db_file, rec, sizeof(*rec)) != sizeof(*rec)) {
		logbt("db_limit_event");
		goto failed;
	}

	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_limit_event: close(%s)", db_file);
		return (-1);
	}
	return (0);

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_limit_event: close(%s)", db_file);
	return (-1);
}

static int
db_set_limit_active(unsigned int ruleno, unsigned int limitno, int active)
{
	const struct rule *rule;
	struct limit *limit;

	rule = RULE(ruleno);

	if (rule->close_fd) {
		/* The "current" file is already closed. */
		return (0);
	}

	limit = LIMIT(rule, limitno);

	if (limit->start_year == 0) {
		/* The "current" file was not linked to database file. */
		return (0);
	}

	if (active == 0) {
		/* Close "current" file. */
		if (close(limit->fd) < 0) {
			logmsg(IPA_LOG_ERR, "db_set_limit_active: close(%s)",
			    limit->db_file);
			return (-1);
		}
		limit->fd = -1;
	} else {
		/* Open "current" file. */
		limit->fd = open(limit->db_file, O_RDWR);
		if (limit->fd < 0) {
			logmsg(IPA_LOG_ERR, "db_set_limit_active: "
			    "open(%s, O_RDWR)", limit->db_file);
			return (-1);
		}
	}

	return (0);
}
#else
# define db_get_limit_state NULL
# define db_set_limit_state NULL
# define db_deinit_limit NULL
# define db_update_limit NULL
# define db_limit_event NULL
# define db_set_limit_active NULL
#endif /* WITH_LIMITS */

/*
 * Deinitialize a rule
 */
static int
db_deinit_rule(unsigned int ruleno)
{
	struct rule *rule;
	int rv;

	rv = 0;
	if (marray_check_index(rules_ptr_marray, ruleno)) {
		/* Such rule was registered. */
		rule = RULE(ruleno);
		if (!rule->close_fd && rule->fd >= 0 && close(rule->fd) < 0) {
			logmsg(IPA_LOG_ERR, "db_deinit_rule: "
			    "close(%s)", rule->db_file);
			rv = -1;
		}
		free_rule(rule);
	}
	return (rv);
}

/*
 * Deinitialize database.
 */
static int
db_deinit(void)
{
	struct rule *rule;
#ifdef WITH_AUTORULES
	struct autorule *autorule;
	unsigned int i;
#endif
	unsigned int n;

	if (global_db_dir != db_dir_default)
		mem_free(global_db_dir, m_parser);

	if (nrulesalloced != 0)
		TAILQ_FOREACH(rule, &rules_list, link) {
			if (rule->flags & RULE_INITED) {
				logmsgx(IPA_LOG_ERR, "internal error: "
				    "db_deinit: rule %s: rule was initialized, "
				    "but was not deinitialized (ref_count %u)",
				    rule->name, rule->ref_count);
				return (-1);
			}
#ifdef WITH_RULES
			free_rule_fast(rule);
			if (nrulesalloced == 0)
				break;
#endif
		}

	n = memfunc->marray_nused(rules_ptr_marray);
	if (n != 0) {
		logmsgx(IPA_LOG_ERR, "internal error: db_deinit: "
		    "rules_ptr_marray is not empty: %u", n);
		return (-1);
	}
	marray_deinit(rules_ptr_marray);

#ifdef WITH_AUTORULES
	if (nautorules != 0) {
		autorule = autorules;
		for (i = 0; i < nautorules; ++autorule, ++i)
			if (autorule->db_dir != global_db_dir)
				mem_free(autorule->db_dir, m_parser);
	}
	marray_deinit(autorules_marray);
#endif

	n = mzone_nused(rule_mzone);
	if (n != 0) {
		logmsgx(IPA_LOG_ERR, "internal error: db_deinit: rule_mzone "
		    "is not empty: %u", n);
		return (-1);
	}
	mzone_deinit(rule_mzone);

#ifdef WITH_LIMITS
	if (limit_mzone != NULL) {
		n = mzone_nused(limit_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: db_deinit: "
			    "limit_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(limit_mzone);
	}
#endif

#ifdef WITH_THRESHOLDS
	if (threshold_mzone != NULL) {
		n = mzone_nused(threshold_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: db_deinit: "
			    "threshold_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(threshold_mzone);
	}
#endif

	return (0);
}

/*
 * Append a new record for a rule to the database.
 */
static int
db_append_rule(unsigned int ruleno, const uint64_t *cnt, const ipa_tm *ctm)
{
	struct ipa_sdb_rule_record rec;
	struct stat statbuf;
	const char *db_file;
	struct rule *rule;
	int fd;

	rule = RULE(ruleno);
	db_file = rule->db_file;

	if (rule->cur_year != ctm->tm_year || rule->cur_mon != ctm->tm_mon) {
		/* New year or new month came. */
		if (!rule->close_fd && rule->fd >= 0) {
			if (close(rule->fd) < 0) {
				logmsg(IPA_LOG_ERR, "db_append_rule: "
				    "close(%s)", db_file);
				return (-1);
			}
			rule->fd = -1;
		}
		mem_free(rule->db_file, m_anon);
		if (mem_asprintf(m_anon, &rule->db_file, "%s/%s/%d%02d",
		    rule->db_dir, rule->name, ctm->tm_year, ctm->tm_mon) < 0) {
			logmsgx(IPA_LOG_ERR, "db_append_rule: "
			    "mem_asprintf failed");
			return (-1);
		}
		db_file = rule->db_file;
		if (stat_func(db_file, &statbuf) < 0) {
			if (errno != ENOENT) {
				logmsg(IPA_LOG_ERR, "db_append_rule: %s(%s)",
				    stat_func_name, db_file);
				return (-1);
			}
			fd = create_db_file(db_file);
			if (fd < 0) {
				logbt("db_append_rule");
				return (-1);
			}
		} else {
			if (check_file(db_file, &statbuf) < 0) {
				logbt("db_append_rule");
				return (-1);
			}
			fd = open(db_file, O_WRONLY);
			if (fd < 0) {
				logmsg(IPA_LOG_ERR, "db_append_rule: "
				    "open(%s, O_WRONLY)", db_file);
				return (-1);
			}
			if (statbuf.st_size % sizeof(rec) != 0) {
				logmsgx(IPA_LOG_ERR, "db_append_rule: "
				    "wrong size of %s", db_file);
				goto failed;
			}
		}
		if (!rule->close_fd)
			rule->fd = fd;
		rule->cur_year = ctm->tm_year;
		rule->cur_mon = ctm->tm_mon;
	} else {
		if (rule->close_fd) {
			fd = open(db_file, O_WRONLY);
			if (fd < 0) {
				logmsg(IPA_LOG_ERR, "db_append_rule: "
				    "open(%s, O_WRONLY)", db_file);
				return (-1);
			}
		} else
			fd = rule->fd;
	}

	if (lseek(fd, (off_t)0, SEEK_END) == (off_t)-1) {
		logmsg(IPA_LOG_ERR, "db_append_rule: lseek(%s, 0, SEEK_END)",
		    db_file);
		goto failed;
	}

	rec.mday = (uint8_t)ctm->tm_mday;
	rec.h1 = rec.h2 = (uint8_t)ctm->tm_hour;
	rec.m1 = rec.m2 = (uint8_t)ctm->tm_min;
	rec.s1 = rec.s2 = (uint8_t)ctm->tm_sec;
	cnt_to_base(*cnt, rec.c_high, rec.c_low);

	if (writen(fd, rule->db_file, &rec, sizeof(rec)) != sizeof(rec)) {
		logbt("db_append_rule");
		goto failed;
	}

	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_append_rule: close(%s)", db_file);
		return (-1);
	}
	return (0);

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_append_rule: close(%s)", db_file);
	return (-1);
}

/*
 * Update statistics for a rule.
 */
static int
db_update_rule(unsigned int ruleno, const uint64_t *cnt, const ipa_tm *ctm)
{
#define UPDATE_SIZE (sizeof(struct ipa_sdb_rule_record) - \
	offsetof(struct ipa_sdb_rule_record, h2))

	struct ipa_sdb_rule_record rec;
	const char *db_file;
	struct rule *rule;
	int fd;

	rule = RULE(ruleno);
	db_file = rule->db_file;

	if (rule->close_fd) {
		fd = open(db_file, O_WRONLY);
		if (fd < 0) {
			logmsg(IPA_LOG_ERR, "db_update_rule: "
			    "open(%s, O_WRONLY)", db_file);
			return (-1);
		}
	} else
		fd = rule->fd;

	rec.h2 = (uint8_t)ctm->tm_hour;
	rec.m2 = (uint8_t)ctm->tm_min;
	rec.s2 = (uint8_t)ctm->tm_sec;
	cnt_to_base(*cnt, rec.c_high, rec.c_low);

	if (lseek(fd, -(off_t)UPDATE_SIZE, SEEK_END) == (off_t)-1) {
		logmsg(IPA_LOG_ERR, "db_update_rule: lseek(%s, -%lu, SEEK_END)",
		    db_file, (unsigned long)UPDATE_SIZE);
		goto failed;
	}
	if (writen(fd, db_file, &rec.h2, UPDATE_SIZE) != UPDATE_SIZE) {
		logbt("db_update_rule");
		goto failed;
	}
	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_update_rule: close(%s)", db_file);
		return (-1);
	}
	return (0);

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_update_rule: close(%s)", db_file);
	return (-1);
#undef UPDATE_SIZE
}

static int
db_set_rule_active(unsigned int ruleno, int active)
{
	struct rule *rule;

	rule = RULE(ruleno);

	if (rule->close_fd) {
		/* Rule's database file is already closed. */
		return (0);
	}

	if (rule->cur_year == 0) {
		/* rule->db_file was not built. */
		return (0);
	}

	if (active == 0) {
		/* Close rule's database file. */
		if (close(rule->fd) < 0) {
			logmsg(IPA_LOG_ERR, "db_set_rule_active: close(%s)",
			    rule->db_file);
			return (-1);
		}
		rule->fd = -1;
	} else {
		/* Open rule's database file. */
		rule->fd = open(rule->db_file, O_WRONLY);
		if (rule->fd < 0) {
			logmsg(IPA_LOG_ERR, "db_set_rule_active: "
			    "open(%s, O_WRONLY)", rule->db_file);
			return (-1);
		}
	}

	return (0);
}

#ifdef WITH_THRESHOLDS
static int
db_get_threshold_state(unsigned int ruleno, unsigned int thresholdno,
    struct ipa_threshold_state *state)
{
	struct ipa_sdb_threshold_record rec;
	struct stat statbuf;
	const struct rule *rule;
	const char *db_file;
	struct threshold *threshold;
	int fd;

	rule = RULE(ruleno);
	threshold = THRESHOLD(rule, thresholdno);
	db_file = threshold->db_file;

	if (stat_func(db_file, &statbuf) < 0) {
		if (errno != ENOENT) {
			logmsg(IPA_LOG_ERR, "db_get_threshold_state: %s(%s)",
			    stat_func_name, db_file);
			return (-1);
		}
#if IPA_DB_MOD_API_VERSION == 1
		state->thr = 0;
#endif
		return (0);
	}

	if (check_file(db_file, &statbuf) < 0) {
		logbt("db_get_threshold_state");
		return (-1);
	}
	if (statbuf.st_size == 0) {
#if IPA_DB_MOD_API_VERSION == 1
		state->thr = 0;
#endif
		return (0);
	}
	if (statbuf.st_size != sizeof(rec)) {
		logmsgx(IPA_LOG_ERR, "db_get_threshold_state: "
		    "wrong size of %s", db_file);
		return (-1);
	}

	if (rule->close_fd || threshold->fd < 0) {
		threshold->fd = open(db_file, O_RDWR);
		if (threshold->fd < 0) {
			logmsg(IPA_LOG_ERR, "db_get_threshold_state: "
			    "open(%s, O_RDWR)", db_file);
			return (-1);
		}
	}
	fd = threshold->fd;
	if (lseek(fd, (off_t)0, SEEK_SET) == (off_t)-1) {
		logmsg(IPA_LOG_ERR, "db_get_threshold_state: "
		    "lseek(%s, 0, SEEK_SET)", db_file);
		goto failed;
	}
	if (readn(fd, db_file, &rec, sizeof(rec)) != sizeof(rec)) {
		logbt("db_get_threshold_state");
		goto failed;
	}
	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_get_threshold_state: close(%s)",
		    db_file);
		return (-1);
	}

	cnt_from_base(state->thr, rec.t_high, rec.t_low);
	cnt_from_base(state->cnt, rec.c_high, rec.c_low);
	copy_db_date_to_tm(&rec.tm_started, &state->tm_from);
	copy_db_date_to_tm(&rec.tm_updated, &state->tm_updated);
#if IPA_DB_MOD_API_VERSION == 1
	return (0);
#else
	return (1);
#endif

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_get_threshold_state: close(%s)",
		    db_file);
	return (-1);
}

static int
db_set_threshold_state(unsigned int ruleno, unsigned int thresholdno,
    const struct ipa_threshold_state *state)
{
	struct ipa_sdb_threshold_record rec;
	const struct rule *rule;
	const char *db_file;
	struct threshold *threshold;
	int fd;

	cnt_to_base(state->thr, rec.t_high, rec.t_low);
	cnt_to_base(state->cnt, rec.c_high, rec.c_low);
	if (copy_tm_to_db_date(&state->tm_from, &rec.tm_started) < 0 ||
	    copy_tm_to_db_date(&state->tm_updated, &rec.tm_updated) < 0) {
		logbt("db_set_threshold_state");
		return (-1);
	}

	rule = RULE(ruleno);
	threshold = THRESHOLD(rule, thresholdno);
	db_file = threshold->db_file;

	if (rule->close_fd) {
		fd = open(db_file, O_WRONLY);
		if (fd < 0) {
			logmsg(IPA_LOG_ERR, "db_set_threshold_state: "
			    "open(%s, O_WRONLY)", db_file);
			return (-1);
		}
	} else
		fd = threshold->fd;

	if (lseek(fd, (off_t)0, SEEK_SET) == (off_t)-1) {
		logmsg(IPA_LOG_ERR, "db_set_threshold_state: "
		    "lseek(%s, 0, SEEK_SET)", db_file);
		goto failed;
	}
	if (writen(fd, db_file, &rec, sizeof(rec)) != sizeof(rec)) {
		logbt("db_set_threshold_state");
		goto failed;
	}
	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_set_threshold_state: close(%s)",
		    db_file);
		return (-1);
	}
	return (0);

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_set_threshold_state: close(%s)",
		    db_file);
	return (-1);
}

/*
 * Deinitialize a threshold.
 */
static int
db_deinit_threshold(unsigned int ruleno, unsigned int thresholdno)
{
	struct threshold *threshold;
	struct rule *rule;
	int rv;

	rv = 0;
	if (marray_check_index(rules_ptr_marray, ruleno)) {
		/* Such rule was registered. */
		rule = RULE(ruleno);
		if (thresholdno < rule->nthresholds) {
			/*
			 * Such threshold was registered, but free_threshold()
			 * could already be called for it.
			 */
			threshold = THRESHOLD(rule, thresholdno);
			if (threshold != NULL) {
				if (!rule->close_fd && threshold->fd >= 0 &&
				    close(threshold->fd) < 0) {
					logmsg(IPA_LOG_ERR,
					    "db_deinit_threshold: close(%s)",
					    threshold->db_file);
					rv = -1;
				}
				free_threshold(rule, threshold, thresholdno);
			}
		}
	}
	return (rv);
}

static int
db_update_threshold(unsigned int ruleno, unsigned int thresholdno,
    const uint64_t *cnt, const ipa_tm *tm_from, const ipa_tm *tm_updated)
{
#define UPDATE_SIZE (sizeof(struct ipa_sdb_threshold_record) - \
	offsetof(struct ipa_sdb_threshold_record, c_high))

	struct ipa_sdb_threshold_record rec;
	const struct rule *rule;
	const char *db_file;
	struct threshold *threshold;
	int fd;

	cnt_to_base(*cnt, rec.c_high, rec.c_low);
	if (copy_tm_to_db_date(tm_from, &rec.tm_started) < 0 ||
	    copy_tm_to_db_date(tm_updated, &rec.tm_updated) < 0) {
		logbt("db_update_threshold");
		return (-1);
	}

	rule = RULE(ruleno);
	threshold = THRESHOLD(rule, thresholdno);
	db_file = threshold->db_file;

	if (rule->close_fd) {
		fd = open(db_file, O_WRONLY);
		if (fd < 0) {
			logmsg(IPA_LOG_ERR, "db_update_threshold: "
			    "fopen(%s, O_WRONLY)", db_file);
			return (-1);
		}
	} else
		fd = threshold->fd;

	if (lseek(fd, -(off_t)UPDATE_SIZE, SEEK_END) == (off_t)-1) {
		logmsg(IPA_LOG_ERR, "db_update_threshold: lseek(%s, -%lu, "
		    "SEEK_END)", db_file, (unsigned long)UPDATE_SIZE);
		goto failed;
	}
	if (writen(fd, db_file, &rec.c_high, UPDATE_SIZE) != UPDATE_SIZE) {
		logbt("db_update_threshold");
		goto failed;
	}
	if (rule->close_fd && close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "db_update_threshold: close(%s)", db_file);
		return (-1);
	}
	return (0);

failed:
	if (rule->close_fd && close(fd) < 0)
		logmsg(IPA_LOG_ERR, "db_update_threshold: close(%s)", db_file);
	return (-1);
#undef UPDATE_SIZE
}

static int
db_set_threshold_active(unsigned int ruleno, unsigned int thresholdno,
    int active)
{
	const struct rule *rule;
	struct threshold *threshold;

	rule = RULE(ruleno);

	if (rule->close_fd) {
		/* Threshold's database file is already closed. */
		return (0);
	}

	threshold = THRESHOLD(rule, thresholdno);

	if (active == 0) {
		/* Close threshold's database file. */
		if (close(threshold->fd) < 0) {
			logmsg(IPA_LOG_ERR, "db_set_threshold_active: "
			    "close(%s)", threshold->db_file);
			return (-1);
		}
		threshold->fd = -1;
	} else {
		/* Open threshold's database file. */
		threshold->fd = open(threshold->db_file, O_RDWR);
		if (threshold->fd < 0) {
			logmsg(IPA_LOG_ERR, "db_set_threshold_active: "
			    "open(%s, O_RDWR)", threshold->db_file);
			return (-1);
		}
	}

	return (0);
}
#else
# define db_get_threshold_state NULL
# define db_set_threshold_state NULL
# define db_deinit_threshold NULL
# define db_update_threshold NULL
# define db_set_threshold_active NULL
#endif /* WITH_THRESHOLDS */

#define SDB_CONF_ROOT_SECT (IPA_CONF_SECT_CUSTOM_OFFSET + 1)

static const unsigned int sect_root[] = { IPA_CONF_SECT_ROOT, 0 };
static const unsigned int sect_cust_root[] = { SDB_CONF_ROOT_SECT, 0 };
static const unsigned int sect_any_rule[] = { IPA_CONF_SECT_GLOBAL,
#ifdef WITH_RULES
	IPA_CONF_SECT_RULE,
#endif
#ifdef WITH_AUTORULES
	IPA_CONF_SECT_AUTORULE,
#endif
	0
};

static ipa_conf_sect conf_sect_tbl[] = {
	{ "", SDB_CONF_ROOT_SECT, 0, NULL, NULL, IPA_CONF_TYPE_MISC,
	  sect_root, parse_config_section
	},
	{ NULL, 0, 0, NULL, NULL, IPA_CONF_TYPE_MISC,
	  NULL, NULL
	}
};

static ipa_conf_param conf_param_tbl[] = {
	{ "db_dir", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_any_rule, parse_db_dir
	},
	{ "db_group", 1, NULL, NULL, IPA_CONF_TYPE_MISC,
	  sect_any_rule, parse_db_group
	},
	{ "close_fd", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_any_rule, parse_close_fd
	},
	{ "allow_symlinks", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_cust_root, parse_allow_symlinks
	},
	{ "check_version", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_cust_root, parse_check_version
	},
	{ NULL, 0, NULL, NULL, IPA_CONF_TYPE_MISC,
	  NULL, NULL
	}
};

struct ipa_db_mod IPA_DB_MOD_ENTRY(ipa_db_sdb) = {
	IPA_DB_MOD_API_VERSION,	/* api_ver			*/
	MOD_FLAGS,		/* mod_flags			*/
	MOD_DB_NAME,		/* db_name			*/
	NULL,			/* suppfunc			*/
	NULL,			/* memfunc			*/
	MOD_CONF_PREFIX,	/* conf_prefix			*/
	conf_sect_tbl,		/* conf_sect_tbl		*/
	conf_param_tbl,		/* conf_param_tbl		*/
	conf_init,		/* conf_init			*/
	conf_deinit,		/* conf_deinit			*/
	conf_event,		/* conf_event			*/
	conf_mimic_real,	/* conf_mimic_real		*/
	NULL,			/* conf_inherit			*/
	conf_show,		/* conf_show			*/
	db_pre_init,		/* db_pre_init			*/
	db_init_dynrule,	/* db_init_dynrule		*/
	db_init_statrule,	/* db_init_statrule		*/
	db_init_dynlimit,	/* db_init_dynlimit		*/
	db_init_statlimit,	/* db_init_statlimit		*/
	db_init_dynthreshold,	/* db_init_dynthreshold		*/
	db_init_statthreshold,	/* db_init_statthreshold	*/
	db_init,		/* db_init			*/
	db_get_limit_state,	/* db_get_limit_state		*/
	db_set_limit_state,	/* db_set_limit_state		*/
	db_get_threshold_state,	/* db_get_threshold_state	*/
	db_set_threshold_state,	/* db_set_threshold_state	*/
	db_deinit_threshold,	/* db_deinit_threshold		*/
	db_deinit_limit,	/* db_deinit_limit		*/
	db_deinit_rule,		/* db_deinit_rule		*/
	db_deinit,		/* db_deinit			*/
	db_append_rule,		/* db_append_rule		*/
	db_update_rule,		/* db_update_rule		*/
	db_update_limit,	/* db_update_limit		*/
	db_limit_event,		/* db_limit_event		*/
	db_update_threshold,	/* db_update_threshold		*/
	db_set_rule_active,	/* db_set_rule_active		*/
	db_set_limit_active,	/* db_set_limit_active		*/
	db_set_threshold_active	/* db_set_threshold_active	*/
};
