package webcrypto

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestGetRandomValues(t *testing.T) {
	t.Parallel()

	ts := newConfiguredRuntime(t)

	_, gotErr := ts.RunOnEventLoop(`
		var input = new Uint8Array(10);
		var output = crypto.getRandomValues(input);

		if (output.length != 10) {
			throw new Error("output.length != 10");
		}

		// Note that we're comparing references here, not values.
		// Thus we're testing that the same typed array is returned.
		if (input !== output) {
			throw new Error("input !== output");
		}
	`)

	assert.NoError(t, gotErr)
}

// TODO: Add tests for DataView

// TestGetRandomValues tests that crypto.getRandomValues() supports the expected types
// listed in the [specification]:
// - Int8Array
// - Int16Arrays
// - Int32Array
// - Uint8Array
// - Uint8ClampedArray
// - Uint16Array
// - Uint32Array
//
// It stands as the k6 counterpart of the [official test suite] on that topic.
//
// [specification]: https://www.w3.org/TR/WebCryptoAPI/#Crypto-method-getRandomValues
// [official test suite]: https://github.com/web-platform-tests/wpt/blob/master/WebCryptoAPI/getRandomValues.any.js#L1
func TestGetRandomValuesSupportedTypedArrays(t *testing.T) {
	t.Parallel()

	ts := newConfiguredRuntime(t)

	type testCase struct {
		name       string
		typedArray string
		wantErr    bool
	}

	testCases := []testCase{
		{
			name:       "filling a Int8Array typed array with random values should succeed",
			typedArray: "Int8Array",
			wantErr:    false,
		},
		{
			name:       "filling a Int16Array typed array with random values should succeed",
			typedArray: "Int16Array",
			wantErr:    false,
		},
		{
			name:       "filling a Int32Array typed array with random values should succeed",
			typedArray: "Int32Array",
			wantErr:    false,
		},
		{
			name:       "filling a Uint8Array typed array with random values should succeed",
			typedArray: "Uint8Array",
			wantErr:    false,
		},
		{
			name:       "filling a Uint8ClampedArray typed array with random values should succeed",
			typedArray: "Uint8ClampedArray",
			wantErr:    false,
		},
		{
			name:       "filling a Uint16Array typed array with random values should succeed",
			typedArray: "Uint16Array",
			wantErr:    false,
		},
		{
			name:       "filling a Uint32Array typed array with random values should succeed",
			typedArray: "Uint32Array",
			wantErr:    false,
		},

		// Unsupported typed arrays
		{
			name:       "filling a BigInt64Array typed array with random values should succeed",
			typedArray: "BigInt64Array",
			wantErr:    true,
		},
		{
			name:       "filling a BigUint64Array typed array with random values should succeed",
			typedArray: "BigUint64Array",
			wantErr:    true,
		},
		{
			name:       "filling a Float32Array typed array with random values should fail",
			typedArray: "Float32Array",
			wantErr:    true,
		},
		{
			name:       "filling a Float64Array typed array with random values should fail",
			typedArray: "Float64Array",
			wantErr:    true,
		},
	}

	for _, tc := range testCases {
		_, gotErr := ts.RunOnEventLoop(fmt.Sprintf(`
				var buf = new %s(10);
				crypto.getRandomValues(buf);

				if (buf.length != 10) {
					throw new Error("buf.length != 10");
				}
			`, tc.typedArray))

		if tc.wantErr != (gotErr != nil) {
			t.Fatalf("unexpected error: %v", gotErr)
		}

		assert.Equal(t, tc.wantErr, gotErr != nil, tc.name)
	}
}

// TestGetRandomValuesQuotaExceeded tests that crypto.getRandomValues() returns a
// QuotaExceededError when the requested size is too large. As described in the
// [specification], the maximum size is 65536 bytes.
//
// It stands as the k6 counterpart of the [official test suite] on that topic.
//
// [specification]: https://www.w3.org/TR/WebCryptoAPI/#Crypto-method-getRandomValues
// [official test suite]: https://github.com/web-platform-tests/wpt/blob/master/WebCryptoAPI/getRandomValues.any.js#L1
func TestGetRandomValuesQuotaExceeded(t *testing.T) {
	t.Parallel()

	ts := newConfiguredRuntime(t)

	_, gotErr := ts.RunOnEventLoop(`
		var buf = new Uint8Array(1000000000);
		crypto.getRandomValues(buf);
	`)

	assert.Error(t, gotErr)
	assert.Contains(t, gotErr.Error(), "QuotaExceededError")
}

// TestRandomUUIDIsTheNamespaceFormat tests that the UUID generated by
// crypto.randomUUID() is in the correct format.
//
// It stands as the k6 counterpart of the equivalent [WPT test].
//
// [WPT test]: https://github.com/web-platform-tests/wpt/blob/master/WebCryptoAPI/randomUUID.https.any.js#L16
func TestRandomUUIDIsInTheNamespaceFormat(t *testing.T) {
	t.Parallel()

	ts := newConfiguredRuntime(t)

	_, gotErr := ts.RunOnEventLoop(`
		const iterations = 256;
		const uuids = new Set();

		function randomUUID() {
			const uuid = crypto.randomUUID();
			if (uuids.has(uuid)) {
				throw new Error("UUID collision: " + uuid);
			}
			uuids.add(uuid);
			return uuid
		}

		const UUIDRegex = /^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/
		for (let i = 0; i < iterations; i++) {
			// Assert that the UUID is in the correct format and
			// that it is unique.
			UUIDRegex.test(randomUUID());
		}
	`)

	assert.NoError(t, gotErr)
}

// TestRandomUUIDIVersion tests that the UUID generated by
// crypto.randomUUID() has the correct version 4 bits set
// (4 most significant bits of the bytes[6] set to `0100`).
//
// It stands as the k6 counterpart of the equivalent [WPT test].
//
// [WPT test]: https://github.com/web-platform-tests/wpt/blob/master/WebCryptoAPI/randomUUID.https.any.js#L25
func TestRandomUUIDVersion(t *testing.T) {
	t.Parallel()

	ts := newConfiguredRuntime(t)

	_, gotErr := ts.RunOnEventLoop(`
		const iterations = 256;
		const uuids = new Set();

		function randomUUID() {
			const uuid = crypto.randomUUID();
			if (uuids.has(uuid)) {
				throw new Error("UUID collision: " + uuid);
			}
			uuids.add(uuid);
			return uuid
		}

		for (let i = 0; i < iterations; i++) {
			let value = parseInt(randomUUID().split('-')[2].slice(0, 2), 16)
			value &= 0b11110000
			if (value !== 0b01000000) {
				throw new Error("UUID version is not 4: " + value);
			}
		}
	`)

	assert.NoError(t, gotErr)
}

// TestRandomUUIDIVariant tests that the UUID generated by
// crypto.randomUUID() has the correct variant 2 bits set
// (2 most significant bits of the bytes[8] set to `10`).
//
// It stands as the k6 counterpart of the equivalent [WPT test].
//
// [WPT test]: https://github.com/web-platform-tests/wpt/blob/master/WebCryptoAPI/randomUUID.https.any.js#L35
func TestRandomUUIDVariant(t *testing.T) {
	t.Parallel()

	ts := newConfiguredRuntime(t)

	_, gotErr := ts.RunOnEventLoop(`
		const iterations = 256;
		const uuids = new Set();

		function randomUUID() {
			const uuid = crypto.randomUUID();
			if (uuids.has(uuid)) {
				throw new Error("UUID collision: " + uuid);
			}
			uuids.add(uuid);
			return uuid
		}

		for (let i = 0; i < iterations; i++) {
			let value = parseInt(randomUUID().split('-')[3].slice(0, 2), 16)
			value &= 0b11000000
			if (value !== 0b10000000) {
				throw new Error("UUID variant is not 1: " + value);
			}
		}
	`)

	assert.NoError(t, gotErr)
}
