// Package line provides a datatype for text lines that are generated by package tag.
package lines

import "strings"

// Line is a line of text.
type Line struct {
	contents    string
	indentable  bool
	indentCount uint
}

// IndentableFrom returns a line that you can and will indent. Use for most tags.
func IndentableFrom(contents string) Line {
	return Line{
		contents:    contents,
		indentable:  true,
		indentCount: 0,
	}
}

// UnindentableFrom returns a line that you cannot indent. Use for codeblocks and stuff like that.
func UnindentableFrom(contents string) Line {
	return Line{
		contents:    contents,
		indentable:  false,
		indentCount: 0,
	}
}

// Indented returns a copy of the line with one more indentation (for indentable lines) or the same line (for unindentable lines).
func (l Line) Indented() Line {
	if l.indentable {
		l.indentCount++
		return l
	}
	return l
}

// Contents returns line's contents
func (l Line) Contents() string {
	return l.contents
}

// IsIndentable is true if the line is indentable.
func (l Line) IsIndentable() bool {
	return l.indentable
}

// String returns the line with the appropriate number of indentations (\t) and a newline.
func (l Line) String() string {
	return strings.Repeat("\t", int(l.indentCount)) + l.contents + "\n"
}
