/* The GIMP -- an image manipulation program
 * Copyright (C) 1995-1997 Spencer Kimball and Peter Mattis
 *
 * GIMP Color Manager Plug-In
 * Copyright (C) 2000, 2001  Karl Heinz Kreme <khk@khk.net>
 * Gimp 2.x Port 2004 Jordi Cantón <jordinitrox@virtual-sub.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#include "config.h"

#include <string.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include <lcms.h>

#include "main.h"
#include "interface.h"
#include "render.h"

#include "plugin-intl.h"


/*  Constants  */

#define PROCEDURE_NAME   "gimp_color_manager"

#define DATA_KEY_VALS    "plug_in_colormanager"
#define DATA_KEY_UI_VALS "plug_in_colormanager_ui"

#define PARASITE_KEY     "plug-in-colormanager-options"


/*  Local function prototypes  */

static void   query (void);
static void   run   (const gchar      *name,
		     gint              nparams,
		     const GimpParam  *param,
		     gint             *nreturn_vals,
		     GimpParam       **return_vals);


/*  Local variables  */

PlugInVals default_vals =
{
  INPUT_PROFILE,
  "",
  "scanner.icm",
  "printer.icm",
  "display.icm",
  "sRGB.icm",
  INTENT_PERCEPTUAL,
  0,				/* preview */
  1,				/* use embedded */
  0				/* show gamut */
};

const PlugInImageVals default_image_vals =
{
  0
};

const PlugInDrawableVals default_drawable_vals =
{
  0
};

const PlugInUIVals default_ui_vals =
{
  TRUE
};

static PlugInVals         vals;
static PlugInImageVals    image_vals;
static PlugInDrawableVals drawable_vals;
static PlugInUIVals       ui_vals;


GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,  /* init_proc  */
  NULL,  /* quit_proc  */
  query, /* query_proc */
  run,   /* run_proc   */
};

MAIN ()

static void
query (void)
{
  gchar *help_path;
  gchar *help_uri;

  static GimpParamDef args[] =
  {
    { GIMP_PDB_INT32,    "run_mode",   N_("Interactive, non-interactive")},
    { GIMP_PDB_IMAGE,    "image",      N_("Input image")                     },
    { GIMP_PDB_DRAWABLE, "drawable",   N_("Input drawable")                  },
    {GIMP_PDB_INT32, "cm_direction",   N_("Transformation direction - Input=0, Output=1")},
    {GIMP_PDB_STRING, "cm_directory", N_("Directory containing profiles")},
    {GIMP_PDB_STRING, "cm_input_profile",N_("Input profile to use for transformation")},
    {GIMP_PDB_STRING, "cm_output_profile",N_("Output profile to use for transformation")},
    {GIMP_PDB_STRING, "cm_display_profile",N_("Display profile to use for transformation")},
    {GIMP_PDB_STRING, "cm_work_space", N_("Work space color system profile")},
    {GIMP_PDB_INT32, "cm_rendering_intent", N_("Rendering intent")},
    {GIMP_PDB_INT32, "cm_preview", N_("Preview function")},
    {GIMP_PDB_INT32, "cm_use_embedded", N_("Use embedded Profiles")},
    {GIMP_PDB_INT32, "cm_gamut_check", N_("Perform gamut check")},
  };

  gimp_plugin_domain_register (PLUGIN_NAME, LOCALEDIR);

  help_path = g_build_filename (DATADIR, "help", NULL);
  help_uri = g_filename_to_uri (help_path, NULL, NULL);
  g_free (help_path);


  gimp_plugin_help_register ("http://www.freecolormanagement.com/color/gimp.html",
                             help_uri);

  gimp_install_procedure (PROCEDURE_NAME,
			  N_("Apply a color transformation based on ICC profiles to an image"),
			  N_("Help"),
			  "Karl Heinz Kremer <khk@khk.net>",
			  "Karl Heinz Kremer <khk@khk.net>",
			  "2000-2004",
			  N_("<Image>/Filters/Colors/Apply ICC profile..."),
			  "RGB*, GRAY*, INDEXED*",
			  GIMP_PLUGIN,
			  G_N_ELEMENTS (args), 0,
			  args, NULL);
}

static void
run (const gchar      *name,
     gint              n_params,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static GimpParam   values[1];
  GimpDrawable      *drawable;
  gint32             image_ID;
  GimpRunMode        run_mode;
  GimpPDBStatusType  status = GIMP_PDB_SUCCESS;

  *nreturn_vals = 1;
  *return_vals  = values;

  /*  Initialize i18n support  */
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
  textdomain (GETTEXT_PACKAGE);

  run_mode = param[0].data.d_int32;
  image_ID = param[1].data.d_int32;
  drawable = gimp_drawable_get (param[2].data.d_drawable);

  /*  Initialize with default values  */
    /*sets the default directory to users's gimp directory\color*/		  
  g_snprintf (default_vals.cm_directory, sizeof(default_vals.cm_directory), "%s%c%s%c",
                          gimp_directory (), G_DIR_SEPARATOR, "color", G_DIR_SEPARATOR);			  
  /*sets the default workspace profile to users's gimp directory\color\sRGB.icm*/
  /*g_snprintf (default_vals.cm_work_space,sizeof(default_vals.cm_work_space), "%s%c%s%c%s",
                          gimp_directory (), G_DIR_SEPARATOR, "color", G_DIR_SEPARATOR,"sRGB.icm");*/
  vals          = default_vals;
  image_vals    = default_image_vals;
  drawable_vals = default_drawable_vals;
  ui_vals       = default_ui_vals;

  if (strcmp (name, PROCEDURE_NAME) == 0)
    {
      switch (run_mode)
	{
	case GIMP_RUN_NONINTERACTIVE:
	  if (n_params != 8)
	    {
	      status = GIMP_PDB_CALLING_ERROR;
	    }
	  else
	    {
	      
	      vals.cm_direction = (gint) param[3].data.d_int32;
	      strncpy (vals.cm_directory, (guchar *) param[4].data.d_string,MAXPATHLEN);
	      strncpy (vals.cm_input_profile,(guchar *) param[5].data.d_string, MAXPATHLEN);
	      strncpy (vals.cm_output_profile,(guchar *) param[6].data.d_string, MAXPATHLEN);
	      strncpy (vals.cm_display_profile,(guchar *) param[7].data.d_string, MAXPATHLEN);
	      strncpy (vals.cm_work_space, (guchar *) param[8].data.d_string,MAXPATHLEN);
	      vals.cm_rendering_intent = (gint) param[9].data.d_int32;
	      vals.cm_preview = (gint) param[10].data.d_int32;
	      vals.cm_use_embedded = (gint) param[11].data.d_int32;
	      vals.cm_gamut_check = (gint) param[12].data.d_int32;
	      
	    }
	  break;

	case GIMP_RUN_INTERACTIVE:
	  /*  Possibly retrieve data  */
	  gimp_get_data (DATA_KEY_VALS,    &vals);
	  gimp_get_data (DATA_KEY_UI_VALS, &ui_vals);

	  if (! dialog (image_ID, drawable,
			&vals, &image_vals, &drawable_vals, &ui_vals))
	    {
	      status = GIMP_PDB_CANCEL;
	    }
	  break;

	case GIMP_RUN_WITH_LAST_VALS:
	  /*  Possibly retrieve data  */
	  gimp_get_data (DATA_KEY_VALS, &vals);


	  break;

	default:
	  break;
	}
    }
  else
    {
      status = GIMP_PDB_CALLING_ERROR;
    }

  if (status == GIMP_PDB_SUCCESS)
    {
       gimp_get_data (DATA_KEY_UI_VALS, &ui_vals);
       status = render (image_ID, drawable, &vals, &image_vals, &ui_vals);

      if (run_mode != GIMP_RUN_NONINTERACTIVE)
	gimp_displays_flush ();

      if (run_mode == GIMP_RUN_INTERACTIVE)
	{
	  gimp_set_data (DATA_KEY_VALS,    &vals,    sizeof (vals));
	  gimp_set_data (DATA_KEY_UI_VALS, &ui_vals, sizeof (ui_vals));
	}

      gimp_drawable_detach (drawable);
    }

  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = status;
}
