
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

/* Unix stream I/O module */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>

#include "runtime.h"


/* This implementation of stream I/O is based on the paper:
 *
 *  "Exploiting the advantages of mapped files for stream I/O",
 *  O. Krieger, M. Stumm and R. Umrau, "Proceedings of the 1992 Winter
 *  USENIX conference", p. 27-42.
 *
 * The basic idea was to use mmap() to avoid the copying overhead of
 * read() and write() calls.  For a fortran runtime library, this
 * turned out not to have much of a speed advantage at all.  The
 * implementation remains because we want to avoid entanglements with
 * the C library, and the alloc() model fits well with fortran's
 * record oriented I/O model.
 *
 * This implementation differs in a number of ways from the version
 * described in the paper.  First of all, threads are not an issue
 * during I/O and we also don't have to worry about having multiple
 * regions, since fortran's I/O model only allows you to be one place
 * at a time.
 *
 * On the other hand, we have to be able to writing at the end of a
 * stream, read from the start of a stream or read and write blocks of
 * bytes from an arbitrary position.  After opening a file, a pointer
 * to a stream structure is returned, which is used to handle file
 * accesses until the file is closed.
 *
 * salloc_r(stream, len)-- Given a stream pointer, return a pointer to
 * a block of memory that mirrors the file at the current position that
 * is 'len' bytes long.  The len integer is updated to reflect how
 * many bytes were actually read.  The only reason for a short read is
 * end of file.  The current file position is updated.  Returns a NULL
 * pointer on I/O error.  The pointer is valid until the next call to
 * salloc_*.
 *
 * salloc_rline(stream, len)-- Given a stream pointer, return a
 * pointer to a block of memory that mirrors the file at the current
 * position up a newline, end of file or the given length, whichever
 * comes first.  The current position is updated and the trailing
 * newline is the last byte read if all went well.
 *
 * salloc_w(stream, len, direct)-- Given the stream pointer, returns a
 * pointer to a block of memory that is updated to reflect the state
 * of the file.  The length of the buffer is always equal to that
 * requested.  The buffer must be completely set by the caller.  When
 * data has been written, the sfree() function must be called to
 * indicate that the caller is done writing data to the buffer.  This
 * may or may not cause an immediate physical write.  The 'direct'
 * flag is used to properly set the end of file.
 *
 * sseek(stream, position)-- Changes the current position of the file.
 * Returns SUCCESS or FAILURE.  */ 



/* writen()-- Write a buffer to a descriptor, allowing for short writes. */

static try writen(int fd, char *buffer, int len) {
int n;

    while(len > 0) {
	n = WRITE(fd, buffer, len);
	if (n < 0)
	    return FAILURE;

	buffer += n;
	len -= n;
    }

    return SUCCESS;
}



/* line_length()-- Given a pointer to a buffer and a maximum line
 * length, look for the end of line character and return the number of
 * characters in the line, including the newline.  If a newline is not
 * found, the maximum length is returned. */

static int line_length(char *buffer, int max_length) {
int n;

    n = 0;
    for(;;) {
	n++;
	if (*buffer++ == '\n')
	    break;

	if (n >= max_length)
	    return max_length;
    }

    return n;
}


/* flush_stream()-- Write bytes that need to be written. */

try flush_stream(stream *s) {

    if (s->ndirty == 0)
	return SUCCESS;

    if (s->physical_offset != s->dirty_offset) {
	if (LSEEK_ABS(s->fd, s->dirty_offset) < 0)
	    return FAILURE;

	s->physical_offset = s->dirty_offset;
    }

    if (writen(s->fd, s->buffer + (s->dirty_offset - s->buffer_offset),
	       s->ndirty) == FAILURE)
	return FAILURE;

    s->physical_offset += s->ndirty;
    s->ndirty = 0;

    return SUCCESS;
}



/* fd_alloc()-- Arrange a buffer such that the salloc() request can be
 * satisfied.  This subroutine gets the buffer ready for whatever is
 * to come next. */

static void fd_alloc(stream *s, int len) {
int n, buffer_len;
char *new_buffer;

    if (len <= BUFFER_SIZE) {
	new_buffer = s->small_buffer;
	buffer_len = BUFFER_SIZE;

    } else {
	new_buffer = get_mem(len);
	buffer_len = len;
    }

    /* Salvage bytes currently within the buffer.  This is important for
     * devices that cannot seek. */

    if (s->buffer != NULL && s->buffer_offset <= s->logical_offset &&
	s->logical_offset <= s->buffer_offset + s->active) {

	n = s->active - (s->logical_offset - s->buffer_offset);
	memmove(new_buffer,
		s->buffer + (s->logical_offset - s->buffer_offset), n);

	s->active = n;

    } else  /* new buffer starts off empty */
	s->active = 0;

    s->buffer_offset = s->logical_offset;

    /* free the old buffer if necessary */

    if (s->buffer != NULL && s->buffer != s->small_buffer)
	free_mem(s->buffer);

    s->buffer = new_buffer;
    s->len = buffer_len;
}



/* salloc_rline()-- Allocate memory for a line. */

char *salloc_rline(stream *s, unsigned *len) {
int first, m, n;
off_t w;
char *p;

    if (s->ndirty != 0 && flush_stream(s) == FAILURE)
	return NULL;

    first = 1;

    p = s->buffer + s->logical_offset - s->buffer_offset;

    if (s->buffer == NULL || p < s->buffer || p > s->buffer + s->active) {
	fd_alloc(s, BUFFER_SIZE);

	p = s->buffer + s->logical_offset - s->buffer_offset;
    }

    w = s->buffer_offset + s->active;
    if (s->physical_offset != w) {
	if (LSEEK_ABS(s->fd, w) < 0)
	    return NULL;

	s->physical_offset = w;
    }

    for(;;) {
	n = s->active - (s->logical_offset - s->buffer_offset);
	if (n > 0) {   /* See if we already have a line */
	    m = line_length(p, n);
	    if (m < n || p[n-1] == '\n')
		break;
	}

	if (first) {
	    fd_alloc(s, *len);
	    p = s->buffer;
	    first = 0;
	}

	n = READ(s->fd, s->buffer + s->active, s->len - s->active);
	if (n < 0)
	    return NULL;

	s->active += n;
	s->physical_offset += n;

	m = line_length(p, s->active);

	if (m < s->active || p[s->active-1] == '\n')
	    break;

	if (n == 0) {
	    m = s->active;
	    break;
	}
    }

    *len = m;
    s->logical_offset += m;

    return p;
}



/* salloc_w()-- Allocate a stream buffer for writing.  Either
 * we've already buffered the data or we need to load it. */

char *salloc_w(stream *s, unsigned len, int direct) {
off_t n;
char *p;

    if (s->buffer == NULL || s->buffer_offset > s->logical_offset ||
	s->logical_offset + len > s->buffer_offset + s->len) {

	if (flush_stream(s) == FAILURE)
	    return NULL;

	fd_alloc(s, len);
    }

    direct = 0;     /* should always be zero already. */

    /* Return a position within the current buffer */
    p = s->buffer + s->logical_offset - s->buffer_offset;
    if (s->ndirty == 0)
	s->dirty_offset = s->logical_offset;

    s->ndirty += len;
    s->logical_offset += len;

    n = s->logical_offset - s->buffer_offset;
    if (n > s->active)
	s->active = n;

    return p;
}



/* salloc_r()-- Allocate a stream buffer for reading.  Either we've
 * already buffered the data or we need to load it.  Returns NULL on
 * I/O error. */

char *salloc_r(stream *s, unsigned *len) {
off_t m;
char *p;
int n;

    if (s->ndirty != 0 && flush_stream(s) == FAILURE)
	return NULL; 

    if (s->buffer != NULL && s->buffer_offset <= s->logical_offset &&
	s->logical_offset + *len <= s->buffer_offset + s->active) {

	/* Return a position within the current buffer */
	p = s->buffer + s->logical_offset - s->buffer_offset;
	s->logical_offset += *len;
	return p;
    }

    fd_alloc(s, *len);

    m = s->buffer_offset + s->active;
    if (s->physical_offset != m) {
	if (LSEEK_ABS(s->fd, m) < 0)
	    return NULL;

	s->physical_offset = m;
    }

    /* Try to fill the buffer and get at least enough bytes to satisfy
     * the request */

    do {
	n = READ(s->fd, s->buffer + s->active, s->len - s->active);
	if (n < 0)
	    return NULL;

	if (n == 0)
	    break;

	s->physical_offset += n;
	s->active += n;
    } while(s->active < *len);

    if (s->active <= *len)
	*len = s->active;  /* Short read */

    s->logical_offset += *len;
    return s->buffer;
}



/* sfree()-- free a buffer */

try sfree(stream *s) {

    if (s->ndirty != 0 &&
	(s->buffer != s->small_buffer || options.all_unbuffered ||
	 s->unbuffered))
	return flush_stream(s);

    return SUCCESS;
}



/* sclose()-- Close a stream */

try sclose(stream *s) {

    if (flush_stream(s) == FAILURE)
	return FAILURE;

    if (s->buffer != NULL && s->buffer != s->small_buffer)
	free_mem(s->buffer);

    if (!standard_desc(s->fd) && CLOSE(s->fd) < 0)
	return FAILURE;

    free_mem(s);
    return SUCCESS;
}



/* sseek()-- Seek within a buffer */

try sseek(stream *s, off_t offset) {

    if (s->logical_offset == offset)
	return SUCCESS;

    if (flush_stream(s) == FAILURE)
	return FAILURE;

    s->physical_offset = s->logical_offset = offset;
    s->active = 0;

    return (LSEEK_ABS(s->fd, offset) < 0) ? FAILURE : SUCCESS;
}



/* fd_to_stream()-- Given an open file descriptor, build a stream
 * around it. */

stream *fd_to_stream(OS_HANDLE fd, int unbuffered) {
stream *s;

    s = get_mem(sizeof(stream));

    s->fd = fd;
    s->buffer_offset = 0;
    s->physical_size = 0;
    s->logical_offset = 0;
    s->truncate = !standard_desc(fd);

#if HAVE_WINDOWS
    s->unbuffered = unbuffered;
#else
    s->unbuffered = isatty(s->fd);
#endif

    s->buffer = NULL;
    return s;
}



/* unpack_filename()-- Given a fortran string and a pointer to a
 * buffer that is PATH_LENGTH characters, convert the fortran string to a
 * C string in the buffer.  Returns nonzero if this is not possible.  */

int unpack_filename(char *cstring, char *fstring, int fstring_len) {
G95_DINT len;

    len = fstrlen(fstring, fstring_len);
    if (len >= PATH_LENGTH)
	return 1;

    memmove(cstring, fstring, len);
    cstring[len] = '\0';

#ifdef __CYGWIN__
    if (cygwin_posix_path_list_p(cstring)) {
	char buffer[PATH_LENGTH];

	cygwin_posix_to_win32_path_list(cstring, buffer);
	strcpy(cstring, buffer);
    }
#endif

    return 0;
}



/* truncate_file()-- Given a unit, truncate the file at the current
 * position.  Sets the physical location to the new end of the file.
 * Returns FAILURE on error. */

try truncate_file(stream *u) {

    if (u->ndirty != 0 && flush_stream(u) == FAILURE)
	return FAILURE;

    if (FTRUNCATE(u->fd, u->logical_offset))
	return FAILURE;

    u->physical_offset = u->physical_size = u->logical_size =u->logical_offset;

    u->active = u->logical_offset - u->buffer_offset;

    if (u->active < 0)
	u->active = 0;
    else if (u->active > u->len)
	u->active = u->len;

    return (LSEEK_END(u->fd, 0) < 0)
	? FAILURE
	: SUCCESS;
}



/* open_external()-- Open an external file, unix specific version.
 * Returns NULL on operating system error. */

stream *open_external(unit_action action, unit_status status, char *path) {
OS_HANDLE fd;
int rc;

    rc = (status == STATUS_SCRATCH)
	? open_tempfile(path, &fd)
	: open_file(action, status, &fd, path);

    if (rc)
	return NULL;

    /* If this is a scratch file, we can unlink it now and the file will
     * go away when it is closed.  Windows can't delete open files. */

    if (!HAVE_WINDOWS && status == STATUS_SCRATCH)
	delete_file(path);

    return fd_to_stream(fd, 0);
}




/* file_length()-- Return the file length in bytes, -1 if unknown */

off_t file_length(stream *s) {

    return fd_length(s->fd);
}



/* file_position()-- Return the current position of the file */

off_t file_position(stream *s) {

    return s->logical_offset;
}




/* How files are stored:  This is an operating-system specific issue,
   and therefore belongs here.  There are three cases to consider.

   Direct Access:
      Records are written as block of bytes corresponding to the record
      length of the file.  This goes for both formatted and unformatted
      records.  Positioning is done explicitly for each data transfer,
      so positioning is not much of an issue.

   Sequential Formatted:
      Records are separated by newline characters.  The newline character
      is prohibited from appearing in a string.  If it does, this will be
      messed up on the next read.  End of file is also the end of a record.

   Sequential Unformatted:
      In this case, we are merely copying bytes to and from main storage,
      yet we need to keep track of varying record lengths.  We adopt
      the solution used by f2c.  Each record contains a pair of length
      markers:

        Length of record n in bytes
        Data of record n
        Length of record n in bytes

        Length of record n+1 in bytes
        Data of record n+1
        Length of record n+1 in bytes

     The length is stored at the end of a record to allow backspacing to the
     previous record.  Between data transfer statements, the file pointer
     is left pointing to the first length of the current record.

     ENDFILE records are never explicitly stored.

*/
