
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


#include <unistd.h>
#include <stdio.h>
#include <string.h>

#include "runtime.h"


static st_option
    access_opt[] = {
	{ "sequential",  ACCESS_SEQUENTIAL },
	{ "direct",      ACCESS_DIRECT },
	{ "stream",      ACCESS_STREAM },
	{ "transparent", ACCESS_STREAM },
	{ "append",      ACCESS_APPEND },
	{ NULL } },

    action_opt[] = {
	{ "read",        ACTION_READ },
	{ "write",       ACTION_WRITE },
	{ "readwrite",   ACTION_READWRITE },
	{ NULL } },

    blank_opt[] = {
	{ "null",   BLANK_NULL },
	{ "zero",   BLANK_ZERO },
	{ NULL } },

    delim_opt[] = {
	{ "none",        DELIM_NONE },
	{ "apostrophe",  DELIM_APOSTROPHE },
	{ "quote",       DELIM_QUOTE },
	{ NULL } },

    form_opt[] = {
	{ "formatted",    FORM_FORMATTED },
	{ "unformatted",  FORM_UNFORMATTED },
	{ NULL } },

    position_opt[] = {
	{ "asis",    POSITION_ASIS },
	{ "rewind",  POSITION_REWIND },
	{ "append",  POSITION_APPEND },
	{ NULL } },

    status_opt[] = {
	{ "unknown",   STATUS_UNKNOWN },
	{ "old",       STATUS_OLD },
	{ "new",       STATUS_NEW },
	{ "replace",   STATUS_REPLACE },
	{ "scratch",   STATUS_SCRATCH },
	{ NULL } },

    pad_opt[] = {
	{ "yes",  PAD_YES },
	{ "no",   PAD_NO  },
	{ NULL } },

    decimal_opt[] = {
	{ "point",  DECIMAL_POINT },
	{ "comma",  DECIMAL_COMMA },
	{ NULL } },

    convert_opt[] = {
	{ "native",  ENDIAN_NATIVE        },
	{ "swap",    ENDIAN_SWAP          },
	{ "little_endian",  ENDIAN_LITTLE },
	{ "big_endian",     ENDIAN_BIG    },
    };


/* test_endfile()-- Given a unit, test to see if the file is
 * positioned at the terminal point, and if so, change state from
 * NO_ENDFILE flag to AT_ENDFILE.  This prevents us from changing the
 * state from AFTER_ENDFILE to AT_ENDFILE. */

void test_endfile(iounit_t *u) {

    if (u->endfile == NO_ENDFILE && file_length(u->s) == file_position(u->s))
	u->endfile = AT_ENDFILE;
}



/* position_file()-- Reposition the file if necessary. */

static void position_file(iounit_t *u, unit_position position) {

    switch(position) {
    case POSITION_UNSPECIFIED:
    case POSITION_ASIS:
	break;

    case POSITION_REWIND:
	if (sseek(u->s, 0) == FAILURE)
	    goto seek_error;

	u->last_record = 0;

	test_endfile(u);     /* We might be at the end */
	break;

    case POSITION_APPEND:
	flush_stream(u->s);

	if (sseek(u->s, file_length(u->s)) == FAILURE)
	    goto seek_error;

	u->endfile = AT_ENDFILE;   /* We are at the end */
	break;

    seek_error:
	generate_error(ERROR_OS, NULL);
	break;
    }
}



/* edit_modes()-- Change the modes of a file, those that are allowed
 * to be changed. */

static void edit_modes(iounit_t *u, unit_flags *flags) {

    /* Complain about attempts to change the unchangeable */

    if (flags->access != ACCESS_UNSPECIFIED &&
	u->flags.access != flags->access)
	generate_error(ERROR_BAD_OPTION,
		       "Cannot change ACCESS parameter in OPEN statement");

    if (flags->form != FORM_UNSPECIFIED &&
	u->flags.form != flags->form)
	generate_error(ERROR_BAD_OPTION,
		       "Cannot change FORM parameter in OPEN statement");

    if (ioparm->recl_in != NULL &&
	extract_mint(ioparm->recl_in, ioparm->recl_in_kind) != u->recl)
	generate_error(ERROR_BAD_OPTION,
		       "Cannot change RECL parameter in OPEN statement");

    if (flags->action != ACTION_UNSPECIFIED &&
	u->flags.access != flags->access)
	generate_error(ERROR_BAD_OPTION,
		       "Cannot change ACTION parameter in OPEN statement");

    /* Status must be OLD if present */

    if (flags->status != STATUS_UNSPECIFIED && flags->status != STATUS_OLD &&
	flags->status != STATUS_UNKNOWN)
	generate_error(ERROR_BAD_OPTION,
		       "OPEN statement must have a STATUS of OLD");

    if (u->flags.form == FORM_UNFORMATTED) {
	if (flags->delim != DELIM_UNSPECIFIED)
	    generate_error(ERROR_OPTION_CONFLICT,
			   "DELIM parameter conflicts with UNFORMATTED form "
			   "in OPEN statement");

	if (flags->blank != BLANK_UNSPECIFIED)
	    generate_error(ERROR_OPTION_CONFLICT,
			   "BLANK parameter conflicts with UNFORMATTED form "
			   "in OPEN statement");

	if (flags->pad != PAD_UNSPECIFIED)
	    generate_error(ERROR_OPTION_CONFLICT,
			   "PAD paramter conflicts with UNFORMATTED form in "
			   "OPEN statement");
    }

    if (ioparm->library_rc == LIBRARY_OK) {  /* Change the changeable */
	if (flags->blank != BLANK_UNSPECIFIED)   u->flags.blank = flags->blank;
	if (flags->delim != DELIM_UNSPECIFIED)   u->flags.delim = flags->delim;
	if (flags->pad   != PAD_UNSPECIFIED)     u->flags.pad   = flags->pad;
	if (flags->decimal != DECIMAL_UNSPECIFIED)
	    u->flags.decimal = flags->decimal;
    }

    if (flags->position == POSITION_UNSPECIFIED)
	flags->position = POSITION_ASIS;

    position_file(u, flags->position);
}



/* new_unit()-- Open an unused unit */

static void new_unit(unit_flags *flags) {
iounit_t *u;

    /* Change unspecifieds to defaults */

    if (flags->access == ACCESS_APPEND) {
	flags->access = ACCESS_SEQUENTIAL;
	flags->position = POSITION_APPEND; /*Should precede POSITION handling*/

    } else if (flags->access == ACCESS_UNSPECIFIED)
	flags->access = ACCESS_SEQUENTIAL;

    if (flags->action == ACTION_UNSPECIFIED)
	flags->action = default_action();    /* Processor dependent */

    if (flags->form == FORM_UNSPECIFIED)
	flags->form = (flags->access == ACCESS_SEQUENTIAL)
	    ? FORM_FORMATTED : FORM_UNFORMATTED;

    if (flags->decimal == DECIMAL_UNSPECIFIED)
	flags->decimal = options.decimal_comma ? DECIMAL_COMMA : DECIMAL_POINT;

    if (flags->delim == DELIM_UNSPECIFIED)
	flags->delim = DELIM_NONE;

    else if (flags->form == FORM_UNFORMATTED) {
	generate_error(ERROR_OPTION_CONFLICT,
		       "DELIM parameter conflicts with UNFORMATTED form in "
		       "OPEN statement");
	return;
    }

    if (flags->blank == BLANK_UNSPECIFIED)
	flags->blank = BLANK_NULL;

    else if (flags->form == FORM_UNFORMATTED) {
	generate_error(ERROR_OPTION_CONFLICT,
		       "BLANK parameter conflicts with UNFORMATTED form in "
		       "OPEN statement");
	return;
    }

    if (flags->pad == PAD_UNSPECIFIED)
	flags->pad = PAD_YES;

    else if (flags->form == FORM_UNFORMATTED) {
	generate_error(ERROR_OPTION_CONFLICT,
		       "PAD parameter conflicts with UNFORMATTED form in "
		       "OPEN statement");
	return;
    }

    if (flags->position == POSITION_UNSPECIFIED)
	flags->position = POSITION_ASIS;

    else if (flags->access == ACCESS_DIRECT) {
	generate_error(ERROR_OPTION_CONFLICT,
		       "ACCESS parameter conflicts with SEQUENTIAL access in "
		       "OPEN statement");
	return;
    }

    if (flags->status == STATUS_UNSPECIFIED)
	flags->status = STATUS_UNKNOWN;

    /* Checks */

    if (flags->access == ACCESS_DIRECT && ioparm->recl_in == NULL) {
	generate_error(ERROR_MISSING_OPTION,
		       "Missing RECL parameter in OPEN statement");
	return;
    }

    if (ioparm->recl_in != NULL &&
	extract_mint(ioparm->recl_in, ioparm->recl_in_kind) <= 0) {
	generate_error(ERROR_BAD_OPTION,
		       "RECL parameter is non-positive in OPEN statement");
	return;
    }

    if (flags->status == STATUS_SCRATCH && ioparm->file != NULL) {
	generate_error(ERROR_BAD_OPTION,
		       "FILE parameter incompatible with STATUS='SCRATCH' "
		       "in OPEN statement");
	return;
    }

    /* The standard requires that an open without a filename must also
     * have a status of SCRATCH, but it appears customary to not check
     * this. */

    u = open_unit(flags);
    if (u == NULL)
	return;

    /* The file is now connected.  Errors after this point leave the
     * file connected.  Curiously, the standard requires that the
     * position specifier be ignored for new files so a newly connected
     * file starts out that the initial point.  We still need to figure
     * out if the file is at the end or not. */

    position_file(u, flags->position);
}



/* already_open()-- Open a unit which is already open.  This involves
 * changing the modes or closing what is there now and opening the new
 * file. */

static void already_open(iounit_t *u, unit_flags *flags) {

    if (flags->position != POSITION_UNSPECIFIED &&
	u->flags.access == ACCESS_DIRECT) {
	generate_error(ERROR_BAD_OPTION,
		       "Cannot use POSITION with direct access files");
	return;
    }

    if (ioparm->file == NULL && flags->status != STATUS_SCRATCH) {
	edit_modes(u, flags);
	return;
    }

    /* If the file is connected to something else, close it and open a
     * new unit */

    if (!compare_file_filename(u, ioparm->file, ioparm->file_len)) {
	if (close_unit(u)) {
	    generate_error(ERROR_OS, "Error closing file in OPEN statement");
	    return;
	}

	new_unit(flags);
	return;
    }

    edit_modes(u, flags);
}



/* check_f_open_rules()-- Enforce addition constraints on F opens. */

static void check_f_open_rules(unit_flags *flags) {

    if ((flags->status == STATUS_NEW || flags->status == STATUS_REPLACE) &&
	flags->action == ACTION_READ)
	generate_error(ERROR_BAD_OPTION,
		       "If status is NEW or REPLACE, action must not be READ");

    if (flags->status == STATUS_SCRATCH && flags->action != ACTION_READWRITE)
	generate_error(ERROR_BAD_OPTION,
		       "If status is SCRATCH, action must be READWRITE");

    if (flags->status == STATUS_UNKNOWN)
	generate_error(ERROR_BAD_OPTION,
		       "Status must be NEW, OLD, or REPLACE");

    if ((flags->access == ACCESS_UNSPECIFIED ||
	 flags->access == ACCESS_SEQUENTIAL) &&
	flags->status == STATUS_OLD &&
	flags->position == POSITION_UNSPECIFIED) 
	generate_error(ERROR_BAD_OPTION,
		    "POSITION must be specified for an OLD SEQUENTIAL file");
}



/* st_open()-- Open a unit */

void st_open(void) {
unit_flags flags;
iounit_t *u;

    library_start();

    /* Decode options */

    flags.access = (ioparm->access == NULL) ? ACCESS_UNSPECIFIED :
	find_option(ioparm->access, ioparm->access_len, access_opt,
		    "Bad ACCESS parameter in OPEN statement");

    flags.action = (ioparm->action == NULL) ? ACTION_UNSPECIFIED :
	find_option(ioparm->action, ioparm->action_len, action_opt,
		    "Bad ACTION parameter in OPEN statement");

    flags.blank = (ioparm->blank == NULL) ? BLANK_UNSPECIFIED :
	find_option(ioparm->blank, ioparm->blank_len, blank_opt,
		    "Bad BLANK parameter in OPEN statement");

    flags.delim = (ioparm->delim == NULL) ? DELIM_UNSPECIFIED :
	find_option(ioparm->delim, ioparm->delim_len, delim_opt,
		    "Bad DELIM parameter in OPEN statement");

    flags.pad = (ioparm->pad == NULL) ? PAD_UNSPECIFIED :
	find_option(ioparm->pad, ioparm->pad_len, pad_opt,
		    "Bad PAD parameter in OPEN statement");

    flags.form = (ioparm->form == NULL) ? FORM_UNSPECIFIED :
	find_option(ioparm->form, ioparm->form_len, form_opt,
		    "Bad FORM parameter in OPEN statement");

    flags.position = (ioparm->position == NULL) ? POSITION_UNSPECIFIED :
	find_option(ioparm->position, ioparm->position_len, position_opt,
		    "Bad POSITION parameter in OPEN statement");

    flags.status = (ioparm->status == NULL) ? STATUS_UNSPECIFIED :
	find_option(ioparm->status, ioparm->status_len, status_opt,
		    "Bad STATUS parameter in OPEN statement");

    flags.decimal = (ioparm->decimal == NULL) ? DECIMAL_UNSPECIFIED :
	find_option(ioparm->decimal, ioparm->decimal_len, decimal_opt,
		    "Bad DECIMAL parameter in OPEN statement");

    flags.endian = (ioparm->convert == NULL) ? ENDIAN_UNSPECIFIED :
        find_option(ioparm->convert, ioparm->convert_len, convert_opt, 
	    "Bad CONVERT parameter in OPEN statement");

    if (ioparm->unit < 0)
	generate_error(ERROR_BAD_OPTION, "Bad unit number in OPEN statement");

    if (ioparm->library_rc == LIBRARY_OK) {
	u = find_unit(ioparm->unit, ioparm->unit_kind);

	if (STD_F) {
	    if (u == NULL) {
		check_f_open_rules(&flags);
		new_unit(&flags);

	    } else
		generate_error(ERROR_BAD_OPTION,
			       "Can't open an F file that is open");

	} else if (u == NULL)
	    new_unit(&flags);

	else
	    already_open(u, &flags);
    }

    library_end();
}

