
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


#include <setjmp.h>
#include <string.h>
#include <limits.h>

#include "safe-ctype.h"
#include "runtime.h"


/* List directed input.  Several parsing subroutines are practically
 * reimplemented from formatted input, the reason being that there are
 * all kinds of small differences between formatted and list directed
 * parsing. */


/* Subroutines for reading characters from the input.  Because a
 * repeat count is ambiguous with an integer, we have to read the
 * whole digit string before seeing if there is a '*' which signals
 * the repeat count.  Since we can have a lot of potential leading
 * zeros, we have to be able to back up by arbitrary amount.  Because
 * the input might not be seekable, we have to buffer the data
 * ourselves.  Data is buffered in scratch[] until it becomes too
 * large, after which we start allocating memory on the heap.  */

static int repeat_count, saved_length, saved_used, input_complete;

static int last_char, last_char2, last_char3;
static char *saved_string;
static bt saved_type;

static jmp_buf eof_jump;

/* Storage area for values except for strings.  Must be large enough
 * to hold a complex value (two reals) of the largest kind */

static char value[20];


typedef struct {
    int start, end, stride;

    unsigned range:1, have_start:1, have_end:1;
} dimen_ref;


typedef struct {
    int rank, range;
    dimen_ref sub[G95_MAX_DIMENSIONS];
} array_ref;


typedef struct {
    bt type;
    int kind, offset, have_section, tentative_section, array, seen_substring;
    char *pointer;

    array_ref ref;

    derived_info *dt_desc;
    g95_array_descriptor desc, section;
} namelist_read_info;

static namelist_read_info info;


/* Buffer for namelist context. */

#define NAMELIST_CONTEXT_SIZE 60

static char namelist_context[NAMELIST_CONTEXT_SIZE];
static int namelist_csize;
static G95_DINT namelist_line;

static char namelist_eof[] = "Unexpected end of namelist";


/* Maximum repeat count.  Less than ten times the maximum value of the
 * default integer. */

#define MAX_REPEAT 200000000

#define EOF (-1)


/* These are the 'constant' separators */

#define CASE_SEPARATORS  case ' ':  case '/': case '\n': case '\t': case '\r'

/* This macro assumes that we're operating on a variable */

#define is_whitespace(c) \
    (c == ' ' || c == '\t' || c == '\n' || c == '\r')


#define CASE_DIGITS   case '0': case '1': case '2': case '3': case '4': \
                      case '5': case '6': case '7': case '8': case '9'


/* is_separator()-- Return nonzero if a character is a separator. */ 

static int is_separator(int c) {

    if (c == '/'  || c == '\n' || c == ' ' || \
	c == '\r' || c == '\t' || c == EOF)
	return 1;

    if (c == ',' && ioparm->current_decimal == DECIMAL_POINT)
	return 1;

    if (c == ';' && ioparm->current_decimal == DECIMAL_COMMA)
	return 1;

    return 0;
}



/* push_char()-- Save a character to a string buffer, enlarging it as
 * necessary. */

static void push_char(char c) {
char *new;

    if (saved_string == NULL) {
	saved_string = scratch;
	saved_length = SCRATCH_SIZE;
	saved_used = 0;
    }

    if (saved_used >= saved_length) {
	saved_length = 2*saved_length;
	new = get_mem(2*saved_length);

	memcpy(new, saved_string, saved_used);
	if (saved_string != scratch)
	    free_mem(saved_string);

	saved_string = new;
    }

    saved_string[saved_used++] = c;
}



/* free_saved()-- Free the input buffer if necessary. */

static void free_saved(void) {

    if (saved_string == NULL)
	return;

    if (saved_string != scratch)
	free_mem(saved_string);

    saved_string = NULL;
    saved_length = 0;
    saved_used = 0;
}



/* next_char()-- Get the next character for a list read. */

static int next_char(void) {
int c;

    if (last_char != '\0') {
	c = last_char;
	last_char = last_char2;
	last_char2 = last_char3;
	last_char3 = '\0';

    } else {
	c = next_list_char();

	if (c == '\n')
	    namelist_csize = 0;

	else {
	    namelist_context[namelist_csize++] = c;
	    if (namelist_csize >= NAMELIST_CONTEXT_SIZE) {
		memmove(namelist_context, namelist_context+20,
			NAMELIST_CONTEXT_SIZE-20);
		namelist_csize -= 20;
	    }

	    namelist_context[namelist_csize] = 0;
	}
    }

    return c;
}



/* unget_char()-- Push a character back onto the input */

static void unget_char(char c) {

    last_char3 = last_char2;
    last_char2 = last_char;
    last_char = c;
}



/* eat_whitespace()-- Skip over whitespace in the input.  Returns the
 * nonspace character that terminated the eating and also places it
 * back on the input. */

static int eat_whitespace(void) {
int c;

    do
	c = next_char();
    while(c == ' ' || c == '\t' || c == '\n' || c == '\r');

    unget_char(c);

    return c;
}



/* eat_separator()-- Skip over a separator. */

static void eat_separator(void) {
int c;

    eat_whitespace();

    c = next_char();
    switch(c) {
    case ',':
	if (ioparm->current_decimal != DECIMAL_POINT)
	    unget_char(c);
	else
	    eat_whitespace();

	break;

    case ';':
	if (ioparm->current_decimal != DECIMAL_COMMA)
	    unget_char(c);
	else
	    eat_whitespace();

	break;

    case '/':
	input_complete = 1;
	break;

    case '!':
	if (ioparm->namelist != NULL) {    /* Eat a namelist comment */
	    do
		c = next_char();
	    while(c != '\n');

	    break;
	}

	/* Fall Through */

    default:
	unget_char(c);
	break;
    }
}



/* convert_repeat()-- Convert a repeat count. */

static int convert_repeat(void) {
char c, *buffer, message[100];
int m, max, max10, v;

    buffer = saved_string;
    v = 0;

    max = MAX_REPEAT;
    max10 = max / 10;

    for(;;) {
	c = *buffer++;
	if (c == '\0')
	    break;

	c -= '0';

	if (v > max10)
	    goto overflow;

	v = 10*v;

	if (v > max-c)
	    goto overflow;

	v += c;
    }

    m = 0;
    repeat_count = v;

    if (repeat_count == 0) {
	st_sprintf(message, "Zero repeat count in item %d of list input",
		   ioparm->item_count);

	generate_error(ERROR_READ_VALUE, message);
	m = 1;
    }

    free_saved();
    return m;

overflow:
    st_sprintf(message, "Repeat count overflow in item %d of list input",
	       ioparm->item_count);

    free_saved();
    generate_error(ERROR_READ_VALUE, message);

    return 1;
}



/* parse_repeat()-- Parse a repeat count for logical and complex
 * values which cannot begin with a digit.  Returns nonzero if we are
 * done, zero if we should continue on. */

static int parse_repeat(void) {
char message[100];
int c, repeat;

    c = next_char();
    switch(c) {
    CASE_DIGITS:
	repeat = c - '0';
	break;

    CASE_SEPARATORS:
	unget_char(c);
	return 1;

    case ',':
	unget_char(c);
	return ioparm->current_decimal == DECIMAL_POINT;

    case ';':
	unget_char(c);
	return ioparm->current_decimal == DECIMAL_COMMA;

    default:
	unget_char(c);
	return 0;

    case EOF:
	longjmp(eof_jump, 1);
    }

    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    repeat = 10*repeat + c - '0';

	    if (repeat > MAX_REPEAT) {
		st_sprintf(message,
			   "Repeat count overflow in item %d of list input",
			   ioparm->item_count);

		generate_error(ERROR_READ_VALUE, message);
		return 1;
	    }

	    break;

	case '*':
	    if (repeat == 0) {
		st_sprintf(message,
			   "Zero repeat count in item %d of list input",
			   ioparm->item_count);

		generate_error(ERROR_READ_VALUE, message);
		return 1;
	    }

	    goto done;

	default:
	    goto bad_repeat;
	}
    }

done:
    repeat_count = repeat;
    return 0;

bad_repeat:
    st_sprintf(message, "Bad repeat count in item %d of list input",
	       ioparm->item_count);

    generate_error(ERROR_READ_VALUE, message);
    return 1;
}



/* read_character()-- Read a character variable */

static int read_character(int length) {
char quote;
int c;

    quote = ' ';    /* Space means no quote character */

    c = next_char(); 
    switch(c) {
    CASE_DIGITS:
	push_char(c);
	break;

    CASE_SEPARATORS:
	unget_char(c);     /* NULL value */
	return BT_NULL;

    case ',':
	if (ioparm->current_decimal == DECIMAL_POINT) {
	    unget_char(c);
	    return BT_NULL;
	}

	push_char(c);
	goto get_string;

    case ';':
	if (ioparm->current_decimal == DECIMAL_COMMA) {
	    unget_char(c);
	    return BT_NULL;
	}

	push_char(c);
	goto get_string;

    case '"':  case '\'':
	quote = c;
	goto get_string;

    case EOF:
	longjmp(eof_jump, 1);

    default:
	if (ioparm->namelist != NULL) {
	    unget_char(c);
	    return BT_NULL;
	}

	push_char(c);
	goto get_string;
    }

/* Deal with a possible repeat count */

    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    push_char(c);
	    break;

	case EOF:
	CASE_SEPARATORS:
	    unget_char(c);
	    goto done;      /* String was only digits! */

	case ',':
	    if (ioparm->current_decimal == DECIMAL_POINT) {
		unget_char(c);
		goto done;
	    }

	    push_char(c);
	    goto get_string;

	case ';':
	    if (ioparm->current_decimal == DECIMAL_COMMA) {
		unget_char(c);
		goto done;
	    }

	    push_char(c);
	    goto get_string;

	case '*':
	    push_char('\0');
	    goto got_repeat;

	default:
	    push_char(c);
	    goto get_string;   /* Not a repeat count after all */
	}
    }

got_repeat:
    if (convert_repeat())
	return BT_NULL;

    /* Now get the real string */

    c = next_char();
    switch(c) {
    case EOF:
    CASE_SEPARATORS:
	unget_char(c);     /* repeated NULL values */
	return BT_NULL;

    case ',':
	if (ioparm->decimal == DECIMAL_POINT) {
	    unget_char(c);
	    return BT_NULL;
	}

	push_char(c);
	break;

    case ';':
	if (ioparm->current_decimal == DECIMAL_COMMA) {
	    unget_char(c);
	    return BT_NULL;
	}

	push_char(c);
	break;

    case '"':  case '\'':
	quote = c;
	break;

    default:
	push_char(c);
	break;
    }

get_string:
    for(;;) {
	c = next_char();
	switch(c) {
	case '"':
	case '\'':
	    if (c != quote) {
		push_char(c);
		break;
	    }

	    /* See if we have a doubled quote character or the end of
	     * the string */

	    c = next_char();
	    if (c == quote) {
		push_char(quote);
		break;
	    }

	    unget_char(c);
	    goto done;

	case ',':
	    if (ioparm->current_decimal == DECIMAL_POINT)
		goto sep;

	    push_char(c);
	    break;

	case ';':
	    if (ioparm->current_decimal == DECIMAL_COMMA)
		goto sep;

	    push_char(c);
	    break;

	case EOF:
	    unget_char(c);
	    goto done;

	CASE_SEPARATORS:
	sep:
	    if (quote == ' ') {
		unget_char(c);
		goto done;
	    }

	    if (c != '\n')
		push_char(c);

	    break;

	default:
	    push_char(c);
	    break;
	}
    }

done:
    return BT_CHARACTER;
}



/* read_logical()-- Read a logical character on the input */

static int read_logical(int length) {
char message[100];
int c, v;

    if (parse_repeat())
	return BT_NULL;

    c = next_char();
    switch(c) {
    case 't': case 'T':   v = 1;    break;
    case 'f': case 'F':   v = 0;    break;

    case '.':
	c = next_char();
	switch(c) {
	case 't': case 'T':   v = 1;    break;
	case 'f': case 'F':   v = 0;    break;
	default:
	    goto bad_logical;
	}

	break;

    case ',':
	if (ioparm->current_decimal == DECIMAL_POINT)
	    goto sep;

	goto bad_logical;

    case ';':
	if (ioparm->current_decimal == DECIMAL_COMMA)
	    goto sep;

	goto bad_logical;

    case EOF:
    CASE_SEPARATORS:
    sep:
	unget_char(c);
	return BT_NULL;    /* Null value */

    default:
	goto bad_logical;
    }

    saved_length = length;

    /* Eat trailing garbage */

    do
	c = next_char();
    while(!is_separator(c));

    unget_char(c);
    set_integer(v, value, length);

    return BT_LOGICAL;

bad_logical:
    st_sprintf(message, "Bad logical value while reading item %d",
	       ioparm->item_count);
 
    generate_error(ERROR_READ_VALUE, message);
    return BT_LOGICAL;
}



/* read_integer()-- Reading integers is tricky because we can actually
 * be reading a repeat count.  We have to store the characters in a
 * buffer because we could be reading an integer that is larger than the
 * default int used for repeat counts.  */

static int read_integer(int length) {
char message[100];
int c, negative;

    negative = 0; 

    c = next_char();
    switch(c) {
    case '-':
	negative = 1;
	/* Fall through */

    case '+':
	c = next_char();
	goto get_integer;

    case ',':
	if (ioparm->current_decimal == DECIMAL_POINT) {
	    unget_char(c);
	    return BT_NULL;
	}

	goto bad_integer;

    case ';':
	if (ioparm->current_decimal == DECIMAL_COMMA) {
	    unget_char(c);
	    return BT_NULL;
	}

	goto bad_integer;

    CASE_SEPARATORS:   /* Single null */
	unget_char(c);
	return BT_NULL;
 
    CASE_DIGITS:
	push_char(c);
	break;

    case EOF:
	longjmp(eof_jump, 1);

    default:
	goto bad_integer;
    }

    /* Take care of what may be a repeat count */

    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    push_char(c);
	    break;

	case '*':
	    push_char('\0');
	    goto repeat;

	case EOF:
	CASE_SEPARATORS:   /* Not a repeat count */
	    goto done;

	default:
	    if (c == COMMA_SEP())
		goto done;

	    goto bad_integer;
	}
    }

repeat:
    if (convert_repeat())
	return BT_NULL;

/* Get the real integer */

    c = next_char();
    switch(c) {
    CASE_DIGITS:
	break;

    case EOF:
    CASE_SEPARATORS:
	unget_char(c);
	return BT_NULL;
  
    case '-':
	negative = 1;
	/* Fall through */

    case '+':
	c = next_char();
	break;

    default:
	if (c == COMMA_SEP()) {
	    unget_char(c);
	    return BT_NULL;
	}

	break;
    }

get_integer:
    if (!isdigit(c))
	goto bad_integer;

    push_char(c);

    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    push_char(c);
	    break;

	case ',':
	    if (ioparm->current_decimal == DECIMAL_POINT)
		goto done;

	    goto bad_integer;

	case ';':
	    if (ioparm->current_decimal == DECIMAL_POINT)
		goto done;

	    goto bad_integer;

	case EOF:
	CASE_SEPARATORS:
	    goto done;

	default:
	    goto bad_integer;
	}
    }

bad_integer:
    free_saved();

    st_sprintf(message, "Bad integer for item %d in list input",
	       ioparm->item_count);
    generate_error(ERROR_READ_VALUE, message);

    return BT_NULL;

done:
    unget_char(c);
    push_char('\0');

    c = read_list_integer(saved_string, negative, length, value);
    free_saved();

    return c ? BT_NULL : BT_INTEGER;
}



/* parse_exceptional()-- Parse an IEEE exceptional values, Infinity or
 * Not-a-number. */

static int parse_exceptional(int sign, char first, char *buffer, int length) {
char c, message[100];
int mantissa;

    if (first == 'i' || first == 'I') {
	if (tolower(next_char()) != 'n' ||
	    tolower(next_char()) != 'f')
	    goto bad_infinity;

	c = tolower(next_char());
	if (c != 'i')
	    unget_char(c);
	else {
	    if (tolower(next_char()) != 'n' ||
		tolower(next_char()) != 'i' ||
		tolower(next_char()) != 't' ||
		tolower(next_char()) != 'y')
		goto bad_infinity;
	}

	mantissa = 0;

    } else {
	if (tolower(next_char()) != 'a' ||
	    tolower(next_char()) != 'n')
	    goto bad_nan;

	c = next_char();
	if (c != '(') {
	    unget_char(c);
	    mantissa = 959595;
	} else {
	    mantissa = 0;

	loop:
	    c = next_char();
	    switch(c) {
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		mantissa = (mantissa << 4) | (c - '0');
		goto loop;

	    case 'a': case 'b': case 'c': case 'd': case 'e':
		mantissa = (mantissa << 4) | (c - 'a' + 10);
		goto loop;

	    case 'A': case 'B': case 'C': case 'D': case 'E':
		mantissa = (mantissa << 4) | (c - 'A' + 10);
		goto loop;

	    case ')':
		break;

	    default:
		goto bad_nan;
	    }
	}
    }

    build_nan(sign, mantissa, buffer, length);
    return 0;

bad_infinity:
    st_sprintf(message, "Bad infinity in item %d of list input",
	       ioparm->item_count);

    generate_error(ERROR_READ_VALUE, message);
    return 1;

bad_nan:
    st_sprintf(message, "Bad not-a-number in item %d of list input",
	       ioparm->item_count);

    generate_error(ERROR_READ_VALUE, message);
    return 1;
}



/* read_real()-- Parse a real number with a possible repeat count. */

static int read_real(int length) {
int c, d, seen_digit, seen_dp;
char message[100];

    seen_digit = 0;
    seen_dp = 0;

    c = next_char(); 
    switch(c) {
    CASE_DIGITS:
	seen_digit = 1;
	push_char(c);
	break;

    case '+': case '-':
	push_char(c);

	d = next_char();
	if (d == 'i' || d == 'I' || d == 'n' || d == 'N')
	    return parse_exceptional(c == '-', d, value, length)
		? BT_NULL
		: BT_REAL;

	unget_char(d);
	goto real_loop;

    case 'i': case 'I': case 'n': case 'N':
	return parse_exceptional(0, c, value, length) ? BT_NULL : BT_REAL;

    CASE_SEPARATORS:
	unget_char(c);       /* Single null */
	return BT_NULL;

    case EOF:
	longjmp(eof_jump, 1);

    default:
	if (c == DECIMAL_CHAR()) {
	    push_char('.');
	    seen_dp = 1;
	    goto real_loop;
	}

	if (c == COMMA_SEP()) {
	    unget_char(c);
	    return BT_NULL;
	}

	goto bad_real;
    }

    /* Get the digit string that might be a repeat count */

    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    seen_digit = 1;
	    push_char(c);
	    break;

	case 'E': case 'e':
	case 'D': case 'd':
	    goto exp1;

	case '+': case '-':
	    push_char('e');
	    push_char(c);
	    c = next_char();
	    goto exp2;

	case '*':
	    push_char('\0');
	    goto got_repeat;

	case EOF:
	CASE_SEPARATORS:
	    goto done;    /* Real number that is just a digit-string */

	default:
	    if (c == DECIMAL_CHAR()) {
		seen_dp = 1;
		push_char('.');
		goto real_loop;
	    }

	    if (c == COMMA_SEP())
		goto done;

	    goto bad_real;
	}
    }

got_repeat:
    if (convert_repeat())
	return BT_NULL;

    seen_digit = 0;

/* Now get the number itself */

    c = next_char();

    switch(c) {
    case EOF:
    CASE_SEPARATORS:
	unget_char(c);
	return BT_NULL;

    case '-': case '+':
	push_char(c);
	break;

    default:
	unget_char(c);

	if ((ioparm->current_decimal == DECIMAL_POINT && c == ',') ||
	    (ioparm->current_decimal == DECIMAL_COMMA && c == ';')) {
	    return BT_NULL;
	}

	break;
    }

real_loop:
    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    seen_digit = 1;
	    push_char(c);
	    break;

	case EOF:
	CASE_SEPARATORS:
	    goto done;

	case 'E': case 'e':
	case 'D': case 'd':
	    goto exp1;

	case '+': case '-':
	    push_char('e');
	    push_char(c);
	    c = next_char();
	    goto exp2;

	default:
	    if (c == DECIMAL_CHAR()) {
		if (seen_dp)
		    goto bad_real;

		seen_dp = 1;
		push_char('.');
		break;
	    }

	    if (c == COMMA_SEP())
		goto done;

	    goto bad_real;
	}
    }

exp1:
    push_char('e');

    c = next_char();
    if (c != '+' && c != '-') 
	push_char('+');

    else {
	push_char(c);
	c = next_char();
    }

exp2:
    if (!isdigit(c))
	goto bad_real;

    push_char(c);

    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    push_char(c);
	    break;

	case EOF:
	CASE_SEPARATORS:
	    goto done;
  
	default:
	    if (c == COMMA_SEP())
		goto done;

	    goto bad_real;
	}
    }

done:
    unget_char(c);
    push_char('\0');

    convert_real(value, saved_string, length);
    return BT_REAL;

bad_real:
    st_sprintf(message, "Bad real number in item %d of list input",
	       ioparm->item_count);

    generate_error(ERROR_READ_VALUE, message);
    return BT_NULL;
}


/* parse_real()-- Parse a component of a complex constant or a real
 * number that we are sure is already there.  This is a straight real
 * number parser. */

static int parse_real(void *buffer, int length) {
int c, d, seen_digit, seen_dp;
char message[100];

    c = next_char();

    if (c != '-' && c != '+')
	unget_char(c);
    else {
	d = next_char();
	if (d == 'i' || d == 'I' || d == 'n' || d == 'N')
	    return parse_exceptional(c == '-', d, buffer, length);

	push_char(c);
	unget_char(d);
    }

    seen_digit = 0;
    seen_dp = 0;

    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    seen_digit = 1;
	    push_char(c);
	    break;

	case ',':
	    if (ioparm->current_decimal != DECIMAL_COMMA)
		goto done;

	    /* Fall through */

	case '.':
	    if (seen_dp)
		goto bad;

	    seen_dp = 1;
	    push_char('.');
	    break;

	case 'i': case 'I': case 'n': case 'N':
	    return parse_exceptional(0, c, buffer, length);

	case 'e': case 'E':
	case 'd': case 'D':
	    push_char('e');
	    goto exp1;

	case '-': case '+':
	    push_char('e');
	    push_char(c);
	    c = next_char();
	    goto exp2;

	default:
	    goto done;
	}
    }

exp1:
    c = next_char();
    if (c != '-' && c != '+')
	push_char('+');
    else {
	push_char(c);
	c = next_char();
    }

exp2:
    if (!isdigit(c))
	goto bad;

    push_char(c);

    for(;;) {
	c = next_char();
	if (c < '0' || c > '9')
	    break;

	push_char(c);
    }

done:
    if (!seen_digit)
	goto bad;

    unget_char(c);
    push_char('\0');

    convert_real(buffer, saved_string, length);
    free_saved();

    return 0;

bad:
    free_saved();
    st_sprintf(message, "Bad floating point number for item %d",
	       ioparm->item_count);
    generate_error(ERROR_READ_VALUE, message);

    return 1;
}



/* read_complex()-- Read a complex number.  We allow reading a real
 * and append a zero imaginary part. */

static int read_complex(int length) {
char message[100];
int seen_dp, c;

    c = next_char();

    switch(c) {
    case '(':
	goto have_complex;

    case '+': case '-':
    case 'i': case 'I': case 'n': case 'N':
	unget_char(c);
	parse_real(value, length);
	goto real_extended;

    CASE_DIGITS:
	push_char(c);
	break;

    CASE_SEPARATORS:
	unget_char(c);
	return BT_NULL;

    case EOF:
	longjmp(eof_jump, 1);

    default:
	if (c == COMMA_SEP()) {
	    unget_char(c);
	    return BT_NULL;
	}

	goto bad_complex;
    }

/* We've seen a digit.  It's either a repeat count or a real number */

    seen_dp = 0;

    for(;;) {
	c = next_char();
	switch(c) {
	CASE_DIGITS:
	    push_char(c);
	    break;

	case '.':
	    push_char(c);
	    if (seen_dp)
		goto bad_complex;

	    seen_dp = 1;
	    break;

	case 'E': case 'e':
	case 'D': case 'd':
	    goto exp1;

	case '+': case '-':
	    push_char('e');
	    push_char(c);
	    goto exp2;

	case '*':
	    if (seen_dp)
		goto bad_complex;

	    push_char('\0');
	    if (convert_repeat())
		return BT_NULL;

	    c = next_char();
	    if (c == '(')
		goto have_complex;

	    unget_char(c);

	    if (c != '+' && c != '-' && c != 'i' && c != 'I' &&
		c != 'n' && c != 'N' && (c < '0' || c > '9'))
		return BT_NULL;

	    if (parse_real(value, length))
		return BT_NULL;

	    goto real_extended;

	case EOF:
	CASE_SEPARATORS:
	    push_char('\0');
	    goto real_done;

	default:
	    if (c == COMMA_SEP()) {
		push_char('\0');
		goto real_done;
	    }

	    goto bad_complex;
	}
    }

exp1:
    c = next_char();

    if (c != '-' && c != '+')
	push_char('+');
    else {
	push_char(c);
	c = next_char();
    }

exp2:
    if (!isdigit(c))
	goto bad_complex;

    push_char(c);

    for(;;) {
	c = next_char();
	if (c < '0' || c > '9')
	    break;

	push_char(c);
    }

real_done:
    convert_real(value, saved_string, length);

real_extended:
    memset(value+length, '\0', length);
    return BT_COMPLEX;


/* Definitely a complex number, the leading left paren has been seen */

have_complex:
    eat_whitespace();
    if (parse_real(value, length))
	return BT_NULL;

    eat_whitespace();
    if (next_char() != COMMA_SEP())
	goto bad_complex;

    eat_whitespace();
    free_saved();
    if (parse_real(value + REAL_SIZE(length), length))
	return BT_NULL;

    eat_whitespace();
    if (next_char() != ')')
	goto bad_complex;

    c = next_char();
    if (!is_separator(c))
	goto bad_complex;

    unget_char(c);
    return BT_COMPLEX;

bad_complex:
    st_sprintf(message, "Bad complex value in item %d of list input",
	       ioparm->item_count);

    generate_error(ERROR_READ_VALUE, message);
    return BT_NULL;
}



/* check_type()-- Check the current type against the saved type to
 * make sure they are compatible.  Returns nonzero if incompatible. */

static int check_type(bt type, int len) {
char message[100];

    if (saved_type != BT_NULL && saved_type != type) {
	st_sprintf(message, "Read type %s where %s was expected for item %d",
		   type_name(saved_type), type_name(type), ioparm->item_count);

	generate_error(ERROR_READ_VALUE, message);
	return 1;
    }

    if (saved_type == BT_NULL || saved_type == BT_CHARACTER)
	return 0;

    if (saved_length != len) {
	st_sprintf(message,
		   "Read kind %d %s where kind %d is required for item %d",
		   saved_length, type_name(saved_type), len,
		   ioparm->item_count);
	generate_error(ERROR_READ_VALUE, message);
	return 1;
    }

    return 0;
}



/* list_formatted_read()-- Top level data transfer subroutine for list
 * reads.  Because we have to deal with repeat counts, the data item
 * is always saved after reading, usually in the value[] array.  If a
 * repeat count is greater than one, we copy the data item multiple
 * times. */

void list_formatted_read(bt type, void *p, int len) {
int m;

    if (ioparm->first_item) {
	ioparm->first_item = 0;
	input_complete = 0;
	repeat_count = 1;

	if (eat_whitespace() == EOF) {
	    next_char();

	    if (!options.ignore_endfile) {
		generate_error(ERROR_END, NULL);
		current_unit->endfile = AFTER_ENDFILE;
	    }

	    return;
	}

    } else {
	if (input_complete)
	    return;

	if (repeat_count > 0) {
	    if (check_type(type, len))
		return;

	    goto set_value;
	}

	if (ioparm->namelist == NULL)
	    eat_separator();

	if (input_complete)
	    return;

	repeat_count = 1;
    }

    if (setjmp(eof_jump)) {
	generate_error(ERROR_END, NULL);
	current_unit->endfile = AFTER_ENDFILE;
	return;
    }

    switch(type) {
    case BT_INTEGER:    saved_type = read_integer(len);    break;
    case BT_LOGICAL:    saved_type = read_logical(len);    break;
    case BT_CHARACTER:  saved_type = read_character(len);  break;
    case BT_REAL:       saved_type = read_real(len);       break;
    case BT_COMPLEX:    saved_type = read_complex(len);    break;
    default:            internal_error("Bad type for list read");
    }

    if (saved_type != BT_CHARACTER && saved_type != BT_NULL)
	saved_length = len;

    if (ioparm->library_rc != LIBRARY_OK)
	return;

set_value:
    switch(saved_type) {
    case BT_COMPLEX:
	len = 2*REAL_SIZE(len);

	/* Fall through */

    case BT_INTEGER:
    case BT_REAL:
    case BT_LOGICAL:
	memcpy(p, value, len);
	break;

    case BT_CHARACTER:
	m = (len < saved_used) ? len : saved_used;
	if (m > 0)
	    memcpy(p, saved_string, m);

	if (m < len)
	    memset(((char *) p) + m, ' ', len - m);

	break;

    case BT_NULL:
    default:
	break;
    }

    if (--repeat_count <= 0)
	free_saved();
}



/* finish_list_read()-- Finish a list read */

void finish_list_read(void) {
int c;

    free_saved();

    do
	c = next_char();
    while(c != '\n' && c != EOF);
}



/* letter_to_type()-- Convert a type in a derived type structure to an
 * internal enum. */

static int letter_to_type(int m) {
int n;

    switch(m) {
    case 'r':  n = BT_REAL;       break;
    case 'i':  n = BT_INTEGER;    break;
    case 'l':  n = BT_LOGICAL;    break;
    case 'c':  n = BT_CHARACTER;  break;
    case 'z':  n = BT_COMPLEX;    break;
    case 'd':  n = BT_DERIVED;    break;
    default:
	n = 0;
	internal_error("parse_part_spec(): Corrupt derived type");
    }

    return n;
}



/* next_namelist_char()-- Return the next interesting character as far
 * as namelists are concerned. */

static int next_namelist_char(void) {
int c;

    do {
	c = next_char();
	if (c == '!')
	    do
		c = next_char();
	    while (c != '\n' && c != EOF);

    } while(c == ' ' || c == '\n' || c == '\r' || c == '\t');

    return c;
}


/* namelist_error()-- Something has going wrong with a namelist read */

static void namelist_error(char *message) {
char *p, buffer[500];
int i;

    st_sprintf(buffer, "In line %d of namelist %s:\n%s\n%s\n", namelist_line,
	       ioparm->namelist, message, namelist_context);

    p = strchr(buffer, '\0');

    for(i=1; i<namelist_csize; i++)
	*p++ = ' ';

    *p++ = '^';
    *p++ = '\0';

    if (message == namelist_eof) {
	generate_error(ERROR_END, buffer);
	current_unit->endfile = AFTER_ENDFILE;

    } else
	generate_error(ERROR_READ_VALUE, buffer);
}



/* namelist_end()-- See if it is time to end a namelist input.
 * Returns nonzero if so and eats the end marker.  Otherwise, the
 * input is preserved. */

static int namelist_end(void) {
int c, end;

    c = next_namelist_char();

    end = (c == '/' || c == '&' || c == '$');
    if (!end)
	unget_char(c);

    return end;
}



/* read_derived()-- Read a derived type in a namelist.  This can
 * include a bunch of things. */

static void read_derived(derived_info *d, char *p) {
int i, extent, n, t;
char *q;

    while(d->name != NULL && !input_complete) {
	q = p + d->offset;
	t = letter_to_type(d->type);

	if (d->rank == 0) {
	    if (t == BT_DERIVED)
		read_derived(d->info, q);
	    else {
		list_formatted_read(t, q, d->kind);
		ioparm->item_count++;
	    }

	} else {
	    n = 1;
	    for(i=0; i<d->rank; i++) {
		extent = d->shape[2*i+1] - d->shape[2*i] + 1;
		if (extent < 0)
		    extent = 0;

		n *= extent;
	    }

	    for(i=0; i<n; i++) {
		if (t == BT_DERIVED)
		    read_derived(d->info, q);
		else {
		    list_formatted_read(t, q, d->kind);
		    ioparm->item_count++;
		}

		if (i != n-1)
		    eat_separator();

		q += d->kind;
	    }
	}

	if (d[1].name != NULL)
	    eat_separator();

	d++;
    }
}



/* parse_integer()-- Parse an integer array index. */

static int parse_integer(int *value) {
G95_DINT max10, v;
int negative;
char c;

    c = next_char();
    negative = 0;

    switch(c) {
    case '-':
	negative = 1;
	c = next_char();
	break;

    case '+':
	negative = 0;
	c = next_char();
	break;

    default:
	break;
    }

    if (!isdigit(c)) {
	generate_error(ERROR_READ_VALUE, "Bad integer in array index");
	return 1;
    }

    max10 = G95_DINT_MAX / 10;
    v = c - '0';

    for(;;) {
	c = next_char();
	if (!isdigit(c))
	    break;

	if (v > max10)
	    goto overflow;

	v = 10 * v;
	c -= '0';

	if (v > G95_DINT_MAX - c)
	    goto overflow;

	v += c;
    }

    unget_char(c);

    if (negative)
	v = -v;

    *value = v;
    return 0;

overflow:
    namelist_error("Integer overflow in array subscript");
    return 1;
}



/* parse_name()-- Parse a variable or component name.  Returns nonzero
 * if something went wrong. */

static int parse_name(char *name) {
int i, c;

    c = next_char();

    if (c == EOF) {
	input_complete = 1;
	return 1;
    }

    c = tolower(c);

    if (!isalpha(c)) {
	namelist_error("Bad name");
	return 1;
    }

    name[0] = c;
    i = 1;

    for(;;) {
	c = tolower(next_char());
	if (!isalpha(c) && !isdigit(c) && c != '_' && c != '$')
	    break;

	name[i++] = c;
	if (i == G95_MAX_SYMBOL_LEN) {
	    namelist_error("Name too long");
	    return 1;
	}
    }

    name[i] = '\0';
    unget_char(c);

    return 0;
}



/* read_array()-- Read elements of an array.  Returns nonzero if
 * something is messed up, which can only be an out of bounds array
 * reference. */

static int read_array(void) {
int rank, c, d, m, i, index[G95_MAX_DIMENSIONS], start[G95_MAX_DIMENSIONS],
    end[G95_MAX_DIMENSIONS];
char *p;

    rank = info.ref.rank;

    for(i=0; i<rank; i++) {
	start[i] = info.ref.sub[i].have_start
	    ? info.ref.sub[i].start
	    : info.desc.info[i].lbound;

	end[i] = info.ref.sub[i].have_end
	    ? info.ref.sub[i].end
	    : info.desc.info[i].ubound;

	m = index[i] = (info.tentative_section == 1)
	    ? info.ref.sub[i].start
	    : start[i];

	if (m < info.desc.info[i].lbound || m > info.desc.info[i].ubound) {
	    namelist_error("Array reference out of bounds");
	    return 1;
	}
    }

    for(;;) {
	p = ((char *) info.desc.offset) + info.offset;
	for(i=0; i<rank; i++)
	    p += index[i] * info.desc.info[i].mult;

	if (repeat_count == 0) {
	    if (input_complete)
		return 0;

	    c = next_namelist_char();

	    if (c == '/' || c == '&' || c == '$' || c == EOF) {
		unget_char(c);
		return 0;
	    }

	    if (isdigit(c) || c == COMMA_SEP())
		unget_char(c);

	    else
		switch(info.type) {
		case BT_CHARACTER:
		    unget_char(c);
		    if (c !='\'' && c != '\"')
			return 0;

		    break;

		case BT_LOGICAL:
		    if (c != 't' && c != 'T' && c != 'f' && c != 'F' &&
			c != '.') {
			unget_char(c);
			return 0;
		    }

		    d = next_char();

		    if (d == ' ' || d == '\n' || d == '\r' || c == '\t') {
			d = next_namelist_char();

			unget_char(d);
			unget_char(' ');
		    } else
			unget_char(d);

		    unget_char(c);

		    if (d == '=')
			return 0;

		    break;

		default:
		    unget_char(c);

		    if (isalpha(c))
			return 0;

		    break;
		}
	}

	if (info.type == BT_DERIVED)
	    read_derived(info.dt_desc, p);

	else
	    list_formatted_read(info.type, p, info.kind);

	ioparm->item_count++;
	if (repeat_count == 0)
	    eat_separator();

	/* Bump the index to the next element */

	i = 0;
	for(;;i++) {
	    if (i >= rank)
		goto done;

	    if (!info.ref.sub[i].range && info.tentative_section != 1)
		continue;

	    index[i] += info.ref.sub[i].stride;

	    if (info.ref.sub[i].stride > 0) {
		if (index[i] <= end[i])
		    break;

	    } else if (index[i] >= end[i])
		break;

	    index[i] = info.tentative_section == 1
		? info.desc.info[i].lbound
		: start[i];
	}
    }

done:
    if (repeat_count != 0) {
	namelist_error("Repeat count too large in namelist");
	return 1;
    }

    return 0;
}



/* parse_array_subscript()-- Parse a single array element, which can
 * be a single index or a range. */

static int parse_array_subscript(dimen_ref *ar) {
char c;

    c = eat_whitespace();

    ar->have_start = 0;
    ar->have_end = 0;
    ar->range = 0;
    ar->stride = 1;

    if (c != ':') {
	ar->have_start = 1;

	if (parse_integer(&ar->start))
	    return 1;

	c = eat_whitespace();
	if (c != ':')
	    return 0;
    }

    next_char();

/* Match an end element */

    ar->range = 1;

    c = eat_whitespace();
    if (c == ')' || c == ',')
	return 0;

    if (c != ':') {
	ar->have_end = 1;
	if (parse_integer(&ar->end))
	    return 1;

	c = eat_whitespace();
	if (c != ':')
	    return 0;
    }

    next_char();
    eat_whitespace();

/* Match a stride */

    return parse_integer(&ar->stride);
}



/* parse_array_ref()-- Parse an array reference.  The initial
 * parenthesis has already been seen.  Returns nonzero if something
 * goes wrong. */

static int parse_array_ref(array_ref *ref) {
char c;

    ref->rank = 0;
    ref->range = 0;

    for(;;) {
	if (parse_array_subscript(&ref->sub[ref->rank]))
	    return 1;

	ref->range |= ref->sub[ref->rank++].range;

	eat_whitespace();
	c = next_char();

	if (c == ')')
	    break;

	if (c != ',')
	    namelist_error("Syntax error in array subscript");

	if (ref->rank >= G95_MAX_DIMENSIONS)
	    namelist_error("Too many subscripts");
    }

    return 0;
}



/* full_array_ref()-- Generate a reference for a full array */

static void full_array_ref(void) {
int i;

    info.ref.rank = info.desc.rank;
    info.ref.range = 1;
    info.have_section = 1;

    for(i=0; i<info.ref.rank; i++) {
	info.ref.sub[i].range = 1;
	info.ref.sub[i].have_start = 1;
	info.ref.sub[i].have_end = 1;

	info.ref.sub[i].start = info.desc.info[i].lbound;
	info.ref.sub[i].end = info.desc.info[i].ubound;
	info.ref.sub[i].stride = 1;
    }
}



/* parse_array_spec()-- Parse an array specification in a namelist */

static int parse_array_spec(int c) {
array_ref ref;
int i, m;
char *p;

    if (c != '(')
	namelist_error("Missing array specification");

    if (parse_array_ref(&ref))
	return 1;

    /* Make sure the reference agrees with the array */

    if (ref.rank != info.desc.rank) {
	namelist_error("Array reference has the wrong rank");
	return 1;
    }

    if (ref.range) {
	if (info.have_section) {
	    namelist_error("More than one array section specified");
	    return 1;
	}

	info.have_section = 1;
	info.section = info.desc;
	info.ref = ref;

    } else { /* Scalar array reference */

	switch(info.tentative_section) {
	case 0:
	    if (!info.have_section) {
		info.section = info.desc;
		info.ref = ref;
		info.tentative_section = 1;
	    }
	    break;

	case 1:
	    info.tentative_section = 2;
	    break;
	}

	p = info.desc.offset;
	for(i=0; i<info.desc.rank; i++) {
	    m = ref.sub[i].start;
	    if (m < info.desc.info[i].lbound || m > info.desc.info[i].ubound) {
		namelist_error("Array reference out of bounds");
		return 1;
	    }

	    p += info.desc.info[i].mult * m;
	}

	info.pointer = p;
	info.offset = 0;
    }

    info.array = 0;
    return 0;
}



/* parse_part_spec()-- Parse a part specification within a derived
 * type.  Returns nonzero if something goes wrong. */

static int parse_part_spec(int c) {
char name[G95_MAX_SYMBOL_LEN+1], msg[200];
derived_info *d;
int i, rank;

    if (c != '%')
	return 1;

    if (parse_name(name))
	return 1;

    /* Go look for the name within the type */

    d = info.dt_desc;

    while(d->name != NULL && strcmp(d->name, name) != 0)
	d++;

    if (d->name == NULL) {
	st_sprintf(msg, "Component name '%s' not found", name);
	namelist_error(msg);
	return 1;
    }

    info.type = letter_to_type(d->type);

    info.kind = d->kind;
    info.dt_desc = d->info;
    info.offset += d->offset;

    if (d->rank == 0)
	info.array = 0;

    else {
	info.array = 1;
	rank = info.desc.rank = d->rank;

	for(i=0; i<rank; i++) {
	    info.desc.info[i].lbound = d->shape[2*i];
	    info.desc.info[i].ubound = d->shape[2*i+1];
	}

	info.desc.base = info.pointer + info.offset;
	info.desc.element_size = d->kind;

	info.offset = 0;
	info.pointer = NULL;

	if (info.type == BT_REAL && info.desc.element_size == 10)
	    info.desc.element_size = REAL10_SIZE;

	init_multipliers(&info.desc);
    }

    return 0;
}


/* parse_substring()-- Parse a substring qualifier. */

static int parse_substring(char c) {
int start, end;

    if (c != '(')
	return 0;

    if (info.seen_substring)
	goto syntax;

    info.seen_substring = 1;

    c = eat_whitespace();
    if (c == ':')
	start = 1;

    else {
	if (parse_integer(&start))
	    return 1;

	c = next_char();
	if (c != ':')
	    goto syntax;
    }

    c = eat_whitespace();
    if (c == ')')
	end = info.kind;

    else {
	if (parse_integer(&end))
	    return 1;

	c = next_char();
	if (c != ')')
	    goto syntax;
    }

    if (start > end) {
	info.kind = 0;
	return 0;
    }

    if (start < 1 || start > info.kind) {
	namelist_error("Substring start out of range");
	return 1;
    }

    if (end < 1 || end > info.kind) {
	namelist_error("Substring end out of range");
	return 1;
    }

    info.kind = end - start + 1;
    info.offset += start- 1;

    return 0;

syntax:
    namelist_error("Syntax error in substring specification");
    return 1;
}



/* parse_qualifiers()-- Parse components, array specifications and
 * sections.  Returns nonzero if something goes wrong.  We return an
 * array descriptor and a corresponding section, or a pointer.  We
 * also return the ultimate type of the array or scalar.  For ultimate
 * types that are derived, a pointer to the structure descriptor is
 * also set. */

static int parse_qualifiers(namelist_info *n) {
int c, rc;

    info.offset = 0;

    info.dt_desc = NULL;
    info.pointer = n->pointer; 
    info.kind = n->kind;
    info.have_section = 0;
    info.tentative_section = 0;
    info.seen_substring = 0;

    info.dt_desc = n->dt_info;

    switch(n->type) {
    case 'r': info.type = BT_REAL;       info.array = 0;  break;
    case 'i': info.type = BT_INTEGER;    info.array = 0;  break;
    case 'l': info.type = BT_LOGICAL;    info.array = 0;  break;
    case 'c': info.type = BT_CHARACTER;  info.array = 0;  break;
    case 'z': info.type = BT_COMPLEX;    info.array = 0;  break;
    case 'd': info.type = BT_DERIVED;    info.array = 0;  break;

    case 'R': info.type = BT_REAL;       info.array = 1;  break;
    case 'I': info.type = BT_INTEGER;    info.array = 1;  break;
    case 'L': info.type = BT_LOGICAL;    info.array = 1;  break;
    case 'C': info.type = BT_CHARACTER;  info.array = 1;  break;
    case 'Z': info.type = BT_COMPLEX;    info.array = 1;  break;
    case 'D': info.type = BT_DERIVED;    info.array = 1;  break;

    default:
	internal_error("parse_qualifiers(): corrupt namelist");
    }

    if (info.array)
	info.desc = *((g95_array_descriptor *) (n->pointer));

    rc = 1;

    for(;;) {
	c = next_namelist_char();

	if (c == EOF) {
	    namelist_error(namelist_eof);
	    break;
	}

	if (c == '=') {
	    if (info.array)
		full_array_ref();

	    rc = 0;
	    eat_whitespace();
	    break;
	}

	if (!info.array && info.type == BT_CHARACTER) {
	    rc = parse_substring(c);
	    continue;
	}

	if (!info.array && info.type != BT_DERIVED) { /* Have to be done */
	    namelist_error("Expected '=' sign in namelist");
	    break;
	}

	if (info.array) {  /* See if we have an array spec */
	    if (parse_array_spec(c))
		break;

	    continue;
	}

	if (info.type == BT_DERIVED && parse_part_spec(c))
	    break;
    }

    return rc;
}



/* match_variable()-- Find the namelist node corresponding to the name
 * on the input. */

static int match_variable(void) {
char buffer[G95_MAX_SYMBOL_LEN+1], message[200];
namelist_info *n;
int m; 

    if (namelist_end())
	return 1;

    if (parse_name(buffer))
	return 1;

    for(n=first_namelist(); n; n=n->next)
	if (strcmp(buffer, n->name) == 0)
	    break;

    if (n == NULL) {
	st_sprintf(message, "Namelist variable '%s' not found", buffer);
	namelist_error(message);

	return 1;
    }

    parse_qualifiers(n);
    ioparm->first_item = 1;
    ioparm->item_count = 1;

    if (info.have_section || info.tentative_section == 1)
	m = read_array();

    else if (info.type == BT_DERIVED) {
	read_derived(info.dt_desc, info.pointer + info.offset);
	m = 0;

    } else {
	list_formatted_read(info.type, info.pointer + info.offset, info.kind);
	m = 0;
    }

    return m;
}



/* find_namelist_start()-- Skip initial comments and whitespace
 * leading up to the next namelist to read.  If the wrong namelist is
 * found, we keep looking.  Returns nonzero if we hit end of file. */

static int find_namelist_start(void) {
char *name;
int c, len;

restart:
    c = next_namelist_char();
    switch(c) {
    case '&':
    case '$':
	name = ioparm->namelist;
	len = strlen(name);

	while(len > 0) {
	    c = tolower(next_char());
	    if (c != tolower(*name++))
		goto restart;

	    len--;
	}

	c = next_char();
	if (!is_whitespace(c) && c != COMMA_SEP())
	    goto restart;

	goto found;

    case EOF:
	input_complete = 1;
	generate_error(ERROR_END, NULL);
	return 1;

    default:
	break; /* If we're here, we have a nonstandard namelist. */
    }

    goto restart;

found:
    namelist_line = 1;
    namelist_csize = 0;
    return 0;
}



/* namelist_read()-- Process a namelist read. */

void namelist_read(void) {

    input_complete = 0;

    if (!find_namelist_start())
	eat_whitespace();

    while(!input_complete) {
	if (match_variable())
	    break;

	if (!input_complete)
	    eat_separator();
    }

    /* Remove any queued EOFs */

    while(last_char == EOF)
	next_char();

    namelist_done();
}
