/* eLectrix - a pdf viewer
 * Copyright (C) 2010, 2011 Martin Linder <mali2297@users.sf.net>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/gpl-2.0.html>.
 */
#include "e6x-common.h"
#include "e6x-pref.h"
#include "e6x-enc-document.h"
#include "e6x-util.h"

#ifdef ENABLE_PDF
#include "e6x-pdf-document.h"
#endif

#ifdef ENABLE_PS
#include "e6x-ps-document.h"
#endif

#define E6X_MAX_DATLEN 32

E6xDocument *
e6x_util_open_document (const gchar *arg,
                        const gchar *passwd,
                        GError **error)
{
  GFile *file = NULL;
  gchar *path = NULL;
  GError *err = NULL;
  gchar *mime_type = NULL;
  GObject *object = NULL;
  
  file = g_file_new_for_commandline_arg (arg);
  path = g_file_get_path (file);
  g_object_unref (file);
  
  if (path == NULL)
  {
    g_set_error (&err, G_FILE_ERROR, G_FILE_ERROR_FAILED,
                 "File path could not be retrieved");
    return NULL;
  }
  
  mime_type = e6x_util_get_mime_type (path, NULL);

#ifdef ENABLE_PDF
  if (g_strcmp0 (mime_type, "application/pdf") == 0)
    object = e6x_pdf_document_new (path, passwd, &err);
  else 
#endif
#ifdef ENABLE_PS
  if (g_strcmp0 (mime_type, "application/postscript") == 0)
    object = e6x_ps_document_new (path, &err);
  else
#endif    
    g_set_error (&err, G_FILE_ERROR, G_FILE_ERROR_FAILED,
                 "File type is not supported");
  
  if (object == NULL &&
      err != NULL &&
      err->code == E6X_FILE_ERROR_ENCRYPTED)
    object = e6x_enc_document_new (path);
  
  g_free (mime_type);
  g_free (path);
  
  if (error != NULL && *error == NULL)
    *error = err;
  else
    g_clear_error (&err);
  
  return E6X_DOCUMENT (object);
}


gboolean
e6x_util_launch_app (GAppInfo *appinfo,
                     const gchar *path,
                     GError **error)
{
  gboolean retval = FALSE;
  GFile *file = NULL;
  GList *filelist = NULL;
      
  file = g_file_new_for_path (path);
  if (file != NULL)
    filelist = g_list_prepend (NULL, file);
  retval = g_app_info_launch (appinfo, filelist, NULL, error);
  if (filelist != NULL)
    g_list_free_full (filelist, &g_object_unref);
  
  return retval;
}


gboolean 
e6x_util_launch_cmd (const gchar *cmd,
                     const gchar *path,
                     GError **error)
{
  gboolean retval = FALSE;
  GAppInfo *appinfo = NULL;
  
  appinfo = g_app_info_create_from_commandline (cmd, NULL, 0, error);
  if (appinfo != NULL)
  {
    retval = e6x_util_launch_app (appinfo, path, error);
    g_object_unref (appinfo);
  }
  
  return retval;
}


gboolean
e6x_util_open_uri (const gchar *url,
                   const gchar *cmd,
                   GError **error)
{
  gboolean retval = FALSE;
  gchar *scheme = NULL;
  gchar *content_type = NULL;
  GList *applist = NULL;
    
  if (cmd != NULL)
  {
    retval = e6x_util_launch_cmd (cmd, url, error);
    return retval;
  }
  
  scheme = g_uri_parse_scheme (url);
  if (scheme != NULL)
  {
    retval = g_app_info_launch_default_for_uri (url, NULL, error);
    g_free (scheme);
    return retval;
  }
    
  content_type = e6x_util_get_content_type (url, error);
  if (content_type != NULL)
  {
    applist = g_app_info_get_all_for_type (content_type);
    g_free (content_type);
  }
  if (applist != NULL)
  {
    retval = e6x_util_launch_app (G_APP_INFO (applist->data), url, error);
    g_list_free_full (applist, &g_object_unref);
  }
  
  return retval;
}


gchar *
e6x_util_get_content_type (const gchar *path,
                           GError **error)
{
  GFile *file = NULL;
  GFileInfo *info = NULL;
  gchar *content_type = NULL;
  
  file = g_file_new_for_path (path);
  info = g_file_query_info (file, 
                            "standard::content-type", 
                            G_FILE_QUERY_INFO_NONE,
                            NULL,
                            error);
  g_object_unref (file);
  if (info != NULL)
  {
    content_type = g_strdup (g_file_info_get_content_type (info));
    g_object_unref (info);
  }
  
  return content_type;
}


gchar *
e6x_util_get_mime_type (const gchar *path,
                        GError **error)
{
  gchar *content_type = NULL;
  gchar *mime_type = NULL;
  
  content_type = e6x_util_get_content_type (path, error);
  if (content_type != NULL)
  {
    mime_type = g_content_type_get_mime_type (content_type);
    g_free (content_type);
  }
  
  return mime_type;
}


gchar *
e6x_util_dattostr (time_t t)
{
  gchar buffer[E6X_MAX_DATLEN];

  /* ISO 8601 format for date and time */
  strftime (buffer, sizeof (buffer), "%F %R", localtime (&t));
  
  return g_strdup (buffer);
}
