/*
  This program downloads a specified file to the SHARC DSP and possibly
  provides host services to the DSP code.

  (C) 1998 PinPoint Corporation
  This software is available for unlimited personal use and may only
  be redistributed in unmodified form.  Above all, this notice may
  not be modified or removed.
  Contact tim.wall@pinpointco.com for commercial use

  History:

  980901        trw     Add fixes noted by Arno
  980831        trw     released updated linux version with printf
                        changes from Arno Dabekaussen (arnodab@pie.nl) and
                        Philippe Catin (cattin@ifr.mavt.ethz.ch)

 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "coff21k.h"
#include "serial.h"

#include <termios.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include "kb.h"
#define kbhit kb_kbhit
#define getch kb_getkey

#ifdef UNIXKEY
extern int UNIX_getc_nowait(FILE *);
#endif

static char version[] = "0.8.1";
static void proxy_printf (unsigned long, unsigned long, unsigned long);

static void
usage (char *name)
{
    fprintf (stderr, "\nusage: %s [-s] [-debug] [-b<baud_rate>] "
             "[-port <serial port 1-6>] <filename>\n\n", name);
#ifdef __NetBSD__
    fprintf(stderr, "for /dev/tty00 choose port 1 (tty01 = 2 etc)\n\n");
#endif
    exit (1);
}

static void
sigint_handler (int sig)
{
    static int sigint_received = 0;

    if (sigint_received)
        return;

    sigint_received = 1;
    printf ("^C\n");
    exit (sig);
}

int
main (int argc, char *argv[])
{
    baud_t baud;
#define DEFAULT_SPEED 115200
    long speed = DEFAULT_SPEED;
    int detach = 0;
    char *filename = NULL;
	int arg;
	FILE *fp;
    unsigned long has_host_services;
    unsigned long kbhit_flag;
    unsigned long msg_type;
    unsigned long msg_length;
    unsigned long msg_buf;
    unsigned long msg_hbuf;
    unsigned long clear;
    unsigned long set;
    unsigned long flag;
    int exitflg;
    int port = 1;/* default ttyS0/COM1 */

    /* set up keyboard handling (allow ^C to generate SIGINT) */
    /* we require this library in order to get single-keypress response */
    /* under DOS, this is handled with kbhit() and getch() */
#ifndef UNIXKEY
    kb_install (KB_FLAG_SIGINT);
#endif

    /* add ^C handling */
    signal (SIGINT, sigint_handler);

    for (arg=1;arg < argc;arg++)
    {
        if (strncmp (argv[arg], "-b", 2)==0)
        {
            /* cmd-line argument overrides environment setting */
            speed = atol (argv[arg]+2);
        }
        else if (strcmp (argv[arg], "-port") == 0)
        {
            port = atoi (argv[++arg]);
            if (port < 1 || port > 6)
            {
                fprintf (stderr,
                         "Bad port (%d), defaulting to port 1\n", port);
                port = 1;
            }
        }
        else if (strncmp (argv[arg], "-debug", strlen (argv[arg])) == 0)
        {
            DSPDebug (1);
        }
        else if (strncmp (argv[arg], "-s", 2)==0)
        {
            detach = 1;
        }
        else if ((arg == argc-1) && *argv[arg] != '-' && arg != 0)
        {
            filename = argv[arg];
        }
        else
        {
            char *slash = strrchr (argv[0], '/');
            usage (slash ? slash + 1 : argv[0]);
        }
    }

    printf ("\n"
            "\t***************************************\n"
            "\t SHARC EZ-KIT downloader version %s\n"
            "\t       (c) 1998 PinPoint Corp.       \n\n"
            "\t patched for BSD by Peter Berg, \n"
            "\t           1998 FutureGroove Music\n\n"
            "\t***************************************\n\n"
            "\t        Hit ^C to quit\n\n", version);

    switch (speed)
    {
    case 300:  baud = Baud300; break;
    case 1200:  baud = Baud1200; break;
    case 2400:  baud = Baud2400; break;
    case 4800:  baud = Baud4800; break;
    case 19200:  baud = Baud19200; break;
    case 38400:  baud = Baud38400; break;
    case 57600:  baud = Baud57600; break;
    case 115200: baud = Baud115200; break;
    default:     baud = Baud9600; break;
    }

    if (!filename)
        usage (argv[0]);

    fp = fopen (filename, "rb");
    if (!fp)
    {
        fprintf (stderr, "Error opening %s\n", filename);
        exit (1);
    }

    upload (fp, baud, 1);

    /* lookup the host services variables */
    has_host_services = sym_lookup (fp, "__hs");
    kbhit_flag = sym_lookup (fp, "__kbhit");
    msg_type = sym_lookup (fp, "__mtype");
    msg_length = sym_lookup (fp, "__mlen");
    msg_buf = sym_lookup (fp, "__mbuf");
    msg_hbuf = sym_lookup (fp, "__rbuf");
    fclose (fp);
        
    if (!has_host_services && !detach)
    {
        printf ("Warning: host services not found\n");
        detach = 1;
    }

    if (!detach)
    {
        DSPInit (1, baud, 1, DEFAULT_TIMEOUT);
        printf ("------------------\n");
    }

    clear = 0;
    set = 1;
	exitflg = 0;
    if (!detach) while (!exitflg)
    {
        flag = 0;
        do 
        {
            /* check the keyboard */
 #ifdef UNIXKEY
            if(UNIX_getc_nowait(stdin))
            {
                extern int last_char;
                unsigned long ch = last_char;
 #else
            if (kbhit ())
            {
                unsigned long ch = getch ();
 #endif 
                /* FIXME - don't print char in kb_getkey,
                   instead of erasing here */
                printf ("\b \b");fflush (stdout);

                if (ch == 26 || ch == 27)
                {
                    exit (0);
                }
                DSPWrite (msg_hbuf, &ch, 1);
                DSPWrite (kbhit_flag, &set, 1);
            }
            DSPRead (msg_type, &flag, 1);
            usleep (100000);

        } while (flag == 0);
            
        switch (flag)
        {
        case 0x0:
            break;
        case 0x1:
        {
            proxy_printf (msg_length, msg_buf, msg_type);
            break;
        }
        case 0x2:
            exitflg = 1;
            DSPRead (msg_buf, &flag, 1);
            DSPWrite (msg_type, &clear, 1);
            break;
        default:
            printf ("Unknown code 0x%lx\n", flag);
            DSPWrite (msg_type, &clear, 1);
            break;
        }
    }
    if (!detach)
    {
        printf ("\n------------------\n");
        printf ("Program exited with code %ld\n", flag);
    }
    else
        printf ("Program running in standalone mode.  "
                "Host services are disabled\n");

    exit (0);
}

/* perform formatted print on behalf of the SHARC processor */
#define MAX_STRING 256
static void
proxy_printf (unsigned long msg_lenp,
              unsigned long msg_bufp,
              unsigned long msg_typep)
{
    int               arg_cnt, fmt_cnt, out_cnt;
    int               result;
    char              fmt[MAX_STRING + 44];
    char              out[MAX_STRING], tmp[MAX_STRING];
    unsigned long     len;
    static char       *fname = "proxy_printf";
    unsigned long clear = 0;
    /* SHARC is little-endian, so if this host is big-endian,
       we need to swap */
    static int big_endian = -1;

    if (big_endian == -1)
    {
        unsigned long test = 0x12345678;
        unsigned char *ctest = (unsigned char *)&test;
        big_endian = (*ctest == 0x12);
    }

    /*
     * First get the printf format string and all the arguments
     * in the fmt buffer. The length that is returned is the
     * length of the format string and arguments. From the number of
     * % in the format is known how many arguments should be found,
     * after the format string in the buffer.
     */
    DSPRead(msg_lenp, &len, 1);
    DSPRead(msg_bufp, (unsigned long *) fmt, len);
    DSPWrite(msg_typep, &clear, 1);

#define int32_fmt ((unsigned long *)fmt)
#define SHUFFLE(X) ((((X)&0xFF)<<24)|\
                    (((X)&0xFF00)<<8)|\
                    (((X)&0xFF0000)>>8)|\
                    (((X)&0xFF000000)>>24))

    /*
     * Some error checking.
     */

    if( (len <= 0) || (len >= MAX_STRING))
    {
        fprintf(stderr, "%s: ERROR: invalid SHARC message detected\n",
                fname);
        exit(EXIT_FAILURE);
    }

    /*
     * revise the byte ordering, if necessary
     */
    if (big_endian)
    {
        int word;
        /* reorder the bytes that we uploaded */
        for (word=0;word<len;word++)
        {
            int32_fmt[word] = SHUFFLE(int32_fmt[word]);
        }
    }

    /*
     * Find index in message of first arguments. This is the next
     * 32 bit value after termination of the index string.
     */

    fmt_cnt = 0;

    while((fmt[fmt_cnt] != '\0') && (fmt_cnt < MAX_STRING))
    {
        fmt_cnt++;
    }

    if (fmt_cnt >= MAX_STRING)
    {
        fprintf(stderr, "%s: ERROR: string overflow detected\n",
                fname);
        exit(EXIT_FAILURE);
    }

    arg_cnt = fmt_cnt / 4 + 1;

    /*
     * Parse the format string, set arg_cnt to the index in message
     * that contains the first argument.
     */

    out_cnt = 0;

    for(fmt_cnt = 0; (fmt[fmt_cnt] != '\0') && (fmt_cnt < MAX_STRING);
        fmt_cnt++)
    {
        if (fmt[fmt_cnt] != '%')
        {
            /*
             * Copy characters to output string.
             */
            out[out_cnt] = fmt[fmt_cnt];
            out_cnt++;
        }
        else
        {
            /*
             * Skip the % then perform the string conversion.
             * This done piece by piece because the SHARC transmits
             * 32 bits arguments (even for chars) and this should
             * be matches against the format on the PC.
             */

            fmt_cnt++;
            result = 0;

            switch (fmt[fmt_cnt])
            {
            case 'd':
                result = sprintf(tmp, "%ld", int32_fmt[arg_cnt]);
                arg_cnt++;
                break;

            case 'i':
                result = sprintf(tmp, "%li", int32_fmt[arg_cnt]);
                arg_cnt++;
                break;

            case 'o':
                result = sprintf(tmp, "%lo", int32_fmt[arg_cnt]);
                arg_cnt++;
                break;

            case 'x':
                result = sprintf(tmp, "%lx", int32_fmt[arg_cnt]);
                arg_cnt++;
                break;

            case 'X':
                result = sprintf(tmp, "%lX", int32_fmt[arg_cnt]);
                arg_cnt++;
                break;

            case 'u':
                result = sprintf(tmp, "%lu", int32_fmt[arg_cnt]);
                arg_cnt++;
                break;

            case 'c':
                result = sprintf(tmp, "%c", (char) int32_fmt[arg_cnt]);
                arg_cnt++;
                break;

            case 's':
                /*
                 * Collect the string, including '\0' but do not
                 * copy '\0' to the out string.
                 */

                do
                {
                    out[out_cnt] = (char) int32_fmt[arg_cnt];
                    out_cnt++;
                    arg_cnt++;

                    if (out_cnt >= MAX_STRING)
                    {
                        fprintf(stderr,
                                "%s: ERROR: string overflow detected\n",
                                fname);
                        exit(EXIT_FAILURE);
                    }
                }
                while(int32_fmt[arg_cnt-1] != 0);

                /*
                 * Step back out_cnt to overwrite the '\0'
                 */
                out_cnt--;

                result = 0;
                break;

            case 'f':
                /*
                 * This works only if host has also internal float
                 * size of 32 bits.
                 */

                result = sprintf(tmp, "%f", *(float *) (int32_fmt+arg_cnt));
                arg_cnt++;
                break;

            case 'e':
                result = sprintf(tmp, "%e", *(float *) (int32_fmt+arg_cnt));
                arg_cnt++;
                break;

            case 'E':
                result = sprintf(tmp, "%E", *(float *) (int32_fmt+arg_cnt));
                arg_cnt++;
                break;

            case 'g':
                result = sprintf(tmp, "%g", *(float *) (int32_fmt+arg_cnt));
                arg_cnt++;
                break;

            case 'G':
                result = sprintf(tmp, "%G", *(float *) (int32_fmt+arg_cnt));
                arg_cnt++;
                break;

            case 'p':
                result = sprintf(tmp, "%p", (void *)int32_fmt[arg_cnt]);
                arg_cnt++;
                break;

            case '%':
                result = sprintf(tmp, "%%");
                break;

            case 'l':
                fmt_cnt++;

                switch (fmt[fmt_cnt])
                {
                    long  swap;

                case 'f':
                    /*
                     * Swap the two 32 bit portions of a 64 bit
                     * double.
                     */
                    swap = *((long *) (int32_fmt+arg_cnt));
                    *((long *) (int32_fmt+arg_cnt)) =
                        *((long *) (int32_fmt+arg_cnt+1));
                    *((long *) (int32_fmt+arg_cnt+1)) = swap;

                    result = sprintf(tmp, "%f",
                                     *((double *) (int32_fmt+arg_cnt)));
                    arg_cnt += 2;
                    break;

                default:
                    fprintf(stderr,
                            "%s: ERROR: unrecognised format %%l%c argument\n",
                            fname, fmt[fmt_cnt]);
                    exit(EXIT_FAILURE);
                }
                break;

            default:
                fprintf(stderr,
                        "%s: ERROR: unrecognised format %%%c argument\n",
                        fname, fmt[fmt_cnt]);
                exit(EXIT_FAILURE);
            }

            /*
             * Do some error checking to avoid string overflow.
             */

            if (result > 0)
            {
                if ((out_cnt + result) < MAX_STRING)
                {
                    /*
                     * Copy without '\0'.
                     */

                    memcpy(&out[out_cnt], tmp, result);
                    out_cnt += result;
                }
                else
                {
                    fprintf(stderr,
                            "%s: ERROR: string overflow detected\n",
                            fname);
                    exit(EXIT_FAILURE);
                }
            }
        }
    }

    /*
     * Now
     terminate the string and print the result.
     */

    out[out_cnt] = '\0';
    printf(out);

    /*
     * Flush message to screen incase there are no \n.
     */
    fflush(stdout);

}
