/*
Copyright (c) 2005 Aristotle Pagaltzis

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#define _GNU_SOURCE
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <err.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <search.h>

char *unitspec[] = { "", "K", "M", "G" };

char *
humanreadable( long long size, int prec )
{
	double dsize = size;
	int unit = 0;
	int max_unit = sizeof( unitspec ) / sizeof( unitspec[ 0 ] );
	char *size_human;
	char *size_unit;

	while( dsize > 1024.0 && unit < max_unit )
		dsize /= 1024.0, ++unit;

	if( asprintf( &size_human, "%.*f", prec, dsize ) == -1 )
		errx( 255, "Memory allocation failed" );

	if( rindex( size_human, '.' ) ) {
		int i = strlen( size_human ) - 1;
		while( size_human[ i ] == '0' )
			size_human[ i-- ] = '\0';
		if( size_human[ i ] == '.' )
			size_human[ i-- ] = '\0';
	}

	if( asprintf( &size_unit, "%s%s", size_human, unitspec[ unit ] ) == -1 )
		errx( 255, "Memory allocation failed" );

	free( size_human );

	return size_unit;
}

int
compare_inodes( const void *pa, const void *pb )
{
	return *( ino_t * )pa - *( ino_t * )pb;
}

void
destroy( const void *nodep, const VISIT which, const int depth )
{
	if( which == endorder ) {
		free( *( ino_t ** )nodep );
		free( ( void * )nodep );
	}
}


int opt_humanreadable = 0;
int opt_brief = 0;
int opt_precision = 1;

int
dirsize( char *dir )
{
	DIR *dh;
	struct dirent *ent;
	long long total = 0;
	void *inodes = NULL;

	dh = opendir( dir );

	if( ! dh ) {
		warn( "Could not open %s", dir );
		return 0;
	}

	while( ( ent = readdir( dh ) ) ) {
		char *path;
		struct stat buf;

		if( strcmp( ent->d_name, ".." ) == 0 )
			continue;

		if( asprintf( &path, "%s/%s", dir, ent->d_name ) == -1 )
			errx( 255, "Memory allocation failed" );

		if( lstat( path, &buf ) )
			warn( "Could not stat %s", ent->d_name );
		else {
			void *tnode;
			ino_t *inode = ( ino_t * )malloc( sizeof( ino_t ) );

			if( ! inode )
				errx( 255, "Memory allocation failed" );

			*inode = buf.st_ino;

			tnode = tsearch( ( void * )inode, &inodes, compare_inodes );

			if( tnode == NULL )
				errx( 255, "Memory allocation failed" );

			if( *( ino_t ** )tnode == inode )
				total += buf.st_size;
			else
				free( inode );
		}

		free( path );
	}

	twalk( inodes, destroy );

	if( ! opt_brief ) printf( "%s: ", dir );

	if( closedir( dh ) )
		warn( "Could not close directory %s", dir );

	if( opt_humanreadable ) {
		char *hr = humanreadable( total, opt_precision );
		printf( "%s\n", hr );
		free( hr );
	}
	else
		printf( "%lld\n", total );
	
	return 1;
}

void
usage( void )
{
	fprintf( stderr,
		"usage: dirsize [-b] [-H] [-h] [-p precision] [dirname ...]\n"
		"         -b  brief output, excludes dirnames\n"
		"         -H  human readable, prettier size display\n"
		"         -h  this help\n"
		"         -p  (implies -H) number of fractional digits displayed [default: 1]\n"
	);
	exit( 1 );
}

int
main( int argc, char *argv[] )
{
	int error = 0;
	int i = 0;

	while( ( i = getopt( argc, argv, "bHhp:" ) ) != -1 ) {
		char *end;
		switch( i ) {
			case 'b':
				opt_brief = 1;
				break;
			case 'H':
				opt_humanreadable = 1;
				break;
			case 'p':
				opt_humanreadable = 1;
				opt_precision = strtol( optarg, &end, 10 );
				if( errno || opt_precision < 0 )
					warnx( "precision must be a non-negative integer" ), usage();
				break;
			case 'h':
			default:
				usage();
				break;
		}
	}

	if( optind < argc )
		for( i = optind ; i < argc ; ++i )
			dirsize( argv[ i ] ) || ++error;
	else
		dirsize( "." ) || ++error;

	return error ? 2 : 0;
}
