/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/mgsubr.c,v $
 * $Id: mgsubr.c,v 3.6 1999/07/20 21:04:31 heiner Exp $
 *
 *	move generator subroutines
 */

#include "types.h"
#include "board.h"
#include "mgsubr.h"


/*
 * Create sequential list linkage in a move list.
 * Needed as last operation of move list generation.
 */
    Eximpl void
mg_link( Movelist* lp )
{
    register Move*	mp;
    register Move*	mp0;
    register Move*	last;

    mp0 = lp->l_m;
    mp  = lp->l_free;
    last = (Move*)0;
    while( --mp >= mp0 ) {
	mp->m_next = last;
	last = mp;
    }
    lp->l_first = last;
}


/*
 * mg_link_attr()
 *	Same as "mg_link()", except that all those moves
 *	which have some of the specified attribute bits set,
 *	are to be sorted before all others.
 */

    Eximpl void
mg_link_attr(
    Movelist*		lp,
    register ruint8	attr)
{
    register Move*	mp;
    register Move*	mp0;
    register Move*	last;
    register Move*	spec;
    register Move*	speclast;

    mp0 = lp->l_m;			/* first allocated */
    mp  = lp->l_free;			/* behind last allocated */
    last = (Move*)0;			/* currently build list */
    while( --mp >= mp0 ) {
	if( mp->m_attr & attr ) {	/* found last of the special ones */
	    speclast = spec = mp;
	    while( --mp >= mp0 ) {
		if( mp->m_attr & attr ) {
		    mp->m_next = spec;
		    spec = mp;
		}else {
		    mp->m_next = last;
		    last = mp;
		}
	    }
	    speclast->m_next = last;	/* append normal to special */
	    lp->l_first = spec;		/* and start with special */
	    return;
	}
	mp->m_next = last;
	last = mp;
    }
    lp->l_first = last;
}


/*
 * is_pinned()
 *	Return whether a piece at the specified position
 *	would be pinned to the king to move.
 */
    Eximpl Bool
is_pinned(
    register Xconst Board*	bp,
    register rPosition		pos)
{
    register rPosition		kpos;
    register int		dir;
    register PieceSet		enemies;
    register const Field*	p;

    kpos = K_POS(bp, bp->b_tomove);
    dir = att_dir(kpos, pos);
    if( dam_dir(dir) ) {
	enemies = F_IATT(&(bp->b_f[kpos])) & F_DATT(&(bp->b_f[pos]))
		& COLOUR_MASK(opp_colour(bp->b_tomove));
	if( enemies ) {
	    if( (kpos + dam_mov[dir]) != pos ) {	/* space between */
		if( F_IATT(&(bp->b_f[pos - dam_mov[dir]])) & enemies ) {
		    return TRUE;
		}
	    }else {					/* no space between */
		p = &( bp->b_f[pos] );
		do {
		    p += dam_mov[dir];
		}while( p->f_c == empty );
		if( (p->f_c == opp_colour(bp->b_tomove))
		 && (enemies & SET1(p->f_idx)) ) {
		    return TRUE;
		}
	    }
	}
    }
    return FALSE;
}


/*
 * ep_legal()
 *	Returns, whether in the specified board the B at the specified position
 *	may legally beat e.p. The third position is the Bs king.
 *	We must not be in check, currently.
 *	Note:	Here is checked the possible double uncovering through both B.
 *		Other uncoverings through the beaten B are impossible,
 *		and simple uncoverings through the beating B have to be checked
 *		outside this function.
 */
    Eximpl Bool
ep_legal(
    const Board*	bp,
    Position		pos,
    Position		tpos,
    Position		kpos)
{
    int			delta;
    int			xpos;
    int			j;
    int			enemy;

    if( bp->b_ep != (tpos - bau_mov[bp->b_tomove]) )
	return FALSE;
    /*
     * uncovering by beat bp->b_ep is possible
     * only horizontally through both B:
     */
    if( LIN(kpos) != LIN(pos) )
	return TRUE;
    if( kpos < pos ) {
	delta = MK_DELTA( 1,0);
    }else {
	delta = MK_DELTA(-1,0);
    }
    enemy = opp_colour(bp->b_tomove);
    xpos = kpos;
    j = 0;
    for(;;) {
	xpos += delta;
	if( bp->b_f[xpos].f_c == empty )
	    continue;
	if( bp->b_f[xpos].f_c == border )
	    break;
	if( xpos == pos || xpos == bp->b_ep ) {
	    ++j;
	    continue;
	}
	if( j == 2 && bp->b_f[xpos].f_c == enemy ) {
	    switch( bp->b_f[xpos].f_f ) {
	     case turm:
	     case dame:
		++j;
		break;
	    }
	}
	break;
    }
    return j < 3;
}


/*
 * mg_app_castle()
 *	Append the legal castling moves to the list.
 *	No entry conditions are really necessary.
 */
    Eximpl void
mg_app_castle(
    register Xconst Board*	bp,
    register Movelist*		lp)
{
    register Xconst Field*	p;
    register PieceSet		emask;
    register rPosition		kpos;
    Move			m;

    emask = COLOUR_MASK(opp_colour(bp->b_tomove));
    kpos = K_POS(bp, bp->b_tomove);
    p = &(bp->b_f[kpos]);
    if( ! (F_DATT(p) & emask) ) {
	m.m_from  = kpos;
	m.m_fig   = p->f_f;
	m.m_idx   = p->f_idx;
	m.m_value = 0;
	m.m_attr  = 0;
	if( bp->b_castle[bp->b_tomove] & castle00 ) {
	    if( (p[MK_DELTA(1,0)].f_c == empty)
	     && (p[MK_DELTA(2,0)].f_c == empty)
	     && ! (F_DATT(&(p[MK_DELTA(1,0)])) & emask)
	     && ! (F_DATT(&(p[MK_DELTA(2,0)])) & emask) ) {
		m.m_to = kpos + MK_DELTA(2,0);
		app_move(&m, lp);
	    }
	}
	if( bp->b_castle[bp->b_tomove] & castle000 ) {
	    if( (p[-MK_DELTA(1,0)].f_c == empty)
	     && (p[-MK_DELTA(2,0)].f_c == empty)
	     && (p[-MK_DELTA(3,0)].f_c == empty)
	     && ! (F_DATT(&(p[-MK_DELTA(1,0)])) & emask)
	     && ! (F_DATT(&(p[-MK_DELTA(2,0)])) & emask) ) {
		m.m_to = kpos - MK_DELTA(2,0);
		app_move(&m, lp);
	    }
	}
    }
}


/*
 * mg_app_ep()
 *	Append all legal e.p. moves to the specified list.
 *	No conditions need be assured.
 */
    Eximpl void
mg_app_ep(
    register Xconst Board*	bp,
    register Movelist*		lp)
{
    register rColour		self;
    register rPosition		tpos;
    register rPosition		pos;
    register rPosition		kpos;
    register int		i;
    register Xconst Field*	p;
    register rPieceSet		emask;
    register int		dir;
    Move			m;

    if( no_pos(bp->b_ep) ) {
	return;				/* no (double step) target */
    }
    self = bp->b_tomove;
    tpos = bp->b_ep + bau_mov[self];
    for( i=0 ; i<2 ; ++i ) {
	pos = tpos - (i ? bau_left : bau_right)[self];
	p = &( bp->b_f[pos] );
	if( (p->f_c != self) || (p->f_f != bauer) ) {
	    continue;
	}
	/*
	 * Well, we have a target for e.p. (made double step),
	 * and a B in the correct position to beat it.
	 * Whether this goes into the move list depends now
	 * on the legality of the move, i.e.
	 *	- an existing check must be removed
	 *	- no new check must be opened.
	 * When there is an existing check, which is not directly
	 * by the target we cannot remove it by beating e.p.
	 * (topologically impossible).
	 * It is also impossible for a B to say double check.
	 */
	kpos = K_POS(bp, self);
	emask = COLOUR_MASK(opp_colour(self));
	if( F_DATT(&(bp->b_f[kpos])) & emask ) {		/* self in check */
	    if( F_DATT(&(bp->b_f[kpos])) & emask
	      & ~SET1(bp->b_f[bp->b_ep].f_idx) ) {
		continue;			/* cannot be removed */
	    }
	    /*
	     * In direct check by the B to be beaten e.p.
	     * This reduces the possible uncoverings:
	     */
	    if( dam_dir(att_dir(pos, kpos))
	     && (F_IATT(&(bp->b_f[kpos])) & emask & F_DATT(p)) ) {
		/*	. * .
		 *	B b .
		 *	K . .
		 * Pinned vertically => illegal
		 */
		continue;
	    }
	}else {						/* self not in check */
	    /*
	     * When removing just the enemy at "b_ep" would discover
	     * a check on a L line, our king could have been beaten
	     * instead of the double step (=> illegal condition).
	     * Discovering vertically through "b_ep" is impossible
	     * as blocked by our own B.
	     * Discovering horizontally through "b_ep" is possible
	     * through both B, only.
	     * Remains to test uncovering through our own B.
	     */
	    if( F_IATT(&(bp->b_f[kpos])) & emask & F_DATT(p) ) {
		dir = att_dir(kpos, pos);
		if( dam_dir(dir)
		 && (lin_dir(dir) != lin_dir(att_dir(pos,tpos)))
		 && is_pinned(bp, pos)
		  ) {
		    continue;		/* pinned, and leaves pinning */
		}
	    }
	    if( ! ep_legal(bp, pos, tpos, kpos) ) {
		continue;
	    }
	}
					/* no legality objections to the move */
	m.m_from  = pos;
	m.m_to    = tpos;
	m.m_fig   = bauer;
	m.m_idx   = p->f_idx;
	m.m_value = 0;
	m.m_attr  = 0;
	app_move(&m, lp);
    } /* loop over left/right */
}


/*
 * app_4_proms()
 *	Append all 4 promotion moves to the list.
 */
    static void
app_4_proms(
    register Movelist*	lp,
    Position		pos,
    Position		tpos,
    int8		idx)
{
    Move	m;

    m.m_value = 0;
    m.m_attr  = 0;
    m.m_from  = pos;
    m.m_to    = tpos;
    m.m_idx   = idx;
    m.m_fig   = dame    ; app_move(&m, lp);
    m.m_fig   = springer; app_move(&m, lp);
    m.m_fig   = laeufer ; app_move(&m, lp);
    m.m_fig   = turm    ; app_move(&m, lp);
}


/*
 * app_ck_proms()
 *	Append all those promotion moves to the list (out of all 4),
 *	which are able to check on a direction "ckdir".
 */
    static void
app_ck_proms(
    register Movelist*	lp,
    Position		pos,
    Position		tpos,
    int8		idx,
    register int	ckdir)
{
    Move	m;

    m.m_value = 0;
    m.m_attr  = 0;
    m.m_from  = pos;
    m.m_to    = tpos;
    m.m_idx   = idx;
    if( dam_dir(ckdir) ) {
	m.m_fig = dame;
	app_move(&m, lp);
	if( lfr_dir(ckdir) ) {
	    m.m_fig = laeufer;
	    app_move(&m, lp);
	}else {
	    m.m_fig = turm;
	    app_move(&m, lp);
	}
    }else if( spr_dir(ckdir) ) {
	m.m_fig = springer;
	app_move(&m, lp);
    }
}


/*
 * mg_app_prom()
 *	Append all legal promotion moves to the specified list.
 *	No conditions need be assured.
 *	Optionally there are left out obviously non-checking moves.
 */
    Eximpl void
mg_app_prom(
    register Xconst Board*	bp,
    Movelist*			lp,
    Bool			needcheck)
{
    register rColour		self;
    register rColour		enemy;
    register Xconst Field*	p;
    register Xconst Field*	tp;
    register rPosition		pos;
    register rPosition		tpos;
    register rPosition		kpos;
    register rPieceSet		ckset;
    register int		pindir;
    register const Field*	ep;
    register int		delta;
    register int		i;
    register Bool		needckdirect;
    register rPosition		ekpos;
    register Xconst Field*	ekp;
    register int		ckdir;
    PieceSet			amask;

    self  = bp->b_tomove;
    enemy = opp_colour(self);
    kpos = K_POS(bp, self);
    ckset = F_DATT(&(bp->b_f[kpos])) & COLOUR_MASK(enemy);	/* those say check */
    if( min2elems(ckset) ) {
	return;						/* double check */
    }
    if( needcheck ) {
	amask = COLOUR_MASK(self);
	ekp = &(bp->b_f[ ekpos = K_POS(bp, enemy) ]);
    }
    /*
     * Scan enemies base line to find own promotion candidates:
     */
    p = &(bp->b_f[ pos = MK_POS(0, BAS_LIN(enemy)) ]);
    for( ; p->f_c != border ; pos += MK_DELTA(1,0), p += MK_DELTA(1,0) ) {
	if( (p->f_c == self) && (p->f_f == bauer) ) {
	    pindir = att_dir(kpos, pos);
	    needckdirect =
		    (  needcheck
		    && ( ! dam_dir(att_dir(pos, ekpos))
		      || ! (F_DATT(p) & F_IATT(ekp) & amask)
		       )
		    );
	    for( i=0 ; i<2 ; ++i ) {
		tp = &(bp->b_f[ tpos = pos + (i ? bau_left:bau_right)[self] ]);
		if( tp->f_c == enemy ) {		/* pseudo-legal */
		    if( ( !ckset
		       || (ckset == SET1(tp->f_idx))
			)			/* existing check removed */
		     && ( no_dir(pindir)
		       || (pindir == att_dir(kpos, tpos))
		       || ! is_pinned(bp, pos)
			)			/* no check uncovered */
		       ) {
			if( ! needckdirect ) {
			    app_4_proms(lp, pos, tpos, p->f_idx);
			}else if( ! no_dir(ckdir = att_dir(tpos, ekpos)) ) {
			    app_ck_proms(lp, pos, tpos, p->f_idx, ckdir);
			}
		    }
		}
	    }
	    tp = &(bp->b_f[ tpos = pos + bau_mov[self] ]);
	    if( tp->f_c == empty ) {		/* pseudo-legal */
		if( ( !ckset
		   || ( (ckset & F_DATT(tp))
		     && trm_dir(att_dir(kpos, tpos))
		      )
		    )			/* existing check may be removed */
		 && ( no_dir(pindir)
		   || (pindir == att_dir(kpos, tpos))
		   || ! is_pinned(bp, pos)
		    )			/* no check uncovered */
		   ) {
		    if( ckset ) {
			delta = dam_mov[att_dir(kpos, tpos)];
			ep = tp;
			do {
			    ep += delta;
			}while( ep->f_c == empty );
			if( (ep->f_c == enemy)
			 && (ckset == SET1(ep->f_idx)) ) {
			    if( ! needckdirect ) {
				app_4_proms(lp, pos, tpos, p->f_idx);
			    }else if( ! no_dir(ckdir = att_dir(tpos, ekpos)) ) {
				app_ck_proms(lp, pos, tpos, p->f_idx, ckdir);
			    }
			}
		    }else {
			if( ! needckdirect ) {
			    app_4_proms(lp, pos, tpos, p->f_idx);
			}else if( ! no_dir(ckdir = att_dir(tpos, ekpos)) ) {
			    app_ck_proms(lp, pos, tpos, p->f_idx, ckdir);
			}
		    }
		}
	    }
	}
    } /* for pos, p */
}
