#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>

enum nagios_state {
	STATE_OK = 0,
	STATE_WARNING = 1,
	STATE_CRITICAL = 2,
	STATE_UNKNOWN = 3,
};

int get_ip_states_default(int *, int *);
int get_nat_states_default(int *, int *);
int get_frag_states_default(int *, int *);
int get_ip_states_inet6(int *, int *);
int get_nat_states_inet6(int *, int *);
int get_frag_states_inet6(int *, int *);

static int
add_state(int st, int new)
{
	int ret; 

	switch(st) {
	case STATE_OK:	
		ret = new;
		break;
	case STATE_WARNING:	
		ret = (new == STATE_CRITICAL) ? new : st;
		break;
	case STATE_CRITICAL:	
		ret = st;
		break;
	case STATE_UNKNOWN:	
	default:
		ret = (new == STATE_OK) ? st : new;
		break;
	}

	return ret;
}

int
main(int argc, char **argv)
{
	int (*get_ip_states)(int *, int*);
	int (*get_nat_states)(int *, int*);
	int (*get_frag_states)(int *, int*);
	enum nagios_state st = STATE_OK;
	char *ststr;
	int ip_cur, ip_max;
	int nat_cur, nat_max;
	int frag_cur, frag_max;
	char ip_msg[512];
	char nat_msg[512];
	char frag_msg[512];
	int warn = 80;
	int crit = 95;
	char *progname = argv[0];
	int ch;
	int s;

	if ((s = socket(PF_INET6, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		get_ip_states = get_ip_states_default;
		get_nat_states = get_nat_states_default;
		get_frag_states = get_frag_states_default;
	} else {
		(void)close(s);
		get_ip_states = get_ip_states_inet6;
		get_nat_states = get_nat_states_inet6;
		get_frag_states = get_frag_states_inet6;
	}

	while ((ch = getopt(argc, argv, "c:w:")) != -1) {
		switch (ch) {
		case 'c':
			crit = atoi(optarg);
			break;
		case 'w':
			warn = atoi(optarg);
			break;
		default:
			printf("usage: %s [-c crit] [-w warn]\n", progname);
			exit(1);
			break;
		}
	}

	if (warn > crit)
		warn = crit - 1;

	if (get_ip_states(&ip_cur, &ip_max) == 0) {
		snprintf(ip_msg, sizeof(ip_msg), 
			 "IP states: %d/%d", ip_cur, ip_max);
		if (100 * ip_cur / ip_max > crit)
			st = add_state(st, STATE_CRITICAL);
		if (100 * ip_cur / ip_max > warn)
			st = add_state(st, STATE_WARNING);
	} else {
		snprintf(ip_msg, sizeof(ip_msg),
			 "cannot get IP states");
		st = STATE_UNKNOWN;
	}

	if (get_nat_states(&nat_cur, &nat_max) == 0) {
		snprintf(nat_msg, sizeof(nat_msg), 
			 "NAT states: %d/%d", nat_cur, nat_max);
		if (100 * nat_cur / nat_max > crit)
			st = add_state(st, STATE_CRITICAL);
		if (100 * nat_cur / nat_max > warn)
			st = add_state(st, STATE_WARNING);
	} else {
		snprintf(nat_msg, sizeof(nat_msg),
			 "cannot get NAT states");
		st = STATE_UNKNOWN;
	}

	if (get_frag_states(&frag_cur, &frag_max) == 0) {
		snprintf(frag_msg, sizeof(frag_msg), 
			 "frag states: %d/%d", frag_cur, frag_max);
		if (100 * frag_cur / frag_max > crit)
			st = add_state(st, STATE_CRITICAL);
		if (100 * frag_cur / frag_max > warn)
			st = add_state(st, STATE_WARNING);
	} else {
		snprintf(frag_msg, sizeof(frag_msg),
			 "cannot get frag states");
		st = STATE_UNKNOWN;
	}

	switch(st) {
	case STATE_OK:
		ststr = "OK";
		break;
	case STATE_WARNING:
		ststr = "WARNING";
		break;
	case STATE_CRITICAL:
		ststr = "CRITICAL";
		break;
	case STATE_UNKNOWN:
	default:
		ststr = "UNKNOWN";
		break;
	}

	printf("%s: %s, %s, %s\n", ststr, ip_msg, nat_msg, frag_msg);

	return st;
}
