/*
 * Copyright (c) 2012,2017 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <getopt.h>
#include <err.h>
#include <errno.h>
#include <sysexits.h>
#include <pthread.h>
#include <netinet/in.h>
#if defined(AF_INET6) && defined(__NetBSD__)
#include <netinet6/in6.h>
#endif
#include <arpa/nameser.h>
#include <arpa/inet.h>
#include <resolv.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/queue.h>
#include <sys/utsname.h>

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif

#define DEFAULT_TIMEOUT 120

struct dnsrbl {
	char *d_domain;
	char *d_positive;
};

struct dnsrbl default_dnsrbl[] = {
	/*
	 * From check_dnsrbl 0.2 
	 * As of 2015/01/07
	 */
	{ "blackholes.five-ten-sg.com",	"127.0.0.1/8" },
	{ "no-more-funn.moensted.dk",	"127.0.0.1/8" },
	{ "dnsrbl.swinog.ch",		"127.0.0.1/8" },
	/* 
	 * From https://en.wikipedia.org/wiki/Comparison_of_DNS_blacklists
	 * As of 2017-08-18
	 */
	{ "zen.spamhaus.org",		"127.0.0.1/8" },
	{ "dnsbl.sorbs.net",		"127.0.0.1/8" },
	{ "dnsbl-1.uceprotect.net",	"127.0.0.1/8" },
	{ "dnsbl-2.uceprotect.net",	"127.0.0.1/8" },
	{ "dnsbl-3.uceprotect.net",	"127.0.0.1/8" },
	{ "psbl.surriel.com",		"127.0.0.1/8" },
	{ "all.rbl.webiron.net",	"127.0.0.1/8" },
	{ "truncate.gbudb.net",		"127.0.0.1/8" },
	{ "db.wpbl.info",		"127.0.0.1/8" },
	{ "cbl.abuseat.org",		"127.0.0.1/8" },
	{ "dnsbl.cobion.com",		"127.0.0.1/8" },
	{ "bad.psky.me",		"127.0.0.1/8" },
	{ "bl.spamcop.net",		"127.0.0.1/8" },
	{ "noptr.spamrats.com",		"127.0.0.1/8" },
	{ "dyna.spamrats.com",		"127.0.0.1/8" },
	{ "spam.spamrats.com",		"127.0.0.1/8" },
	{ "auth.spamrats.com",		"127.0.0.1/8" },
	{ "hostkarma.junkemailfilter.com","127.0.0.2/32" },
	{ "ix.dnsbl.manitu.net",	"127.0.0.1/8" },
	{ "dnsbl.inps.de",		"127.0.0.1/8" },
	{ "bl.blocklist.de",		"127.0.0.1/8" },
	{ "srnblack.surgate.net",	"127.0.0.1/8" },
	{ "all.s5h.net",		"127.0.0.1/8" },
	/*
	 * Broken as of 2025-11-15
	 * { "rbl.megarbl.net",		"127.0.0.1/8" },
	 */
	{ "rbl.realtimeblacklist.com",	"127.0.0.1/8" },
	{ "b.barracudacentral.org",	"127.0.0.1/8" },
	{ "dnsbl.spfbl.net",		"127.0.0.1/8" },
	{ "ubl.unsubscore.com",		"127.0.0.1/8" },
	{ NULL,				NULL, },
};

struct dnsrbl *config_dnsrbl = default_dnsrbl;

struct addrlist {
	char al_addr[MAXHOSTNAMELEN + 1];
	char *al_query;
	SLIST_ENTRY(addrlist) al_next;
};
SLIST_HEAD(addrlist_head, addrlist);

struct rev_thread_args {
	struct addrlist_head	*rta_alp;
	pthread_mutex_t		*rta_mtx;
	char			*rta_query;
	unsigned char		*rta_addr;
	int			rta_family;
	pthread_cond_t		*rta_cond;
	int			*rta_count;
};

struct check_dnsrbl_args {
	char			*cda_revaddr;
	struct dnsrbl		*cda_d;
	struct addrlist_head	*cda_alp;
	pthread_mutex_t		*cda_mtx;
	pthread_cond_t		*cda_cond;
	int			*cda_retval;
	int			*cda_count;
};

struct addrlist_head *revaddrs = NULL;
pthread_mutex_t revaddrs_mtx = PTHREAD_MUTEX_INITIALIZER;

int verbose = 0;
struct timespec timeout = {0, 0};
struct __res_state *res = NULL;

void panic(char *);
void free_addrlist(struct addrlist_head *);
void usage(char *);
void parse_options(int, char **);
void dump_dnsrbl(void);
struct dnsrbl *load_dnsrbl(char *);
void rev_addr(struct addrlist_head *, pthread_mutex_t *, char *, unsigned char *, int);
void rev_dns(struct addrlist_head *, pthread_mutex_t *, char *);
void *rev_addr_thread(void *);
void *rev_dns_thread(void *);
void load_addr(struct addrlist_head *, pthread_mutex_t *, char *);
int check_dnsrbl(char *, struct dnsrbl *, struct addrlist_head *,
		 pthread_mutex_t *, pthread_cond_t *, int *, int *);
void *check_dnsrbl_thread(void *);
int check_subnet(in_addr_t, in_addr_t, int);
int main(int, char **);

void
panic(msg)
	char *msg;
{
	printf("%s\n", msg); 
	exit(STATE_UNKNOWN);
}

void
free_addrlist(alp)
	struct addrlist_head *alp;
{
	struct addrlist *al;
     
	while (!SLIST_EMPTY(alp)) {
		al = SLIST_FIRST(alp);
		SLIST_REMOVE_HEAD(alp, al_next);
		free(al);
	}

	free(alp);
}


void
usage(name)
	char *name;
{
	printf("%s [-H address[,address...]] [-c config_file] [-v]\n", name);
	printf("%s [-c config_file] [-l]\t list DNSRBL\n", name);
	printf("%s [-h]\tThis help\n", name);

	exit(EX_USAGE);
}

void
parse_options(argc, argv)
	int argc;
	char **argv;
{
	time_t now;
	char *progname = argv[0];
	int dump = 0;
	int ch;

	now = time(NULL);
	timeout.tv_sec = now + DEFAULT_TIMEOUT;
	timeout.tv_nsec = 0;

	while ((ch = getopt(argc, argv, "H:c:lhvt:")) != -1) {
		switch (ch) {
		case 'v':
			verbose = 1;
			break;
		case 'l':
			dump = 1;
			break;
		case 'c':
			config_dnsrbl = load_dnsrbl(optarg);
			break;
		case 't':
			timeout.tv_sec = now + atoi(optarg);
			timeout.tv_nsec = 0;
			break;
		case 'H':
			load_addr(revaddrs, &revaddrs_mtx, optarg);
			break;
		case 'h':
		default:
			usage(progname);
			/* NOTREACHED */
			break;
		}

	}

	argc -= optind;
	argv += optind;

	if (dump) {
		dump_dnsrbl();
		exit(EX_OK);
	}

	return;
}

void
print_rev4(s, a)
	char *s;
	unsigned char *a;
{
	sprintf(s, "%d.%d.%d.%d",  
		a[3], a[2], a[1], a[0]);
}

void
print_rev6(s, a)
	char *s;
	unsigned char *a;
{
	sprintf(s, "%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x.%x",  
	        a[15] & 0xf, a[15] >> 4,
	        a[14] & 0xf, a[14] >> 4,
	        a[13] & 0xf, a[13] >> 4,
	        a[12] & 0xf, a[12] >> 4,
	        a[11] & 0xf, a[11] >> 4,
	        a[10] & 0xf, a[10] >> 4,
	        a[9] & 0xf, a[9] >> 4,
	        a[8] & 0xf, a[8] >> 4,
	        a[7] & 0xf, a[7] >> 4,
	        a[6] & 0xf, a[6] >> 4,
	        a[5] & 0xf, a[5] >> 4,
	        a[4] & 0xf, a[4] >> 4,
	        a[3] & 0xf, a[3] >> 4,
	        a[2] & 0xf, a[2] >> 4,
	        a[1] & 0xf, a[1] >> 4,
	        a[0] & 0xf, a[0] >> 4);
}

void *
rev_addr_thread(void *args)
{
	struct rev_thread_args *rta = args;
	int *count = rta->rta_count;

	rev_addr(rta->rta_alp, rta->rta_mtx, rta->rta_query, rta->rta_addr, rta->rta_family);

	if (pthread_mutex_lock(rta->rta_mtx) != 0)
		panic("pthread_mutex_lock failed");

	(*count)--;

	if (pthread_mutex_unlock(rta->rta_mtx) != 0)
		panic("pthread_mutex_unlock failed");

	if (pthread_cond_signal(rta->rta_cond) != 0)
		panic("pthread_cond_signal failed");

	free(rta->rta_query);
	free(rta->rta_addr);
	free(rta);
	return NULL;
}

void
rev_addr(alp, mtx, query, addr, family)
	struct addrlist_head *alp;
	pthread_mutex_t *mtx;
	char *query;
	unsigned char *addr;
	int family;
{
	struct addrlist *al;


	if ((al = malloc(sizeof(*al))) == NULL)
		panic("malloc failed");

	if ((al->al_query = strdup(query)) == NULL)
		panic("strdup failed");

	if (family == AF_INET)
		print_rev4(al->al_addr, addr);
	else if (family == AF_INET6)
		print_rev6(al->al_addr, addr);
	else
		panic("rta_family");

	if (pthread_mutex_lock(mtx) != 0)
		panic("pthread_mutex_lock failed");

	SLIST_INSERT_HEAD(alp, al, al_next);

	if (pthread_mutex_unlock(mtx) != 0)
		panic("pthread_mutex_unlock failed");

	return;
}

void
dump_dnsrbl(void) 
{
	struct dnsrbl *d;

	for (d = config_dnsrbl; d->d_domain; d++)
		printf("%s\t%s\n", d->d_domain, d->d_positive);

	return;
}

struct dnsrbl *
load_dnsrbl(file)
	char *file;
{
	FILE *f;
	char *line;
	size_t len;
	int lcount;
	struct dnsrbl *d;

	if ((f = fopen(file, "r")) == NULL)
		panic("cannot open dnsrbl file");

	for (lcount = 0; fgetln(f, &len) != NULL; lcount++); 

	if (verbose)
		printf("dnsrbl file \"%s\" has %d lines\n", file, lcount);

	if ((d = malloc((lcount + 1) * sizeof(*d))) == NULL)
		panic("malloc failed");

	rewind(f);

	d[lcount].d_domain = NULL; 
	d[lcount].d_positive = NULL; 

	while(lcount-- > 0) {
		const char *s = " \t\n";
		if ((line = fgetln(f, &len)) == NULL)
			panic( "unexpected empty line in dnsrbl");
			
		if ((d[lcount].d_domain = strdup(strtok(line, s))) == NULL)
			panic("strdup failed");

		if ((d[lcount].d_positive =  strdup(strtok(NULL, s))) == NULL)
			panic("strdup failed");

		if (verbose)
			printf("readen from \"%s\", %s:%s\n",
			       file, d[lcount].d_domain, d[lcount].d_positive);  
	}

	(void)fclose(f);

	return d;
}

void
load_addr(alp, mtx, addrs)
	struct addrlist_head *alp;
	pthread_mutex_t *mtx;
	char *addrs;
{	
	pthread_cond_t wait_cond = PTHREAD_COND_INITIALIZER;
	const char *s = " ,";
	char *a;
	int count = 0;
	int done = 0;

	if ((a = strtok(addrs, s)) == NULL)
		a = addrs;

	do {
		struct rev_thread_args *rta;
		pthread_t thread;
		int res;

		/* thread is to free it */
		if ((rta = malloc(sizeof(*rta))) == NULL)
			panic("malloc failed");

		rta->rta_alp = alp;
		rta->rta_mtx = mtx;
		rta->rta_query = strdup(a); /* thread is to free it */
		rta->rta_cond = &wait_cond;
		rta->rta_count = &count;

		if (rta->rta_query == NULL)
			panic("strdup failed");
		if ((rta->rta_addr = malloc(16)) == NULL)
			panic("malloc failed");

		if (verbose)
			printf("lookup %s\n", rta->rta_query);

		if (inet_pton(AF_INET, rta->rta_query, rta->rta_addr) == 1) {
			rta->rta_family = AF_INET;
			res = pthread_create(&thread, NULL,
					     *rev_addr_thread, rta);
		} else if (inet_pton(AF_INET6, rta->rta_query, rta->rta_addr) == 1) {
			rta->rta_family = AF_INET6;
			res = pthread_create(&thread, NULL,
					     *rev_addr_thread, rta);
		} else {
			rta->rta_family = AF_UNSPEC;
			free(rta->rta_addr);
			res = pthread_create(&thread, NULL,
					     *rev_dns_thread, rta);
		}
			
		if (res != 0)
			panic("pthread_create failed");

		count++;
	} while ((a = strtok(NULL, s)));

	if (pthread_mutex_lock(mtx) != 0)
		panic("pthread_mutex_lock failed");

	while (!done) {
		int res;

		res = pthread_cond_timedwait(&wait_cond, mtx, &timeout);
		if (res != 0 && errno == ETIMEDOUT) {
			if (verbose)
				printf("timeout in %s, %d lookups left\n",
				       __func__, count);
			done = 1;
			break;
		}

		done = (count == 0);

	}

	if (pthread_mutex_unlock(mtx) != 0)
		panic("pthread_mutex_unlock failed");

	return;
}

void *
rev_dns_thread(void *args)
{
	struct rev_thread_args *rta = args;
	int *count = rta->rta_count;

	rev_dns(rta->rta_alp, rta->rta_mtx, rta->rta_query);

	if (pthread_mutex_lock(rta->rta_mtx) != 0)
		panic("pthread_mutex_lock failed");

	(*count)--;

	if (pthread_mutex_unlock(rta->rta_mtx) != 0)
		panic("pthread_mutex_unlock failed");

	if (pthread_cond_signal(rta->rta_cond) != 0)
		panic("pthread_cond_signal failed");

	free(rta->rta_query);
	free(rta);
	return NULL;
}

void
rev_dns(alp, mtx, name)
	struct addrlist_head *alp;
	pthread_mutex_t *mtx;
	char *name;
{
	struct addrlist *al;
        unsigned char ans4[NS_MAXMSG + 1];
        int anslen4;
#ifdef AF_INET6
        unsigned char ans6[NS_MAXMSG + 1];
	int anslen6;
#endif /* AF_INET6 */
	ns_msg handle;
	ns_rr rr;
	int i;

	anslen4 = res_nquery(res, name, C_IN, T_A, ans4, NS_MAXMSG + 1);

	/*
	 * Collect IPv4 ansswers
	 */
	if (anslen4 != -1) {
		if (ns_initparse(ans4, anslen4, &handle) != 0)
			panic("ns_initparse IPv4 failed");

		for (i = 0; i < ns_msg_count(handle, ns_s_an); i++) {
			if ((ns_parserr(&handle, ns_s_an, i, &rr)) != 0) 
				continue;

			if ((rr.rdlength != 4) ||(rr.type != T_A))
				continue;

			if ((al = malloc(sizeof(*al))) == NULL)
				panic("malloc addrlist failed");

			if ((al->al_query = strdup(name)) == NULL)
				panic("strdup failed");
			print_rev4(al->al_addr, rr.rdata);
			
			if (pthread_mutex_lock(mtx) != 0)
				panic("pthread_mutex_lock failed");

			SLIST_INSERT_HEAD(alp, al, al_next);

			if (pthread_mutex_unlock(mtx) != 0)
				panic("pthread_mutex_unlock failed");

		}
	}

#ifdef AF_INET6
	anslen6 = res_nquery(res, name, C_IN, T_AAAA, ans6, NS_MAXMSG + 1);


	/*
	 * Collect IPv6 ansswers
	 */
	if (anslen6 != -1) {
		if (ns_initparse(ans6, anslen6, &handle) != 0)
			panic("ns_initparse IPv6 failed");

		for (i = 0; i < ns_msg_count(handle, ns_s_an); i++) {
			if ((ns_parserr(&handle, ns_s_an, i, &rr)) != 0) 
				continue;

			if ((rr.rdlength != 16) || (rr.type != T_AAAA))
				continue;

			if ((al = malloc(sizeof(*al))) == NULL)
				panic("malloc addrlist failed");

			if ((al->al_query = strdup(name)) == NULL)
				panic("strdup failed");
			print_rev6(al->al_addr, rr.rdata);

			if (pthread_mutex_lock(mtx) != 0)
				panic("pthread_mutex_lock failed");

			SLIST_INSERT_HEAD(alp, al, al_next);

			if (pthread_mutex_unlock(mtx) != 0)
				panic("pthread_mutex_unlock failed");
		}
	}
#endif /* AF_INET6 */

	return;
}

int
check_dnsrbl(revaddr, d, alp, mtx, wait_cond, retval, count) 
	char *revaddr;
	struct dnsrbl *d;
	struct addrlist_head *alp;
	pthread_mutex_t *mtx;
	pthread_cond_t *wait_cond;
	int *retval;
	int *count;
{
	struct check_dnsrbl_args *cda;
	pthread_t thread;

	/* thread is to free it */
	if ((cda = malloc(sizeof(*cda))) == NULL)
		panic("malloc failed");

	cda->cda_revaddr = revaddr;
	cda->cda_d = d;
	cda->cda_alp = alp;
	cda->cda_mtx = mtx;
	cda->cda_cond = wait_cond;
	cda->cda_retval = retval;
	cda->cda_count = count;

	if (pthread_create(&thread, NULL, *check_dnsrbl_thread, cda) != 0)
		panic("pthread_create failed");

	return 0;
}

void *
check_dnsrbl_thread(void *args)
{
	struct check_dnsrbl_args *cda = args;
	char *revaddr = cda->cda_revaddr;
	struct dnsrbl *d = cda->cda_d;
	struct addrlist_head *alp = cda->cda_alp;
	pthread_mutex_t *mtx = cda->cda_mtx;
	pthread_cond_t *wait_cond = cda->cda_cond;
	int *retval = cda->cda_retval;
	int *count = cda->cda_count;

	char queryname[MAXHOSTNAMELEN + 1];
	char positive_addrstr[MAXHOSTNAMELEN + 1];
	in_addr_t positive_addr;
	struct addrlist *al;
	struct addrlist *altmp;
        unsigned char ans4[NS_MAXMSG + 1];
        int anslen4;
	int mask;
	char *cp;
	ns_msg handle;
	ns_rr rr;
	int i;

	sprintf(queryname, "%s.%s", revaddr, d->d_domain);

	strcpy(positive_addrstr, d->d_positive);
	if (strcmp(positive_addrstr, "*") == 0) {
		positive_addr = inet_addr("0.0.0.0");	
		mask = 0;
	} else if ((cp = strchr(positive_addrstr, '/')) != NULL) {
		*cp = '\0';
		positive_addr = inet_addr(positive_addrstr);	
		mask = atoi(cp + 1);
	} else {
		positive_addr = inet_addr(positive_addrstr);	
		mask = 32;
	}

	/* 
	 * IPv4 resolution
	 */

	anslen4 = res_nquery(res, queryname, C_IN, T_A, 
			     ans4, NS_MAXMSG + 1);

	if (anslen4 == -1 && verbose)
		printf("+ %s is not listed\n", queryname);

	/*
	 * Collect IPv4 ansswers
	 */
	if (anslen4 != -1) {
		int msg_count;

		if (ns_initparse(ans4, anslen4, &handle) != 0)
			panic("ns_initparse IPv4 failed");

		msg_count = ns_msg_count(handle, ns_s_an);
		for (i = 0; i < msg_count; i++) {
			in_addr_t resaddr;

			if ((ns_parserr(&handle, ns_s_an, i, &rr)) != 0) 
				continue;

			if ((rr.rdlength != 4) ||(rr.type != T_A))
				continue;

			resaddr = (rr.rdata[3] << 24)
			        + (rr.rdata[2] << 16)
			        + (rr.rdata[1] << 8)
			        + (rr.rdata[0] << 0);

			if (!check_subnet(resaddr, positive_addr, mask))
				continue;

			if ((al = malloc(sizeof(*al))) == NULL)
				panic("malloc failed");

			if ((al->al_query = strdup(queryname)) == NULL)
				panic("strdup failed");
			sprintf(al->al_addr, "%d.%d.%d.%d", 
			       rr.rdata[0], rr.rdata[1],
			       rr.rdata[2], rr.rdata[3]);

			/*
			 * Avoid dups
			 */
			if (pthread_mutex_lock(mtx) != 0)
				panic("pthread_mutex_lock failed");

			SLIST_FOREACH(altmp, alp, al_next) {
				if (strcmp(al->al_addr, altmp->al_addr) == 0) {
					free(al);
					al = NULL;
					break;
				}
			}

			if (al != NULL) {
				SLIST_INSERT_HEAD(alp, al, al_next);
				(*retval)++;
				if (verbose) 
					printf("- %s blacklisted\n", queryname);
			}

			if (pthread_mutex_unlock(mtx) != 0)
				panic("pthread_mutex_unlock failed");
		}
	}

	if (pthread_mutex_lock(mtx) != 0)
		panic("pthread_mutex_lock failed");

	(*count)++;

	if (pthread_mutex_unlock(mtx) != 0)
		panic("pthread_mutex_unlock failed");

	if (pthread_cond_signal(wait_cond) != 0)
		panic("pthread_cond_signal failed");

	free(cda);
	return NULL;
}

int
check_subnet(addr, subnet, maskint)
	in_addr_t addr;
	in_addr_t subnet;
	int maskint;
{
	in_addr_t mask;

	mask = (maskint == 32) ? ~0 : (1 << (maskint - 32)) - 1;

	return ((addr & mask) == (subnet & mask));
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	int status_code = STATE_UNKNOWN;
	struct addrlist *al;
	struct dnsrbl *d;
	struct addrlist_head *alp;
	pthread_mutex_t mtx = PTHREAD_MUTEX_INITIALIZER;
	pthread_cond_t wait_cond = PTHREAD_COND_INITIALIZER;
	struct addrlist *ral;
	int retval = 0;
	int count = 0;
	int total_count = 0;
	int done = 0;


	/*
	 * initialize resolver
	 */
	if ((res = malloc(sizeof(*res))) == NULL)
		panic("Cannot malloc resolver");

	if (res_ninit(res) != 0)
		panic("Cannot init resolzer");
		
	/*
	 * Initialize address list
	 */
	if ((revaddrs = malloc(sizeof(*revaddrs))) == NULL)
		err(EX_OSERR, "failed to malloc addrlist head");

	SLIST_INIT(revaddrs);

	/* 
	 * Initialize output list
	 */
	if ((alp = malloc(sizeof(*alp))) == NULL)
		panic("failed to malloc addrlist head");

	SLIST_INIT(alp);

	parse_options(argc, argv);

	if (SLIST_EMPTY(revaddrs)) {
		struct utsname utsname;

		if (uname(&utsname) != 0)
			err(EX_OSERR, "failed to get my own name");

		load_addr(revaddrs, &revaddrs_mtx,  utsname.nodename);
	}

	if (SLIST_EMPTY(revaddrs))
		panic("No address to check");
	
	SLIST_FOREACH(al, revaddrs, al_next) {
		for (d = config_dnsrbl; d->d_domain; d++) {
			total_count++;

			if (verbose)
				printf("trying %s.%s\n",
				       al->al_addr, d->d_domain);

			check_dnsrbl(al->al_addr, d, alp, &mtx, 
				     &wait_cond, &retval, &count);
		}
	}

	/*
	 * Wait replies 
	 */
	if (pthread_mutex_lock(&mtx) != 0)
		panic("pthread_mutex_lock failed");

	while (!done) {
		int res;

		res = pthread_cond_timedwait(&wait_cond,
					     &mtx, &timeout);
		if (res != 0 && errno == ETIMEDOUT) {
			if (verbose)
				printf("timeout in %s, "
				       "got %d/%d replies\n",
				       __func__, count, total_count);
			done = 1;
			break;
		}

		done = (count == total_count);
	}

	/*
	 * Display failures.
	 * 
	 * Keep mutex locked in case another 
	 * thread gets a late result
	 */
	if (retval > 0) {
		printf("FAILED ");

		SLIST_FOREACH(ral, alp, al_next)
			printf("%s%s returns %s",
				(ral != SLIST_FIRST(alp)) ? ", " : "",
				ral->al_query, ral->al_addr); 
		printf("\n");
	}

	if (pthread_mutex_unlock(&mtx) != 0)
		panic("pthread_mutex_unlock failed");

	if (retval == 0) {
		if (count != total_count)
			printf("status ok (with %d/%d DNSRBL replies)\n",
			       count, total_count);
		else
			printf("status ok \n");
		status_code = STATE_OK;
	} else if (count == 0) {
		status_code = STATE_UNKNOWN;
	} else {
		status_code = STATE_CRITICAL;
	}

	/* 
	 * Do not free anything as some other thread may still use it.
	 * exit() will cleanup anyway
	 */

	return status_code;
}
