/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 *  $Id: bsdavdemux.c,v 1.9 2006/03/14 03:41:20 jakemsr Exp $
 */

#include "includes.h"

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/shm.h>

#include <err.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "bsdav.h"

extern char *__progname;

int open_files(void);
int stream(void);

void cleanup(void);
void usage(void);

char	 in_path[FILENAME_MAX];
FILE	*in_file;
int	 bsdavdemux_quit;
int	 do_audio;
int	 do_video;
int	 verbose;

char	 vid_path[FILENAME_MAX];
FILE	*vid_file;
uint8_t	*vid_buffer;
size_t	 vid_buffer_size;

char	 aud_path[FILENAME_MAX];
FILE	*aud_file;
uint8_t	*aud_buffer;
size_t	 aud_buffer_size;


void
usage(void)
{
extern char *__progname;

	fprintf(stderr,
		"Usage: %s [-lv] [-i input] [-O aout] [-o vout]\n",
		__progname);
	exit(1);
}


void
cleanup(void)
{
	if (in_file != NULL)
		fclose(in_file);
	in_file = NULL;

	if (vid_file != NULL)
		fclose(vid_file);
	vid_file = NULL;

	if (aud_file != NULL)
		fclose(aud_file);
	aud_file = NULL;
}


int
stream(void)
{
struct bsdav_frame_header frmhdr;
struct timeval start_tv;
struct timeval end_tv;
struct timeval run_tv;
double	 run_time;
int	 count;
int	 sequence;
long	 vid_frame_count;
long	 aud_frame_count;

	vid_frame_count = 0;
	aud_frame_count = 0;
	sequence = 200;
	count = 0;

	gettimeofday(&start_tv, NULL);

	while (bsdavdemux_quit == 0) {

		count++;

		if (bsdav_read_frame_header(in_file, &frmhdr) != 0) {
			if (feof(in_file))
				warnx("EOF");
			else
				warnx("bsdav_read_frame_header failed   ");
			break;
		}

		if (verbose > 2)
			bsdav_dump_frame_header(&frmhdr);

		switch (frmhdr.type) {
		case BSDAV_FRMTYP_VIDEO:
			if (bsdav_read_frame_data(in_file, vid_buffer,
			    (size_t)frmhdr.size, ((do_video == 1) ? 0 : 1)) != 0) {
				if (feof(in_file) == 1)
					warnx("EOF");
				else
					warnx("video read error   ");
				bsdavdemux_quit = 1;
			}
			if (do_video == 1) {
				if (bsdav_write_frame_data(vid_file,
				    vid_buffer, (size_t)frmhdr.size, 0) != 0) {
					warnx("video write error   ");
					break;
				} else
					vid_frame_count++;
			}
			break;
		case BSDAV_FRMTYP_AUDIO:
			if (bsdav_read_frame_data(in_file, aud_buffer,
			    (size_t)frmhdr.size, ((do_audio == 1) ? 0 : 1)) != 0) {
				if (feof(in_file) == 1)
					warnx("EOF");
				else
					warnx("audio read error   ");
				bsdavdemux_quit = 1;
			}
			if (do_audio == 1) {
				if (bsdav_write_frame_data(aud_file,
				    aud_buffer, (size_t)frmhdr.size, 0) != 0) {
					warnx("audio write error   ");
					break;
				} else
					aud_frame_count++;
			}
			break;
		default:
			warnx("invalid frame header type");
			bsdavdemux_quit = 1;
			break;
		}

		if ((bsdavdemux_quit == 0) && (count == sequence)) {
			count = 0;
			gettimeofday(&end_tv, NULL);
			timersub(&start_tv, &end_tv, &run_tv);
			run_time = run_tv.tv_sec + (double)run_tv.tv_usec / 1000000;
			if (verbose > 1) {
				fprintf(stderr, "%s: ", __progname);
				if (do_video == 1)
					fprintf(stderr,
					    " vf: %06ld, vfps: %04.4f",
				    	    vid_frame_count, vid_frame_count /
					    run_time);
				if (do_audio == 1)
					fprintf(stderr,
					    " af: %06ld, afps: %04.4f",
			    		    aud_frame_count, aud_frame_count /
					    run_time);
				fprintf(stderr, "\r");
				fflush(stderr);
			}
		}
	}

	gettimeofday(&end_tv, NULL);

	/*  last character was probably a carriage return */
	if (verbose > 1)
		fprintf(stderr, "\n");

	if (verbose > 0) {
		timersub(&start_tv, &end_tv, &run_tv);
		run_time = run_tv.tv_sec + (double)run_tv.tv_usec /
		    1000000;
		if (do_video == 1)
			warnx("video: frames: %ld, fps: %f",
			    vid_frame_count, vid_frame_count / run_time);
		if (do_audio == 1)
			warnx("audio: frames: %ld, fps: %f",
			    aud_frame_count, aud_frame_count / run_time);
	}

	return(0);
}


int
main(int argc, char *argv[])
{
struct bsdav_stream_header strhdr;
int	 ch;
int	 sret;
int	 lflag;

	/* defaults */
	vid_file = NULL;
	vid_buffer = NULL;
	vid_buffer_size = 0;
	aud_file = NULL;
	aud_buffer = NULL;
	aud_buffer_size = 0;

	in_file = NULL;
	bsdavdemux_quit = 0;
	do_audio = 0;
	do_video = 0;
	verbose = 0;
	lflag = 0;

	while ((ch = getopt(argc, argv, "lvi:O:o:")) != -1) {
		switch (ch) {
		case 'i':
			sret = snprintf(in_path, FILENAME_MAX, optarg);
			if (sret >= FILENAME_MAX)
				errx(1, "input file name is too long");
			break;
		case 'l':
			lflag++;
			break;
		case 'O':
			sret = snprintf(aud_path, FILENAME_MAX, optarg);
			if (sret >= FILENAME_MAX)
				errx(1, "audio file name is too long");
			do_audio = 1;
			break;
		case 'o':
			sret = snprintf(vid_path, FILENAME_MAX, optarg);
			if (sret >= FILENAME_MAX)
				errx(1, "video file name is too long");
			do_video = 1;
			break;
		case 'v':
			verbose++;
			break;
		default:
			usage();
			break;
		}
	}
	argc -= optind;
	argv += optind;

	if (in_path[0] == '\0') {
		warnx("no input file specified");
		exit(1);
	}

	if (lflag > 0) {
		if ((in_file = fopen(in_path, "r")) == NULL) {
			warn("%s", in_path);
			exit(1);
		}
		if (bsdav_read_stream_header(in_file, &strhdr) != 0) {
			warnx("bsdav_read_stream_header failed");
			cleanup();
			exit(1);
		}
		bsdav_dump_stream_header(&strhdr);
		cleanup();
		exit(0);
	} else if ((do_video == 0) && (do_audio == 0)) {
		warnx("demuxing neither audio nor video, exiting");
		exit(1);
	}

	if (verbose > 1)
		warnx("opening input: %s", in_path);

	if ((in_file = fopen(in_path, "r")) == NULL) {
		warn("%s", in_path);
		cleanup();
		exit(1);
	}

	if (bsdav_read_stream_header(in_file, &strhdr) != 0) {
		warnx("bsdav_read_stream_header failed");
		cleanup();
		exit(1);
	}

	if ((strhdr.vidfmt == BSDAV_VIDFMT_NONE) || (strhdr.vidmfs == 0))
		do_video = 0;

	if ((strhdr.audfmt == BSDAV_AUDFMT_NONE) || (strhdr.audmfs == 0))
		do_audio = 0;

	if (do_video == 1) {
		vid_buffer_size = strhdr.vidmfs;
		if ((vid_buffer = malloc(vid_buffer_size)) == NULL) {
			warn("vid_buffer");
			cleanup();
			exit(1);
		}
		if ((vid_file = fopen(vid_path, "w")) == NULL) {
			warn("%s", vid_path);
			cleanup();
			exit(1);
		}
	}

	if (do_audio == 1) {
		aud_buffer_size = strhdr.audmfs;
		if ((aud_buffer = malloc(aud_buffer_size)) == NULL) {
			warn("aud_buffer");
			cleanup();
			exit(1);
		}
		if ((aud_file = fopen(aud_path, "w")) == NULL) {
			warn("%s", aud_path);
			cleanup();
			exit(1);
		}
	}

	if (stream() != 0) {
		warnx("failed to stream");
		cleanup();
		exit(1);
	}

	cleanup();

	return (0);
}
