/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 *  $Id: avplay.c,v 1.23 2006/03/30 09:21:26 jakemsr Exp $
 */

#include "includes.h"

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/mman.h>
#include <sys/shm.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include <X11/extensions/Xvlib.h>
#include <X11/Xatom.h>

#include "bsdav.h"

struct avp_video {
	Display		*dpy;
	Window		 window;
	Window		 rootwindow;
	XEvent		 event;
	GC		 gc;
	Atom		 wmdelwin;
	XWindowAttributes *norm_winatt;
	XvPortID	 port;
	XvImage		*xv_image;
	XShmSegmentInfo	 shminfo;
	char		*pixels;
	unsigned int	 adaptor;
	int		 display_width;
	int		 display_height;
	int		 screen_id;
	char		 in_path[FILENAME_MAX];
	FILE		*in_file;
	int		 format;
	int		 norm;
	int		 width;
	int		 height;
	struct bsdav_ratio  aspect;
	struct bsdav_ratio fps;
	size_t		 buffer_size;
	uint8_t		*buffer;
	Bool		 full_screen;
	Bool		 aspect_lock;
	long		 frame_count;
	int		 sync;
};

struct avp_audio {
	char		 dev[FILENAME_MAX];
	int	 	 fd;
	char		 in_path[FILENAME_MAX];
	FILE		*in_file;
	int	 	 format;
	int		 channels;
	int		 sample_rate;
	int		 samples_per_frame;
	size_t		 buffer_size;
	uint8_t		*buffer;
	int		 fill_buffs;
	long		 refill_buffs;
	long		 frame_count;
};

struct avplay_t {
	struct avp_video	*vid;
	struct avp_audio	*aud;
	int		 shutdown;
	long		 usecs_per_frame;
	int		 verbose;
	Bool		 pause;
	Bool		 do_audio;
	Bool		 do_video;
};

extern int errno;
volatile sig_atomic_t missed_ticks;
int playing;

void sig_handler(int);
int open_files(struct avplay_t *);
int get_usecs_per_frame(struct avplay_t *);
int init_buffers(struct avplay_t *);
int sync_ticks(struct avplay_t *, int);
int get_audio_buf(struct avplay_t *);
int play_audio_buf(struct avplay_t *);
int fill_audio_buf(struct avplay_t *, int);
int stream(struct avplay_t *);
int set_video_format(struct avplay_t *);
int create_win(struct avplay_t *);
int get_video_buf(struct avplay_t *);
int play_video_buf(struct avplay_t *);
int display_event(struct avplay_t *);
int resize_window(struct avplay_t *, Bool);
int switch_input(struct avplay_t *, int);
int switch_pause(struct avplay_t *);

void cleanup(struct avplay_t *);
void usage(void);

void list_xv_enc(struct avplay_t *);

extern char *__progname;

void
usage(void)
{
	fprintf(stderr,
		"Usage: %s [-l] [-v] [-C achans] [-E aenc] [-I afile]\n"
		"	   [-R arate] [-a vaspect] [-e venc] [-f file]\n"
		"          [-h vheight] [-i vfile] [-n vnorm] [-r vrate]\n"
		"          [-w vwidth]\n",
		__progname);
	return;
}


void
sig_handler(int signal)
{
	if (signal == SIGALRM)
		if (playing == 1)
			missed_ticks++;
	return;
}


void
list_xv_enc(struct avplay_t *avp)
{
XvImageFormatValues *xvformats;
XvAdaptorInfo *ainfo;
unsigned int p;
int i, j, k;
int num_xvformats;
int num_adaptors;


	if ((avp->vid->dpy = XOpenDisplay(NULL)) == NULL) {
		warnx("cannot open display %s", XDisplayName(NULL));
		return;
	}

	if (Success != XvQueryExtension(avp->vid->dpy, &p, &p, &p, &p, &p)) {
		warnx("Xv not available");
		return;
	}
	avp->vid->screen_id = DefaultScreen(avp->vid->dpy);

	avp->vid->rootwindow = DefaultRootWindow(avp->vid->dpy);

	if (Success != XvQueryAdaptors(avp->vid->dpy, avp->vid->rootwindow,
	    &num_adaptors, &ainfo)) {
		warnx("no Xv adaptors present");
		return;
	}

	printf("Video Formats\n");
	for (i = 0; i < num_adaptors; i++) {
		if ((ainfo[i].type & XvInputMask) &&
		    (ainfo[i].type & XvImageMask)) {
			printf("  Xv Adaptor %d: %s\n", i, ainfo[i].name);
			printf("    %-12s  %16s\n", "name", "bits per pixel");
			xvformats = XvListImageFormats(avp->vid->dpy,
			    ainfo[i].base_id, &num_xvformats);
			for (j = 0; j < num_xvformats; j++)
				for (k = 0; bsdav_vid_fmts[k].name; k++)
					if (xvformats[j].id ==
					    bsdav_vid_fmts[k].xv_id)
						printf("    %12s  %16d\n",
						    bsdav_vid_fmts[k].name,
						    bsdav_vid_fmts[k].bpp);
			if (xvformats != NULL)
				XFree(xvformats);
		}
	}
	XvFreeAdaptorInfo(ainfo);

	return;
}


int
get_usecs_per_frame(struct avplay_t *avp)
{

	if (avp->vid->norm != BSDAV_VIDNORM_NONE) {
		if (avp->vid->fps.n == 0) {
			avp->vid->fps.n =
			    bsdav_vid_nrms[avp->vid->norm].frame_rate.n;
			avp->vid->fps.d =
			    bsdav_vid_nrms[avp->vid->norm].frame_rate.d;
		}
	}

	if (avp->vid->fps.n != 0)
		avp->usecs_per_frame = 1000000 * (double)avp->vid->fps.d /
		    avp->vid->fps.n;
	else {
		warnx("no frame rate, exiting");
		return (1);
	}

	if (avp->verbose > 1)
		warnx("usecs_per_frame = %ld", avp->usecs_per_frame);

	avp->aud->samples_per_frame = (double)avp->aud->sample_rate *
	    avp->vid->fps.d / avp->vid->fps.n;

	return (0);
}


int
open_files(struct avplay_t *avp)
{
unsigned int p;

	if (avp->do_video == True) {
		if ((avp->vid->dpy = XOpenDisplay(NULL)) == NULL) {
			warnx("cannot open display %s", XDisplayName(NULL));
			return (1);
		}

		if (Success != XvQueryExtension(avp->vid->dpy, &p, &p, &p,
		    &p, &p)) {
			warnx("Xv not available");
			return (1);
		}
		avp->vid->screen_id = DefaultScreen(avp->vid->dpy);

		if (avp->verbose > 1)
			warnx("opening video input: %s", avp->vid->in_path);

		if ((avp->vid->in_file = fopen(avp->vid->in_path, "r")) == NULL) {
			warn("%s", avp->vid->in_path);
			return (1);
		}
	}

	if (avp->do_audio == True) {
		if ((avp->aud->fd = open(avp->aud->dev, O_WRONLY)) < 0) {
			warnx("could not open %s", avp->aud->dev);
			close(avp->aud->fd);
			return (1);
		}

		if (avp->verbose > 1)
			warnx("opening audio input: %s", avp->aud->in_path);

		if ((avp->aud->in_file = fopen(avp->aud->in_path, "r")) == NULL) {
			warn("%s", avp->aud->in_path);
			return (1);
		}
	}

	return (0);
}


int
set_video_format(struct avplay_t *avp)
{
XvImageFormatValues *xvformats;
XvAdaptorInfo *ainfo;
int i, j, k;
int num_xvformats;
int num_adaptors;


	if (avp->vid->norm != BSDAV_VIDNORM_NONE) {
		if (avp->vid->width == 0)
			avp->vid->width =
			    bsdav_vid_nrms[avp->vid->norm].width;
		if (avp->vid->height == 0)
			avp->vid->height =
			    bsdav_vid_nrms[avp->vid->norm].height;
	}

	avp->vid->rootwindow = DefaultRootWindow(avp->vid->dpy);

	if (Success != XvQueryAdaptors(avp->vid->dpy, avp->vid->rootwindow,
	    &num_adaptors, &ainfo)) {
		warnx("no Xv adaptors present");
		return (1);
	}

	for (i = 0; i < num_adaptors && !avp->vid->port; i++) {
		if ((ainfo[i].type & XvInputMask) &&
		    (ainfo[i].type & XvImageMask)) {
			if (avp->verbose > 1)
				warnx("adaptor %d can PutImage", i);
			xvformats = XvListImageFormats(avp->vid->dpy,
			    ainfo[i].base_id, &num_xvformats);
			for (j = 0; j < num_xvformats; j++) {
				if (xvformats[j].id ==
				    bsdav_vid_fmts[avp->vid->format].xv_id) {
					avp->vid->adaptor = i;
					break;
				}
			}
			if (xvformats != NULL)
				XFree(xvformats);
		}
		for (k = 0; k < ainfo[i].num_ports; ++k) {
			if (Success == XvGrabPort(avp->vid->dpy,
			    ainfo[i].base_id, CurrentTime)) {
				avp->vid->port = ainfo[i].base_id;
				if (avp->verbose > 1)
					warnx("grabbed Xv port %ld",
					    avp->vid->port);
				break;
			}
		}
	}

	XvFreeAdaptorInfo(ainfo);

	if (avp->vid->port == 0) {
		warnx("could not find valid Xv port");
		return (1);
	}

	return (0);
}


int
create_win(struct avplay_t * avp)
{
XGCValues values;
XTextProperty WinName;
XSizeHints szhints;
XWMHints wmhints;
char *name;
extern char *__progname;

	name = __progname;
	XStringListToTextProperty(&name, 1, &WinName);

	szhints.flags = PSize | PMaxSize | PMinSize;
	szhints.width = avp->vid->display_width;
	szhints.height = avp->vid->display_height;
	szhints.max_width = 2048;  /* get this from xv_init + DisplayWidth */
	szhints.max_height = 2048;  /* get this from xv_init + DisplayHeight*/
	szhints.min_width = 160;
	szhints.min_height = 120;

	wmhints.flags = InputHint | StateHint;
	wmhints.input = True;
	wmhints.initial_state = NormalState;

	if (avp->vid->full_screen == True) {
		avp->vid->display_width = DisplayWidth(avp->vid->dpy,
		    avp->vid->screen_id);
		avp->vid->display_height = DisplayHeight(avp->vid->dpy,
		    avp->vid->screen_id);
	} else {
		avp->vid->display_width = avp->vid->width;
		avp->vid->display_height = avp->vid->height;
	}

	avp->vid->window = XCreateSimpleWindow(avp->vid->dpy,
	    avp->vid->rootwindow, 0, 0, avp->vid->display_width,
	    avp->vid->display_height, 0,
	    XWhitePixel(avp->vid->dpy, avp->vid->screen_id),
	    XBlackPixel(avp->vid->dpy, avp->vid->screen_id));

	XSetWMProperties(avp->vid->dpy, avp->vid->window, &WinName, &WinName,
	    NULL, 0, &szhints, &wmhints, NULL);

	XSelectInput(avp->vid->dpy, avp->vid->window,
	    KeyPressMask | ButtonPressMask | StructureNotifyMask);

	avp->vid->wmdelwin = XInternAtom(avp->vid->dpy, "WM_DELETE_WINDOW", False);
	XSetWMProtocols(avp->vid->dpy, avp->vid->window, &avp->vid->wmdelwin, 1);

	XMapRaised(avp->vid->dpy, avp->vid->window);

	avp->vid->gc = XCreateGC(avp->vid->dpy, avp->vid->window, 0, &values);

	avp->vid->xv_image = XvShmCreateImage(avp->vid->dpy, avp->vid->port,
	    bsdav_vid_fmts[avp->vid->format].xv_id, avp->vid->pixels,
	    avp->vid->width, avp->vid->height, &avp->vid->shminfo);

	avp->vid->shminfo.shmid = shmget(IPC_PRIVATE, avp->vid->buffer_size,
	    IPC_CREAT | 0777);

	if (avp->vid->shminfo.shmid < 0) {
		warn("shmget");
		return (1);
	}
	avp->vid->xv_image->data = avp->vid->pixels =
	    avp->vid->shminfo.shmaddr = shmat(avp->vid->shminfo.shmid, 0, 0);
	if (avp->vid->shminfo.shmaddr == (void *)-1) {
		warn("shmat");
		return (1);
	}
	XShmAttach(avp->vid->dpy, &avp->vid->shminfo);

	XSync(avp->vid->dpy, False);

	return (0);
}


int
init_buffers(struct avplay_t *avp)
{
size_t	 hw_buf_size;

	if (avp->do_video == True) {
		avp->vid->buffer_size = avp->vid->width * avp->vid->height *
		    bsdav_vid_fmts[avp->vid->format].bpp / 8;

		if ((avp->vid->buffer =
		    malloc(avp->vid->buffer_size)) == NULL) {
			warn("avp->vid->buffer");
			return (1);
		}

		if ((avp->vid->norm_winatt =
		    malloc(sizeof(XWindowAttributes))) == NULL) {
			warn("avp->vid->norm_winatt");
			return (1);
		}
	}

	if (avp->do_audio == True) {
		avp->aud->buffer_size = avp->aud->samples_per_frame *
		    (bsdav_aud_fmts[avp->aud->format].bps / 8) *
		    avp->aud->channels;

		if ((avp->aud->buffer =
		    malloc(avp->aud->buffer_size)) == NULL) {
			warn("avp->aud->buffer");
			return (1);
		}

		hw_buf_size = bsdav_get_audhw_buf_size(avp->aud->fd,
		    BSDAV_AUDMODE_PLAY);

		avp->aud->fill_buffs = hw_buf_size / avp->aud->buffer_size;

		avp->aud->refill_buffs = hw_buf_size / 4;
	}

	return (0);
}


int
get_video_buf(struct avplay_t *avp)
{
	if (fread(avp->vid->buffer, avp->vid->buffer_size, 1,
	    avp->vid->in_file) == 1)
		return (0);
	else {
		if (feof(avp->vid->in_file) == 1) {
			warnx("video EOF");
			avp->shutdown = 1;
			return (0);
		} else {
			warn("video fread");
			return (1);
		}
	}
}


int
get_audio_buf(struct avplay_t *avp)
{
	if (fread(avp->aud->buffer, avp->aud->buffer_size, 1,
	    avp->aud->in_file) == 1)
		return (0);
	else {
		if (feof(avp->aud->in_file) == 1) {
			warnx("audio EOF");
			avp->shutdown = 1;
			return (0);
		} else {
			warn("audio fread");
			return (1);
		}
	}
}


int
play_video_buf(struct avplay_t *avp)
{
	memcpy(avp->vid->pixels, avp->vid->buffer, avp->vid->buffer_size);

	XvShmPutImage(avp->vid->dpy, avp->vid->port, avp->vid->window,
	    avp->vid->gc, avp->vid->xv_image, 0, 0, avp->vid->width,
	    avp->vid->height, 0, 0, avp->vid->display_width,
	    avp->vid->display_height, True);

	return (0);
}


int
play_audio_buf(struct avplay_t *avp)
{
size_t	left;
off_t	offset;
ssize_t	wrote;

	for (left = avp->aud->buffer_size, offset = 0; left > 0;) {
		wrote = write(avp->aud->fd, avp->aud->buffer + offset, left);
		if (wrote == 0) {
			if (avp->verbose > 1)
				warnx("audio play: wrote == 0");
		}
		if (wrote < 0) {
			if (errno == EINTR) {
				wrote = 0;
				if (avp->verbose > 1)
					warn("audio play");
			} else if (errno == EAGAIN) {
				wrote = 0;
				if (avp->verbose > 1)
					warn("audio play");
			} else {
				warn("audio play");
				return (1);
			}
		}
		if (wrote > left) {
			warnx("audio: write returns more bytes than requested");
			warnx("audio: requested: %llu, returned: %lld",
			    (unsigned long long)left, (long long)wrote);
			return (1);
		}
		offset += wrote;
		left -= wrote;
	}
	return (left);
}


int
fill_audio_buf(struct avplay_t *avp, int fill_buffs)
{
size_t	left;
off_t	offset;
ssize_t	wrote;
uint8_t	*buffer;
size_t	buffer_size;

	buffer_size = avp->aud->refill_buffs * fill_buffs;

	if ((buffer = malloc(buffer_size)) == NULL) {
		warn("audio fill buffer");
		return (1);
	}

	if (fread(buffer, buffer_size, 1, avp->aud->in_file) != 1) {
		if (feof(avp->aud->in_file) == 1) {
			warnx("audio EOF");
			avp->shutdown = True;
			return (0);
		}
	}
	for (left = buffer_size, offset = 0; left > 0;) {
		wrote = write(avp->aud->fd, buffer + offset, left);
		if (wrote == 0) {
			if (avp->verbose > 1)
				warn("audio play");
		}
		if (wrote < 0) {
			if (errno == EINTR) {
				wrote = 0;
				if (avp->verbose > 1)
					warn("audio play");
			} else {
				warn("audio play");
				return (1);
			}
		}
		if (wrote > left) {
			warnx("audio: write returns more bytes than requested");
			warnx("audio: requested: %llu, returned: %lld",
			    (unsigned long long)left, (long long)wrote);
			return (1);
		}
		offset += wrote;
		left -= wrote;
	}

	return (0);
}


int
sync_ticks(struct avplay_t *avp, int ticks)
{
int	 skip;

	if (ticks > 0) {
		/* fell behind, catch up */
		avp->vid->sync++;

		if (avp->do_audio == True) {
			/* write an audio frame */
			if (get_audio_buf(avp) != 0) {
				warnx("audio read errror");
				return (1);
			}
			if (play_audio_buf(avp) != 0) {
				warnx("audio play error");
				return (1);
			} else
				avp->aud->frame_count++;
		}

		/* far behind? */
		if (ticks > 1) {
			if (avp->verbose > 1)
				warnx("missed ticks = %d", ticks);
			/* don't go too far, ie pause */
			skip = ticks / 10;
			if (skip > 2)
				skip = 2;
			if (skip > 0)
				avp->vid->sync += skip;
		}
		missed_ticks -= ticks;
	}
	return (0);
}


int
stream(struct avplay_t *avp)
{
struct sigaction act;
sigset_t sa_mask;
struct itimerval tim;
struct timeval tp_start;
struct timeval tp_end;
struct timeval tp_run;
double	 run_time;
int	 skip;
size_t	 hw_pos;
int	 count;
int	 sequence;

	avp->pause = False;

	avp->vid->frame_count = 0;
	avp->aud->frame_count = 0;
	sequence = 200;
	count = 0;

	/* for sigsuspend() */
	sigfillset(&sa_mask);
	sigdelset(&sa_mask, SIGALRM);

	memset(&act, 0, sizeof(act));
	sigemptyset(&act.sa_mask);
	act.sa_handler = sig_handler;
	sigaction(SIGALRM, &act, NULL);

	memset(&tim, 0, sizeof(tim));
	tim.it_value.tv_usec = 1;
	tim.it_interval.tv_usec = avp->usecs_per_frame - 1;
	setitimer(ITIMER_REAL, &tim, NULL);

	if (avp->do_audio == True) {
		if (fill_audio_buf(avp, 3) != 0) {
			warnx("couldn't prefill audio buffer");
			return (1);
		}
	}

	gettimeofday(&tp_start, NULL);

	while (avp->shutdown == False) {

		count++;

		if (avp->do_video == True)
			display_event(avp);

		if ((avp->pause == False) && (avp->shutdown == False)) {

			if (avp->do_video == True) {
				if (sync_ticks(avp, missed_ticks) != 0)
					break;

				if (avp->vid->sync > 0) {
					skip = avp->vid->sync;
					if (avp->verbose > 0)
						warnx("skipping %d video "
						    "frames", skip);
					fseek(avp->vid->in_file,
					    avp->vid->buffer_size * skip,
					    SEEK_CUR);
					avp->vid->sync -= skip;
				}
				if (avp->vid->sync == 0) {
					if (get_video_buf(avp) != 0) {
						warnx("video read error");
						break;
					}
				} else {
					warnx("duplicating %d video frames",
					    avp->vid->sync * -1);
					avp->vid->sync++;
				}
			}

			if (avp->do_audio == True) {
				if (get_audio_buf(avp) != 0) {
					warnx("audio read errror");
					break;
				}
			}

			playing = 0;
			if (missed_ticks == 0)
				sigsuspend(&sa_mask);
			else
				missed_ticks--;
			playing = 1;

			if (avp->do_video == True) {
				if (play_video_buf(avp) != 0) {
					warnx("video play error");
					break;
				} else
					avp->vid->frame_count++;
			}

			if (avp->do_audio == True) {
				if (play_audio_buf(avp) != 0) {
					warnx("audio play error");
					break;
				} else
					avp->aud->frame_count++;
			}
		}

		/* check if audio buffer is low */
		if ((avp->do_audio == True) &&
		    (avp->aud->frame_count % 50 == 0)) {
			hw_pos = bsdav_get_audhw_buf_pos(avp->aud->fd,
			    BSDAV_AUDMODE_PLAY);
			if (hw_pos < avp->aud->refill_buffs) {
				if (avp->verbose > 0)
					warnx("audio buffer low");
				if (fill_audio_buf(avp, 1) != 0) {
					warnx("couldn't refill audio buffer");
				}
				avp->aud->frame_count += avp->aud->fill_buffs / 4;

				/* skip a video frame */
				// avp->vid->sync++;
			}
		}

		if ((count == sequence) && (avp->verbose > 1)) {
			count = 0;
			gettimeofday(&tp_end, NULL);
			timersub(&tp_end, &tp_start, &tp_run);
			run_time = tp_run.tv_sec + (double)tp_run.tv_usec /
			    1000000;
			fprintf(stderr, "%s: sec: %08.3f,", __progname,
			    run_time);
			if (avp->do_video == True)
				fprintf(stderr, " vf: %06ld, fps: %0.3f,",
				    avp->vid->frame_count,
				    ((double)avp->vid->frame_count) /
				    run_time);
			if (avp->do_audio == True)
				fprintf(stderr,
				    " as: %09ld, srate: %ld",
		    		    avp->aud->frame_count *
				    avp->aud->samples_per_frame,
				    (long)((avp->aud->frame_count *
				    avp->aud->samples_per_frame) / run_time));
			fprintf(stderr, "\r");
			fflush(stderr);
		}
	}

	/*  last character was probably a carriage return */
	if (avp->verbose > 1)
		fprintf(stderr, "\n");

	gettimeofday(&tp_end, NULL);

	tim.it_value.tv_usec = 0;
	tim.it_interval.tv_usec = 0;
	setitimer(ITIMER_REAL, &tim, NULL);

	act.sa_handler = SIG_DFL;
	sigaction(SIGALRM, &act, NULL);

	timersub(&tp_end, &tp_start, &tp_run);
	run_time = tp_run.tv_sec + (double)tp_run.tv_usec / 1000000;

	if (avp->verbose > 0) {
		warnx("total time: %f", run_time);
		if (avp->do_video == 1) {
			warnx("video: frames: %ld", avp->vid->frame_count);
			warnx("video: average fps: %f",
			    ((double)avp->vid->frame_count) / run_time);
		}
		if (avp->do_audio == 1) {
			warnx("audio: samples: %ld", avp->aud->frame_count *
			    avp->aud->samples_per_frame);
			warnx("audio: average rate: %f Hz",
			    (avp->aud->frame_count *
			    avp->aud->samples_per_frame) / run_time);
		}
	}

	return (0);
}


int
display_event(struct avplay_t *avp)
{
char str;

	if (XPending(avp->vid->dpy) > 0) {
		XNextEvent(avp->vid->dpy, &avp->vid->event);
		switch (avp->vid->event.type) {
		case KeyPress:
			if (avp->verbose > 1)
				warnx("got keypress event");
			XLookupString(&avp->vid->event.xkey, &str, 1, NULL, NULL);
			switch (str) {
			case 'a':
				if (avp->vid->aspect_lock == True) {
					avp->vid->aspect_lock = False;
				} else {
					avp->vid->aspect_lock = True;
					resize_window(avp, False);
				}
				break;
			case 'f':
				resize_window(avp, True);
				break;
			case 'p':
				switch_pause(avp);
				break;
			case 'q':
				avp->shutdown = True;
				break;
			case '+':
				avp->vid->sync++;
				break;
			case '-':
				avp->vid->sync--;
				break;
			default:
				break;
			}
			break;
		case ClientMessage:
			if (avp->verbose > 1)
				warnx("got ClientMessage event");
			if (avp->vid->event.xclient.data.l[0] ==
			    avp->vid->wmdelwin) {
				avp->shutdown = True;
				break;
			}
			break;
		case ConfigureNotify:
			if (avp->verbose > 1)
				warnx("got ConigureNotify event");
			resize_window(avp, False);
			break;
		default:
			break;
		}
	}
	XSync(avp->vid->dpy, False);

	return (0);
}


int
switch_pause(struct avplay_t *avp)
{
	if (avp->pause == True)
		avp->pause = False;
	else
		avp->pause = True;

	return (0);
}


int
resize_window(struct avplay_t *avp, Bool fullscreen)
{
XWindowAttributes *winatt;
double aspect;
int new_width;
int new_height;
int new_x;
int new_y;

	aspect = (double)avp->vid->aspect.n / avp->vid->aspect.d;

	if (fullscreen == True) {
		if (avp->vid->full_screen == True) {
			new_width = avp->vid->norm_winatt->width;
			new_height = avp->vid->norm_winatt->height;
			new_x = avp->vid->norm_winatt->x;
			new_y = avp->vid->norm_winatt->y;
			avp->vid->full_screen = False;
		} else {
			XGetWindowAttributes(avp->vid->dpy, avp->vid->window,
			    avp->vid->norm_winatt);
			new_width = DisplayWidth(avp->vid->dpy,
			    avp->vid->screen_id);
			new_height = DisplayHeight(avp->vid->dpy,
			    avp->vid->screen_id);
			new_x = 0;
			new_y = 0;
			if (avp->vid->aspect_lock == True) {
				if (new_width <= new_height * aspect) {
					new_width -= new_width % 16;
					new_height = new_width / aspect;
				} else {
					new_height -= new_height % 16;
					new_width = new_height * aspect;
				}
			}
			avp->vid->full_screen = True;
		}
		XMoveResizeWindow(avp->vid->dpy, avp->vid->window, new_x,
		    new_y, new_width, new_height);
	} else {
		if ((winatt = malloc(sizeof(XWindowAttributes))) == NULL) {
			warn("winatt");
			return (1);
		}
		XGetWindowAttributes(avp->vid->dpy, avp->vid->window, winatt);
		new_x = winatt->x;
		new_y = winatt->y;
		new_width = winatt->width;
		new_height = winatt->height;
		free(winatt);
		if (avp->vid->aspect_lock == True) {
			new_width = (new_width + (new_height * aspect)) / 2;
			new_width -= new_width % 16;
			new_height = new_width / aspect;
		}
		XResizeWindow(avp->vid->dpy, avp->vid->window, new_width,
		    new_height);
	}
	XSync(avp->vid->dpy, False);
	avp->vid->display_width = new_width;
	avp->vid->display_height = new_height;
	XNextEvent(avp->vid->dpy, &avp->vid->event);
	XSync(avp->vid->dpy, True);

	return (0);
}


void
cleanup(struct avplay_t *avp)
{
	if (avp->vid->shminfo.shmaddr != NULL)
		shmdt(avp->vid->shminfo.shmaddr);
	avp->vid->shminfo.shmaddr = NULL;

	if (avp->vid->shminfo.shmid > 0)
		shmctl(avp->vid->shminfo.shmid, IPC_RMID, 0);
	avp->vid->shminfo.shmid = 0;

	if (avp->vid->xv_image != NULL)
		XFree(avp->vid->xv_image);
	avp->vid->xv_image = NULL;

	if (avp->vid->gc != NULL)
		XFreeGC(avp->vid->dpy, avp->vid->gc);
	avp->vid->gc = NULL;

	if (avp->vid->window != 0)
		XDestroyWindow(avp->vid->dpy, avp->vid->window);
	avp->vid->window = 0;

	if (avp->vid->port != 0)
		XvUngrabPort(avp->vid->dpy, avp->vid->port, CurrentTime);
	avp->vid->port = 0;

	if (avp->vid->dpy != NULL)
		XCloseDisplay(avp->vid->dpy);
	avp->vid->dpy = NULL;

	if (avp->vid->in_file != NULL)
		fclose(avp->vid->in_file);
	avp->vid->in_file = NULL;

	if (avp->aud->in_file != NULL)
		fclose(avp->aud->in_file);
	avp->aud->in_file = NULL;

}


int
main(int argc, char *argv[])
{
struct avplay_t *avp;
const char *errstr;
int	 ch;
int	 sret;
int	 err;
int	 lflag;


	if ((avp = malloc(sizeof(struct avplay_t))) == NULL) {
		warn("avp");
		cleanup(avp);
		return (1);
	}
	memset(avp, 0, sizeof(struct avplay_t));

	if ((avp->vid = malloc(sizeof(struct avp_video))) == NULL) {
		warn("avp->vid");
		cleanup(avp);
		return (1);
	}
	memset(avp->vid, 0, sizeof(struct avp_video));

	if ((avp->aud = malloc(sizeof(struct avp_audio))) == NULL) {
		warn("avp->aud");
		cleanup(avp);
		return (1);
	}
	memset(avp->aud, 0, sizeof(struct avp_audio));

	/* defaults */
	snprintf(avp->aud->dev, FILENAME_MAX, DEFAULT_AUDIO_DEVICE);
	avp->vid->norm = BSDAV_VIDNORM_NTSC;
	avp->vid->format = BSDAV_VIDFMT_UYVY;
	avp->vid->aspect.n = 4;
	avp->vid->aspect.d = 3;
	avp->vid->aspect_lock = True;
	avp->vid->in_file = NULL;
	avp->aud->channels = 2;
	avp->aud->sample_rate = 48000;
	avp->aud->format = BSDAV_AUDFMT_SLLE;
	avp->aud->in_file = NULL;

	avp->aud->fd = -1;

	avp->shutdown = False;
	avp->do_audio = False;
	avp->do_video = False;
	avp->verbose = 0;

	lflag = 0;
	err = 0;

	while ((ch = getopt(argc, argv, "lva:C:e:E:f:h:i:I:n:r:R:w:")) != -1) {
		switch (ch) {
		case 'a':
			if (bsdav_parse_ratio(optarg, &avp->vid->aspect) != 0) {
				warnx("invalid vaspect: %s", optarg);
				err++;
			} else
				if (avp->vid->aspect.n == 0)
					avp->vid->aspect_lock = False;
			break;
		case 'C':
			avp->aud->channels = (int)strtonum(optarg, 1, 2,
			    &errstr);
			if (errstr != NULL) {
				warnx("achans is %s: %s:", errstr, optarg);
				err++;
			}
			break;
		case 'e':
			avp->vid->format = bsdav_find_vid_fmt(optarg);
			if (avp->vid->format < 0) {
				warnx("invalid video encoding: %s", optarg);
				err++;
			}
			break;
		case 'E':
			avp->aud->format = bsdav_find_aud_fmt(optarg);
			if (avp->aud->format < 0) {
				warnx("invalid audio encoding: %s", optarg);
				err++;
			}
			break;
		case 'f':
			sret = snprintf(avp->aud->dev, FILENAME_MAX, optarg);
			if (sret >= FILENAME_MAX) {
				warnx("file name is too long");
				err++;
			}
			break;
		case 'h':
			avp->vid->height = (int)strtonum(optarg, 1, UINT_MAX,
			    &errstr);
			if (errstr != NULL) {
				warnx("vheight is %s: %s", errstr, optarg);
				err++;
			}
			break;
		case 'i':
			sret = snprintf(avp->vid->in_path, FILENAME_MAX,
			    optarg);
			if (sret >= FILENAME_MAX) {
				warnx("vfile name is too long");
				err++;
			} else
				avp->do_video = True;
			break;
		case 'I':
			sret = snprintf(avp->aud->in_path, FILENAME_MAX,
			    optarg);
			if (sret >= FILENAME_MAX) {
				warnx("afile name is too long");
				err++;
			} else
				avp->do_audio = True;
			break;
		case 'l':
			lflag++;
			break;
		case 'n':
			avp->vid->norm = bsdav_find_vid_norm(optarg);
			if (avp->vid->norm < 0) {
				warnx("invalid norm: %s", optarg);
				err++;
			}
			break;
		case 'r':
			if (bsdav_parse_ratio(optarg, &avp->vid->fps) != 0) {
				warnx("invalid vrate: %s", optarg);
				err++;
			}
			break;
		case 'R':
			avp->aud->sample_rate = (int)strtonum(optarg, 0,
			    96000, &errstr);
			if (errstr != NULL) {
				warnx("arate is %s: %s", errstr, optarg);
				err++;
			}
			break;
		case 'v':
			avp->verbose++;
			break;
		case 'w':
			avp->vid->width = (int)strtonum(optarg, 1, UINT_MAX,
			    &errstr);
			if (errstr != NULL) {
				warnx("vwidth is %s: %s", errstr, optarg);
				err++;
			}
			break;
		default:
			err++;
			break;
		}
		if (err > 0)
			break;
	}
	argc -= optind;
	argv += optind;

	if (err > 0) {
		usage();
		cleanup(avp);
		return (1);
	}

	if (lflag > 0) {
		bsdav_list_audio_formats(avp->aud->dev, avp->aud->fd);
		list_xv_enc(avp);
		cleanup(avp);
		return (0);
	} else if ((avp->do_video == False) && (avp->do_audio == False)) {
		warnx("input not specified, exiting");
		cleanup(avp);
		return (1);
	}

	if (open_files(avp) != 0) {
		warnx("failed opening files");
		cleanup(avp);
		return (1);
	}

	if (avp->do_video == True) {
		if (set_video_format(avp)) {
			warnx("failed to set video format");
			cleanup(avp);
			return (1);
		}
	}

	if (avp->do_audio == True) {
		if (bsdav_audio_init(avp->aud->fd, BSDAV_AUDMODE_PLAY,
		    avp->aud->format, avp->aud->channels,
		    avp->aud->sample_rate) != 0) {
			warnx("failed to set audio format");
			cleanup(avp);
			return (1);
		}
	}

	if (get_usecs_per_frame(avp)) {
		warnx("failed to set timing info");
		cleanup(avp);
		return (1);
	}

	if (init_buffers(avp)) {
		warnx("failed to init buffers");
		cleanup(avp);
		return (1);
	}

	if (avp->do_video == True) {
		if (create_win(avp)) {
			warnx("failed to create window");
			cleanup(avp);
			return (1);
		}
	}

	if (stream(avp) != 0) {
		warnx("failed to stream");
		cleanup(avp);
		return (1);
	}

	cleanup(avp);

	return (0);
}
