/*
  bdf2kanjirom.c  2021.4.4 version
    convert .bdf font file to KANJIROM format for PC-6001 series
    by AKIKAWA, Hisashi
    This software is redistributable under the LGPLv2.1 or any later version.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define NCHR 1024
#define SIZEX_DEFAULT 16
#define SIZEY_DEFAULT 16

int main(int argc, char *argv[])
{
  int i, j;
  int opt;
  int use_ascent = 0;
  int count = 0;
  int code = 0, chr;
  int x, y;
  int ascent = -1, descent = -1, pixel_x, pixel_y;
  int xmin = 999, xmax = -1;
  int ymax;
  int size_x_out = SIZEX_DEFAULT, size_y_out = SIZEY_DEFAULT;
  int size_x, size_y;
  int offset_x, offset_y;
  double ratio_x, ratio_y;
  char line[100];
  char charset[100] = "";
  char outfile[] = "KANJIROM.66";
  unsigned char bmpbdf[64][64];
  unsigned char bmp60_left[NCHR][16] = {0};
  unsigned char bmp60_right[NCHR][16] = {0};
  unsigned long long data;
  FILE *fp;

  int *tbl;
  int jistbl[] = {
    0x3026, 0x302D, 0x3035, 0x3042, 0x3045, 0x3046, 0x304A, 0x304C,
    0x304F, 0x3051, 0x3055, 0x3057, 0x305B, 0x305C, 0x305F, 0x3061,
    0x3064, 0x3065, 0x3068, 0x3069, 0x306C, 0x306D, 0x3071, 0x3075,
    0x3077, 0x3078, 0x307A, 0x307B, 0x3121, 0x3126, 0x3127, 0x3129,
    0x312B, 0x313F, 0x3140, 0x3144, 0x3147, 0x3149, 0x314A, 0x314B,
    0x3151, 0x3152, 0x3155, 0x3157, 0x3158, 0x315F, 0x3160, 0x3164,
    0x3168, 0x3169, 0x3173, 0x3176, 0x317B, 0x317D, 0x317E, 0x3223,
    0x3226, 0x322B, 0x322C, 0x322D, 0x322F, 0x3230, 0x3238, 0x3239,
    0x323B, 0x323C, 0x323D, 0x323E, 0x323F, 0x3241, 0x3243, 0x3244,
    0x3246, 0x3248, 0x324A, 0x324C, 0x324E, 0x324F, 0x3250, 0x3256,
    0x3259, 0x325D, 0x325F, 0x3261, 0x3266, 0x3268, 0x326A, 0x326C,
    0x3271, 0x3272, 0x3273, 0x3277, 0x327E, 0x3323, 0x3324, 0x3325,
    0x3326, 0x3328, 0x332B, 0x332C, 0x332D, 0x3330, 0x3332, 0x3339,
    0x3346, 0x3348, 0x334A, 0x334E, 0x3350, 0x3351, 0x3355, 0x3357,
    0x3358, 0x335A, 0x335B, 0x3363, 0x3364, 0x3368, 0x3374, 0x3428,
    0x3429, 0x342B, 0x342C, 0x3430, 0x3431, 0x3433, 0x3434, 0x3436,
    0x3437, 0x343F, 0x3441, 0x3447, 0x3449, 0x344A, 0x3451, 0x3456,
    0x3458, 0x345B, 0x345D, 0x345F, 0x3463, 0x3464, 0x3469, 0x346A,
    0x346D, 0x346E, 0x346F, 0x3470, 0x3473, 0x3474, 0x3475, 0x3478,
    0x3479, 0x347A, 0x347C, 0x3521, 0x3522, 0x3524, 0x3525, 0x3528,
    0x352A, 0x352C, 0x352D, 0x352E, 0x352F, 0x353B, 0x353F, 0x3541,
    0x3544, 0x3552, 0x3555, 0x3557, 0x3559, 0x355B, 0x355C, 0x355D,
    0x355E, 0x355F, 0x3561, 0x3563, 0x3565, 0x3566, 0x3569, 0x356B,
    0x356C, 0x356D, 0x356E, 0x356F, 0x3573, 0x3576, 0x3579, 0x357B,
    0x357E, 0x3621, 0x3625, 0x3626, 0x3628, 0x362D, 0x362F, 0x3635,
    0x3636, 0x363B, 0x363D, 0x363F, 0x3640, 0x3648, 0x3649, 0x364A,
    0x364B, 0x364C, 0x3650, 0x3651, 0x3658, 0x365A, 0x3661, 0x3662,
    0x3664, 0x3665, 0x3667, 0x3668, 0x366C, 0x3671, 0x3675, 0x3727,
    0x372F, 0x3731, 0x3732, 0x3733, 0x3734, 0x3738, 0x373B, 0x373C,
    0x373F, 0x3741, 0x3742, 0x3749, 0x374A, 0x374F, 0x3750, 0x3757,
    0x3759, 0x375A, 0x375D, 0x3760, 0x3767, 0x3768, 0x3769, 0x376A,
    0x376B, 0x376C, 0x376E, 0x376F, 0x3772, 0x3773, 0x3774, 0x377A,
    0x377B, 0x3821, 0x3822, 0x3824, 0x3826, 0x3828, 0x3829, 0x382B,
    0x3831, 0x3833, 0x3835, 0x3836, 0x3837, 0x383A, 0x383B, 0x383D,
    0x3840, 0x3842, 0x3844, 0x3845, 0x3846, 0x3847, 0x384A, 0x384B,
    0x384D, 0x384E, 0x3850, 0x385E, 0x3861, 0x3865, 0x386C, 0x386D,
    0x386E, 0x3872, 0x3875, 0x3877, 0x3878, 0x3879, 0x387A, 0x387C,
    0x387D, 0x387E, 0x3921, 0x3925, 0x3927, 0x3929, 0x392C, 0x392D,
    0x392F, 0x393B, 0x393D, 0x3941, 0x3943, 0x3944, 0x3948, 0x394C,
    0x394D, 0x3952, 0x3954, 0x3956, 0x395B, 0x395D, 0x395F, 0x3961,
    0x3962, 0x3966, 0x3967, 0x396F, 0x3970, 0x3971, 0x3972, 0x3975,
    0x397C, 0x3A23, 0x3A24, 0x3A2C, 0x3A2E, 0x3A34, 0x3A34, 0x3A39,
    0x3A3A, 0x3A3D, 0x3A42, 0x3A46, 0x3A47, 0x3A4A, 0x3A4D, 0x3A4E,
    0x3A51, 0x3A52, 0x3A57, 0x3A59, 0x3A5A, 0x3A5B, 0x3A5D, 0x3A5F,
    0x3A60, 0x3A61, 0x3A62, 0x3A64, 0x3A65, 0x3A6A, 0x3A6B, 0x3A6E,
    0x3A72, 0x3A76, 0x3A7D, 0x3A7E, 0x3B21, 0x3B25, 0x3B26, 0x3B28,
    0x3B30, 0x3B32, 0x3B33, 0x3B36, 0x3B3A, 0x3B3B, 0x3B3D, 0x3B3F,
    0x3B40, 0x3B44, 0x3B45, 0x3B48, 0x3B4A, 0x3B4B, 0x3B4D, 0x3B4E,
    0x3B4F, 0x3B50, 0x3B51, 0x3B52, 0x3B54, 0x3B55, 0x3B56, 0x3B57,
    0x3B58, 0x3B59, 0x3B5F, 0x3B60, 0x3B61, 0x3B64, 0x3B65, 0x3B66,
    0x3B6A, 0x3B6B, 0x3B6C, 0x3B6D, 0x3B6E, 0x3B6F, 0x3B71, 0x3B75,
    0x3B76, 0x3B77, 0x3B79, 0x3B7A, 0x3B7B, 0x3B7D, 0x3B7E, 0x3C21,
    0x3C22, 0x3C23, 0x3C27, 0x3C28, 0x3C2A, 0x3C2B, 0x3C2D, 0x3C2F,
    0x3C30, 0x3C31, 0x3C37, 0x3C37, 0x3C3C, 0x3C41, 0x3C42, 0x3C4B,
    0x3C4C, 0x3C4D, 0x3C4E, 0x3C52, 0x3C54, 0x3C55, 0x3C56, 0x3C5A,
    0x3C5C, 0x3C61, 0x3C63, 0x3C65, 0x3C67, 0x3C68, 0x3C69, 0x3C6A,
    0x3C6F, 0x3C72, 0x3C73, 0x3C75, 0x3C78, 0x3C79, 0x3C7B, 0x3C7D,
    0x3C7E, 0x3D21, 0x3D22, 0x3D23, 0x3D24, 0x3D26, 0x3D29, 0x3D2A,
    0x3D2C, 0x3D30, 0x3D35, 0x3D38, 0x3D3B, 0x3D3D, 0x3D3E, 0x3D44,
    0x3D49, 0x3D49, 0x3D4B, 0x3D4C, 0x3D4F, 0x3D50, 0x3D51, 0x3D52,
    0x3D55, 0x3D60, 0x3D63, 0x3D67, 0x3D68, 0x3D69, 0x3D6A, 0x3D6B,
    0x3D70, 0x3D71, 0x3D74, 0x3D75, 0x3D77, 0x3D78, 0x3D7C, 0x3D7D,
    0x3E21, 0x3E26, 0x3E27, 0x3E2D, 0x3E2E, 0x3E2F, 0x3E35, 0x3E37,
    0x3E3C, 0x3E43, 0x3E46, 0x3E48, 0x3E4A, 0x3E4E, 0x3E4F, 0x3E50,
    0x3E5A, 0x3E5D, 0x3E5E, 0x3E63, 0x3E65, 0x3E68, 0x3E6B, 0x3E6C,
    0x3E6F, 0x3E70, 0x3E72, 0x3E75, 0x3E78, 0x3F22, 0x3F25, 0x3F26,
    0x3F27, 0x3F29, 0x3F2E, 0x3F34, 0x3F37, 0x3F39, 0x3F3C, 0x3F3D,
    0x3F3F, 0x3F40, 0x3F43, 0x3F46, 0x3F48, 0x3F4A, 0x3F48, 0x3F4D,
    0x3F4E, 0x3F5E, 0x3F62, 0x3F64, 0x3F65, 0x3F74, 0x4023, 0x4024,
    0x4027, 0x4029, 0x402A, 0x402D, 0x402E, 0x402F, 0x4030, 0x4031,
    0x4032, 0x4035, 0x4036, 0x4038, 0x4039, 0x403A, 0x403B, 0x403C,
    0x403D, 0x403E, 0x403F, 0x4044, 0x4045, 0x4047, 0x404A, 0x4050,
    0x4051, 0x4053, 0x4055, 0x4056, 0x405A, 0x405C, 0x405E, 0x405F,
    0x4061, 0x4062, 0x4063, 0x4064, 0x4065, 0x4067, 0x4068, 0x4069,
    0x406B, 0x406C, 0x406E, 0x406F, 0x4074, 0x4075, 0x4076, 0x4077,
    0x407E, 0x4125, 0x412A, 0x412C, 0x4130, 0x4131, 0x4133, 0x4134,
    0x4144, 0x4147, 0x4148, 0x414F, 0x4152, 0x4155, 0x4158, 0x415B,
    0x4160, 0x4161, 0x4168, 0x416A, 0x416B, 0x416D, 0x4170, 0x4176,
    0x4177, 0x417C, 0x417D, 0x4221, 0x4222, 0x4224, 0x4226, 0x4227,
    0x4229, 0x422C, 0x422D, 0x422E, 0x422F, 0x4230, 0x4232, 0x4233,
    0x4234, 0x4238, 0x4239, 0x423A, 0x423B, 0x423C, 0x423E, 0x423F,
    0x4240, 0x4247, 0x424E, 0x4250, 0x4250, 0x4254, 0x4256, 0x425F,
    0x4260, 0x4262, 0x4265, 0x4266, 0x4267, 0x4268, 0x426A, 0x4270,
    0x4323, 0x432B, 0x4331, 0x4334, 0x4335, 0x433A, 0x433B, 0x4344,
    0x4347, 0x4348, 0x434A, 0x434B, 0x434C, 0x434D, 0x434E, 0x434F,
    0x4353, 0x4356, 0x435B, 0x435D, 0x4363, 0x4365, 0x4366, 0x4367,
    0x4368, 0x4369, 0x436B, 0x436C, 0x436D, 0x436E, 0x4378, 0x4379,
    0x437A, 0x437B, 0x4422, 0x4423, 0x4425, 0x442B, 0x442C, 0x442E,
    0x4432, 0x4434, 0x4439, 0x443A, 0x443B, 0x443E, 0x4442, 0x4445,
    0x4449, 0x444B, 0x444C, 0x4463, 0x4464, 0x4469, 0x446A, 0x446C,
    0x446D, 0x446F, 0x4478, 0x4528, 0x452A, 0x452C, 0x4534, 0x4535,
    0x4537, 0x4538, 0x4539, 0x453E, 0x4540, 0x4541, 0x4544, 0x4545,
    0x454C, 0x4550, 0x4554, 0x4558, 0x4559, 0x455A, 0x455E, 0x455F,
    0x4561, 0x4567, 0x456A, 0x456C, 0x4572, 0x4575, 0x4576, 0x4579,
    0x457A, 0x457C, 0x457D, 0x4624, 0x462C, 0x462F, 0x4630, 0x4631,
    0x4632, 0x4633, 0x4638, 0x463B, 0x463C, 0x4640, 0x4641, 0x4643,
    0x4647, 0x4648, 0x4649, 0x464A, 0x464F, 0x4660, 0x4661, 0x4662,
    0x466C, 0x466E, 0x4671, 0x4673, 0x4675, 0x4679, 0x467C, 0x467D,
    0x467E, 0x4724, 0x4727, 0x472E, 0x472F, 0x4730, 0x4733, 0x473C,
    0x473D, 0x473E, 0x4740, 0x4746, 0x4748, 0x4749, 0x474B, 0x474F,
    0x4750, 0x4752, 0x4754, 0x4758, 0x4759, 0x475B, 0x475C, 0x4763,
    0x4764, 0x476E, 0x4772, 0x477E, 0x482A, 0x482C, 0x482F, 0x483D,
    0x483E, 0x483F, 0x4844, 0x4847, 0x4848, 0x4849, 0x4853, 0x4855,
    0x4856, 0x485D, 0x4861, 0x4863, 0x4866, 0x4869, 0x486B, 0x486E,
    0x4871, 0x4873, 0x4874, 0x4877, 0x487E, 0x4921, 0x492C, 0x492E,
    0x4932, 0x4934, 0x4936, 0x4938, 0x4939, 0x493C, 0x493D, 0x493E,
    0x4942, 0x4943, 0x494A, 0x494F, 0x4954, 0x4955, 0x4957, 0x4958,
    0x4959, 0x495B, 0x495C, 0x4963, 0x4969, 0x496C, 0x4970, 0x4974,
    0x4977, 0x497B, 0x497C, 0x497E, 0x4A21, 0x4A22, 0x4A23, 0x4A29,
    0x4A2A, 0x4A2C, 0x4A33, 0x4A34, 0x4A38, 0x4A39, 0x4A3C, 0x4A3F,
    0x4A44, 0x4A45, 0x4A46, 0x4A4C, 0x4A51, 0x4A52, 0x4A54, 0x4A55,
    0x4A56, 0x4A58, 0x4A59, 0x4A5B, 0x4A5D, 0x4A62, 0x4A64, 0x4A68,
    0x4A6C, 0x4A71, 0x4A73, 0x4A75, 0x4A7C, 0x4A7D, 0x4B21, 0x4B2C,
    0x4B2D, 0x4B34, 0x4B3A, 0x4B3D, 0x4B3E, 0x4B40, 0x4B47, 0x4B49,
    0x4B4C, 0x4B52, 0x4B5A, 0x4B5C, 0x4B65, 0x4B67, 0x4B68, 0x4B6B,
    0x4B76, 0x4B7C, 0x4B7E, 0x4C23, 0x4C24, 0x4C29, 0x4C2E, 0x4C31,
    0x4C33, 0x4C35, 0x4C3E, 0x4C3F, 0x4C40, 0x4C41, 0x4C42, 0x4C44,
    0x4C4A, 0x4C4C, 0x4C4F, 0x4C53, 0x4C5A, 0x4C5C, 0x4C64, 0x4C67,
    0x4C6B, 0x4C6E, 0x4C70, 0x4C72, 0x4C73, 0x4C74, 0x4C75, 0x4C7D,
    0x4D22, 0x4D25, 0x4D26, 0x4D27, 0x4D2D, 0x4D33, 0x4D37, 0x4D39,
    0x4D3C, 0x4D3D, 0x4D3E, 0x4D42, 0x4D44, 0x4D46, 0x4D4B, 0x4D4D,
    0x4D4E, 0x4D51, 0x4D53, 0x4D55, 0x4D57, 0x4D5B, 0x4D5C, 0x4D5F,
    0x4D61, 0x4D62, 0x4D68, 0x4D6E, 0x4D70, 0x4D71, 0x4D77, 0x4D78,
    0x4D7C, 0x4D7D, 0x4E22, 0x4E24, 0x4E26, 0x4E27, 0x4E28, 0x4E29,
    0x4E2C, 0x4E2E, 0x4E31, 0x4E39, 0x4E3E, 0x4E41, 0x4E49, 0x4E4C,
    0x4E4E, 0x4E4F, 0x4E50, 0x4E53, 0x4E57, 0x4E58, 0x4E60, 0x4E61,
    0x4E63, 0x4E64, 0x4E69, 0x4E72, 0x4E73, 0x4E7D, 0x4F22, 0x4F29,
    0x4F2B, 0x4F2F, 0x4F37, 0x4F3B, 0x4F3F, 0x4F40, 0x4F42, 0x4F43
  };
  int unitbl[] = {
    0x611b, 0x60aa, 0x5727, 0x5b89, 0x6697, 0x6848, 0x4ee5, 0x4f4d,
    0x56f2, 0x59d4, 0x610f, 0x6613, 0x7570, 0x79fb, 0x80c3, 0x8863,
    0x907a, 0x533b, 0x57df, 0x80b2, 0x4e00, 0x58f1, 0x8328, 0x5370,
    0x54e1, 0x56e0, 0x5f15, 0x98f2, 0x9662, 0x53f3, 0x5b87, 0x7fbd,
    0x96e8, 0x904b, 0x96f2, 0x55b6, 0x6620, 0x6804, 0x6c38, 0x6cf3,
    0x82f1, 0x885b, 0x6db2, 0x76ca, 0x99c5, 0x5186, 0x5712, 0x5ef6,
    0x6cbf, 0x6f14, 0x9060, 0x5869, 0x592e, 0x5f80, 0x5fdc, 0x6a2a,
    0x738b, 0x9ec4, 0x5ca1, 0x6c96, 0x5104, 0x5c4b, 0x6069, 0x6e29,
    0x97f3, 0x4e0b, 0x5316, 0x4eee, 0x4f55, 0x4fa1, 0x52a0, 0x53ef,
    0x590f, 0x5bb6, 0x79d1, 0x679c, 0x6b4c, 0x6cb3, 0x706b, 0x82b1,
    0x8377, 0x8ab2, 0x8cb8, 0x904e, 0x6211, 0x753b, 0x82bd, 0x8cc0,
    0x4f1a, 0x89e3, 0x56de, 0x5feb, 0x6539, 0x68b0, 0x6d77, 0x7070,
    0x754c, 0x7d75, 0x958b, 0x968e, 0x8c9d, 0x5916, 0x5bb3, 0x8857,
    0x5404, 0x62e1, 0x683c, 0x78ba, 0x899a, 0x89d2, 0x95a3, 0x9769,
    0x5b66, 0x697d, 0x984d, 0x6f5f, 0x5272, 0x6d3b, 0x682a, 0x5bd2,
    0x520a, 0x52e7, 0x5dfb, 0x5b8c, 0x5b98, 0x5e72, 0x5e79, 0x611f,
    0x6163, 0x6b53, 0x6f22, 0x770b, 0x7ba1, 0x7c21, 0x89b3, 0x9593,
    0x95a2, 0x9928, 0x4e38, 0x5cb8, 0x773c, 0x5ca9, 0x9854, 0x9858,
    0x5371, 0x559c, 0x5668, 0x57fa, 0x5bc4, 0x5c90, 0x5e0c, 0x63ee,
    0x673a, 0x65d7, 0x671f, 0x6a5f, 0x5e30, 0x6c17, 0x6c7d, 0x5b63,
    0x7d00, 0x898f, 0x8a18, 0x8cb4, 0x8d77, 0x6280, 0x7591, 0x7fa9,
    0x8b70, 0x5ba2, 0x9006, 0x4e45, 0x4f11, 0x5438, 0x5bae, 0x5f13,
    0x6025, 0x6551, 0x6c42, 0x6ce3, 0x7403, 0x7a76, 0x7d1a, 0x7d66,
    0x65e7, 0x725b, 0x53bb, 0x5c45, 0x6319, 0x8a31, 0x6f01, 0x9b5a,
    0x4eac, 0x4f9b, 0x7af6, 0x5171, 0x5354, 0x5883, 0x5f37, 0x6559,
    0x6a4b, 0x80f8, 0x8208, 0x90f7, 0x93e1, 0x696d, 0x5c40, 0x66f2,
    0x6975, 0x7389, 0x52e4, 0x5747, 0x7981, 0x7b4b, 0x8fd1, 0x91d1,
    0x9280, 0x4e5d, 0x53e5, 0x533a, 0x82e6, 0x5177, 0x7a7a, 0x718a,
    0x541b, 0x8a13, 0x7fa4, 0x8ecd, 0x90e1, 0x4fc2, 0x5144, 0x5553,
    0x578b, 0x5f62, 0x5f84, 0x656c, 0x666f, 0x7cfb, 0x7d4c, 0x8a08,
    0x8b66, 0x8efd, 0x82b8, 0x5287, 0x6b20, 0x6c7a, 0x6f54, 0x7a74,
    0x7d50, 0x8840, 0x6708, 0x4ef6, 0x5065, 0x517c, 0x5238, 0x5efa,
    0x61b2, 0x691c, 0x6a29, 0x72ac, 0x7814, 0x7d79, 0x770c, 0x898b,
    0x967a, 0x9a13, 0x5143, 0x539f, 0x53b3, 0x6e1b, 0x6e90, 0x73fe,
    0x8a00, 0x9650, 0x500b, 0x53e4, 0x547c, 0x56fa, 0x5df1, 0x5eab,
    0x6238, 0x6545, 0x6e56, 0x4e94, 0x5348, 0x5f8c, 0x8a9e, 0x8aa4,
    0x8b77, 0x4ea4, 0x5019, 0x5149, 0x516c, 0x529f, 0x52b9, 0x539a,
    0x53e3, 0x5411, 0x540e, 0x597d, 0x5b5d, 0x5de5, 0x5e78, 0x5e83,
    0x5eb7, 0x6821, 0x69cb, 0x6e2f, 0x7532, 0x7687, 0x7d05, 0x8015,
    0x8003, 0x822a, 0x884c, 0x8b1b, 0x9271, 0x92fc, 0x964d, 0x9999,
    0x9ad8, 0x53f7, 0x5408, 0x523b, 0x544a, 0x56fd, 0x7a40, 0x9ed2,
    0x9aa8, 0x4eca, 0x56f0, 0x6839, 0x6df7, 0x4f50, 0x5de6, 0x5dee,
    0x67fb, 0x7802, 0x5ea7, 0x518d, 0x6700, 0x59bb, 0x624d, 0x63a1,
    0x6e08, 0x707d, 0x796d, 0x7d30, 0x83dc, 0x683d, 0x969b, 0x5728,
    0x6750, 0x7f6a, 0x8ca1, 0x5742, 0x962a, 0x5d0e, 0x57fc, 0x4f5c,
    0x6628, 0x7b56, 0x518a, 0x5237, 0x5bdf, 0x672d, 0x6bba, 0x96d1,
    0x4e09, 0x53c2, 0x5c71, 0x6563, 0x7523, 0x7b97, 0x8695, 0x8cdb,
    0x9178, 0x6b8b, 0x4ed5, 0x4f7f, 0x53f8, 0x53f2, 0x56db, 0x58eb,
    0x59cb, 0x59c9, 0x59ff, 0x5b50, 0x5e02, 0x5e2b, 0x5fd7, 0x601d,
    0x6307, 0x652f, 0x6b62, 0x6b7b, 0x6c0f, 0x79c1, 0x7cf8, 0x7d19,
    0x81f3, 0x8996, 0x8a5e, 0x8a69, 0x8a66, 0x8a8c, 0x8cc7, 0x6b6f,
    0x4e8b, 0x4f3c, 0x5150, 0x5b57, 0x5bfa, 0x6301, 0x6642, 0x6b21,
    0x6ecb, 0x6cbb, 0x78c1, 0x793a, 0x8033, 0x81ea, 0x8f9e, 0x9e7f,
    0x5f0f, 0x8b58, 0x4e03, 0x5931, 0x5ba4, 0x8cea, 0x5b9f, 0x820e,
    0x5199, 0x5c04, 0x6368, 0x793e, 0x8005, 0x8b1d, 0x8eca, 0x501f,
    0x5c3a, 0x91c8, 0x82e5, 0x5f31, 0x4e3b, 0x53d6, 0x5b88, 0x624b,
    0x7a2e, 0x9152, 0x9996, 0x53d7, 0x6388, 0x6a39, 0x9700, 0x53ce,
    0x5468, 0x5b97, 0x5c31, 0x5dde, 0x4fee, 0x62fe, 0x79cb, 0x7d42,
    0x7fd2, 0x8846, 0x9031, 0x96c6, 0x4f4f, 0x5341, 0x5f93, 0x7e26,
    0x91cd, 0x5bbf, 0x795d, 0x7e2e, 0x719f, 0x51fa, 0x8853, 0x8ff0,
    0x6625, 0x6e96, 0x7d14, 0x9806, 0x51e6, 0x521d, 0x6240, 0x6691,
    0x7f72, 0x66f8, 0x8af8, 0x52a9, 0x5973, 0x5e8f, 0x9664, 0x50b7,
    0x52dd, 0x5546, 0x5531, 0x5c06, 0x5c0f, 0x5c11, 0x627f, 0x62db,
    0x662d, 0x6d88, 0x713c, 0x7167, 0x7701, 0x79f0, 0x7ae0, 0x7b11,
    0x8a3c, 0x8c61, 0x8cde, 0x969c, 0x4e0a, 0x4e57, 0x57ce, 0x5834,
    0x5e38, 0x60c5, 0x6761, 0x72b6, 0x84b8, 0x690d, 0x7e54, 0x8077,
    0x8272, 0x98df, 0x4fe1, 0x5fc3, 0x65b0, 0x68ee, 0x6df1, 0x7533,
    0x771f, 0x795e, 0x81e3, 0x89aa, 0x8eab, 0x9032, 0x91dd, 0x4eba,
    0x4ec1, 0x56f3, 0x5782, 0x63a8, 0x6c34, 0x6570, 0x5bf8, 0x4e16,
    0x662f, 0x5236, 0x52e2, 0x6027, 0x6210, 0x653f, 0x6574, 0x661f,
    0x6674, 0x6b63, 0x6e05, 0x751f, 0x76db, 0x7cbe, 0x8056, 0x58f0,
    0x88fd, 0x897f, 0x8aa0, 0x9752, 0x9759, 0x7a0e, 0x5e2d, 0x77f3,
    0x7a4d, 0x7e3e, 0x8cac, 0x8d64, 0x5207, 0x63a5, 0x6298, 0x8a2d,
    0x7bc0, 0x8aac, 0x96ea, 0x7d76, 0x820c, 0x4ed9, 0x5148, 0x5343,
    0x5ba3, 0x5c02, 0x5ddd, 0x6226, 0x6cc9, 0x6d45, 0x6d17, 0x67d3,
    0x7dda, 0x8239, 0x9078, 0x92ad, 0x524d, 0x5584, 0x7136, 0x5168,
    0x7956, 0x7d20, 0x7d44, 0x5275, 0x5009, 0x594f, 0x5c64, 0x60f3,
    0x64cd, 0x65e9, 0x4e89, 0x76f8, 0x7a93, 0x7dcf, 0x8349, 0x8d70,
    0x9001, 0x50cf, 0x5897, 0x81d3, 0x8535, 0x9020, 0x5074, 0x5247,
    0x606f, 0x6e2c, 0x8db3, 0x901f, 0x4fd7, 0x5c5e, 0x65cf, 0x7d9a,
    0x5352, 0x5b58, 0x5b6b, 0x5c0a, 0x640d, 0x6751, 0x4ed6, 0x591a,
    0x592a, 0x6253, 0x4f53, 0x5bfe, 0x5e2f, 0x5f85, 0x614b, 0x8cb8,
    0x9000, 0x968a, 0x4ee3, 0x53f0, 0x5927, 0x7b2c, 0x984c, 0x5b85,
    0x9054, 0x8c37, 0x5358, 0x62c5, 0x63a2, 0x70ad, 0x77ed, 0x56e3,
    0x65ad, 0x6696, 0x6bb5, 0x7537, 0x8ac7, 0x5024, 0x77e5, 0x5730,
    0x6c60, 0x7f6e, 0x7bc9, 0x7af9, 0x8336, 0x7740, 0x4e2d, 0x4ef2,
    0x5b99, 0x5fe0, 0x663c, 0x67f1, 0x6ce8, 0x866b, 0x8457, 0x8caf,
    0x4e01, 0x5146, 0x5e33, 0x5e81, 0x5f35, 0x671d, 0x6f6e, 0x753a,
    0x8178, 0x8abf, 0x9577, 0x9802, 0x9ce5, 0x76f4, 0x8cc3, 0x6d25,
    0x8ffd, 0x75db, 0x901a, 0x4f4e, 0x505c, 0x63d0, 0x5b9a, 0x5e95,
    0x5ead, 0x5f1f, 0x7a0b, 0x6575, 0x7684, 0x9069, 0x9244, 0x5178,
    0x5929, 0x5c55, 0x5e97, 0x8ee2, 0x70b9, 0x4f1d, 0x7530, 0x96fb,
    0x5f92, 0x767b, 0x90fd, 0x52aa, 0x5ea6, 0x571f, 0x515a, 0x51ac,
    0x5200, 0x5cf6, 0x6295, 0x6771, 0x6e6f, 0x71c8, 0x5f53, 0x7b49,
    0x7b54, 0x7cd6, 0x7d71, 0x8a0e, 0x982d, 0x50cd, 0x52d5, 0x540c,
    0x5802, 0x5c0e, 0x7ae5, 0x9053, 0x9285, 0x5f97, 0x5fb3, 0x7279,
    0x6bd2, 0x72ec, 0x8aad, 0x6803, 0x5c4a, 0x5948, 0x90a3, 0x5185,
    0x7e04, 0x5357, 0x96e3, 0x4e8c, 0x5f10, 0x8089, 0x65e5, 0x4e73,
    0x5165, 0x4efb, 0x8a8d, 0x71b1, 0x5e74, 0x5ff5, 0x71c3, 0x7d0d,
    0x80fd, 0x8133, 0x8fb2, 0x8987, 0x6ce2, 0x6d3e, 0x7834, 0x99ac,
    0x4ff3, 0x62dd, 0x6557, 0x80cc, 0x80ba, 0x914d, 0x500d, 0x8cb7,
    0x58f2, 0x535a, 0x767d, 0x9ea6, 0x7551, 0x516b, 0x767a, 0x5224,
    0x534a, 0x53cd, 0x677f, 0x7248, 0x72af, 0x73ed, 0x98ef, 0x6669,
    0x756a, 0x5426, 0x60b2, 0x6279, 0x6bd4, 0x76ae, 0x79d8, 0x80a5,
    0x8cbb, 0x975e, 0x98db, 0x5099, 0x7f8e, 0x9f3b, 0x5fc5, 0x7b46,
    0x5a9b, 0x767e, 0x4ff5, 0x6a19, 0x6c37, 0x7968, 0x8868, 0x8a55,
    0x75c5, 0x79d2, 0x54c1, 0x8ca7, 0x4e0d, 0x4ed8, 0x592b, 0x5a66,
    0x5bcc, 0x5e03, 0x5e9c, 0x7236, 0x8ca0, 0x961c, 0x6b66, 0x90e8,
    0x98a8, 0x526f, 0x5fa9, 0x670d, 0x798f, 0x8179, 0x8907, 0x4ecf,
    0x7269, 0x5206, 0x596e, 0x7c89, 0x6587, 0x805e, 0x5175, 0x5e73,
    0x9589, 0x965b, 0x7c73, 0x5225, 0x5909, 0x7247, 0x7de8, 0x8fba,
    0x8fd4, 0x4fbf, 0x52c9, 0x5f01, 0x4fdd, 0x6b69, 0x88dc, 0x5893,
    0x6bcd, 0x5305, 0x5831, 0x5b9d, 0x653e, 0x65b9, 0x6cd5, 0x8a2a,
    0x8c4a, 0x4ea1, 0x5fd8, 0x66b4, 0x671b, 0x68d2, 0x8cbf, 0x9632,
    0x5317, 0x7267, 0x5e4c, 0x672c, 0x59b9, 0x679a, 0x6bce, 0x5e55,
    0x672b, 0x4e07, 0x6e80, 0x5473, 0x672a, 0x5bc6, 0x8108, 0x6c11,
    0x52d9, 0x7121, 0x540d, 0x547d, 0x660e, 0x76df, 0x8ff7, 0x9cf4,
    0x7dbf, 0x9762, 0x6478, 0x6bdb, 0x6728, 0x76ee, 0x554f, 0x9580,
    0x591c, 0x91ce, 0x77e2, 0x5f79, 0x7d04, 0x85ac, 0x8a33, 0x6cb9,
    0x8f38, 0x512a, 0x52c7, 0x53cb, 0x6709, 0x7531, 0x904a, 0x90f5,
    0x5915, 0x4e88, 0x4f59, 0x9810, 0x5e7c, 0x5bb9, 0x66dc, 0x69d8,
    0x6d0b, 0x7528, 0x7f8a, 0x8449, 0x8981, 0x967d, 0x990a, 0x6b32,
    0x6d74, 0x7fcc, 0x6765, 0x843d, 0x4e71, 0x5375, 0x89a7, 0x5229,
    0x68a8, 0x7406, 0x88cf, 0x91cc, 0x9678, 0x5f8b, 0x7387, 0x7acb,
    0x7565, 0x6d41, 0x7559, 0x65c5, 0x4e21, 0x6599, 0x826f, 0x91cf,
    0x9818, 0x529b, 0x7dd1, 0x6797, 0x81e8, 0x8f2a, 0x985e, 0x4ee4,
    0x4f8b, 0x51b7, 0x793c, 0x6b74, 0x5217, 0x7df4, 0x9023, 0x8def,
    0x52b4, 0x6717, 0x8001, 0x516d, 0x9332, 0x8ad6, 0x548c, 0x8a71
  };

  for (opt = 1; argc > opt && argv[opt][0] == '-'; opt++) {
    if (strcmp(argv[opt], "-ascent") == 0) {
      use_ascent = 1;
    } else {
      printf("usage: bdf2kanjirom [-ascent] fontfile.bdf\n");
      exit(1);
    }
  }

  if (argc < opt + 1) {
    printf("usage: bdf2kanjirom [-ascent] fontfile.bdf\n");
    exit(1);
  }


  if (strcmp(argv[opt], "-") == 0) {
    fp = stdin;
  } else {
    fp = fopen(argv[opt], "rb");
    if (fp == NULL) {
      printf("cannot open %s\n", argv[opt]);
      exit(1);
    }
  }

  while (fgets(line, sizeof(line), fp)) {
    sscanf(line, "ENCODING %d", &code);
    if (sscanf(line, "BBX %d %d %d %d", &size_x, &size_y, &offset_x, &offset_y) == 4) {
      /* except for two- or three- em dash */
      if (tbl != unitbl || (code != 0x2e3a && code != 0x2e3b)) {
	if (offset_x < xmin) {
	  xmin = offset_x;
	}
	if (offset_x + size_x > xmax) {
	  xmax = offset_x + size_x;
	}
      }
    }
    if (sscanf(line, "FONTBOUNDINGBOX %d %d %d %d", &pixel_x, &pixel_y, &offset_x, &offset_y) == 4) {
      ymax = pixel_y + offset_y;
    }
    sscanf(line, "FONT_ASCENT %d", &ascent);
    sscanf(line, "FONT_DESCENT %d", &descent);
    if (sscanf(line, "CHARSET_REGISTRY %s", charset) == 1) {
      if (strncasecmp(charset, "\"JISX0208", 9) == 0 ||
	  strncasecmp(charset, "\"JISX0213", 9) == 0) {
	tbl = jistbl;
      } else if (strcasecmp(charset, "\"ISO10646\"") == 0) {
	tbl = unitbl;
      } else {
	printf("cannot convert for charset %s\n", charset);
	exit(1);
      }
    }
  }
  pixel_x = xmax - xmin;
  if (use_ascent) {
    pixel_y = ascent + descent;
  }
  printf("xmin=%d, xmax=%d\n", xmin, xmax);
  printf("pixel=%d x %d\n", pixel_x, pixel_y);
  if (ascent < 0 || descent < 0 || pixel_x < 0 || strlen(charset) == 0) {
    printf("illegal bdf\n");
    exit(1);
  }
  rewind(fp);

  while (count < NCHR) {
    if (fgets(line, sizeof(line), fp) == NULL) {
      break;
    }
    chr = -1;
    if (sscanf(line, "ENCODING %d", &code) == 1 && code) {
      for (i = 0; i < NCHR; i++) {
	if (code == tbl[i]) {
	  chr = i;
	  break;
	}
      }
    }
    if (chr != -1) {
      count++;
      memset(bmpbdf, 0, sizeof(bmpbdf));
      do {
	fgets(line, sizeof(line), fp);
      } while (sscanf(line, "BBX %d %d %d %d", &size_x, &size_y, &offset_x, &offset_y) != 4);

      if (pixel_x > size_x_out) {
	ratio_x = (double)pixel_x / size_x_out;
      } else {
	ratio_x = 1;
	offset_x += (size_x_out - pixel_x) / 2;
      }
      if (pixel_y > size_y_out) {
	ratio_y = (double)pixel_y / size_y_out;
      } else {
	ratio_y = 1;
	offset_y -= (size_y_out - pixel_y) / 2;
      }

      if (offset_x < 0) {
	offset_x = 0;
      }

      do {
	fgets(line, sizeof(line), fp);
      } while (strncmp(line, "BITMAP", 6) != 0);

      for (i = 0; i < size_y; i++) {
	fgets(line, sizeof(line), fp);
	while (sscanf(line, "%llx", &data) != 1) {
	  ;
	}
	for (j = 0; j < size_x; j++) {
	  x = j + offset_x;
	  if (use_ascent) {
	    y = i + ascent - size_y - offset_y;
	  } else {
	    y = i + ymax - size_y - offset_y;
	  }
	  bmpbdf[x][y] = (data >> ((64 - size_x) % 8 + size_x - j - 1)) & 1;
	}
      }

      for (i = 0; i < size_y_out; i++) {
	for (j = 0; j < size_x_out; j++) {
	  data = 0;
	  for (x = j * ratio_x; x < (int)((j + 1) * ratio_x + .01); x++) {
	    for (y = i * ratio_y; y < (int)((i + 1) * ratio_y + .01); y++) {
	      data += bmpbdf[x][y];
	    }
	  }
	  if ((double)data / (x - (int)(j * ratio_x)) * (y - (int)(i * ratio_y)) > 0) {
	    if (j < size_x_out / 2) {
	      bmp60_left[chr][i] |= 1 << (size_x_out / 2 - j - 1);
	    } else {
	      bmp60_right[chr][i] |= 1 << (size_x_out - j - 1);
	    }
	  }
	}
      }
    }
  }
  fclose(fp);

  fp = fopen(outfile, "wb");
  if (fp == NULL) {
    printf("cannot open %s\n", outfile);
    exit(1);
  }
  for (i = 0; i < NCHR; i++) {
    for (j = 0; j < 16; j++) {
      fputc(bmp60_left[i][j], fp);
    }
  }
  for (i = 0; i < NCHR; i++) {
    for (j = 0; j < 16; j++) {
      fputc(bmp60_right[i][j], fp);
    }
  }
  fclose(fp);

  return 0;
}
