/*
 * nasd_options.c
 *
 * Print NASD options supported by the kernel
 *
 * Author: Jim Zelenka
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <stdio.h>
#include <nasd/nasd_general.h>
#include <nasd/nasd_sys.h>
#include <nasd/nasd_security.h>

char *progname;

int sec_only = 0;
int rpc_only = 0;
int mask_rpc = 0;

void
usage()
{
  fprintf(stderr, "USAGE: %s [options]\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -r RPC options only\n");
  fprintf(stderr, "  -R mask RPC options\n");
  fprintf(stderr, "  -s highest security level only (userlevel)\n");
  fflush(stderr);
  exit(1);
}

int
main(
  int    argc,
  char **argv)
{
  nasd_uint64 rpc_mask, rpc_val;
  nasd_options_t opts;
  int ret;
  char c;

  progname = argv[0];

  while (nasd_getopt(argc, argv, "rRs", &c)) {
    switch(c) {
      case 'r':
        rpc_only = 1;
        break;
      case 'R':
        mask_rpc = 1;
        break;
      case 's':
        sec_only = 1;
        break;
      default:
        fprintf(stderr, "Unknown option '%c'\n", nasd_optopt);
        usage();
    }
  }

  if (nasd_optind < argc)
    usage();

  if(sec_only) {
    printf("%d\n", NASD_MAX_SECURITY_LEVEL);
    fflush(stdout);
    exit(0);
  }

  bzero((char *)&opts, sizeof(opts));

  ret = nasd_srv(NASD_SC_GET_OPTIONS, &opts);
  if (ret) {
    printf("Kernel does not have NASD support loaded.\n");
    fflush(stdout);
    exit(0);
  }

  if (rpc_only == 0) {
    if (opts.opts1 & NASD_OPTS1_SECURITY) {
      printf("Security level %d\n",
             NASD_MIN(NASD_MAX_SECURITY_LEVEL, opts.opts_seclevel));
    } else {
      printf("Security level 0\n");
    }
    if (opts.opts1 & NASD_OPTS1_EDRFS_CLIENT) {
      printf("EDRFS client\n");
    }
    if (opts.opts1 & NASD_OPTS1_EDRFS_FM) {
      printf("EDRFS file manager\n");
    }
    if (opts.opts1 & NASD_OPTS1_DRIVE) {
      printf("Drive\n");
    }
    if (opts.opts1 & NASD_OPTS1_KPDEV) {
      printf("Client kpdev\n");
    }
  }

  rpc_val = opts.opts_rpc;
  if (mask_rpc) {
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
    rpc_mask = NASD_OPTS_RPC_COLOCATE | NASD_OPTS_RPC_DCE;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC
    rpc_mask = NASD_OPTS_RPC_COLOCATE | NASD_OPTS_RPC_SRPC;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC */
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_NONE
    rpc_mask = NASD_OPTS_RPC_COLOCATE;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_NONE */
    rpc_val &= rpc_mask;
  }

  if (rpc_val & NASD_OPTS_RPC_COLOCATE) {
    printf("RPC Colocate\n");
  }
  if (rpc_val & NASD_OPTS_RPC_DCE) {
    printf("RPC DCE\n");
  }
  if (rpc_val & NASD_OPTS_RPC_SRPC) {
    printf("RPC SRPC\n");
  }

  fflush(stdout);
  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
