/*
 * nasd_linux_threads_kernel.h
 *
 * Inside-Linux-kernel variant of threads portability layer
 *
 * Authors: Jim Zelenka, Sean Levy
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#ifndef _NASD__NASD_LINUX_THREADS_KERNEL_H_
#define _NASD__NASD_LINUX_THREADS_KERNEL_H_

#include <linux/config.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/malloc.h>
#include <linux/kernel.h>

#include <nasd/nasd_options.h>

typedef struct nasd_once_s nasd_once_t;
struct nasd_once_s {
  int  done;
};

typedef struct nasd_sys_cond_s nasd_sys_cond_t;
struct nasd_sys_cond_s {
  struct semaphore    m;
  struct semaphore    sem;
  int                 nwaiters;
  int                 broadcast_counter;
};

#define NASD_DECLARE_ONCE(_o_)                     nasd_once_t _o_ = { 0 };

#define NASD_SYS_DECLARE_MUTEX(_m_)               struct semaphore _m_;
#define NASD_SYS_DECLARE_STATIC_MUTEX(_m_) static struct semaphore _m_ = MUTEX;
#define NASD_SYS_DECLARE_EXTERN_MUTEX(_m_) extern struct semaphore _m_;
#define NASD_SYS_DECLARE_COND(_c_)                 nasd_sys_cond_t _c_;
#define NASD_SYS_DECLARE_STATIC_COND(_c_)   static nasd_sys_cond_t _c_;
#define NASD_SYS_DECLARE_EXTERN_COND(_c_)   extern nasd_sys_cond_t _c_;

#define NASD_SYS_LOCK_MUTEX(_m_)                down(&(_m_))
#define NASD_SYS_TRY_LOCK_MUTEX(_m_)            ((down_trylock(&(_m_)) == 0) ? 1 : 0)
#define NASD_SYS_UNLOCK_MUTEX(_m_)              up(&(_m_))

extern int _nasd_once(nasd_once_t *once_block,
  void (*init_routine)(void), char *file, int line);
#define nasd_once(_o_,_i_) _nasd_once(_o_, (void (*)(void))_i_, __FILE__, __LINE__)

extern void _nasd_linux_thread_yield(void);

#define nasd_sys_thread_yield() { \
  _nasd_linux_thread_yield(); \
}

/*
 * Here we build condition variables out of semaphores.
 *
 * Condition algorithm:
 *
 * WAIT:
 *   lock condition
 *   if broadcasting already (broadcast_counter nonzero) unlock+bail
 *   increment nwaiters
 *   unlock condition
 *   unlock mutex
 *   sleep on condition sem
 *   lock mutex
 *   lock condition mutex
 *   decrement nwaiters
 *   if (broadcast_counter) decrement broadcast counter
 *   unlock condition mutex
 *
 * BROADCAST:
 *    lock condition
 *    if broadcasting already (counter nonzero) unlock+bail
 *    set broadcasting counter to nwaiters
 *    raise sem counter times
 *    unlock condition
 *
 * SIGNAL:
 *   lock condition
 *   if (not broadcasting) and (nwaiters nonzero) raise condition sem
 *   unlock condition
 *
 */
#define NASD_SYS_WAIT_COND(_c_,_m_) { \
  int _ret; \
  down(&(_c_).m); \
  if ((_c_).broadcast_counter == 0) { \
    (_c_).nwaiters++; \
    up(&(_c_).m); \
    up(&(_m_)); \
    _ret = down_interruptible(&(_c_).sem); \
    if (_ret) { \
      nasd_sys_thread_yield(); \
    } \
    down(&(_m_)); \
    down(&(_c_).m); \
    (_c_).nwaiters--; \
    if ((_c_).broadcast_counter) { \
      (_c_).broadcast_counter--; \
    } \
    up(&(_c_).m); \
  } \
  else { \
    up(&(_c_).m); \
    nasd_sys_thread_yield(); \
  } \
}
#define NASD_SYS_SIGNAL_COND(_c_) { \
  down(&(_c_).m); \
  if (((_c_).broadcast_counter == 0) && ((_c_).nwaiters)) { \
    up(&(_c_).sem); \
  } \
  up(&(_c_).m); \
}
#define NASD_SYS_BROADCAST_COND(_c_) { \
  register int _i; \
  down(&(_c_).m); \
  if (((_c_).broadcast_counter == 0) && ((_c_).nwaiters != 0)) { \
    (_c_).broadcast_counter = (_c_).nwaiters; \
    for(_i=0;_i<(_c_).nwaiters;_i++) { \
      up(&(_c_).sem); \
    } \
  } \
  up(&(_c_).m); \
}

typedef pid_t nasd_thread_id_t;
#define NASD_THREAD_ID_FMT  "u"
#define NASD_THREAD_ID_NULL ((nasd_thread_id_t)0)

#define nasd_thread_self() ((nasd_thread_id_t)current->pid)

#define NASD_THREAD_KILL_SELF() return /* XXX this seems way bogus */

typedef struct nasd_sys_threadattr_s  nasd_sys_threadattr_t;
struct nasd_sys_threadattr_s {
  int  dummy;
};

typedef int       nasd_sys_thread_t;    /* they don't care so why should i */
typedef void *    nasd_threadarg_t;

void nasd_linux_sys_cond_shutdown(void *cond_arg);
void nasd_linux_set_thread_name(const char *name);
nasd_status_t nasd_sys_mutex_init(struct semaphore *);
nasd_status_t nasd_sys_mutex_destroy(struct semaphore *);
nasd_status_t nasd_sys_cond_init(nasd_sys_cond_t *);
nasd_status_t nasd_sys_cond_destroy(nasd_sys_cond_t *);

#endif /* !_NASD__NASD_LINUX_THREADS_KERNEL_H_ */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
