/*
 * edrfs_marshall_tester.c
 *
 * Test marshalling of NASD-EDRFS types
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_types.h>
#include <nasd/nasd_error.h>
#include <nasd/nasd_edrfs_types.h>
#include <nasd/nasd_drive_types.h>
#include <nasd/nasd_marshall.h>
#include <nasd/nasd_common.h>
#include <nasd/nasd_site.h>
#include <nasd/nasd_general.h>
#include <nasd/nasd_mq.h>
#include <nasd/nasd_marshall_tester.h>

int check_sizes(void);
int check_lookup_args(void);
int check_lookup_res(void);

int check_sizes(void) {
  int fail = 0;

  printf("Struct sizes:\n");
  printf("  sizeof(nasd_edrfs_lookup_args_t) = %d\n", NASD_MT_SizeOf(nasd_edrfs_lookup_args_t));
  printf("  sizeof(nasd_edrfs_lookup_res_t) = %d\n", NASD_MT_SizeOf(nasd_edrfs_lookup_res_t));
  printf("  sizeof(nasd_edrfs_identifier_t) = %d\n", NASD_MT_SizeOf(nasd_edrfs_identifier_t));

  if (fail)
    printf("Size check failed.\n\n");

  return fail;
}

int
main(
  int     argc,
  char  **argv)
{
  int fail = 0;

  if (check_sizes()) fail = 1;
  if (check_lookup_args()) fail = 1;
  if (check_lookup_res()) fail = 1;

  if (fail) {
    printf("Marshalling tests failed.\n");
    return 1;
  } else {
    printf("Marshalling tests succeeded.\n");
    return 0;
  }
}

int check_lookup_args(void) {
  int fail = 0;
  nasd_edrfs_lookup_args_t lookup_args, lookup_args_after;
  nasd_edrfs_lookup_args_otw_t lookup_args_otw;

  lookup_args.in_identifier.nasd_identifier = 1001;
  lookup_args.in_identifier.partnum = 1234;
  lookup_args.in_identifier.disk_identifier = 64;
  lookup_args.in_dirpath[0] = 64;
  lookup_args.in_credential.uid = 12345;
  lookup_args.in_credential.gid = 98765;

  nasd_edrfs_lookup_args_t_marshall(&lookup_args, lookup_args_otw);
  nasd_edrfs_lookup_args_t_unmarshall(lookup_args_otw, &lookup_args_after);

#define CHECKLOOKUPARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(lookup_args,_x_,_fmt_)

  NASD_MT_CHECKCOOKIE(lookup_args,in_cookie)
  NASD_MT_CHECKEDRFSIDENT(lookup_args,in_identifier)
  CHECKLOOKUPARGS(in_credential.uid,"d")
  CHECKLOOKUPARGS(in_credential.gid,"d")
  NASD_MT_CHECKBYTEARR2(lookup_args,lookup_args_after,in_dirpath,NASD_EDRFS_MAX_NAME_LEN)

#undef CHECKLOOKUPARGS

  if (fail) {
    printf("lookup_args checks failed.\n\n");
  }

  return fail;
}

int check_lookup_res(void) {
  int fail = 0;
  nasd_edrfs_lookup_res_t lookup_res, lookup_res_after;
  nasd_edrfs_lookup_res_otw_t lookup_res_otw;

  lookup_res.nasd_status = 123456;
  lookup_res.out_identifier.nasd_identifier = 1001;
  lookup_res.out_identifier.partnum = 1234;
  lookup_res.out_identifier.disk_identifier = 64;
  lookup_res.out_attribute.attribute.block_size = 98765;
  lookup_res.out_attribute.attribute.object_len = nasd_int64cast(9876543210);

  nasd_edrfs_lookup_res_t_marshall(&lookup_res, lookup_res_otw);
  nasd_edrfs_lookup_res_t_unmarshall(lookup_res_otw, &lookup_res_after);

  NASD_MT_CHECKSTATUS(lookup_res,nasd_status)
  NASD_MT_CHECKCOOKIE(lookup_res,out_cookie)
  NASD_MT_CHECKEDRFSIDENT(lookup_res,out_identifier)
  NASD_MT_CHECKEDRFSPOSTATTR(lookup_res,out_attribute)
  NASD_MT_CHECKEDRFSPOSTATTR(lookup_res,post_attribute)

  if (fail) {
    printf("lookup_res checks failed.\n\n");
  }

  return fail;
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
