/*
 * nasd_edrfs_client_srpc.c
 *
 * SRPC module for talking to EDRFS filemanager.
 *
 * Author: Jim Zelenka
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>

#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC

#include <nasd/nasd_threadstuff.h>
#include <nasd/nasd_shutdown.h>
#include <nasd/nasd_types.h>
#include <nasd/nasd_common.h>
#include <nasd/nasd_freelist.h>
#include <nasd/nasd_srpc_types.h>
#include <nasd/nasd_srpc.h>
#include <nasd/nasd_marshall.h>
#include <nasd/nasd_edrfs_rpc.h>
#include <nasd/nasd_edrfs_client.h>
#include <nasd/nasd_edrfs_client_srpc.h>
#include <nasd/nasd_edrfs_server_cstub.h>

/*
 * Pipe stuff
 *
 * XXX if the server ever starts pushing more than 1 ent at a time,
 * this will need to be totally changed
 */
void
nasd_edrfs_client_readdir_alloc(
  void          *state_arg,
  nasd_uint32    bsize,
  nasd_byte_t  **buf,
  nasd_uint32   *bcount)
{
  nasd_edrfs_readdir_pipe_state_t *state;

  state = (nasd_edrfs_readdir_pipe_state_t *)state_arg;

  /* we never alloc more than a single structure. such is life. */

  *buf = (nasd_byte_t *)&(state->partial_buf[state->partial_buf_offset]);
  *bcount =
    NASD_MIN(bsize, (NASD_EDRFS_DIRENT_T_MAX - state->partial_buf_offset));
  state->partial_buf_offset += *bcount;
}

void
nasd_edrfs_client_readdir_push(
  void          *state_arg,
  void          *buf,
  nasd_uint32    count)
{
  nasd_edrfs_readdir_pipe_state_t *state;

  state  = (nasd_edrfs_readdir_pipe_state_t *)state_arg;

  if (count == 0) {
    /* no bytes to grab. time to go home. */
    return;
  }

  /*
   * This code assumes we only get pushed buffers we allocated.
   * Verify that.
   */
{
  unsigned long buf_addr = (unsigned long)buf;
  unsigned long pb_addr = (unsigned long)&state->partial_buf[0];
  unsigned long pb_last = pb_addr + NASD_EDRFS_DIRENT_T_MAX - 1;

  NASD_ASSERT(buf_addr >= pb_addr);
  NASD_ASSERT(buf_addr <= pb_last);
}

  if (state->partial_buf_offset != NASD_EDRFS_DIRENT_T_MAX) {
    /* not a whole entry */
    return;
  }

  state->partial_buf_offset = 0;

  state->dir_ents_pushed++;

  if (state->dir_buf_ents_filled >= state->dir_buf_ents)
    return;

  nasd_edrfs_dirent_t_unmarshall(state->partial_buf,
    &state->dir_buf[state->dir_buf_ents_filled]);

  state->dir_buf_ents_filled++;
}

nasd_status_t
nasd_edrfscli_srpc_init(
  nasd_shutdown_list_t  *sl)
{
  return(NASD_SUCCESS);
}


nasd_status_t
nasd_edrfscli_srpc_bind(
  nasd_edrfs_handle_t   handle,
  char                 *server_name,
  char                 *portnum,
  int                   binding_type,
  void                 *binding_param,
  int                   binding_param_len)
{
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_status_t rc;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  rc = nasd_srpc_bind_to_server(server_name, portnum, &rpch->h);
  if (rc)
    return(rc);

  return(NASD_SUCCESS);
}

nasd_status_t
nasd_edrfscli_srpc_unbind(
  nasd_edrfs_handle_t  handle)
{
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_status_t rc;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  rc = nasd_srpc_unbind_server(&rpch->h);
  if (rc)
    return(rc);

  return(NASD_SUCCESS);
}

void
nasd_edrfscli_srpc_null(
  nasd_edrfs_handle_t   handle,
  nasd_res_t           *res,
  nasd_rpc_status_t    *op_status)
{
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_res_otw_t res_otw;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  src = nasd_null_fm_client(rpch->h, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_mount(
  nasd_edrfs_handle_t       handle,
  nasd_edrfs_mount_args_t  *args,
  nasd_edrfs_mount_res_t   *res,
  nasd_rpc_status_t        *op_status)
{
  nasd_edrfs_mount_args_otw_t args_otw;
  nasd_edrfs_mount_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_mount_args_t_marshall(args, args_otw);
  src = nasd_mount_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_mount_res_t_unmarshall(res_otw, res);
  }
}



void
nasd_edrfscli_srpc_fsstat(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_fsstat_args_t  *args,
  nasd_edrfs_fsstat_res_t   *res,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfs_fsstat_args_otw_t args_otw;
  nasd_edrfs_fsstat_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_fsstat_args_t_marshall(args, args_otw);
  src = nasd_fsstat_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_fsstat_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_fsinfo(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_fsinfo_args_t  *args,
  nasd_edrfs_fsinfo_res_t   *res,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfs_fsinfo_args_otw_t args_otw;
  nasd_edrfs_fsinfo_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_fsinfo_args_t_marshall(args, args_otw);
  src = nasd_fsinfo_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_fsinfo_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_lookup_otw_provided(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_lookup_args_t  *args,
  nasd_edrfs_lookup_res_t   *res,
  nasd_otw_base_t           *args_otw_p,
  nasd_otw_base_t           *res_otw_p,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_lookup_args_t_marshall(args, args_otw_p);

  src = nasd_lookup_fm_client(rpch->h, args_otw_p, res_otw_p);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_lookup_res_t_unmarshall(res_otw_p, res);
  }
}


void
nasd_edrfscli_srpc_lookup(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_lookup_args_t  *args,
  nasd_edrfs_lookup_res_t   *res,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfs_lookup_args_otw_t args_otw;
  nasd_edrfs_lookup_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_lookup_args_t_marshall(args, args_otw);
  src = nasd_lookup_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_lookup_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_readdir(
  nasd_edrfs_handle_t         handle,
  nasd_edrfs_readdir_args_t  *args,
  nasd_edrfs_dirent_t        *out_entries,
  nasd_edrfs_readdir_res_t   *res,
  nasd_rpc_status_t          *op_status)
{
  nasd_edrfs_readdir_args_otw_t args_otw;
  nasd_edrfs_readdir_pipe_state_t state;
  nasd_edrfs_readdir_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_client_pushcb_t pipe;
  nasd_srpc_status_t src;
  nasd_len_t out_len;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  state.dir_buf = out_entries;
  state.dir_buf_ents = args->in_count;
  state.dir_buf_ents_filled = 0;
  state.partial_buf_offset = 0;

  pipe.state = (void *)&state;
  pipe.alloc_cb = (void (*)())nasd_edrfs_client_readdir_alloc;
  pipe.push_cb = (void (*)())nasd_edrfs_client_readdir_push;

  nasd_edrfs_readdir_args_t_marshall(args, args_otw);

  src = nasd_readdir_fm_client(rpch->h, args_otw,
    nasd_srpc_client_run_serverpush_cb, &pipe, &out_len, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_readdir_res_t_unmarshall(res_otw, res);

    if ((state.partial_buf_offset != 0) &&
      (state.partial_buf_offset != NASD_EDRFS_DIRENT_T_MAX))
    {
      /* we have a partial buffer hanging around */
      res->nasd_status = NASD_BAD_TRANSFER_STATE;
    }
  }
}


void
nasd_edrfscli_srpc_access(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_access_args_t  *args,
  nasd_edrfs_access_res_t   *res,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfs_access_args_otw_t args_otw;
  nasd_edrfs_access_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_access_args_t_marshall(args, args_otw);
  src = nasd_access_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_access_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_setattr(
  nasd_edrfs_handle_t         handle,
  nasd_edrfs_setattr_args_t  *args,
  nasd_edrfs_setattr_res_t   *res,
  nasd_rpc_status_t          *op_status)
{
  nasd_edrfs_setattr_args_otw_t args_otw;
  nasd_edrfs_setattr_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_setattr_args_t_marshall(args, args_otw);
  src = nasd_setattr_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_setattr_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_create_otw_provided(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_create_args_t  *args,
  nasd_edrfs_create_res_t   *res,
  nasd_otw_base_t           *args_otw_p,
  nasd_otw_base_t           *res_otw_p,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_create_args_t_marshall(args, args_otw_p);
  src = nasd_create_fm_client(rpch->h, args_otw_p, res_otw_p);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_create_res_t_unmarshall(res_otw_p, res);
  }
}


void
nasd_edrfscli_srpc_create(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_create_args_t  *args,
  nasd_edrfs_create_res_t   *res,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfs_create_args_otw_t args_otw;
  nasd_edrfs_create_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_create_args_t_marshall(args, args_otw);
  src = nasd_create_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_create_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_symlink(
  nasd_edrfs_handle_t         handle,
  nasd_edrfs_symlink_args_t  *args,
  nasd_edrfs_symlink_res_t   *res,
  nasd_rpc_status_t          *op_status)
{
  nasd_edrfs_symlink_args_otw_t args_otw;
  nasd_edrfs_symlink_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_symlink_args_t_marshall(args, args_otw);
  src = nasd_symlink_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_symlink_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_remove(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_remove_args_t  *args,
  nasd_edrfs_remove_res_t   *res,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfs_remove_args_otw_t args_otw;
  nasd_edrfs_remove_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_remove_args_t_marshall(args, args_otw);
  src = nasd_remove_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_remove_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_mkdir(
  nasd_edrfs_handle_t       handle,
  nasd_edrfs_mkdir_args_t  *args,
  nasd_edrfs_mkdir_res_t   *res,
  nasd_rpc_status_t        *op_status)
{
  nasd_edrfs_mkdir_args_otw_t args_otw;
  nasd_edrfs_mkdir_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_mkdir_args_t_marshall(args, args_otw);
  src = nasd_mkdir_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_mkdir_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_rmdir(
  nasd_edrfs_handle_t       handle,
  nasd_edrfs_rmdir_args_t  *args,
  nasd_edrfs_rmdir_res_t   *res,
  nasd_rpc_status_t        *op_status)
{
  nasd_edrfs_rmdir_args_otw_t args_otw;
  nasd_edrfs_rmdir_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_rmdir_args_t_marshall(args, args_otw);
  src = nasd_rmdir_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_rmdir_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_newcookie(
  nasd_edrfs_handle_t           handle,
  nasd_edrfs_newcookie_args_t  *args,
  nasd_edrfs_newcookie_res_t   *res,
  nasd_rpc_status_t            *op_status)
{
  nasd_edrfs_newcookie_args_otw_t args_otw;
  nasd_edrfs_newcookie_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_newcookie_args_t_marshall(args, args_otw);
  src = nasd_newcookie_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_newcookie_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_rename(
  nasd_edrfs_handle_t        handle,
  nasd_edrfs_rename_args_t  *args,
  nasd_edrfs_rename_res_t   *res,
  nasd_rpc_status_t         *op_status)
{
  nasd_edrfs_rename_args_otw_t args_otw;
  nasd_edrfs_rename_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;

  nasd_edrfs_rename_args_t_marshall(args, args_otw);
  src = nasd_rename_fm_client(rpch->h, args_otw, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_rename_res_t_unmarshall(res_otw, res);
  }
}


void
nasd_edrfscli_srpc_getstats(
  nasd_edrfs_handle_t         handle,
  nasd_edrfs_getstats_res_t  *res,
  nasd_rpc_status_t          *op_status)
{
  nasd_edrfs_getstats_res_otw_t res_otw;
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;
  src = nasd_getstats_fm_client(rpch->h, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_edrfs_getstats_res_t_unmarshall(res_otw, res);
  }
}

void
nasd_edrfscli_srpc_resetstats(
  nasd_edrfs_handle_t   handle,
  nasd_res_t           *res,
  nasd_rpc_status_t    *op_status)
{
  nasd_edrfscli_handle_srpc_t *rpch;
  nasd_res_otw_t res_otw;
  nasd_srpc_status_t src;

  rpch = (nasd_edrfscli_handle_srpc_t *)handle->rpc_specific_handle;
  src = nasd_resetstats_fm_client(rpch->h, res_otw);
  *op_status = src;
  if (src) {
    res->nasd_status = NASD_RPC_FAILURE;
  }
  else {
    nasd_res_t_unmarshall(res_otw, res);
  }
}

void
nasd_edrfscli_srpc_error_string(
  nasd_edrfs_handle_t     handle,
  nasd_rpc_status_t       status,
  nasd_error_string_t     str,
  char                   *file,
  int                     line)
{
  nasd_srpc_error_string((nasd_srpc_status_t)status,
    str, sizeof(str), file, line);
}

nasd_edrfscli_rpcmod_tab_t nasd_edrfscli_srpc_mod = {
  NULL,
  NASD_RPCMOD_INIT_FAIL,

  nasd_edrfscli_srpc_bind,
  nasd_edrfscli_srpc_unbind,

  nasd_edrfscli_srpc_null,
  nasd_edrfscli_srpc_mount,
  nasd_edrfscli_srpc_fsstat,
  nasd_edrfscli_srpc_fsinfo,
  nasd_edrfscli_srpc_lookup,
  nasd_edrfscli_srpc_readdir,
  nasd_edrfscli_srpc_access,
  nasd_edrfscli_srpc_setattr,
  nasd_edrfscli_srpc_create,
  nasd_edrfscli_srpc_symlink,
  nasd_edrfscli_srpc_remove,
  nasd_edrfscli_srpc_mkdir,
  nasd_edrfscli_srpc_rmdir,
  nasd_edrfscli_srpc_newcookie,
  nasd_edrfscli_srpc_rename,
  nasd_edrfscli_srpc_getstats,
  nasd_edrfscli_srpc_resetstats,

  nasd_edrfscli_srpc_lookup_otw_provided,
  nasd_edrfscli_srpc_create_otw_provided,

  nasd_edrfscli_srpc_error_string,

  NASD_BIND_SRPC,
  "SRPC",
  sizeof(nasd_edrfscli_handle_srpc_t),

  NULL
};

void
nasd_edrfscli_srpc_register(
  int  counter_lock_held)
{
  nasd_edrfscli_mod_register(&nasd_edrfscli_srpc_mod, counter_lock_held);
}

#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
