/* Copyright 1991 by John Atwood deVries II. */
/* $Id: s_group.c,v 1.29 2001/10/30 05:12:33 jwise Exp $ */
/* For copying and distribution information, see the file COPYING. */

/* primitive to handle group requests */

#include <stdio.h>
#include <string.h>
#include <time.h>
#include "externs.h"
#include "lookup.h"

int is_booting = 0;

void	s_cancel (int, int);
int	s_change (int, int);
void	s_invite (int, int);
void	s_pass (int, int);
void	s_boot (int, int);
void	s_status (int, int);
static void	dump_table (int);
void	s_topic (int, int);
void	s_send_group (int);
void	s_exclude (int, int);
void	s_status_group (int, int, int, char *, char *);
static int	set_name ( int, int, char *);
void	s_talk (int, int);
static int	talk_delete (char *, int);
void	talk_report (int, int);

void
s_cancel(int n, int argc)
{
	int	gi;
	int	is_restricted, is_moderator;
	int	can_do;
	int	i;
	int	mode = -1;
	int	quiet = -1;
	char	*who, *flags;
	char	line[LINE_SIZE];

        if (argc == 2) {
		/* constraints:
			group is restricted & inviter is mod & 
			fields[1] is in invite list
				change in invite list
			otherwise
				complain appropriately
		   side-effects:
			none
		   other:
			fields[1] is nickname of person for cancel
		*/
		/* gi is canceller's groupindex */
		gi = find_group(u_tab[n].group);

		is_restricted = (g_tab[gi].control == RESTRICTED);
		is_moderator =(g_tab[gi].mod == n);
		can_do = (is_restricted && is_moderator) ;
		who = fields[1];
		while (who[0] == '-')  {
			flags = getword(who);
			for (i = 1; i < strlen(flags); i++)
			   switch (flags[i]) {
			   case 'q':
				if (quiet == -1)
					quiet = 1;
				else {
					senderror(n, "Usage: cancel {-q} {-n nickname | -s address}");
					return;
				}
				break;
			   case 's':
				if (mode == -1)
					mode = 1;
				else {
					senderror(n, "Usage: cancel {-q} {-n nickname | -s address}");
					return;
				}
				break;
			   case 'n':
				if (mode == -1)
					mode = 0;
				else {
					senderror(n, "Usage: cancel {-q} {-n nickname | -s address}");
					return;
				}
				break;
			   default:
				senderror(n, "Usage: cancel {-q} {-n nickname | -s address}");
				return;
			   }
			who = get_tail(who);
		}
		if (mode == -1) mode = 0;
		if (mode == 1) ucaseit(who);
		if (quiet == -1) quiet = 0;

		if (can_do) {
			if ((mode == 0) &&
			   ((nldelete(g_tab[gi].n_invites,who) == 0) ||
			   (nldelete(g_tab[gi].nr_invites,who) == 0))) {
				if (quiet == 0) {
				   snprintf(line, LINE_SIZE, "%s cancelled", who);
				   sendstatus(n, "FYI", line);
				   }
				if (find_user(who) < 0) {
					snprintf(line, LINE_SIZE, "%s is not signed on",who);
					sendstatus(n, "Warning", line);
					}
				}
			else if ((mode == 1) &&
			   ((nldelete(g_tab[gi].s_invites,who) == 0) ||
			   (nldelete(g_tab[gi].sr_invites,who) == 0))) {
				if (quiet == 0) {
				   snprintf(line, LINE_SIZE, "%s cancelled", who);
				   sendstatus(n, "FYI", line);
				   }
				}
			else {
				snprintf(line, LINE_SIZE, "%s was not invited", who);
				sendstatus(n, "Warning", line);
				}
			}
		else {
			if ( !is_restricted ) {
				senderror(n,"The group isn't restricted.");
			}
			if ( !is_moderator ) {
				senderror(n,"You aren't the moderator.");
			}
		}
	} else {
		error("cancel: wrong number of arguments");
	}
}

/* change group (if possible) */
/* group 1 is always:
	normal visibility
	public control
   otherwise the group is created:
	normal visibility 
	n is mod
*/

int
s_change(int n, int argc)
{
	int group_exists;
	int is_restricted;
	int is_moderator;
	int visibility;
	int is_invited;
	int target_user;
	int ngi, ogi; /* new and old group indices */
	int len;
	char n_g_n[MAX_GROUPLEN+1]; /* new group name */
	long TheTime;
	char useraddr[LINE_SIZE];
	char line[LINE_SIZE];

        if (argc == 2) {
		/* constraints:
			if group does not exist,
				there must be an empty slot in g_tab.
			if the group does exist, and is public or moderated
				none.
			if the group does exist, and is restricted
				n must have an invitation
				         -or-
				n must be the mod.
			new group can't be the same as the old group.
			new group name has to be at least 1 char.
			new group name is limited in length.

		   notes:
			the old group doesn't exist if this is a login.
			presumes u_tab has an entry set up for n
				(except the group name and login members)

		   side-effects:
			if success, new group gets status "join" message
			if old group existed and still has members,
				old group gets status "leave" message
				otherwise old group gets destroyed if nec.
			return is 0
			for either the old or new groups -- if QUIET,
				then no message.

			if failure, message given and return is -1
		*/

		snprintf(useraddr, LINE_SIZE, "%s@%s", u_tab[n].loginid, u_tab[n].nodeid);
		ucaseit(useraddr);
		len = strlen(fields[1]);

		if (fields[1][0] == '@') {
			if ((target_user = find_user(&fields[1][1])) < 0) {
				senderror(n, "User not found.");
				return(-1);
			} else if (g_tab[find_group(u_tab[target_user].group)].visibility != VISIBLE) {
				senderror(n, "User not found.");
				return(-1);
			} else {
				strcpy (fields[1], u_tab[target_user].group);
				return(s_change(n, argc));
			}
		}
			
		if(len <=0) {
			/* oops, not long enough */
			senderror(n, "Null groupnames are not allowed.");
			return -1;
		}

		if (fields[1][0] == '.') {
		   if (fields[1][1] == '.') {
		      strlcpy(n_g_n, &fields[1][2], MAX_GROUPLEN + 1);
		      visibility = SUPERSECRET;
		      }
		   else {
		      strlcpy(n_g_n, &fields[1][1], MAX_GROUPLEN + 1);
		      visibility = SECRET;
		      }
                   }
		else {
		   strlcpy(n_g_n, fields[1], MAX_GROUPLEN + 1);
		   visibility = VISIBLE;
		   }

		if (strlen(n_g_n) < 1) {
			senderror(n, "Group name too short.");
			return(-1);
			}
		filtergroupname(n_g_n);

		group_exists = ((ngi = find_group(n_g_n)) >= 0);

		if (group_exists)
                   is_invited = ((nlpresent(u_tab[n].nickname, 
			*g_tab[ngi].n_invites) > 0) ||
               		(nlmatch(useraddr, *g_tab[ngi].s_invites)) ||
               		(nlpresent(u_tab[n].nickname, *g_tab[ngi].nr_invites) &&
                  		(strlen(u_tab[n].realname) > 0)) ||
               		(nlmatch(useraddr, *g_tab[ngi].sr_invites) &&
                  		(strlen(u_tab[n].realname) > 0)));
		else
			is_invited = 0;

		if (!group_exists) {
			/* if it doesn't exist, try to create it */
			/* see if there is an empty slot */
			ngi = find_empty_group();
			if (ngi < 0) {
				/* complain and fail */
				senderror(n,"Too many groups!");
				return (-1);
			}
			/* could create it, so fill in the info */
			g_tab[ngi].visibility = visibility;
			strlcpy(g_tab[ngi].name, n_g_n, MAX_GROUPLEN);
			if (strcasecmp("1", n_g_n) != 0) {
				g_tab[ngi].control = MODERATED;
				g_tab[ngi].mod = n;
			} else {
				g_tab[ngi].control = PUBLIC;
			}
			/* initialize the invitation list */
                	nlinit(g_tab[ngi].n_invites, MAX_INVITES);
                	nlinit(g_tab[ngi].nr_invites, MAX_INVITES);
                	nlinit(g_tab[ngi].s_invites, MAX_INVITES);
                	nlinit(g_tab[ngi].sr_invites, MAX_INVITES);
                	nlinit(g_tab[ngi].n_bars, MAX_INVITES);
                	nlinit(g_tab[ngi].n_nr_bars, MAX_INVITES);
                	nlinit(g_tab[ngi].s_bars, MAX_INVITES);
                	nlinit(g_tab[ngi].s_nr_bars, MAX_INVITES);
		}
		/* if we get here, the destination group exists */

		/* is it restricted? */

		is_restricted = (g_tab[ngi].control == RESTRICTED);
		if (is_restricted) {
			/* are we the moderator? */
			is_moderator = (g_tab[ngi].mod == n);
			if (!is_moderator) {
				/* are we invited? */
	                   is_invited = ((nlpresent(u_tab[n].nickname, 
			*g_tab[ngi].n_invites) > 0) ||
               		(nlmatch(useraddr, *g_tab[ngi].s_invites)) ||
               		(nlpresent(u_tab[n].nickname, *g_tab[ngi].nr_invites) &&
                  		(strlen(u_tab[n].realname) > 0)) ||
               		(nlmatch(useraddr, *g_tab[ngi].sr_invites) &&
                  		(strlen(u_tab[n].realname) > 0)));

				if (!is_invited && is_booting == 0) {
					snprintf(line, LINE_SIZE, "%s is restricted.", n_g_n);
					senderror(n, line);
					if ((g_tab[ngi].mod > 0) && (g_tab[ngi].volume == LOUD)) {
					   snprintf(line, LINE_SIZE, "%s tried to enter the group.", u_tab[n].nickname);
					   sendstatus(g_tab[ngi].mod, "Probe", line);
					}
					return(-1);
				}
			}
		}



		/* remember if old group existed, and if so,
                   what was its index
		*/

		if (strlen(u_tab[n].group) == 0) {
			ogi = -1;
		} else {
			ogi = find_group(u_tab[n].group);
			if (ogi < 0) {
				error("Can't find group %s\n", u_tab[n].group);
				}
		}


		/* is the new group different than the old group? */
		if (ngi != ogi) {
			/* the group exists and we are allowed in. */
			strlcpy(u_tab[n].group, g_tab[ngi].name, MAX_GROUPLEN);

			/* tell the new group about the arrival */
			snprintf(line, LINE_SIZE, "%s (%s@%s) entered group", u_tab[n].nickname,
					u_tab[n].loginid, u_tab[n].nodeid);
			if (g_tab[ngi].volume != QUIET) {
			   if (u_tab[n].login == 0) {
				s_status_group(1, 0, n, "Sign-on", line);
			   } else {
				s_status_group(1, 0, n, "Arrive", line);
			   }
			}
			snprintf(line, LINE_SIZE, "You are now in group %s", g_tab[ngi].name);
			if (g_tab[ngi].mod == n)
				strlcat(line, " as moderator", LINE_SIZE);
			sendstatus(n, "Status", line);

			TheTime = time(NULL);
			u_tab[n].t_group = TheTime;

			/* did the old group exist? */
			if (ogi >= 0) {
				/* is there anyone left? */
				if(count_users_in_group(g_tab[ogi].name)>0) {
					/* tell them user left */
					snprintf(line, LINE_SIZE, "%s (%s@%s) just left",
							u_tab[n].nickname, u_tab[n].loginid, u_tab[n].nodeid);
					if (g_tab[ogi].volume != QUIET){
					   s_status_group(2,0,ogi, "Depart", line);
					if (g_tab[ogi].mod == n) {
						snprintf(line, LINE_SIZE, "You are still mod of group %s", g_tab[ogi].name);
						sendimport(n, "Mod", line);
						}
					}
				} else {
					/* zap the old group */
					clear_group_item(ogi);
				}
			return 0;
			}
		} else {
			senderror(n,"You are already in that group!");
			return -1;
		}
        } else {
                error("change: wrong number of arguments");
		return (-1);
        }
	return 0;
}

void
s_invite(int n, int argc)
{
	int gi;
	int is_restricted, is_moderator;
	int dest;
	int mode = -1;
	int r = -1;
	int i;
	int quiet = -1;
	char *who;
	char *flags;
	char line[LINE_SIZE];

        if (argc == 2) {
		/* constraints:
			group is restricted & inviter is mod => 
				messages and change in invite list
			group is not restricted =>
				message only
		   side-effects:
			if invitee is signed on, send status message
		   other:
			fields[1] is nickname of person to invite
		*/

		gi = find_group(u_tab[n].group);

		is_restricted = (g_tab[gi].control == RESTRICTED);
		is_moderator =(g_tab[gi].mod == n);
		who = fields[1];
		while (who[0] == '-')  {
			flags = getword(who);
			for (i = 1; i < strlen(flags); i++)
			   switch (flags[i]) {
			   case 'q':
				if (quiet == -1)
					quiet = 1;
				else {
					senderror(n, "Usage: invite {-q} {-r} {-n nickname | -s address}");
					return;
				}
				break;
			   case 's':
				if (mode == -1)
					mode = 1;
				else {
					senderror(n, "Usage: invite {-q} {-r} {-n nickname | -s address}");
					return;
				}
				break;
			   case 'n':
				if (mode == -1)
					mode = 0;
				else {
					senderror(n, "Usage: invite {-q} {-r} {-n nickname | -s address}");
					return;
				}
				break;
			   case 'r':
				if (r == -1)
					r = 1;
				else {
					senderror(n, "Usage: invite {-q} {-r} {-n nickname | -s address}");
					return;
				}
				break;
			   default:
				senderror(n, "Usage: invite {-q} {-r} {-n nickname | -s address}");
				return;
			   }
			who = get_tail(who);
		}
		if (mode == -1) mode = 0;
		if (r == -1) r = 0;
		if (mode == 1) ucaseit(who);
		if (quiet == -1) quiet = 0;

		if (is_restricted && is_moderator) {
			if ((mode == 0) && (r == 0)) {
				nlput(g_tab[gi].n_invites, who);
				if (quiet == 0) {
				   snprintf(line, LINE_SIZE, "%s invited", who);
				   sendstatus(n,"FYI",line);
				   }
				}
			else if ((mode == 0) && (r == 1)) {
				nlput(g_tab[gi].nr_invites, who);
				if (quiet == 0) {
				   snprintf(line, LINE_SIZE, "%s invited (registered only)", who);
				   sendstatus(n,"FYI",line);
				   }
			        }
			else if ((mode == 1) && (r == 0)) {
				nlput(g_tab[gi].s_invites, who);
				if (quiet == 0) {
				   snprintf(line, LINE_SIZE, "%s invited", who);
				   sendstatus(n,"FYI",line);
				   }
				}
			else if ((mode == 1) && (r == 1)) {
				nlput(g_tab[gi].sr_invites, who);
				if (quiet == 0) {
				   snprintf(line, LINE_SIZE, "%s invited (registered only)", who);
				   sendstatus(n,"FYI",line);
				   }
				}
			}
		if ((mode == 0) && (!is_restricted || is_moderator))
		{
		   if((dest = find_user(who)) >= 0) {
			if ((strlen(u_tab[dest].realname) > 0) || (r == 0)) {
			   snprintf(line, LINE_SIZE, "You are invited to group %s by %s",
				u_tab[n].group, u_tab[n].nickname);
			   sendstatus(dest,"RSVP",line);
			   }
			}
		   else {
			snprintf(line, LINE_SIZE, "%s is not signed on.", who);
			sendstatus(n, "Warning", line);
			}
		}
		if ((mode == 1) && !is_restricted)
			senderror(n, "The group isn't restricted.");
		if (!is_moderator && is_restricted)
			senderror(n, "You aren't the moderator.");
		}
        else {
                error("invite: wrong number of arguments");
        }
}

void
s_pass(int n, int argc)
{
	int is_moderator;
	int gi;
	int dest;
	char * cp;
	char line[LINE_SIZE];

        if (argc == 2) {
		/* constraints:
			user must be mod of group they are in
			if passing, dest. nickname must be on

		   side-effects:
			if passed successfully,
				dest. nickname gets message
			if passed or relinquished successfully,
				user's moderated group gets message
			if the group is QUIET, no messages.
		*/

		/* what is the user's group? */
                gi = find_group(u_tab[n].group);

		/* are they the mod or admin? */
                is_moderator = (check_auth(n) || (g_tab[gi].mod == n));

		if (!is_moderator) {
			senderror(n,"You aren't the moderator.");
			return;
		}

		/* they are the mod */

		/* do they want to pass or relinquish? */
		if( strlen(fields[1]) == 0) {
			/* relinquish mod */
			g_tab[gi].mod = -1;
			g_tab[gi].control = PUBLIC;

			/* tell people */
			snprintf(line, LINE_SIZE, "%s just relinquished the mod.", u_tab[n].nickname);
			if (g_tab[gi].volume != QUIET) {
				s_status_group(1,1,n,"Pass",line);
			}
		} else {
			/* check dest. nickname */
			cp = getword(fields[1]);
			if (strlen(cp) == 0) {
				error("null string in pass");
				return;
			}

			if (strcasecmp(cp, "server") == 0) {
				senderror(n, "Cannot pass to Server.");
				return;
				}

			/* get index of dest. nickname */
			dest = find_user(cp);

			/* do they exist? */
			if (dest >= 0) {
				/* fix the mod-ship */
				g_tab[gi].mod = dest;

				/* send message to dest.*/
				snprintf(line, LINE_SIZE, "%s just passed you moderation of group %s",
						u_tab[n].nickname, g_tab[gi].name);	
				sendstatus(dest,"Pass", line);

				/* send message to group */
				snprintf(line, LINE_SIZE, "%s has passed moderation to %s", u_tab[n].nickname, u_tab[dest].nickname);
				if (g_tab[gi].volume != QUIET) {
				   s_status_group(1,1,n,"Pass",line);
				}
			} else {
				snprintf(line, LINE_SIZE, "%s is not on!", cp);
				senderror(n, line);
			}
		}
        } else {
                error("pass: wrong number of arguments");
        }
}

void
s_boot(int n, int argc)
{
        int gi;
        int is_moderator;
        int dest;
	int destgi;
	char line[LINE_SIZE];

        if (argc == 2) {
                /* constraints:
                       inviter is mod & fields[1] is in this group
                                put fields[1] in group 1
				send boot message to fields[1]
                        otherwise
                                complain appropriately
			if group is QUIET, no messages sent.
                   side-effects:
                        if boot occurs, inform others in group
                   other:
                        fields[1] is nickname of person to boot
                */

                /* gi is canceller's groupindex */
                gi = find_group(u_tab[n].group);
                is_moderator =(g_tab[gi].mod == n);

                if (is_moderator) {
			dest = find_user(fields[1]);
			if (dest < 0) {
				snprintf(line, LINE_SIZE, "%s is not signed on",fields[1]);
                                sendstatus(n,"Warning",line);
			} else {
				destgi = find_group(u_tab[dest].group);
				if (gi == destgi) {
					/* send status to group */
					if ((nldelete(g_tab[destgi].n_invites,
					   fields[1]) == 0) ||
					   (nldelete(g_tab[destgi].nr_invites,
					   fields[1]) == 0)) {
					   snprintf(line, LINE_SIZE, "%s cancelled", fields[1]);
					   sendstatus(n, "FYI", line);
					   }
					snprintf(line, LINE_SIZE, "%s was booted.", u_tab[dest].nickname);
					if(g_tab[destgi].volume != QUIET) {
					   s_status_group(1,1,n,"Boot",line);
					}
					/* tell 'em it happened. */
					snprintf(line, LINE_SIZE, "%s booted you.", u_tab[n].nickname);
                                	sendstatus(dest,"Boot",line);
					/* fake an s_change to group ICB */
					strcpy(fields[1],"ICB");
					is_booting = 1;
					s_change(dest,2);
					is_booting = 0;
					/* note: although s_change returns a result, we presume that we can
         					always join group 1. */
				} else {
					snprintf(line, LINE_SIZE, "%s is not in your group", fields[1]);
					senderror(n,line);
				}
			}
                } else {
                        if ( !is_moderator ) {
                                senderror(n,"You aren't the moderator.");
                        }
                }
        } else {
                error("boot: wrong number of arguments");
        }
}

void
s_status(int n, int argc)
{
	int count;
	int i;
	char * cp;
	char cp2[80];
	char * vis;
	char * con;
	char * volume;
	char * mod;
	int is_moderator;
	int is_public;
	int is_group_1;
	int can_do;
	char * p1;
	char * p2;
	int gi;
	NAMLIST * ip;
	char row[ROW_SIZE];
	char line[LINE_SIZE];

        if (argc == 2) {
		if( strlen(fields[1]) == 0) {
			gi = find_group(u_tab[n].group); /* is n's groupindex */
			
			/* visibility */
        		switch(g_tab[gi].visibility) {

        		case VISIBLE:
				vis = "Visible";
                		break;
        		case SECRET:
				vis = "Secret";
                		break;
        		case SUPERSECRET:
				vis = "Invisible";
                		break;
        		default:
                		error("s_status visibility bad");	
				vis = "Unknown";
		        }

			/* control */
                        switch(g_tab[gi].control) {

                        case PUBLIC:
                                con = "Public";
                                break;
                        case MODERATED:
                                con = "Moderated";
                                break;
                        case RESTRICTED:
                                con = "Restricted";
                                break;
			case CONTROLLED:
				con = "Controlled";
				break;
                        default:
                                error("s_status control bad");
                                con = "Unknown";
                        }

                        /* volume */
                        switch(g_tab[gi].volume) {

                        case QUIET:
                                volume = "Quiet";
                                break;
                        case NORMAL:
                                volume = "Normal";
                                break;
                        case LOUD:
                                volume = "Loud";
                                break;
                        default:
                                error("s_status volume bad");
                                volume = "Unknown";
                        }

			/* moderator */
			if(g_tab[gi].mod >= 0)
				mod = u_tab[g_tab[gi].mod].nickname;
			else if (g_tab[gi].modtimeout == 0.0)
				mod = "None";
			else
				mod = g_tab[gi].missingmod;

			snprintf(line, LINE_SIZE, "Name: %s Mod: %s (%s / %s / %s)",
					g_tab[gi].name, mod, vis, con, volume);
                        sends_cmdout(n, line);

			/* invitations */
			row[0] = '\0';
			ip = g_tab[gi].n_invites;
			count = nlcount(*ip);
        		for (i=0; i<count; i++) {
				if (i == 0)
					add_item(n, "Nicks invited: ", row);
				else
					add_item(n, ", ", row);
                		cp = nlget(ip);
				add_item(n, cp, row);
        		}
			ip = g_tab[gi].nr_invites;
			count = nlcount(*ip);
        		for (i=0; i<count; i++) {
				if (strlen(row) == 0)
					add_item(n, "Nicks invited: ", row);
				else
					add_item(n, ", ", row);
                		cp = nlget(ip);
				snprintf(line, LINE_SIZE, "%s(r)", cp);
				add_item(n, line, row);
        		}
			if (strlen(row) > 0)
				sends_cmdout(n, row);
			row[0] = '\0';
			ip = g_tab[gi].s_invites;
			count = nlcount(*ip);
        		for (i=0; i<count; i++) {
				if (i == 0)
					add_item(n, "Addrs invited: ", row);
				else
					add_item(n, ", ", row);
                		cp = nlget(ip);
				add_item(n, cp, row);
        		}
			ip = g_tab[gi].sr_invites;
			count = nlcount(*ip);
        		for (i=0; i<count; i++) {
				if (strlen(row) == 0)
					add_item(n, "Addrs invited: ", row);
				else
					add_item(n, ", ", row);
                		cp = nlget(ip);
				snprintf(line, LINE_SIZE, "%s(r)", cp);
				add_item(n, line, row);
        		}
			if (strlen(row) > 0)
				sends_cmdout(n, row);

			row[0] = '\0';
			ip = g_tab[gi].n_talk;
			count = nlcount(*ip);
        		for (i=0; i<count; i++) {
				if (i == 0)
				    add_item(n, "Nicks who can talk: ", row);
				else
				    add_item(n, ", ", row);
                		cp = nlget(ip);
				add_item(n, cp, row);
        		}
			ip = g_tab[gi].nr_talk;
			count = nlcount(*ip);
        		for (i=0; i<count; i++) {
				if (strlen(row) == 0)
				    add_item(n, "Nicks who can talk: ", row);
				else
				    add_item(n, ", ", row);
                		cp = nlget(ip);
				snprintf(line, LINE_SIZE, "%s(r)", cp);
				add_item(n, line, row);
        		}
			if (strlen(row) > 0)
				sends_cmdout(n, row);

                } else {
			/* gi is n's group's index */
                        gi = find_group(u_tab[n].group);

			is_public = (g_tab[gi].control == PUBLIC);
			is_moderator = (g_tab[gi].mod == n);
			is_group_1 = (strcasecmp(g_tab[gi].name,
					"1") == 0);
			can_do = ((is_moderator ||
				   is_public    ||
                                   ((g_tab[gi].mod < 0) &&
				   (g_tab[gi].modtimeout == 0.0)))
				   && !is_group_1);

			if (!can_do) {
				if (is_group_1) {
					senderror(n,
						"Can't change group 1!");
				} else {
					senderror(n,
						"You aren't the moderator.");
				}
				return;
			}

			/* note: if the group is public when the
				 command is given, then the whole
				 command is executed even if the first
				 option makes the group restricted.
				 if there is no moderator and it isn't
				 group 1, then the remaining users can
				 still change some of the options,
				 but still can't boot.
			   also: this differs from Sean's server in
				 that the moderator can make the group
				 public while still retaining the mod
				 and thus the ability to boot.
			   but:  group 1 is still always public and
				 visible.
			*/
			p1 = fields[1];
			while(strlen(p2=getword(p1)) > 0) {
				switch(lookup(p2,status_table)){
					case SET_RESTRICT:
						g_tab[gi].control =
							RESTRICTED;
                				nlinit(g_tab[gi].n_bars, 
							MAX_INVITES);
                				nlinit(g_tab[gi].n_nr_bars, 
							MAX_INVITES);
                				nlinit(g_tab[gi].s_bars, 
							MAX_INVITES);
                				nlinit(g_tab[gi].s_nr_bars, 
							MAX_INVITES);
						cp = "restricted";
						for (i=0; i<MAX_USERS; i++)
							if (strcasecmp(u_tab[i].group,g_tab[gi].name) == 0) {
							   if (strlen(u_tab[i].realname) == 0)
								nlput(g_tab[gi].n_invites,u_tab[i].nickname);
							   else
								nlput(g_tab[gi].nr_invites,u_tab[i].nickname);
								if (g_tab[gi].mod > 0) {
								   snprintf(line, LINE_SIZE, "%s invited",u_tab[i].nickname);
								   sendstatus(g_tab[gi].mod, "FYI", line);
								   }
								snprintf(line, LINE_SIZE, "You are invited to group %s by default.", g_tab[gi].name);
								sendstatus(i, "FYI", line);
							}
						break;
					case SET_MODERATE:
						g_tab[gi].control =
							MODERATED;
                				nlinit(g_tab[gi].n_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].nr_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].s_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].sr_invites, 
							MAX_INVITES);
						cp = "moderated";
						break;

					case SET_CONTROL:
						g_tab[gi].control =
							CONTROLLED;
                				nlinit(g_tab[gi].n_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].nr_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].s_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].sr_invites, 
							MAX_INVITES);
						cp = "controlled";
						break;

					case SET_PUBLIC:
						g_tab[gi].control =
							PUBLIC;
                				nlinit(g_tab[gi].n_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].nr_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].s_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].sr_invites, 
							MAX_INVITES);
                				nlinit(g_tab[gi].n_bars, 
							MAX_INVITES);
                				nlinit(g_tab[gi].n_nr_bars, 
							MAX_INVITES);
                				nlinit(g_tab[gi].s_bars, 
							MAX_INVITES);
                				nlinit(g_tab[gi].s_nr_bars, 
							MAX_INVITES);
						cp = "public";
						break;
					case SET_INVISIBLE:
						g_tab[gi].visibility =
							SUPERSECRET;
						cp = "invisible";
						break;
					case SET_SECRET:
						g_tab[gi].visibility =
							SECRET;
						cp = "secret";
						break;
					case SET_VISIBLE:
						g_tab[gi].visibility =
							VISIBLE;
						cp = "visible";
						break;
					case SET_QUIET:
						g_tab[gi].volume =
							QUIET;
						cp = "quiet";
						break;
					case SET_NORMAL:
						g_tab[gi].volume =
							NORMAL;
						cp = "normal";
						break;
					case SET_LOUD:
						g_tab[gi].volume =
							LOUD;
						cp = "loud";
						break;
					case SET_NAME:
						p1 = get_tail(p1);
						p2=getword(p1);
						filtergroupname(p2);
						if (set_name(n, gi, p2) < 0) {
							cp = "";
						} else {
							snprintf (cp2, 80, "named %s", p2);
							cp = cp2;
						}
						break;
					case SET_QUESTION:
						dump_table(n);
						cp = NULL;
						break;
        				default:
						snprintf(line, LINE_SIZE, "Option %s unknown.", p2);
						senderror(n, line);
						cp = NULL;
        			}
				if (cp != NULL) 
				   if (strlen(cp) > 0) {
					snprintf(line, LINE_SIZE, "Group is now %s.", cp);
				        s_status_group(1,1,n,"Change",line);
				}
				p1 = get_tail(p1);
			}
                }
        } else {
                error("status: wrong number of arguments");
        }
}

static void
dump_table(int n)
{
	sendstatus(n,"Information","Table Of Status Commands");
	sendstatus(n,"Information","r - restricted c - controlled m - moderated p - public");
	sendstatus(n,"Information","i - invisible s - secret v - visible");
	sendstatus(n,"Information","q - quiet n - normal l - loud");
	sendstatus(n,"Information","name - change group name");
	sendstatus(n,"Information","? -- this table");
}

void
s_topic(int n, int argc)
{
	int t_group;
	int is_moderator;
	int has_moderator;
	char line[LINE_SIZE];

        if (argc == 2) {
			t_group = find_group(u_tab[n].group);

			if (strlen(fields[1]) == 0) {
				if(strlen(g_tab[t_group].topic) == 0) {
					strlcpy(line, "The topic is not set.", LINE_SIZE);
				} else {
					snprintf(line, LINE_SIZE, "The topic is: %s", g_tab[t_group].topic);
				}
				sends_cmdout(n, line);
				return;
			}

			is_moderator =(g_tab[t_group].mod == n);
			has_moderator = (g_tab[t_group].mod >= 0);

			if (has_moderator && (!is_moderator)) {
				senderror(n,"You aren't the moderator.");
				return;
			}

			strlcpy(g_tab[t_group].topic,fields[1],MAX_TOPICLEN);
			if (g_tab[t_group].volume != QUIET) {
			  snprintf(line, LINE_SIZE, "%s changed the topic to \"%s\"",
				u_tab[n].nickname,
				g_tab[t_group].topic);
			  if(g_tab[t_group].volume != QUIET) {
			     s_status_group(1,1,n,"Topic",line);
			  }
			}
        } else {
                error("topic: wrong number of arguments");
        }
}

/* send open group message */
/* loop through the table... */
/*   make sure they aren't us unless echoback is on */
/*   look for other user being in this user's group */
/* strings comparisons are done case-insensitive */

void
s_send_group(int n)
{
        int i;
        char my_group[MAX_GROUPLEN+1];
	char one[LINE_SIZE], two[LINE_SIZE];

        strlcpy(my_group, u_tab[n].group, MAX_GROUPLEN+1);
	snprintf(one, LINE_SIZE, "%s@%s", u_tab[n].loginid, u_tab[n].nodeid);
	ucaseit(one);
	strlcpy(two, u_tab[n].nickname, LINE_SIZE);
	ucaseit(two);

	/* send it to all of the real users */
        for (i=0; i<MAX_REAL_USERS; i++) {
                if ((i != n) || u_tab[n].echoback)
                        if ((strcasecmp(my_group, u_tab[i].group) == 0) &&
			   (!nlmatch(one, *u_tab[i].pub_s_hushed)) &&
			   (!nlmatch(two, *u_tab[i].pub_n_hushed)) &&
			   (u_tab[i].login == 1))
					sendopen(n, i, fields[0]);
        }
}

void
s_exclude(int n, int argc)
{
        int i, j;
        char my_group[MAX_GROUPLEN+1];
	char one[LINE_SIZE], two[LINE_SIZE];
	char line[LINE_SIZE];

	if (argc == 2) {
		if (strlen(get_tail(fields[1])) > 0) {
			strlcpy(my_group, u_tab[n].group, MAX_GROUPLEN+1);
			snprintf(one, LINE_SIZE, "%s@%s", u_tab[n].loginid, u_tab[n].nodeid);
			ucaseit(one);
			strlcpy(two, u_tab[n].nickname, LINE_SIZE);
			ucaseit(two);

			/* send it to all of the real users */
			for (i=0; i<MAX_REAL_USERS; i++) {
				if ((i != n) || u_tab[n].echoback)
					if ((strcasecmp(my_group, u_tab[i].group) == 0) &&
							(!nlmatch(one, *u_tab[i].pub_s_hushed)) &&
							(!nlmatch(two, *u_tab[i].pub_n_hushed)) &&
							(u_tab[i].login == 1) && 
							(strcasecmp(u_tab[i].nickname, getword(fields[1])))) {
						if ((j = find_user(getword(fields[1]))) > 0)
							snprintf(line, LINE_SIZE, "%s's next message excludes %s:", u_tab[n].nickname, u_tab[j].nickname);
						else
							snprintf(line, LINE_SIZE, "%s's next message excludes %s:", u_tab[n].nickname, getword(fields[1]));
						sendstatus(i, "Exclude", line);
						sendopen(n, i, get_tail(fields[1]));
					}
			}
		} else {
			senderror(n, "Empty message.");
		}
	}
}

/* send group status message */
/* loop through the table... */
/*   look for other user being in this user's group */
/* strings comparisons are done case-insensitive */

void
s_status_group(int k, int tellme, int n, char *class_string, char *message_string)
{
	/* for k,  1 means n is u_tab index, otherwise n is g_tab index */
        int i;
        char my_group[MAX_GROUPLEN+1];

	if (k==1) {
        	strlcpy(my_group, u_tab[n].group, MAX_GROUPLEN+1);
		if (strlen(my_group) == 0)
			error("bad u_tab entry");
	} else {
        	strlcpy(my_group, g_tab[n].name, MAX_GROUPLEN+1);
		if (strlen(my_group) == 0)
			error("bad g_tab entry");
	}

	/* do it only for real users */
        for (i=0; i<MAX_REAL_USERS; i++) {
		if (strcasecmp(my_group, u_tab[i].group) == 0) {
			/* n means something different here */
			/* for k=1 (u_tab entry) vs k=2 ( g_tab entry) */
			/* so n for k=2 doesn't make sense. */
			/* 'cuz it'd mean don't do this when user i was
				equal to group n...*/
			if (tellme || (k != 1) || ((k==1) && (i != n))) {
				/* actually send it */
				sendstatus(i,class_string, message_string);
			}
		}
        }
}

static int
set_name(int n, int group, char *name)
{
	char * cp;
	int i;
	char n_g_n[MAX_GROUPLEN + 1];

	/* make sure they gave us a name to change to */
	if (strlen(name) == 0) {
		senderror(n, "New name may not be null.");
		return -1;
	}

	/* use only as much as needed */
	strlcpy(n_g_n, name, MAX_GROUPLEN+1);

	if (!strcmp(n_g_n, "1")) {
		senderror (n, "Cannot change name to group 1.");
		return -1;
		}

	/* make sure that group doesn't already exist */
	if ((find_group(n_g_n) >= 0) && (find_group(n_g_n) != group)) {
		senderror(n, "That group already exists.");
		return -1;
	}

	/* get a copy to the old name */
	cp = g_tab[group].name;

	/* check through the user table to see who is in that group */
	/* and change their group name entries */
	for (i=0; i< MAX_REAL_USERS; i++) {
		if (strcasecmp(u_tab[i].group, cp) == 0) {
			strlcpy(u_tab[i].group, n_g_n, MAX_GROUPLEN);
		}
	}

	/* finally change the name of the group itself */
	strlcpy(g_tab[group].name, n_g_n, MAX_GROUPLEN);
	return 0;
}


/*
 * s_talk()
 * - function to modify the group list of nicknames with permission to
 *   talk within a controlled group
 */

void
s_talk(int n, int argc)
{
    int is_public,
	is_moderator;
    int gi;
    int dest;
    int r = -1,
	delete = -1,
	addmode = -1;
    int i;
    int quiet = -1;
    char *who;
    char *flags;
    char *talk_usage = "Usage: talk {-a} {-q} {-r} {-d} nickname";
    char line[LINE_SIZE];

    if (argc == 2)
    {
	/* constraints:
		group is public & "talk"er is mod => 
			messages and change in invite list
	   side-effects:
		if talkee is signed on, send status message
	   other:
		fields[1] is nickname of person to invite
	*/

	gi = find_group(u_tab[n].group);

	is_public = (g_tab[gi].control == PUBLIC);
	is_moderator = (g_tab[gi].mod == n);
	who = fields[1];
	while (who[0] == '-') 
	{
	    flags = getword(who);
	    for (i = 1; i < strlen(flags); i++)
	       switch (flags[i]) {
	       case 'q':
		    if (quiet == -1)
			    quiet = 1;
		    else {
			    senderror(n, talk_usage);
			    return;
		    }
		    break;
	       case 'a':
		    if ( addmode == -1 )
			addmode = 1;
		    else
		    {
			senderror (n, talk_usage);
			return;
		    }
		    break;

	       case 'd':
		    if ( delete == -1 )
			delete = 1;
		    else
		    {
			senderror (n, talk_usage);
			return;
		    }
		    break;

	       case 'r':
		    if (r == -1)
			    r = 1;
		    else {
			    senderror(n, talk_usage);
			    return;
		    }
		    break;
	       default:
		    senderror(n, talk_usage);
		    return;
	       }

	    who = get_tail(who);
	}

	if (r == -1) r = 0;
	if (quiet == -1) quiet = 0;

	if ( who[0] == '\0' )
	{
	    (void) senderror (n, talk_usage);
	}
	else if ( is_public )
	{
	    (void) senderror (n,
		"The group is public.");
	}
	else if ( !is_moderator )
	{
	    (void) senderror (n, "You aren't the moderator.");
	}
	else /* !is_public && is_moderator */
	{
	    /* if delete == 1, then we are removing this user */
	    if ( delete == 1 )
	    {
		int	deleted;

		deleted = talk_delete (who, gi);

		if ( deleted == 0 )
		{
		    snprintf(line, LINE_SIZE, "%s is not in talk list", who);
		    senderror (n, line);
		}
		else if ( deleted > 0 )
		{
		    if ( quiet == 0 )
		    {
			snprintf(line, LINE_SIZE, "%s removed from talk list", who);
			sendstatus (n, "FYI", line);
		    }

		}
	    }
	    else
	    {
		/* if not adding to, remove all other entries */
		if ( addmode == -1 )
		{
		    NAMLIST	*nl;
		    int		count, j;
		    char	*cp;

		    nl = g_tab[gi].n_talk;
		    count = nlcount(*nl);
		    for ( j = 0; j < count; j++)
		    {
			cp = nlget (nl);
			if ( talk_delete (cp, gi) > 0
			    && quiet == 0 )
			{
			    snprintf(line, LINE_SIZE, "%s removed from talk list", cp);
			    sendstatus (n, "FYI", line);
			}
		    }

		    nl = g_tab[gi].nr_talk;
		    count = nlcount(*nl);
		    for ( j = 0; j < count; j++)
		    {
			cp = nlget (nl);
			if ( talk_delete (cp, gi) > 0 && quiet == 0 )
			{
			    snprintf(line, LINE_SIZE, "%s removed from registered talk list", cp);
			    sendstatus (n, "FYI", line);
			}
		    }
		}

		if ( r == 0 )
		{
		    nlput(g_tab[gi].n_talk, who);

		    if (quiet == 0)
		    {
		       snprintf(line, LINE_SIZE, "%s can talk", who);
		       sendstatus(n, "FYI", line);
		    }
		}
		else /* ( r == 1 ) */
		{
		    nlput(g_tab[gi].nr_talk, who);
		    if (quiet == 0)
		    {
		       snprintf(line, LINE_SIZE, "%s can talk (registered only)", who);
		       sendstatus(n, "FYI", line);
		    }
		}

		if ( (dest = find_user(who)) >= 0 )
		{
		    if ( (strlen (u_tab[dest].realname) > 0) || (r == 0) )
		    {
			snprintf(line, LINE_SIZE, "You can talk in group %s (courtesy of %s)",
			    		u_tab[n].group, u_tab[n].nickname);
			    sendstatus(dest,"TALK",line);
		    }
		}
		else
		{
		    snprintf(line, LINE_SIZE, "%s is not signed on.", who);
		    sendstatus(n, "Warning", line);
		}
	    }
	}
    }
    else
    {
	error("talk: wrong number of arguments");
    }
}

static int
talk_delete (char *who, int gi)
{
    int		deleted = 0, dest, in_registered = 0, in_normal = 0;
    char	line[LINE_SIZE];

    if ( nlpresent (who, *g_tab[gi].n_talk) != 0 )
    {
	in_normal = 1;
	deleted++;
	nldelete (g_tab[gi].n_talk, who);
    }

    if ( nlpresent (who, *g_tab[gi].nr_talk) != 0 )
    {
	in_registered = 1;
	deleted++;
	nldelete (g_tab[gi].nr_talk, who);
    }

    if ( deleted > 0 )
    {
	if ( (dest = find_user(who)) >= 0 )
	{
	    if ( in_normal ||
		(in_registered && (strlen (u_tab[dest].realname) > 0)) )
	    {
		snprintf(line, LINE_SIZE, "You no longer can talk in group %s (no thanks to %s)",
		    		u_tab[g_tab[gi].mod].group, u_tab[g_tab[gi].mod].nickname);
		    sendstatus(dest, "TALK", line);
	    }
	}
    }

    return (deleted);
}

void
talk_report (int n, int gi)
{
	char	line[LINE_SIZE];

	if ( (nlpresent(u_tab[n].nickname, *g_tab[gi].n_talk) > 0) ||
			(nlpresent(u_tab[n].nickname, *g_tab[gi].nr_talk) &&
			(strlen(u_tab[n].realname) > 0)) ) {
		snprintf(line, LINE_SIZE, "Talk permission in: %s", g_tab[gi].name);
		sends_cmdout (n, line);
	}
}
