# $NetBSD: language.mk,v 1.13 2020/01/21 13:46:57 leot Exp $

# This file contains the necessary definitions to regenerate the news
# and events in the language-specific index.html files. It should be
# included by all top-level directories of a translation.
#
# The following variables may be set in the language-specific Makefile.
#
# CHANGES_DIR
#	The directory that contains the changes. If the NetBSD changes
#	are not translated into the current language, this variable
#	should be set to ../changes, so the English changes are used.
#
# EVENTS_DIR
#	The directory that contains the future events. If the language
#	does not translate the future events, this variable should be
#	set to ../gallery, so the English events are integrated.
#
# The following variables should be set in the language-specific
# Makefile.inc(!).
#
# DOCLANG
#	The language code in which the pages in this directory are
#	written.
#
# CHANGESDOCS
#	The list of documents that contain the latest changes. For most
#	of the languages, simply mentioning "index" will suffice,
#	although the preferred way is to put the changes into the file
#	named after the current year. They will eventually end there
#	anyway, so why should we first put them into index.xml?
#
#	This setting should be in Makefile.inc instead of Makefile
#	because then the changes/Makefile can refer to it by simply
#	saying XMLDOCS+= ${CHANGESDOCS}.

.include "../../share/mk/web.prefs.mk"

.if exists(changes/Makefile)
CHANGES_DIR?=	changes
.else
CHANGES_DIR?=	../changes
.endif
.if exists(gallery/events.xml)
EVENTS_DIR?=	gallery
.else
EVENTS_DIR?=	../gallery
.endif
DOCLANG?=	en
CHANGESDOCS?=	index

# Note: this is _not_ added to GENDOCS because it cannot be regenerated
# after it has been deleted once.
DATA+=		index.html

CHANGES_FILE=	index.chg
EVENTS_FILE=	${EVENTS_DIR}/events.chg

# changes.
.if defined(CHANGESDOCS)
INDEX_DEPS=	${CHANGES_FILE}
.elif exists(${CHANGES_DIR}/index.xml)
INDEX_DEPS=	${CHANGES_DIR}/index.html
.else
INDEX_DEPS=	${HTDOCS_TOPDIR}/changes/index.html
.endif

# Events.
.if !exists(${EVENTS_DIR}/events.xml)
ERRORS+=	"[language.mk] ${EVENTS_DIR}/events.xml does not exist."
.endif
INDEX_DEPS+=	${EVENTS_DIR}/events.html
INDEX_DEPS+=	${EVENTS_FILE}

MAKEINDEX_ENV=	WEB_PREFIX=${WEB_PREFIX}
MAKEINDEX_ENV+=	CHANGES_DIR=${CHANGES_DIR}
MAKEINDEX_ENV+=	EVENTS_DIR=${EVENTS_DIR}
MAKEINDEX_ENV+=	HTDOCS_RELTOPDIR=${HTDOCS_RELTOPDIR}

# The .PHONY prevents make(1) from printing the up-to-date message.
.PHONY: language-index
language-index: index.html
index.html: ${INDEX_DEPS}
	env ${MAKEINDEX_ENV} sh ${HTDOCS_TOPDIR}/makeindex.sh

${CHANGES_DIR}/index.html: ${CHANGES_DIR}/index.xml
	cd ${CHANGES_DIR} && ${MAKE} index.html

${EVENTS_DIR}/events.html: ${EVENTS_DIR}/events.xml
	cd ${EVENTS_DIR} && ${MAKE} events.html

CLEANFILES+=	${CHANGES_FILE} ${CHANGES_FILE}.tmp* ${EVENTS_FILE}
${CHANGES_FILE}: ${CHANGESDOCS:C,.*,${CHANGES_DIR}/&.xml,}
	@${ECHO} "[xsltproc] ${.ALLSRC} -> ${.TARGET}"
	${RUN} ${XSLTPROC} ${XSLTPROCOPTS} \
		--stringparam changesdir ${CHANGES_DIR} \
		${HTDOCS_TOPDIR}/share/xsl/changes.xsl ${.ALLSRC} \
	> ${.TARGET}.tmp1
	${RUN} ${GREP} "^<a " < ${.TARGET}.tmp1 > ${.TARGET}.tmp2
	${RUN} ${MV} -f ${.TARGET}.tmp2 ${.TARGET}
	${RUN} ${RM} -f ${.TARGET}.tmp1

${EVENTS_FILE}: ${EVENTS_DIR}/events.xml
	@${ECHO} "[xsltproc] ${.ALLSRC} -> ${.TARGET}"
	${RUN} ${XSLTPROC} ${XSLTPROCOPTS} \
		--stringparam eventsdir ${EVENTS_DIR} \
		${HTDOCS_TOPDIR}/share/xsl/events.xsl ${.ALLSRC} \
	> ${.TARGET}.tmp1
	${RUN} ${SED} -ne "/^<a /p" < ${.TARGET}.tmp1 > ${.TARGET}.tmp2
	${RUN} ${MV} -f ${.TARGET}.tmp2 ${.TARGET}
	${RUN} ${RM} -f ${.TARGET}.tmp1
