/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 *  $Id: bktrrec.c,v 1.18 2006/03/30 09:21:26 jakemsr Exp $
 */

#include "includes.h"

#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

#include "bsdav.h"

#define BKTR_WIDTH_MAX		768
#define BKTR_HEIGHT_MAX		576
#define BKTR_FPS_MAX		30

volatile sig_atomic_t bktrrec_quit;
volatile sig_atomic_t bktr_frame_waiting;

char	*bktr_dev;
int	 bktr_fd;
char	*outfile;
FILE 	*out;
int	 verbose;

struct bsdav_ringbuf rb;
uint8_t	*buffer;
size_t	 buffer_size;

int	 norm;
double	 usecs_per_frame;

extern char *__progname;

void	 catchsignal(int);
void	 usage(void);

int	 grab(void);
int	 stream(void);
void	 cleanup(int);

/* getopt externs */
extern char *optarg;
extern int opterr;
extern int optind;
extern int optopt;
extern int optreset;


void
catchsignal(int signal)
{
	switch(signal) {
	case SIGUSR1:
		bktr_frame_waiting++;
		break;
	case SIGINT:
		bktrrec_quit = 1;
		break;
	default:
		break;
	}
}


void
usage(void)
{
	printf("usage: %s [-glv] [-e encoding] [-f file] [-h height] [-i field]\n"
	       "          [-n norm] [-o output] [-p pidfile] [-r rate]\n"
	       "          [-s source] [-w width]\n",
	    __progname);
}


int
grab(void)
{
#define GRABBER_SETTLE_TIME 2
int	 c;

	/* allow the grabber brightness to settle down */
	sleep(GRABBER_SETTLE_TIME);

	c = METEOR_CAP_SINGLE;
	if (ioctl(bktr_fd, METEORCAPTUR, &c) < 0) {
		warn("METEORCAPTUR");
		return(1);
	}

	fwrite(buffer, buffer_size, 1, out);

	return(0);
}


int
stream(void)
{
struct sigaction act;
struct timeval tp_start;
struct timeval tp_offset;
struct timeval tp_tmp0;
struct timeval tp_tmp1;
struct timeval ts;
sigset_t sa_mask;
long	 run_time;
long	 frames;
long	 s_frames;
int	 sequence;
uint	 c;
int	 ret;
long	 usecs;
int	 rolling;
int	 wrote_frame;
int	 got_frame;
int	 out_frame_waiting;
int	 count;

	ret = 0;
	frames = 0;
	s_frames = 0;
	rolling = 0;
	bktrrec_quit = 0;
	sequence = 100;

	/* for sigsuspend() */
	sigfillset(&sa_mask);
	sigdelset(&sa_mask, SIGUSR1);
	sigdelset(&sa_mask, SIGINT);
	sigdelset(&sa_mask, SIGALRM);

	/* signal handler setup */
	memset(&act, 0, sizeof(act));
	sigemptyset(&act.sa_mask);
	act.sa_handler = catchsignal;
	sigaction(SIGUSR1, &act, NULL);
	sigaction(SIGINT, &act, NULL);

	/* signal when a buffer is ready */
	c = SIGUSR1;
	if (ioctl(bktr_fd, METEORSSIGNAL, &c) < 0) {
		warn("METEORSSIGNAL");
		return(1);
	}

	/* start grabing frames */
	c = METEOR_CAP_CONTINOUS;
	if (ioctl(bktr_fd, METEORCAPTUR, &c) < 0) {
		warn("METEORCAPTUR");
		return(1);
	}

	/* wait for first signal */
	alarm(1);
	sigsuspend(&sa_mask);
	if (bktr_frame_waiting == 0) {
		warnx("frame grab timeout");
		ret = 1;
		bktrrec_quit = 1;
	}
	alarm(0);

	gettimeofday(&tp_start, NULL);

	bktr_frame_waiting = 0;
	out_frame_waiting = 0;

	while (bktrrec_quit == 0) {

		got_frame = 0;
		wrote_frame = 0;

		if (bktr_frame_waiting == 0)
			usleep((useconds_t)usecs_per_frame / 8);

		if (bktrrec_quit == 1)
			break;

		count = 0;
		while (bktr_frame_waiting > 0) {
			if (rb.bufs_off >= rb.num_bufs) {
				warnx("too many frames buffered   ");
				bktr_frame_waiting--;
				break;
			} else if (rb.bufs_off < 0) {
				warnx("no frames buffered   ");
				bktr_frame_waiting++;
			}
			gettimeofday(&ts, NULL);
			if (count > 0) {
				tp_tmp0.tv_sec = 0;
				tp_tmp0.tv_usec = usecs_per_frame * count;
				timeradd(&ts, &tp_tmp0, &tp_tmp1);
				memcpy(&ts, &tp_tmp1, sizeof(struct timeval));
			}
			if (bsdav_rb_buf_in(&rb, buffer, buffer_size, ts) != 0) {
				warnx("bsdav_rb_buf_in, size = %llu   ",
				    (unsigned long long)buffer_size);
				ret = 1;
				bktrrec_quit = 1;
				break;
			}
			bktr_frame_waiting--;
			got_frame = 1;
			count++;
		}

		if (bktrrec_quit == 1)
			break;

		if (rolling == 0) {
			if (rb.bufs_off >= rb.num_bufs / 2) {
				rolling = 1;
				memcpy(&tp_tmp0, &tp_start,
				    sizeof(struct timeval));
				gettimeofday(&tp_start, NULL);
				timersub(&tp_start, &tp_tmp0, &tp_offset);
			}
		} else if (got_frame == 1) {
			if (rb.bufs_off <= rb.num_bufs / 4) {
				if (verbose > 1)
					warnx("buffers low: %d   ",
					    rb.bufs_off);
				rb.bufs_off++;
			}
			if (rb.bufs_off >= rb.num_bufs * 3 / 4) {
				if (verbose > 1)
					warnx("buffers high: %d   ",
					    rb.bufs_off);
				rb.bufs_off--;
			}
		}

		if (rolling == 1) {
			gettimeofday(&tp_tmp0, NULL);
			timersub(&tp_tmp0, &tp_start, &tp_tmp1);
			s_frames = (((tp_tmp1.tv_sec * 1000000) +
			    tp_tmp1.tv_usec) /
			    ((double)bsdav_vid_nrms[norm].frame_rate.d *
			    1000000 / bsdav_vid_nrms[norm].frame_rate.n)) + 0.5;
			out_frame_waiting = s_frames - frames;
		}

		if (bktrrec_quit == 1)
			break;

		while (out_frame_waiting > 0) {
			gettimeofday(&tp_tmp0, NULL);
			timersub(&tp_tmp0, &tp_offset, &tp_tmp1);
			timersub(&tp_tmp1, &(rb.bufs[rb.buf_out].ts), &tp_tmp0);
			usecs = tp_tmp0.tv_sec * 1000000 + tp_tmp0.tv_usec;
			if (usecs > usecs_per_frame * 2) {
				if (verbose > 1)
					warnx("frame time behind = %ld   ",
					    usecs);
				/* rb.bufs_off--; */
			} else if (usecs < -(usecs_per_frame)) {
				if (verbose > 1)
					warnx("frame time ahead = %ld   ",
					    usecs);
				rb.bufs_off++;
			}
			rb.buf_out = rb.buf_in - (rb.bufs_off - 1);
			if (rb.buf_out < 0)
				rb.buf_out += rb.num_bufs;
			if (bsdav_write_frame_data(out, rb.bufs[rb.buf_out].buf,
			    rb.bufs[rb.buf_out].size, 0) != 0) {
				warnx("bsdav_write_frame_data failed   ");
				ret = 1;
				bktrrec_quit = 1;
			} else {
				out_frame_waiting--;
				rb.bufs_off--;
				frames++;
				wrote_frame = 1;
			}
		}

		if (bktrrec_quit == 1)
			break;

		if ((frames > 0) && (wrote_frame == 1) &&
		    (frames % sequence == 0)) {
			gettimeofday(&tp_tmp0, NULL);
			timersub(&tp_tmp0, &tp_start, &tp_tmp1);
			run_time = tp_tmp1.tv_sec * 1000000 + tp_tmp1.tv_usec;
			if (verbose > 1) {
				fprintf(stderr, "frames: %06ld [%02d] [%06ld],"
				    " seconds: %09.3f, fps: %08.5f\r",
				    frames, rb.bufs_off, s_frames,
				    (double)run_time / 1000000,
				    (double)frames * 1000000 / run_time);
				fflush(stderr);
			}
		}
	}
	gettimeofday(&tp_tmp0, NULL);

	fprintf(stderr, "\n");

	/* close down the signal generation */

	c = METEOR_SIG_MODE_MASK;
	if (ioctl(bktr_fd, METEORSSIGNAL, &c) < 0) {
		warn("METEORSSIGNAL");
		ret = 1;
	}

	c = METEOR_CAP_STOP_CONT;
	if (ioctl(bktr_fd, METEORCAPTUR, &c) < 0) {
		warn("METEORCAPTUR");
		ret = 1;
	}

	act.sa_handler = SIG_DFL;
	sigaction(SIGUSR1, &act, NULL);
	sigaction(SIGINT, &act, NULL);

	fclose(out);

	timersub(&tp_tmp0, &tp_start, &tp_tmp1);
	run_time = tp_tmp1.tv_sec * 1000000 + tp_tmp1.tv_usec;

	if (verbose > 0) {
		warnx("total time: %f", (double)run_time / 1000000);
		warnx("total frames: %ld", frames);
		warnx("total fps: %f", (double)frames * 1000000 / run_time);
	}

	return(ret);
}


void
cleanup(int excode)
{

	if (buffer != MAP_FAILED)
		munmap(buffer, buffer_size);

	if (out != NULL)
		fclose(out);

	if (bktr_fd >= 0)
		close(bktr_fd);

	bsdav_free_ringbuf(&rb);

	bsdav_unmap_vid_buffer(&buffer, buffer_size);

	if (excode != 0)
		exit(excode);
}


int
main(int argc, char *argv[])
{
struct bsdav_crop crop;
const char *errstr;
char	*pidfile;
int	 gflag, lflag;
int	 ch;
int	 source;
int	 fps;
int	 fields;
int	 format;
int	 width;
int	 height;


	/* defaults */
	bktr_dev = DEFAULT_VIDEO_DEVICE;
	fields = 0;
	outfile = NULL;
	norm = BSDAV_VIDNORM_NTSC;
	source = BSDAV_VIDSOURCE_TUNER;
	format = BSDAV_VIDFMT_UYVY;
	fps = 0;
	width = 0;
	height = 0;
	verbose = 0;
	gflag = 0;
	lflag = 0;
	pidfile = NULL;
	rb.num_bufs = 10;

	/* for cleanup() */
	bktr_fd = -1;
	buffer = MAP_FAILED;
	out = NULL;

	while ((ch = getopt(argc, argv, "vglb:e:f:h:i:n:o:p:r:s:w:")) != -1) {
		switch (ch) {
		case 'b':
			rb.num_bufs = (int)strtonum(optarg, 5, 20, &errstr);
			if (errstr != NULL)
				errx(1, "number of buffers is %s: %s", errstr,
				    optarg);
			break;
		case 'e':
			format = bsdav_find_vid_fmt(optarg);
			if (format < 0)
				errx(1, "invalid encoding: %s", optarg);
			break;
		case 'f':
			bktr_dev = optarg;
			break;
		case 'g':
			gflag++;
			break;
		case 'h':
			height = (int)strtonum(optarg, 0, BKTR_HEIGHT_MAX, &errstr);
			if (errstr != NULL)
				errx(1, "height is %s: %s", errstr, optarg);
			break;
		case 'i':
			if (strncmp(optarg, "odd", 3) == 0) {
				fields = METEOR_GEO_ODD_ONLY;
			} else if (strncmp(optarg, "even", 3) == 0) {
				fields = METEOR_GEO_EVEN_ONLY;
			} else
				errx(1, "invalid fields: %s", optarg);
			break;
		case 'l':
			lflag++;
			break;
		case 'n':
			norm = bsdav_find_vid_norm(optarg);
			if (norm < 0)
				errx(1, "invalid norm: %s", optarg);
			break;
		case 'o':
			outfile = optarg;
			break;
		case 'p':
			pidfile = optarg;
			break;
		case 'r':
			fps = (int)strtonum(optarg, 0, BKTR_FPS_MAX, &errstr);
			if (errstr != NULL)
				errx(1, "rate is %s: %s", errstr, optarg);
			break;
		case 's':
			source = bsdav_find_vid_source(optarg);
			if (source < 0)
				errx(1, "invalid source: %s", optarg);
			break;
		case 'v':
			verbose++;
			break;
		case 'w':
			width = (int)strtonum(optarg, 0, BKTR_WIDTH_MAX, &errstr);
			if (errstr != NULL)
				errx(1, "width is %s: %s", errstr, optarg);
			break;
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (pidfile != NULL) {
		if (bsdav_write_pid(pidfile) != 0) {
			warnx("couldn't write pid file");
			cleanup(1);
		}
	}

	bktr_fd = open(bktr_dev, O_RDONLY);
	if (bktr_fd < 0) {
		warn("%s", bktr_dev);
		cleanup(1);
	}

	if (lflag > 0) {
		bsdav_list_bktr_formats(bktr_dev, bktr_fd);
	} else {
		if (outfile != NULL) {
			memset(&crop, 0, sizeof(struct bsdav_crop));
			if (bsdav_bktr_init(bktr_fd, norm, &width, &height,
			    fields, format, source, &fps, &crop) != 0)
				cleanup(1);

			if (verbose > 0) {
				warnx("format: %s",
				    bsdav_vid_fmts[format].name);
				warnx("norm: %s", bsdav_vid_nrms[norm].name);
				warnx("width: %d", width);
				warnx("height: %d", height);
				warnx("source: %s",
				    bsdav_vid_srcs[source].name);
				warnx("fps: %d", fps);
			}

			buffer_size = bsdav_map_vid_buffer(&buffer, bktr_fd,
				width, height, format);
			if (buffer_size == 0) {
				warnx("failed to mmap hardware buffer");
				cleanup(1);
			}
			if (verbose > 1)
				warnx("buffer_size: %llu", (unsigned long long)buffer_size);

			if ((out = fopen(outfile, "w")) == NULL) {
				warn("%s", outfile);
				cleanup(1);
			}

			if (gflag > 0) {
				if (grab() != 0)
					cleanup(1);
			} else {
				/* the ring buffer */

				if (bsdav_init_ringbuf(&rb, buffer_size) != 0) {
					warnx("bsdav_init_ringbuf() failed");
					return(1);
				}

				usecs_per_frame =
				    (double)bsdav_vid_nrms[norm].frame_rate.d *
				     1000000 /
				     bsdav_vid_nrms[norm].frame_rate.n;

				if (stream() != 0)
					cleanup(1);

				bsdav_free_ringbuf(&rb);
			}

			fclose(out);

			bsdav_unmap_vid_buffer(&buffer, buffer_size);

		} else {
			warnx("output file not specified, exiting");
			cleanup(1);
		}
	}

	cleanup(0);

	return (0);
}
