package status

import (
	"context"
	"strings"
	"testing"

	"github.com/emersion/go-imap/v2"
)

func TestGetDatabaseFilename(t *testing.T) {
	tests := []struct {
		accountName string
		want        string
	}{
		{accountName: "personal", want: "personal-4a0a339b0c6d0553.db"},
		{accountName: "雨果", want: "account-553daff18fda62d0.db"},
		{accountName: "user@example.com", want: "user_example.com-b4c9a289323b21a0.db"},
		{accountName: "My Work Account", want: "My_Work_Account-9d319d2ed5c5bac3.db"},
		{accountName: "home/user/mail", want: "home_user_mail-5f46bd776dbd4eac.db"},
		{accountName: "user+tag@example.com", want: "user_tag_example.com-80e9aafb240be8b1.db"},
		{accountName: "this_is_a_very_long_account_name_that_exceeds_thirty_two_characters", want: "this_is_a_very_long_account_name-92e3d9f71b8e8f35.db"},
		{accountName: "", want: "account-e3b0c44298fc1c14.db"},
		{accountName: "@#$%^&*()", want: "account-a821cf5c9ba958c4.db"},
	}

	for _, tt := range tests {
		t.Run(tt.accountName, func(t *testing.T) {
			r := &Repository{
				accountName: tt.accountName,
			}

			got := r.getDatabaseFilename()

			if got != tt.want {
				t.Errorf("getDatabaseFilename() = %q, want %q", got, tt.want)
			}
		})
	}
}

func TestGetDatabaseFilename_Uniqueness(t *testing.T) {
	// Test that different account names produce different filenames
	accounts := []string{
		"personal",
		"work",
		"user@example.com",
		"user@example.org",
		"User@Example.com", // Different case
	}

	filenames := make(map[string]string)

	for _, account := range accounts {
		r := &Repository{
			accountName: account,
		}
		filename := r.getDatabaseFilename()

		if existing, exists := filenames[filename]; exists {
			t.Errorf("getDatabaseFilename() collision: accounts %q and %q both produced %q", account, existing, filename)
		}

		filenames[filename] = account
	}
}

func TestGetDatabaseFilename_Consistency(t *testing.T) {
	// Test that the same account name always produces the same filename
	accountName := "test@example.com"

	r1 := &Repository{accountName: accountName}
	r2 := &Repository{accountName: accountName}

	filename1 := r1.getDatabaseFilename()
	filename2 := r2.getDatabaseFilename()

	if filename1 != filename2 {
		t.Errorf("getDatabaseFilename() not consistent: got %v and %v for same account", filename1, filename2)
	}
}

func TestSanitizeFilename(t *testing.T) {
	tests := []struct {
		name  string
		input string
		want  string
	}{
		{name: "alphanumeric", input: "abc123ABC", want: "abc123ABC"},
		{name: "allowed special chars", input: "test-account_v1.2", want: "test-account_v1.2"},
		{name: "spaces", input: "my account", want: "my_account"},
		{name: "email address", input: "user@example.com", want: "user_example.com"},
		{name: "path-like", input: "/home/user/mail", want: "_home_user_mail"},
		{name: "unicode", input: "用户账户", want: "account"},
		{name: "empty string", input: "", want: "account"},
		{name: "only special chars", input: "@#$%", want: "account"},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			r := &Repository{}
			got := r.sanitizeFilename(tt.input)

			if got != tt.want {
				t.Errorf("sanitizeFilename(%q) = %q, want %q", tt.input, got, tt.want)
			}
		})
	}
}

func TestSanitizeFilename_SafeForFilesystem(t *testing.T) {
	// Test that sanitized names don't contain filesystem-unsafe characters
	unsafeChars := []rune{'/', '\\', ':', '*', '?', '"', '<', '>', '|', '\x00'}

	inputs := []string{
		"user@example.com",
		"path/to/mail",
		"file:name",
		"wild*card?",
		"quotes\"test\"",
		"<brackets>",
		"pipe|test",
	}

	r := &Repository{}

	for _, input := range inputs {
		sanitized := r.sanitizeFilename(input)

		for _, unsafe := range unsafeChars {
			if strings.ContainsRune(sanitized, unsafe) {
				t.Errorf("sanitizeFilename(%q) = %q, contains unsafe character: %q", input, sanitized, unsafe)
			}
		}
	}
}

func TestJoinFlags(t *testing.T) {
	tests := []struct {
		flags []imap.Flag
		want  string
	}{
		{
			flags: nil,
			want:  "",
		},
		{
			flags: []imap.Flag{imap.FlagSeen},
			want:  "\\Seen",
		},
		{
			flags: []imap.Flag{imap.FlagSeen, imap.FlagFlagged, imap.FlagAnswered},
			want:  "\\Answered,\\Flagged,\\Seen",
		},
		{
			flags: []imap.Flag{imap.FlagSeen, imap.FlagAnswered, imap.FlagFlagged, imap.FlagDeleted, imap.FlagDraft},
			want:  "\\Answered,\\Deleted,\\Draft,\\Flagged,\\Seen",
		},
		{
			flags: []imap.Flag{imap.FlagDeleted, imap.FlagAnswered, imap.FlagSeen},
			want:  "\\Answered,\\Deleted,\\Seen",
		},
	}

	for _, tt := range tests {
		t.Run(tt.want, func(t *testing.T) {
			got := joinFlags(tt.flags)
			if got != tt.want {
				t.Errorf("joinFlags() = %q, want %q", got, tt.want)
			}
		})
	}
}

func TestJoinFlags_Consistency(t *testing.T) {
	// Test that joinFlags produces consistent output for same input

	result1 := joinFlags([]imap.Flag{imap.FlagSeen, imap.FlagFlagged})
	result2 := joinFlags([]imap.Flag{imap.FlagFlagged, imap.FlagSeen})

	if result1 != result2 {
		t.Errorf("joinFlags() not consistent: got %q and %q", result1, result2)
	}
}

func TestGetByFilename_MatchesOnUniquePortion(t *testing.T) {
	r := New("test-account")
	if err := r.Init(); err != nil {
		t.Fatalf("Failed to initialize repository: %v", err)
	}
	defer func() {
		_ = r.Close()
	}()

	ctx := context.Background()
	mailbox := "INBOX"
	uid := imap.UID(42)

	// Add a message with the Seen flag.
	originalFilename := "1234567890.abcd1234,U=42:2,S"
	originalFlags := []imap.Flag{imap.FlagSeen}

	if err := r.Add(mailbox, uid, originalFilename, originalFlags); err != nil {
		t.Fatalf("Failed to add message: %v", err)
	}

	// Query with a filename with different flags (Seen + Flagged).
	queryFilename := "1234567890.abcd1234,U=42:2,SF"

	msg, err := r.GetByFilename(ctx, mailbox, queryFilename)
	if err != nil {
		t.Fatalf("GetByFilename failed: %v", err)
	}

	if msg == nil {
		t.Fatal("Expected to find message, got nil")
	}

	if msg.UID != uid {
		t.Errorf("Got UID %d, want %d", msg.UID, uid)
	}

	if msg.Filename != originalFilename {
		t.Errorf("Got filename %q, want %q", msg.Filename, originalFilename)
	}

	if len(msg.Flags) != 1 || msg.Flags[0] != imap.FlagSeen {
		t.Errorf("Got flags %v, want [\\Seen]", msg.Flags)
	}
}

func TestGetByFilename_NoMatch(t *testing.T) {
	r := New("test-account-nomatch")
	if err := r.Init(); err != nil {
		t.Fatalf("Failed to initialize repository: %v", err)
	}
	defer func() {
		_ = r.Close()
	}()

	ctx := context.Background()
	mailbox := "INBOX"

	// Query for a non-existent message
	msg, err := r.GetByFilename(ctx, mailbox, "9999999999.nonexist,U=999:2,S")
	if err != nil {
		t.Fatalf("GetByFilename failed: %v", err)
	}

	if msg != nil {
		t.Errorf("Expected nil, got message with UID %d", msg.UID)
	}
}

func TestGetByFilename_NoFlags(t *testing.T) {
	r := New("test-account-noflags")
	if err := r.Init(); err != nil {
		t.Fatalf("Failed to initialize repository: %v", err)
	}
	defer func() {
		_ = r.Close()
	}()

	ctx := context.Background()
	mailbox := "INBOX"
	uid := imap.UID(123)

	originalFilename := "1234567890.xyz12345,U=123"
	if err := r.Add(mailbox, uid, originalFilename, []imap.Flag{}); err != nil {
		t.Fatalf("Failed to add message: %v", err)
	}

	msg, err := r.GetByFilename(ctx, mailbox, originalFilename)
	if err != nil {
		t.Fatalf("GetByFilename failed: %v", err)
	}

	if msg == nil {
		t.Fatal("Expected to find message, got nil")
	}

	if msg.UID != uid {
		t.Errorf("Got UID %d, want %d", msg.UID, uid)
	}

	// Query with flags added.
	queryFilename := "1234567890.xyz12345,U=123:2,S"
	msg, err = r.GetByFilename(ctx, mailbox, queryFilename)
	if err != nil {
		t.Fatalf("GetByFilename failed: %v", err)
	}

	if msg == nil {
		t.Fatal("Expected to find message, got nil")
	}

	if msg.UID != uid {
		t.Errorf("Got UID %d, want %d", msg.UID, uid)
	}
}
