package maildir

import (
	"fmt"
	"log/slog"
	"os"
	"path/filepath"
	"slices"
	"strings"
	"testing"

	"github.com/emersion/go-imap/v2"
)

func TestGetMaildirPath(t *testing.T) {
	tests := []struct {
		basePath string
		mailbox  string
		want     string
	}{
		{
			basePath: "/home/user/Mail",
			mailbox:  "INBOX",
			want:     "/home/user/Mail/INBOX",
		},
		{
			basePath: "/home/user/Mail",
			mailbox:  "Archive/2024",
			want:     "/home/user/Mail/Archive/2024",
		},
		{
			basePath: "/home/user/Mail",
			mailbox:  "Work/Projects/2024/Q1",
			want:     "/home/user/Mail/Work/Projects/2024/Q1",
		},
		{
			basePath: "/home/user/Mail/",
			mailbox:  "INBOX",
			want:     "/home/user/Mail/INBOX",
		},
	}

	for _, tt := range tests {
		t.Run(tt.want, func(t *testing.T) {
			m := &Maildir{
				path:    tt.basePath,
				mailbox: tt.mailbox,
			}
			got := m.getMaildirPath()
			if got != tt.want {
				t.Errorf("getMaildirPath() = %q, want %q", got, tt.want)
			}
		})
	}
}

func TestParseFilename(t *testing.T) {
	uid42 := imap.UID(42)
	uid100 := imap.UID(100)
	uid5 := imap.UID(5)

	tests := []struct {
		filename  string
		wantUID   *imap.UID
		wantFlags []imap.Flag
	}{
		{
			filename:  "1234567890.abc123,U=42:2,SF",
			wantUID:   &uid42,
			wantFlags: []imap.Flag{imap.FlagSeen, imap.FlagFlagged},
		},
		{
			filename:  "1234567890.abc123,U=100:2,",
			wantUID:   &uid100,
			wantFlags: nil,
		},
		{
			filename:  "1234567890.abc123:2,DFRS",
			wantUID:   nil,
			wantFlags: []imap.Flag{imap.FlagDraft, imap.FlagFlagged, imap.FlagAnswered, imap.FlagSeen},
		},
		{
			filename:  "1234567890.abc123:2,",
			wantUID:   nil,
			wantFlags: nil,
		},
		{
			filename:  "1234567890.abc123,U=5:2,DFRST",
			wantUID:   &uid5,
			wantFlags: []imap.Flag{imap.FlagDraft, imap.FlagFlagged, imap.FlagAnswered, imap.FlagSeen, imap.FlagDeleted},
		},
		{
			filename:  "1234567890.abc123",
			wantUID:   nil,
			wantFlags: nil,
		},
	}

	for _, tt := range tests {
		t.Run(tt.filename, func(t *testing.T) {
			m := &Maildir{}
			gotUID, gotFlags := m.parseFilename(tt.filename)

			if (gotUID == nil) != (tt.wantUID == nil) {
				t.Errorf("parseFilename() UID = %v, want %v", gotUID, tt.wantUID)
			} else if gotUID != nil && tt.wantUID != nil && *gotUID != *tt.wantUID {
				t.Errorf("parseFilename() UID = %v, want %v", *gotUID, *tt.wantUID)
			}

			if len(gotFlags) != len(tt.wantFlags) {
				t.Errorf("parseFilename() flags = %v, want %v", gotFlags, tt.wantFlags)
				return
			}

			for i := range gotFlags {
				if gotFlags[i] != tt.wantFlags[i] {
					t.Errorf("parseFilename() flags[%d] = %v, want %v", i, gotFlags[i], tt.wantFlags[i])
				}
			}
		})
	}
}

func TestParseFlags(t *testing.T) {
	tests := []struct {
		flagChars string
		want      []imap.Flag
	}{
		{
			// "empty",
			flagChars: "",
			want:      nil,
		},
		{
			// "seen",
			flagChars: "S",
			want:      []imap.Flag{imap.FlagSeen},
		},
		{
			// "all flags",
			flagChars: "DFRST",
			want:      []imap.Flag{imap.FlagDraft, imap.FlagFlagged, imap.FlagAnswered, imap.FlagSeen, imap.FlagDeleted},
		},
		{
			// "unordered flags",
			flagChars: "SFR",
			want:      []imap.Flag{imap.FlagSeen, imap.FlagFlagged, imap.FlagAnswered},
		},
		{
			// "unknown flags ignored",
			flagChars: "SXYZ",
			want:      []imap.Flag{imap.FlagSeen},
		},
		{
			//      "only unknown flags",
			flagChars: "XYZ",
			want:      nil,
		},
	}

	for _, tt := range tests {
		t.Run(tt.flagChars, func(t *testing.T) {
			m := &Maildir{}
			got := m.parseFlags(tt.flagChars)

			if len(got) != len(tt.want) {
				t.Errorf("parseFlags() = %v, want %v", got, tt.want)
				return
			}

			for i := range got {
				if got[i] != tt.want[i] {
					t.Errorf("parseFlags()[%d] = %v, want %v", i, got[i], tt.want[i])
				}
			}
		})
	}
}

func TestFormatFlags(t *testing.T) {
	tests := []struct {
		flags []imap.Flag
		want  string
	}{
		{
			// "empty",
			flags: nil,
			want:  "",
		},
		{
			// "single flag",
			flags: []imap.Flag{imap.FlagSeen},
			want:  "S",
		},
		{
			// "multiple flags sorted",
			flags: []imap.Flag{imap.FlagSeen, imap.FlagFlagged},
			want:  "FS",
		},
		{
			// "all flags sorted",
			flags: []imap.Flag{imap.FlagDraft, imap.FlagFlagged, imap.FlagAnswered, imap.FlagSeen, imap.FlagDeleted},
			want:  "DFRST",
		},
		{
			// "unsorted input becomes sorted",
			flags: []imap.Flag{imap.FlagSeen, imap.FlagAnswered, imap.FlagFlagged},
			want:  "FRS",
		},
		{
			// "deleted and draft",
			flags: []imap.Flag{imap.FlagDeleted, imap.FlagDraft},
			want:  "DT",
		},
	}

	for _, tt := range tests {
		t.Run(tt.want, func(t *testing.T) {
			m := &Maildir{}
			got := m.formatFlags(tt.flags)

			if got != tt.want {
				t.Errorf("formatFlags() = %q, want %q", got, tt.want)
			}
		})
	}
}

func TestFormatFlags_AlphabeticalOrder(t *testing.T) {
	// Test that formatFlags always produces alphabetically sorted output
	m := &Maildir{}

	// All possible permutations should produce "DFRST"
	permutations := [][]imap.Flag{
		{imap.FlagDraft, imap.FlagFlagged, imap.FlagAnswered, imap.FlagSeen, imap.FlagDeleted},
		{imap.FlagDeleted, imap.FlagSeen, imap.FlagAnswered, imap.FlagFlagged, imap.FlagDraft},
		{imap.FlagSeen, imap.FlagDeleted, imap.FlagDraft, imap.FlagAnswered, imap.FlagFlagged},
		{imap.FlagFlagged, imap.FlagDraft, imap.FlagDeleted, imap.FlagSeen, imap.FlagAnswered},
	}

	want := "DFRST"

	for i, flags := range permutations {
		got := m.formatFlags(flags)
		if got != want {
			t.Errorf("permutation %d: formatFlags() = %q, want %q", i, got, want)
		}
	}
}

func TestGenerateFilename(t *testing.T) {
	m := &Maildir{
		logger: slog.Default(),
	}

	tests := []struct {
		name  string
		uid   imap.UID
		flags []imap.Flag
	}{
		{
			name:  "with UID and flags",
			uid:   42,
			flags: []imap.Flag{imap.FlagSeen, imap.FlagFlagged},
		},
		{
			name:  "with UID no flags",
			uid:   100,
			flags: nil,
		},
		{
			name:  "zero UID with flags",
			uid:   0,
			flags: []imap.Flag{imap.FlagDraft},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got := m.generateFilename(tt.uid, tt.flags)

			// Check format: timestamp.random,U=uid:2,flags
			if !strings.Contains(got, ".") {
				t.Errorf("generateFilename() = %q, missing timestamp separator", got)
			}
			if !strings.Contains(got, ",U=") {
				t.Errorf("generateFilename() = %q, missing UID marker", got)
			}
			if !strings.Contains(got, ":2,") {
				t.Errorf("generateFilename() = %q, missing Maildir info section", got)
			}

			// Verify UID is present.
			expectedUID := strings.Contains(got, fmt.Sprintf(",U=%d", tt.uid))
			if tt.uid > 0 && !expectedUID {
				if !strings.Contains(got, ",U=") {
					t.Errorf("generateFilename() = %q, missing UID in filename", got)
				}
			}

			// Verify flags.
			if len(tt.flags) > 0 {
				flagStr := m.formatFlags(tt.flags)
				if !strings.HasSuffix(got, ":2,"+flagStr) {
					t.Errorf("generateFilename() = %q, expected to end with :2,%s", got, flagStr)
				}
			} else {
				if !strings.HasSuffix(got, ":2,") {
					t.Errorf("generateFilename() = %q, expected to end with :2,", got)
				}
			}
		})
	}
}

func TestGenerateFilename_Uniqueness(t *testing.T) {
	// Test that generateFilename produces unique filenames
	m := &Maildir{
		logger: slog.Default(),
	}

	uid := imap.UID(42)
	flags := []imap.Flag{imap.FlagSeen}

	filename1 := m.generateFilename(uid, flags)
	filename2 := m.generateFilename(uid, flags)

	if filename1 == filename2 {
		t.Errorf("generateFilename() generated duplicate filenames: %s", filename1)

	}
}

func TestRoundTripFlagConversion(t *testing.T) {
	// Test that flags can be converted to string and back
	m := &Maildir{}

	tests := [][]imap.Flag{
		{},
		{imap.FlagSeen},
		{imap.FlagSeen, imap.FlagFlagged},
		{imap.FlagDraft, imap.FlagFlagged, imap.FlagAnswered, imap.FlagSeen, imap.FlagDeleted},
	}

	for i, originalFlags := range tests {
		flagStr := m.formatFlags(originalFlags)
		parsedFlags := m.parseFlags(flagStr)

		if len(originalFlags) != len(parsedFlags) {
			t.Errorf("test %d: round trip failed, got %v flags, want %v", i, len(parsedFlags), len(originalFlags))
			continue
		}

		// formatFlags sorts, so we need to compare sorted versions.
		// parsed flags should match the formatted order
		for j, flag := range parsedFlags {
			if !slices.Contains(originalFlags, flag) {
				t.Errorf("test %d: flag %v not found in original flags", i, parsedFlags[j])
			}
		}
	}
}

func TestAdd_Placement(t *testing.T) {
	tmpDir := t.TempDir()
	m := &Maildir{
		path:    tmpDir,
		mailbox: "INBOX",
		logger:  slog.Default(),
	}

	if err := m.Init(); err != nil {
		t.Fatalf("Failed to initialize Maildir: %v", err)
	}

	tests := []struct {
		uid        imap.UID
		flags      []imap.Flag
		wantSubdir string // "new" or "cur"
	}{
		{
			uid:        1,
			flags:      []imap.Flag{},
			wantSubdir: "new",
		},
		{
			uid:        2,
			flags:      []imap.Flag{imap.FlagSeen},
			wantSubdir: "cur",
		},
		{
			uid:        3,
			flags:      []imap.Flag{imap.FlagFlagged},
			wantSubdir: "new",
		},
		{
			uid:        4,
			flags:      []imap.Flag{imap.FlagSeen, imap.FlagFlagged},
			wantSubdir: "cur",
		},
	}

	for _, tt := range tests {
		t.Run(m.formatFlags(tt.flags), func(t *testing.T) {
			content := []byte("test message")
			filename, err := m.Add(tt.uid, tt.flags, content)
			if err != nil {
				t.Fatalf("Add() error = %v", err)
			}

			expectedPath := filepath.Join(m.getMaildirPath(), tt.wantSubdir, filename)
			if _, err := os.Stat(expectedPath); err != nil {
				t.Errorf("Message not found in %s/: %v", tt.wantSubdir, err)
			}

			otherSubdir := "cur"
			if tt.wantSubdir == "cur" {
				otherSubdir = "new"
			}
			otherPath := filepath.Join(m.getMaildirPath(), otherSubdir, filename)
			if _, err := os.Stat(otherPath); !os.IsNotExist(err) {
				t.Errorf("Message unexpectedly found in %s/", otherSubdir)
			}
		})
	}
}

func TestResolveAbsPath_NotExist(t *testing.T) {
	tmpDir := t.TempDir()
	m := &Maildir{
		path:    tmpDir,
		mailbox: "INBOX",
		logger:  slog.Default(),
	}

	if err := m.Init(); err != nil {
		t.Fatalf("Failed to initialize Maildir: %v", err)
	}

	_, err := m.ResolveAbsPath("nonexistent.file:2,S")

	if err == nil {
		t.Fatal("ResolveAbsPath() returned nil error for non-existent file")
	}

	if !os.IsNotExist(err) {
		t.Errorf("ResolveAbsPath() error should be detectable by os.IsNotExist()\ngot error: %v\nerror type: %T", err, err)
	}
}

func TestResolveAbsPath_ExistsInCur(t *testing.T) {
	tmpDir := t.TempDir()
	m := &Maildir{
		path:    tmpDir,
		mailbox: "INBOX",
		logger:  slog.Default(),
	}

	if err := m.Init(); err != nil {
		t.Fatalf("Failed to initialize Maildir: %v", err)
	}

	filename := "1234567890.abc123:2,S"
	curPath := filepath.Join(m.getMaildirPath(), "cur", filename)
	if err := os.WriteFile(curPath, []byte("test"), 0600); err != nil {
		t.Fatalf("Failed to create test file: %v", err)
	}

	gotPath, err := m.ResolveAbsPath(filename)
	if err != nil {
		t.Fatalf("ResolveAbsPath() error = %v", err)
	}

	if gotPath != curPath {
		t.Errorf("ResolveAbsPath() = %q, want %q", gotPath, curPath)
	}
}

func TestResolveAbsPath_ExistsInNew(t *testing.T) {
	tmpDir := t.TempDir()
	m := &Maildir{
		path:    tmpDir,
		mailbox: "INBOX",
		logger:  slog.Default(),
	}

	if err := m.Init(); err != nil {
		t.Fatalf("Failed to initialize Maildir: %v", err)
	}

	filename := "1234567890.abc123:2,"
	newPath := filepath.Join(m.getMaildirPath(), "new", filename)
	if err := os.WriteFile(newPath, []byte("test"), 0600); err != nil {
		t.Fatalf("Failed to create test file: %v", err)
	}

	gotPath, err := m.ResolveAbsPath(filename)
	if err != nil {
		t.Fatalf("ResolveAbsPath() error = %v", err)
	}

	if gotPath != newPath {
		t.Errorf("ResolveAbsPath() = %q, want %q", gotPath, newPath)
	}
}

func TestResolveAbsPath_MatchesOnUniquePortion(t *testing.T) {
	// Test that ResolveAbsPath matches files when flags have changed.
	tmpDir := t.TempDir()
	m := &Maildir{
		path:    tmpDir,
		mailbox: "INBOX",
		logger:  slog.Default(),
	}

	if err := m.Init(); err != nil {
		t.Fatalf("Failed to initialize Maildir: %v", err)
	}

	// Create a file with Seen + Flagged flags.
	actualFilename := "1234567890.abc123,U=42:2,SF"
	curPath := filepath.Join(m.getMaildirPath(), "cur", actualFilename)
	if err := os.WriteFile(curPath, []byte("test"), 0600); err != nil {
		t.Fatalf("Failed to create test file: %v", err)
	}

	// Try to resolve using just Seen flag.
	oldFilename := "1234567890.abc123,U=42:2,S"

	gotPath, err := m.ResolveAbsPath(oldFilename)
	if err != nil {
		t.Fatalf("ResolveAbsPath() error = %v", err)
	}

	if gotPath != curPath {
		t.Errorf("ResolveAbsPath(%q) = %q, want %q", oldFilename, gotPath, curPath)
	}
}

func TestUpdateFlags_PreservesFilename(t *testing.T) {
	m := &Maildir{
		logger: slog.Default(),
	}

	tests := []struct {
		origFilename string
		origUID      *imap.UID
		toAdd        []imap.Flag
		toRemove     []imap.Flag
		wantFilename string
	}{
		{
			// "with UID, add flag",
			origFilename: "/tmp/cur/1234567890.abcd1234,U=42:2,S",
			origUID:      func() *imap.UID { u := imap.UID(42); return &u }(),
			toAdd:        []imap.Flag{imap.FlagFlagged},
			toRemove:     []imap.Flag{},
			wantFilename: "1234567890.abcd1234,U=42:2,FS",
		},
		{
			// "without UID, add flag",
			origFilename: "/tmp/cur/1234567890.xyz789:2,S",
			origUID:      nil,
			toAdd:        []imap.Flag{imap.FlagFlagged},
			toRemove:     []imap.Flag{},
			wantFilename: "1234567890.xyz789:2,FS",
		},
		{
			// "no flags initially, add flag",
			origFilename: "/tmp/cur/1234567890.abcd1234",
			origUID:      nil,
			toAdd:        []imap.Flag{imap.FlagSeen},
			toRemove:     []imap.Flag{},
			wantFilename: "1234567890.abcd1234:2,S",
		},
		{
			// "remove all flags",
			origFilename: "/tmp/cur/1234567890.abcd1234:2,DFRS",
			origUID:      nil,
			toAdd:        []imap.Flag{},
			toRemove:     []imap.Flag{imap.FlagDraft, imap.FlagFlagged, imap.FlagAnswered, imap.FlagSeen},
			wantFilename: "1234567890.abcd1234:2,",
		},
		{
			// "mbsync-style filename, add flag",
			origFilename: "/tmp/cur/1234567890.M123456P789.host:2,S",
			origUID:      nil,
			toAdd:        []imap.Flag{imap.FlagAnswered},
			toRemove:     []imap.Flag{},
			wantFilename: "1234567890.M123456P789.host:2,RS",
		},
		{
			// "add and remove at once, remove S, add F",
			origFilename: "/tmp/cur/1234567890.abcd1234:2,SR",
			origUID:      nil,
			toAdd:        []imap.Flag{imap.FlagFlagged},
			toRemove:     []imap.Flag{imap.FlagSeen},
			wantFilename: "1234567890.abcd1234:2,FR",
		},
	}

	for _, tt := range tests {
		t.Run(tt.wantFilename, func(t *testing.T) {
			tmpDir := t.TempDir()
			curDir := filepath.Join(tmpDir, "cur")
			if err := os.MkdirAll(curDir, 0700); err != nil {
				t.Fatalf("Failed to create cur directory: %v", err)
			}

			origBasename := filepath.Base(tt.origFilename)
			origPath := filepath.Join(curDir, origBasename)
			if err := os.WriteFile(origPath, []byte("test"), 0600); err != nil {
				t.Fatalf("Failed to create test file: %v", err)
			}

			newFilename, actualFlags, err := m.UpdateFlags(origPath, tt.toAdd, tt.toRemove)
			if err != nil {
				t.Fatalf("UpdateFlags() error = %v", err)
			}

			if newFilename != tt.wantFilename {
				t.Errorf("UpdateFlags() returned filename = %q, want %q", newFilename, tt.wantFilename)
			}

			newPath := filepath.Join(curDir, tt.wantFilename)
			if _, err := os.Stat(newPath); err != nil {
				t.Errorf("New file not found at %q: %v", newPath, err)
			}

			if _, err := os.Stat(origPath); !os.IsNotExist(err) {
				t.Errorf("Original file still exists at %q", origPath)
			}

			if tt.origUID == nil && strings.Contains(tt.wantFilename, ",U=0") {
				t.Errorf("UpdateFlags() incorrectly added U=0 for nil UID")
			}

			// Verify returned flags match expected
			_, expectedFlags := m.parseFilename(tt.wantFilename)
			if len(actualFlags) != len(expectedFlags) {
				t.Errorf("UpdateFlags() returned flags = %v, want %v", actualFlags, expectedFlags)
			}
		})
	}
}
