package test

import (
	"context"
	"fmt"
	"log/slog"
	"os"
	"path/filepath"
	"slices"
	"strings"
	"testing"
	"time"

	"git.sr.ht/~whynothugo/ImapGoose/internal/daemon"
	"git.sr.ht/~whynothugo/ImapGoose/internal/imap"
	imap2 "github.com/emersion/go-imap/v2"
)

// TestFlagConflictResolution tests that merged flag changes are back-propagated.
// See: e1e9ca2c51cc16c1571a10c35a4f5836c795f8c5
func TestFlagConflictResolution(t *testing.T) {
	dovecot := setupDovecot(t)

	stateDir := t.TempDir()
	localMailDir := t.TempDir()

	cfg := dovecot.MakeConfig("test", localMailDir, stateDir)

	logger := slog.New(slog.NewTextHandler(os.Stdout, &slog.HandlerOptions{
		Level: slog.LevelDebug,
	}))

	// Start daemon and create message.
	ctx, stopDaemon := context.WithCancel(context.Background())

	d := daemon.New(cfg, logger)
	daemonDone := make(chan struct{})
	go func() {
		_ = d.Run(ctx)
		close(daemonDone)
	}()

	time.Sleep(100 * time.Millisecond)

	imapClient, err := imap.Connect(
		context.Background(),
		fmt.Sprintf("127.0.0.1:%d", dovecot.Port),
		"",
		"",
		true, // plaintext
		nil,
		logger,
	)
	if err != nil {
		t.Fatalf("Failed to connect to IMAP server: %v", err)
	}
	defer func() { _ = imapClient.Close() }()

	_, err = imapClient.SelectMailbox(context.Background(), "INBOX", nil)
	if err != nil {
		t.Fatalf("Failed to select INBOX: %v", err)
	}

	// Create a test message with no flags.
	testMessage := []byte("From: test@example.com\r\nTo: user@example.com\r\nSubject: Flag Test\r\n\r\nThis is a flag conflict test.\r\n")
	uid, err := imapClient.AppendMessage(context.Background(), "INBOX", []imap2.Flag{}, testMessage)
	if err != nil {
		t.Fatalf("Failed to append message: %v", err)
	}

	t.Logf("Created message with UID %d", uid)

	// Wait for message to sync to local.
	inboxPath := filepath.Join(localMailDir, "INBOX")
	deadline := time.Now().Add(500 * time.Millisecond)

	var localFilePath string
	var localFileName string
	for time.Now().Before(deadline) {
		for _, subdir := range []string{"cur", "new"} {
			dirPath := filepath.Join(inboxPath, subdir)
			entries, err := os.ReadDir(dirPath)
			if err != nil {
				if os.IsNotExist(err) {
					continue
				}
				t.Fatalf("Failed to read directory %s: %v", dirPath, err)
			}

			if len(entries) > 0 {
				localFileName = entries[0].Name()
				localFilePath = filepath.Join(dirPath, localFileName)
				t.Logf("Found message in %s: %s", subdir, localFileName)
				break
			}
		}

		if localFilePath != "" {
			break
		}

		time.Sleep(10 * time.Millisecond)
	}

	if localFilePath == "" {
		t.Fatal("Message did not appear in local Maildir within timeout")
	}
	t.Logf("Setting local flag: Seen on file: %s", localFileName)

	// Stop daemon
	t.Logf("Stopping daemon")
	stopDaemon()

	select {
	case <-daemonDone:
		t.Logf("Daemon stopped")
	case <-time.After(5 * time.Second):
		t.Fatal("Daemon did not stop within timeout")
	}

	// Add the Seen flag locally.
	parts := strings.Split(localFileName, ":2,")
	var newLocalFileName string
	flags := parts[1]
	if !strings.Contains(flags, "S") {
		// Sort flags alphabetically (Maildir convention)
		newFlags := "S" + flags
		newLocalFileName = parts[0] + ":2," + newFlags
	} else {
		newLocalFileName = localFileName // Already has S
	}
	newLocalPath := filepath.Join(filepath.Dir(localFilePath), newLocalFileName)
	if err := os.Rename(localFilePath, newLocalPath); err != nil {
		t.Fatalf("Failed to rename local file: %v", err)
	}
	t.Logf("Renamed local file from %s to %s", filepath.Base(localFilePath), newLocalFileName)

	// Add the Flagged flag remotely.
	_, err = imapClient.UpdateFlags(context.Background(), uid, []imap2.Flag{imap2.FlagFlagged}, nil)
	if err != nil {
		t.Fatalf("Failed to set remote flag: %v", err)
	}
	t.Logf("Set Flagged flag on remote message UID %d", uid)

	// Start daemon again
	t.Logf("Restarting daemon")
	ctx2, stopDaemon2 := context.WithCancel(context.Background())
	defer stopDaemon2()

	d2 := daemon.New(cfg, logger)
	go func() { _ = d2.Run(ctx2) }()

	// Wait for flags to be synced and check both sides

	// Wait for local file to have both flags.
	deadline = time.Now().Add(2 * time.Second)
	var finalLocalFilename string

	for time.Now().Before(deadline) {
		for _, subdir := range []string{"cur", "new"} {
			dirPath := filepath.Join(inboxPath, subdir)
			entries, err := os.ReadDir(dirPath)
			if err != nil {
				continue
			}

			if len(entries) > 0 {
				filename := entries[0].Name()
				if strings.Contains(filename, ":2,") {
					flagsPart := strings.Split(filename, ":2,")[1]
					if strings.Contains(flagsPart, "S") && strings.Contains(flagsPart, "F") {
						finalLocalFilename = filename
						break
					}
				}
			}
		}

		if finalLocalFilename != "" {
			break
		}

		time.Sleep(10 * time.Millisecond)
	}

	if finalLocalFilename == "" {
		t.Fatal("Message with merged flags did not appear in local maildir within timeout")
	}
	t.Logf("Found message after sync with merged flags: %s", finalLocalFilename)

	// Check that both S (Seen) and F (Flagged) are present
	if !strings.Contains(finalLocalFilename, ":2,") {
		t.Fatal("Local file has no flags suffix")
	}

	flagsPart := strings.Split(finalLocalFilename, ":2,")[1]
	t.Logf("Local flags: %s", flagsPart)
	if !strings.Contains(flagsPart, "S") {
		t.Error("Local file missing Seen flag (S)")
	}
	if !strings.Contains(flagsPart, "F") {
		t.Error("Local file missing Flagged flag (F)")
	}

	// TODO: need to loop and wait as above.
	messages, err := imapClient.ListMessages(context.Background())
	if err != nil {
		t.Fatalf("Failed to list messages: %v", err)
	}

	if len(messages) == 0 {
		t.Fatal("No messages found on remote server")
	}

	// Find any message (UID might have changed due to delete/re-upload)
	var finalMsg *imap.Message
	var finalUID imap2.UID
	for msgUID, msg := range messages {
		finalUID = msgUID
		msgCopy := msg
		finalMsg = &msgCopy
		break
	}

	if finalMsg == nil {
		t.Fatal("No message found in remote messages")
	}

	t.Logf("Found message with UID %d on remote", finalUID)
	remoteFlags := finalMsg.Flags
	t.Logf("Remote flags: %v", remoteFlags)

	if !slices.Contains(remoteFlags, imap2.FlagSeen) {
		t.Error("Remote message missing Seen flag")
	}
	if !slices.Contains(remoteFlags, imap2.FlagFlagged) {
		t.Error("Remote message missing Flagged flag")
	}

	t.Logf("Flag conflict resolution successful: both sides have merged flags (UID: %d)", finalUID)
}
