package test

import (
	"fmt"
	"math/rand"
	"net"
	"os"
	"os/exec"
	"path/filepath"
	"testing"
	"text/template"

	"git.sr.ht/~whynothugo/ImapGoose/internal/config"
)

// DovecotServer represents a test Dovecot instance for testing.
type DovecotServer struct {
	Port       int
	BaseDir    string
	MailDir    string
	ConfigPath string
	listener   net.Listener
	stopChan   chan struct{}
}

// setupDovecot starts a Dovecot server in a temporary directory and returns a handle to stop it.
func setupDovecot(t *testing.T) *DovecotServer {
	t.Helper()

	// t.TempDir() doesn't work because its parents might not be readable by the dovecot user.
	baseDir, err := os.MkdirTemp("/tmp", "dovecot-test-")
	if err != nil {
		t.Fatalf("Failed to create temp directory: %v", err)
	}
	// Make directory world-writable so dovecot user can write logs.
	if err := os.Chmod(baseDir, 0777); err != nil {
		t.Fatalf("Failed to chmod base directory: %v", err)
	}
	t.Cleanup(func() {
		_ = os.RemoveAll(baseDir)
	})

	mailDir := filepath.Join(baseDir, "mail")
	configPath := filepath.Join(baseDir, "dovecot.conf")
	homeDir := filepath.Join(mailDir, "testuser")

	if err := os.MkdirAll(homeDir, 0777); err != nil {
		t.Fatalf("Failed to create home directory: %v", err)
	}
	if err := os.Chmod(mailDir, 0777); err != nil {
		t.Fatalf("Failed to chmod mail directory: %v", err)
	}
	if err := os.Chmod(homeDir, 0777); err != nil {
		t.Fatalf("Failed to chmod home directory: %v", err)
	}

	port := 10000 + rand.Intn(50000)

	// Simplified Dovecot config - only essential settings needed.
	configTemplate := `dovecot_config_version = 2.4.0
dovecot_storage_version = 2.4.0

protocols = imap

# Use temp directory
base_dir = {{.BaseDir}}/dovecot/base
state_dir = {{.BaseDir}}/dovecot/state

# Dovecot 2.4 mail settings
mail_driver = maildir
mail_path = {{.MailDir}}
mail_uid = {{.UID}}
mail_gid = {{.GID}}

first_valid_uid = {{.UID}}
last_valid_uid = {{.UID}}

# Enable mailbox list index for NOTIFY capability
mailbox_list_index = yes

# Namespace
namespace inbox {
  inbox = yes
  separator = /
}

# Explicitly enable all required IMAP capabilities
protocol imap {
  imap_capability = IMAP4rev1 LITERAL+ SASL-IR LOGIN-REFERRALS ID ENABLE IDLE NOTIFY UIDPLUS CONDSTORE QRESYNC AUTH=PLAIN
}

`

	tmpl, err := template.New("dovecot").Parse(configTemplate)
	if err != nil {
		t.Fatalf("Failed to parse template: %v", err)
	}

	// Use current UID/GID, but fall back to 1000 if running as root (UID 0).
	// Dovecot doesn't allow UID 0.
	uid := os.Getuid()
	gid := os.Getgid()
	if uid == 0 {
		// FIXME: should use dovecot user
		uid = 1000
		gid = 1000
	}

	configData := struct {
		MailDir string
		UID     int
		GID     int
		BaseDir string
	}{
		MailDir: mailDir,
		UID:     uid,
		GID:     gid,
		BaseDir: baseDir,
	}

	configFile, err := os.Create(configPath)
	if err != nil {
		t.Fatalf("Failed to create config file: %v", err)
	}
	defer func() { _ = configFile.Close() }()

	if err := tmpl.Execute(configFile, configData); err != nil {
		t.Fatalf("Failed to execute template: %v", err)
	}
	_ = configFile.Close()

	listener, err := net.Listen("tcp", fmt.Sprintf("127.0.0.1:%d", port))
	if err != nil {
		t.Fatalf("Failed to create listener: %v", err)
	}

	stopChan := make(chan struct{})

	// Accept connections and spawn doveadm exec imap for each.
	go func() {
		for {
			conn, err := listener.Accept()
			if err != nil {
				select {
				case <-stopChan: // Shutdown requested.
					return
				default:
					t.Logf("Accept error: %v", err)
					return
				}
			}

			// Spawn doveadm exec imap with stdin/stdout connected to connection.
			homeDir := filepath.Join(mailDir, "testuser")
			cmd := exec.Command("doveadm", "-c", configPath, "exec", "imap")
			cmd.Env = []string{"USER=testuser", "HOME=" + homeDir, "PATH=" + os.Getenv("PATH")}
			cmd.Dir = baseDir
			cmd.Stdin = conn
			cmd.Stdout = conn
			cmd.Stderr = os.Stderr

			if err := cmd.Start(); err != nil {
				t.Logf("Failed to start doveadm: %v", err)
				_ = conn.Close()
				continue
			}

			go func(c net.Conn, cmd *exec.Cmd) {
				if err := cmd.Wait(); err != nil {
					select {
					case <-stopChan: // Shutdown requested.
					default:
						t.Logf("doveadm exec imap failed: %v", err)
					}
				}
				_ = c.Close()
			}(conn, cmd)
		}
	}()

	server := &DovecotServer{
		Port:       port,
		BaseDir:    baseDir,
		MailDir:    mailDir,
		ConfigPath: configPath,
		listener:   listener,
		stopChan:   stopChan,
	}

	t.Cleanup(func() {
		server.Stop()
	})

	t.Logf("Dovecot server listening on port %d, config at %s", port, configPath)

	return server
}

// Stop tells the server to — wait for it — Stop!
func (d *DovecotServer) Stop() {
	close(d.stopChan)
	if d.listener != nil {
		_ = d.listener.Close()
	}
}

// Server returns the server address (host:port).
func (d *DovecotServer) Server() string {
	return fmt.Sprintf("127.0.0.1:%d", d.Port)
}

// MakeConfig creates a config.Config for this Dovecot instance.
func (d *DovecotServer) MakeConfig(accountName string, localPath string, stateDir string) *config.Config {
	return &config.Config{
		Accounts: []config.Account{
			{
				Name:           accountName,
				Server:         d.Server(),
				Username:       "",
				PasswordCmd:    nil,
				LocalPath:      localPath,
				Plaintext:      true,
				StateDir:       stateDir,
				MaxConnections: 2,
			},
		},
	}
}
