// ImapGoose - An offlineimap clone in Go
//
// ImapGoose synchronises IMAP mailboxes to local Maildir storage using the
// offlineimap algorithm. It runs as a daemon and uses IMAP NOTIFY for
// real-time synchronisation.
package main

import (
	"context"
	"fmt"
	"log/slog"
	"os"
	"path/filepath"
	"strings"
	"time"

	"git.sr.ht/~whynothugo/ImapGoose/internal/config"
	"git.sr.ht/~whynothugo/ImapGoose/internal/daemon"
	"git.sr.ht/~whynothugo/ImapGoose/internal/index"
	"git.sr.ht/~whynothugo/ImapGoose/internal/migrate"
	"github.com/lmittmann/tint"
)

var (
	configPath     string
	verbose        bool
	migrateToV2    bool
	indexFromLocal bool
)

func printUsage() {
	fmt.Fprintf(os.Stderr, "Usage: %s [options]\n\n", os.Args[0])
	fmt.Fprintf(os.Stderr, "Options:\n")
	fmt.Fprintf(os.Stderr, "  -c <path>  path to configuration file\n")
	fmt.Fprintf(os.Stderr, "  -v         enable verbose logging\n")
	fmt.Fprintf(os.Stderr, "  -m         migrate from an older status repository format\n")
	fmt.Fprintf(os.Stderr, "  -i         index maildir into status repository using U= hints\n")
	fmt.Fprintf(os.Stderr, "  -h         show this help message\n")
}

func parseFlags() error {
	args := os.Args[1:]
Args:
	for i := 0; i < len(args); i++ {
		arg := args[i]

		if arg == "--" || !strings.HasPrefix(arg, "-") || arg == "-" {
			return fmt.Errorf("bad usage")
		}

		// Parse each character as a flag
		flags := arg[1:]
		for j, flag := range flags {
			switch flag {
			case 'c':
				if j < len(flags)-1 {
					// Handle `-cvalue` and `c=value`.
					remainder := flags[j+1:] // Remaining part after 'c'
					if strings.HasPrefix(remainder, "=") {
						configPath = remainder[1:]
					} else {
						// Handle -cvalue
						configPath = remainder
					}
					break Args
				}
				if i+1 < len(args) {
					// Handle `-c value` (value is next arg).
					i++
					configPath = args[i]
					break Args
				}
				return fmt.Errorf("-c requires a value")
			case 'v':
				verbose = true
			case 'm':
				migrateToV2 = true
			case 'i':
				indexFromLocal = true
			case 'h':
				printUsage()
				os.Exit(0)
			default:
				return fmt.Errorf("unknown flag: -%c", flag)
			}
		}
	}
	return nil
}

func main() {
	if err := parseFlags(); err != nil {
		fmt.Fprintf(os.Stderr, "Error: %v\n\n", err)
		printUsage()
		os.Exit(1)
	}

	// Setup logging.
	logLevel := slog.LevelInfo
	if verbose {
		logLevel = slog.LevelDebug
	}
	logger := slog.New(tint.NewHandler(os.Stdout, &tint.Options{
		Level:      logLevel,
		TimeFormat: time.TimeOnly,
	}))

	// Determine config path.
	cfgPath := configPath
	if cfgPath == "" {
		// Default to $XDG_CONFIG_HOME/imapgoose/config.scfg or ~/.config/imapgoose/config.scfg.
		configDir := os.Getenv("XDG_CONFIG_HOME")
		if configDir == "" {
			home, err := os.UserHomeDir()
			if err != nil {
				logger.Error("Failed to determine home directory", "error", err)
				os.Exit(1)
			}
			configDir = filepath.Join(home, ".config")
		}
		cfgPath = filepath.Join(configDir, "imapgoose", "config.scfg")
	}

	logger.Info("Loading configuration", "path", cfgPath)

	// Load configuration.
	cfg, err := config.Load(cfgPath)
	if err != nil {
		// TODO: man page imapgoose(5).
		logger.Error("Failed to load configuration", "error", err)
		fmt.Fprintf(os.Stderr, "\nConfiguration file format:\n\n")
		fmt.Fprintf(os.Stderr, "account example {\n")
		fmt.Fprintf(os.Stderr, "    server imap.example.com:993\n")
		fmt.Fprintf(os.Stderr, "    username user@example.com\n")
		fmt.Fprintf(os.Stderr, "    password your_password\n")
		fmt.Fprintf(os.Stderr, "    local-path ~/Mail/example\n")
		fmt.Fprintf(os.Stderr, "    max-connections 3  # optional, default: 3\n")
		fmt.Fprintf(os.Stderr, "}\n\n")
		fmt.Fprintf(os.Stderr, "See config.example.scfg for a complete example.\n")
		os.Exit(1)
	}

	ctx := context.Background()

	if migrateToV2 {
		logger.Info("Running migration from V1 to V2 storage format")
		for _, account := range cfg.Accounts {
			if err := migrate.MigrateAccount(ctx, account, logger); err != nil {
				logger.Error("Migration failed", "account", account.Name, "error", err)
				os.Exit(1)
			}
		}
		logger.Info("All accounts migrated successfully")
		return
	}

	if indexFromLocal {
		logger.Info("Indexing status from local Maildir")
		for _, account := range cfg.Accounts {
			if err := index.IndexAccount(ctx, account, logger); err != nil {
				logger.Error("Indexing failed", "account", account.Name, "error", err)
				os.Exit(1)
			}
		}
		logger.Info("All accounts indexed")
		return
	}

	// Create and run daemon.
	d := daemon.New(cfg, logger)

	if err := d.Run(ctx); err != nil {
		logger.Error("Daemon error", "error", err)
		os.Exit(1)
	}
}
