// FB Alpha Atari Nite Driver Module
// Based on MAME driver by Mike Balfour

// to do:
//	hook up samples?

#include "tiles_generic.h"
#include "m6502_intf.h"
#include "watchdog.h"
#include "samples.h"

static UINT8 *AllMem;
static UINT8 *MemEnd;
static UINT8 *AllRam;
static UINT8 *RamEnd;
static UINT8 *DrvM6502ROM;
static UINT8 *DrvGfxROM;
static UINT8 *DrvM6502RAM;
static UINT8 *DrvVidRAM;
static UINT8 *DrvHVCRAM;
static UINT8 *DrvCarBitmap;

static UINT32 *DrvPalette;
static UINT8 DrvRecalc;

static UINT8 crash_en;
static UINT8 crash_data_en;
static UINT8 crash_data;
static INT32 crash_timer;
static UINT8 steering_val;
static UINT8 last_steering_val;
static INT32 steering_buf;
static UINT8 ac_line;
static INT32 m_gear;
static UINT8 m_track;
static INT32 last;

static INT32 vblank;

static INT32 sound_disable;

static UINT8 DrvJoy1[8];
static UINT8 DrvJoy2f[8];
static UINT8 DrvJoy3f[8];
static UINT8 DrvDips[3];
static UINT8 DrvInputs[2];
static UINT8 DrvReset;

static struct BurnInputInfo NitedrvrInputList[] = {
	{"P1 Coin",			BIT_DIGITAL,	DrvJoy1 + 0,	"p1 coin"	},
	{"P1 Start",		BIT_DIGITAL,	DrvJoy1 + 2,	"p1 start"	},
	{"P1 Left",		    BIT_DIGITAL,	DrvJoy3f + 1,	"p1 left"	},
	{"P1 Right",		BIT_DIGITAL,	DrvJoy3f + 0,	"p1 right"	},
	{"P1 Accelerator",	BIT_DIGITAL,	DrvJoy1 + 3,	"p1 fire 1"	},
	{"P1 Gear Up",		BIT_DIGITAL,	DrvJoy2f + 0,	"p1 fire 2"	},
	{"P1 Gear Down",	BIT_DIGITAL,	DrvJoy2f + 1,	"p1 fire 3"	},
	{"P1 Novice Track",	BIT_DIGITAL,	DrvJoy1 + 4,	"p1 fire 4"	},
	{"P1 Expert Track",	BIT_DIGITAL,	DrvJoy1 + 5,	"p1 fire 5"	},
	{"P1 Pro Track",	BIT_DIGITAL,	DrvJoy1 + 6,	"p1 fire 6"	},

	{"Reset",			BIT_DIGITAL,	&DrvReset,		"reset"		},
	{"Dip A",			BIT_DIPSWITCH,	DrvDips + 0,	"dip"		},
	{"Dip B",			BIT_DIPSWITCH,	DrvDips + 1,	"dip"		},
	{"Dip C",			BIT_DIPSWITCH,	DrvDips + 2,	"dip"		},
};

STDINPUTINFO(Nitedrvr)

static struct BurnDIPInfo NitedrvrDIPList[]=
{
	{0x0b, 0xff, 0xff, 0x90, NULL					},
	{0x0c, 0xff, 0xff, 0xa0, NULL					},
	{0x0d, 0xff, 0xff, 0x00, NULL					},

	{0   , 0xfe, 0   ,    3, "Coinage"				},
	{0x0b, 0x01, 0x30, 0x30, "2 Coins 1 Credits"	},
	{0x0b, 0x01, 0x30, 0x10, "1 Coin  1 Credits"	},
	{0x0b, 0x01, 0x30, 0x00, "1 Coin  2 Credits"	},

	{0   , 0xfe, 0   ,    4, "Playing Time"			},
	{0x0b, 0x01, 0xc0, 0x00, "50"					},
	{0x0b, 0x01, 0xc0, 0x40, "75"					},
	{0x0b, 0x01, 0xc0, 0x80, "100"					},
	{0x0b, 0x01, 0xc0, 0xc0, "125"					},

	{0   , 0xfe, 0   ,    2, "Track Set"			},
	{0x0c, 0x01, 0x10, 0x00, "Normal"				},
	{0x0c, 0x01, 0x10, 0x10, "Reverse"				},

	{0   , 0xfe, 0   ,    2, "Bonus Time"			},
	{0x0c, 0x01, 0x20, 0x00, "No"					},
	{0x0c, 0x01, 0x20, 0x20, "Score = 350"			},

	{0   , 0xfe, 0   ,    2, "Service Mode"			},
	{0x0c, 0x01, 0x80, 0x00, "On"					},
	{0x0c, 0x01, 0x80, 0x80, "Off"					},

	{0   , 0xfe, 0   ,    2, "Difficult Bonus"		},
	{0x0d, 0x01, 0x20, 0x00, "Normal"				},
	{0x0d, 0x01, 0x20, 0x20, "Difficult"			},
};

STDDIPINFO(Nitedrvr)

static INT32 nitedrvr_steering()
{
	if (DrvJoy3f[0])
	{
		steering_val = 0xc0;
	}
	else if (DrvJoy3f[1])
	{
		steering_val = 0x80;
	}
	else
	{
		steering_val = 0x00;
	}

	return steering_val;
}

static UINT8 nitedrvr_in0_r(UINT8 offset)
{
	{ // gear logic
		if ((last & (1<<0)) == 0 && DrvInputs[1] & (1<<0)) {
			m_gear++;
		}
		if ((last & (1<<1)) == 0 && DrvInputs[1] & (1<<1)) {
			m_gear--;
		}
		last = DrvInputs[1];

		if (m_gear < 1) m_gear = 1;
		if (m_gear > 4) m_gear = 4;
	}

	switch (offset & 0x03)
	{
		case 0x00:
			return DrvDips[0];

		case 0x01:
			return (DrvDips[1] & ~0x40) | (vblank ? 0x40 : 0);

		case 0x02:
			if (m_gear == 1)
				return 0xe0;
			else if (m_gear == 2)
				return 0xd0;
			else if (m_gear == 3)
				return 0xb0;
			else
				return 0x70;

		case 0x03:
			return ((DrvDips[2] & 0x20) | nitedrvr_steering());
	}

	return 0xff;
}

static UINT8 nitedrvr_in1_r(UINT8 offset)
{
	INT32 port = DrvInputs[0] ^ 0x0f;

	ac_line = (ac_line + 1) % 3;

	if (port & 0x10) m_track = 0;
	else if (port & 0x20) m_track = 1;
	else if (port & 0x40) m_track = 2;

	switch (offset & 0x07)
	{
		case 0x00:
			return ((port & 0x01) << 7);

		case 0x01:
			return ((port & 0x02) << 6);

		case 0x02:
			return ((port & 0x04) << 5);

		case 0x03:
			return ((port & 0x08) << 4);

		case 0x04:
			if (m_track == 1) return 0x80; else return 0x00;

		case 0x05:
			if (m_track == 0) return 0x80; else return 0x00;

		case 0x06:
			if (ac_line == 0) return 0x80; else return 0x00;

		case 0x07:
			return 0x00;
	}

	return 0xff;
}

static void out1_write(UINT8 data)
{
//	led = BIT(data, 4);

	crash_en = data & 0x01;

	sound_disable = data & 0x02;

	if ((data & 1) == 0)
	{
		crash_data_en = 1;
		crash_data = 0x0f;

		DrvPalette[1] = 0;
		DrvPalette[0] = ~0;
	}
	if (crash_data_en) {
		BurnSamplePlay(3);
	}
}

static void nitedrvr_write(UINT16 address, UINT8 data)
{
	static UINT8 screech[2] = { 0, 0 };
	static INT32 rate_target = 0;
	static INT32 rate_adder = 0;

	if ((address & 0xfe00) == 0x0200) {
		DrvVidRAM[address & 0x7f] = data;
		return;
	}

	if ((address & 0xfe00) == 0x0400)
	{
		address &= 0x3f;

		if (address <= 0x042f) {
			DrvHVCRAM[address] = data;
			return;
		}
		BurnWatchdogWrite();
		return;
	}

	if ((address & 0xfe00) == 0x0a00) {
		{ // motor speed
			rate_target = 100 + (0xf - (data & 0xf)) * 100;
			if (rate_adder < rate_target) rate_adder += 10;
			else if (rate_adder > rate_target) rate_adder -= 10;

			if (rate_adder > rate_target) rate_adder = rate_target;

			BurnSampleSetPlaybackRate(0, rate_adder);
		}
		if (data&0x10) {
			if (!BurnSampleGetStatus(1)) {
				screech[0] = 3;
				BurnSamplePlay(1);
			}
		} else {
			if (screech[0] > 0) screech[0] --;
			if (!screech[0])
				BurnSampleStop(1);
		}
		if (data&0x20) {
			if (!BurnSampleGetStatus(2)) {
				screech[1] = 3;
				BurnSamplePlay(2);
			}
		} else {
			if (screech[1] > 0) screech[1] --;
			if (!screech[1])
				BurnSampleStop(2);
		}
		return;
	}

	if ((address & 0xfe00) == 0x0c00) {
		out1_write(data);
		return;
	}

	if ((address & 0xfc00) == 0x8400) {
		steering_val = 0;
		return;
	}
}

static UINT8 nitedrvr_read(UINT16 address)
{
	if ((address & 0xfe00) == 0x0600) {
		return nitedrvr_in0_r(address);
	}

	if ((address & 0xfe00) == 0x0800) {
		return nitedrvr_in1_r(address);
	}

	if ((address & 0xfc00) == 0x8000) {
		return DrvVidRAM[address & 0x7f];
	}

	if ((address & 0xfc00) == 0x8400) {
		steering_val = 0;
		return 0;
	}

	return 0;
}

static tilemap_callback( bg )
{
	TILE_SET_INFO(0, DrvVidRAM[offs], 0, 0);
}

static INT32 DrvDoReset(INT32 clear_mem)
{
	if (clear_mem) {
		memset (AllRam, 0, RamEnd - AllRam);
	}

	M6502Open(0);
	M6502Reset();
	BurnSampleReset();
	BurnSamplePlay(0); // engine
	M6502Close();

	BurnWatchdogReset();

	DrvPalette[1] = ~0; // white

	crash_en = 0;
	crash_data_en = 0;
	crash_data = 0;
	crash_timer = 0;
	steering_buf = 0;
	ac_line = 0;
	m_track = 0;
	steering_val = 0;
	last_steering_val = 0;
	m_gear = 1;
	last = 0;

	sound_disable = 1;

	HiscoreReset();

	return 0;
}

static INT32 MemIndex()
{
	UINT8 *Next; Next = AllMem;

	DrvM6502ROM		= Next; Next += 0x001000;

	DrvGfxROM		= Next; Next += 0x001000*4;
	DrvCarBitmap	= Next; Next += 14080;

	AllRam			= Next;

	DrvPalette		= (UINT32*)Next; Next += 0x0012 * sizeof(UINT32);

	DrvM6502RAM		= Next; Next += 0x000100;
	DrvVidRAM		= Next; Next += 0x000080;
	DrvHVCRAM		= Next; Next += 0x000080;

	RamEnd			= Next;

	MemEnd			= Next;

	return 0;
}

static const UINT8 cardata[6377] = {
	0xDD, 0xAC, 0x11, 0x1A, 0xDD, 0x27, 0x22, 0x2A, 0xDD, 0x52, 0x11, 0x12, 0xDD, 0x31, 0x22, 0x2C, 
	0xDD, 0x4B, 0x11, 0xDD, 0x36, 0x22, 0x21, 0xDD, 0x48, 0x11, 0xDD, 0x38, 0x22, 0x21, 0xDD, 0x47, 
	0x11, 0x22, 0x22, 0x3E, 0xDD, 0x32, 0x22, 0x27, 0x32, 0x22, 0x21, 0xDD, 0x43, 0x11, 0x12, 0x21, 
	0x11, 0x11, 0x22, 0x22, 0x23, 0x22, 0x42, 0xDD, 0x2F, 0x22, 0x42, 0x23, 0x72, 0x22, 0x21, 0x11, 
	0x11, 0x22, 0xDD, 0x39, 0x11, 0xCC, 0xC0, 0xDD, 0x05, 0x00, 0x02, 0x22, 0x00, 0x00, 0x22, 0x22, 
	0x23, 0xF2, 0x4A, 0xDD, 0x1C, 0x22, 0x27, 0x55, 0x57, 0xDD, 0x0F, 0x22, 0x2A, 0x42, 0x73, 0x22, 
	0x22, 0x50, 0x00, 0x02, 0x22, 0xDD, 0x06, 0x00, 0xCC, 0xC1, 0xDD, 0x20, 0x11, 0x41, 0x11, 0x11, 
	0x11, 0xCC, 0x90, 0xDD, 0x12, 0x00, 0x02, 0x22, 0x00, 0x00, 0x62, 0x22, 0x23, 0x32, 0x44, 0xDD, 
	0x1A, 0x22, 0x70, 0xDD, 0x05, 0x00, 0x08, 0xDD, 0x0D, 0x22, 0x24, 0x42, 0x33, 0x22, 0x22, 0x00, 
	0x00, 0x02, 0x22, 0xDD, 0x13, 0x00, 0x0C, 0xC1, 0x11, 0x11, 0x11, 0x41, 0xDD, 0x0E, 0x11, 0x10, 
	0x49, 0xDD, 0x17, 0x00, 0x02, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x32, 0x24, 0x72, 0xDD, 0x18, 
	0x22, 0xDD, 0x09, 0x00, 0x02, 0xDD, 0x0B, 0x22, 0x24, 0x22, 0x32, 0x22, 0x22, 0x00, 0x00, 0x02, 
	0x22, 0xDD, 0x18, 0x00, 0x40, 0xDD, 0x0E, 0x11, 0x10, 0x44, 0xDD, 0x17, 0x00, 0x06, 0x22, 0x00, 
	0x00, 0x02, 0x22, 0x22, 0x33, 0x24, 0x42, 0xDD, 0x17, 0x22, 0xDD, 0x0B, 0x00, 0x82, 0xDD, 0x0A, 
	0x22, 0x44, 0x2F, 0x32, 0x22, 0x22, 0x00, 0x00, 0x02, 0x26, 0xDD, 0x17, 0x00, 0x04, 0x40, 0xDD, 
	0x0E, 0x11, 0x19, 0x44, 0xDD, 0x18, 0x00, 0x22, 0x70, 0x00, 0x02, 0x22, 0x22, 0x33, 0x2A, 0x42, 
	0xDD, 0x16, 0x22, 0x50, 0xDD, 0x04, 0x00, 0x62, 0x22, 0x22, 0x50, 0xDD, 0x04, 0x00, 0xDD, 0x0A, 
	0x22, 0x4B, 0x23, 0x32, 0x22, 0x22, 0x00, 0x00, 0x52, 0x20, 0xDD, 0x17, 0x00, 0x04, 0x40, 0xDD, 
	0x0F, 0x11, 0xB4, 0xDD, 0x18, 0x00, 0x22, 0x20, 0x00, 0x05, 0x22, 0x22, 0x23, 0x22, 0x42, 0xDD, 
	0x15, 0x22, 0x26, 0x00, 0x00, 0x00, 0x07, 0xDD, 0x06, 0x22, 0x00, 0x00, 0x00, 0x02, 0xDD, 0x09, 
	0x22, 0x42, 0x23, 0x52, 0x22, 0x25, 0x00, 0x00, 0x22, 0x20, 0xDD, 0x17, 0x00, 0x04, 0xB1, 0xDD, 
	0x0F, 0x11, 0x04, 0xDD, 0x18, 0x00, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x23, 0x22, 0x44, 0xDD, 
	0x15, 0x22, 0x60, 0x00, 0x00, 0x02, 0xDD, 0x08, 0x22, 0x00, 0x00, 0x00, 0xDD, 0x08, 0x22, 0x2B, 
	0x42, 0x23, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x20, 0xDD, 0x17, 0x00, 0x04, 0x01, 0xDD, 0x0F, 
	0x11, 0x04, 0x40, 0xDD, 0x17, 0x00, 0x52, 0x26, 0x00, 0x00, 0x22, 0x22, 0x23, 0x32, 0xA4, 0xDD, 
	0x14, 0x22, 0x20, 0x00, 0x00, 0x05, 0xDD, 0x0A, 0x22, 0x00, 0x00, 0x07, 0xDD, 0x07, 0x22, 0x24, 
	0xA2, 0x33, 0x22, 0x22, 0x20, 0x00, 0x08, 0x22, 0x70, 0xDD, 0x17, 0x00, 0xB4, 0x01, 0xDD, 0x0F, 
	0x11, 0xC4, 0x40, 0xDD, 0x17, 0x00, 0x02, 0x22, 0x00, 0x00, 0x22, 0x22, 0x22, 0x32, 0x24, 0xE2, 
	0xDD, 0x13, 0x22, 0x20, 0x00, 0x00, 0x42, 0xDD, 0x0A, 0x22, 0x20, 0x00, 0x00, 0xDD, 0x07, 0x22, 
	0x74, 0x22, 0x35, 0x22, 0x22, 0x20, 0x00, 0x02, 0x22, 0xDD, 0x18, 0x00, 0x44, 0x01, 0xDD, 0x0F, 
	0x11, 0x10, 0x40, 0xDD, 0x17, 0x00, 0x02, 0x22, 0x00, 0x00, 0x62, 0x22, 0x22, 0x35, 0x24, 0x42, 
	0xDD, 0x13, 0x22, 0x00, 0x00, 0x04, 0xEA, 0xA2, 0xDD, 0x09, 0x22, 0x44, 0x00, 0x00, 0x02, 0xDD, 
	0x06, 0x22, 0x44, 0x22, 0x32, 0x22, 0x22, 0x60, 0x00, 0x02, 0x22, 0xDD, 0x18, 0x00, 0x40, 0xDD, 
	0x10, 0x11, 0x10, 0x40, 0xDD, 0x17, 0x00, 0x02, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x33, 0x24, 
	0x42, 0xDD, 0x12, 0x22, 0x27, 0x00, 0x00, 0x9E, 0x44, 0x44, 0xA2, 0xDD, 0x07, 0x22, 0xBE, 0xE4, 
	0x40, 0x00, 0x02, 0xDD, 0x06, 0x22, 0x44, 0x23, 0x32, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0xDD, 
	0x18, 0x00, 0x40, 0xDD, 0x10, 0x11, 0x1C, 0x49, 0xDD, 0x17, 0x00, 0x07, 0x22, 0x20, 0x00, 0x02, 
	0x22, 0x22, 0x33, 0x22, 0x44, 0xDD, 0x12, 0x22, 0x20, 0x00, 0x00, 0x2E, 0xA4, 0x47, 0x42, 0xB2, 
	0x22, 0xA4, 0xA2, 0x22, 0x22, 0x44, 0x44, 0x44, 0x40, 0x00, 0x00, 0xDD, 0x05, 0x22, 0x24, 0x42, 
	0x23, 0x32, 0x22, 0x22, 0x00, 0x00, 0x72, 0x22, 0xDD, 0x18, 0x00, 0x40, 0xDD, 0x11, 0x11, 0x94, 
	0xDD, 0x18, 0x00, 0x22, 0x20, 0x00, 0x02, 0x22, 0x22, 0x23, 0xF2, 0x44, 0xDD, 0x12, 0x22, 0x70, 
	0x00, 0x07, 0x22, 0x24, 0x44, 0x42, 0x4E, 0x44, 0x44, 0x44, 0xB2, 0xB2, 0x42, 0x4B, 0xBA, 0x22, 
	0x00, 0x00, 0xDD, 0x05, 0x22, 0x24, 0x42, 0x73, 0x22, 0x22, 0x22, 0x00, 0x00, 0x22, 0x28, 0xDD, 
	0x17, 0x00, 0x04, 0x91, 0xDD, 0x11, 0x11, 0x04, 0xDD, 0x18, 0x00, 0x22, 0x20, 0x00, 0x00, 0x22, 
	0x22, 0x23, 0x32, 0x44, 0xA2, 0xDD, 0x11, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x44, 0x74, 0xA4, 
	0x7E, 0x44, 0xA2, 0x44, 0x4B, 0x42, 0x44, 0x22, 0x22, 0x00, 0x00, 0x72, 0xDD, 0x04, 0x22, 0xE4, 
	0x42, 0x33, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x20, 0xDD, 0x17, 0x00, 0x04, 0x01, 0xDD, 0x11, 
	0x11, 0x94, 0xDD, 0x18, 0x00, 0x22, 0x28, 0x00, 0x00, 0x22, 0x22, 0x23, 0x32, 0x24, 0x42, 0xDD, 
	0x11, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x22, 0x4A, 0x42, 0x44, 0x44, 0x44, 0x74, 0x74, 0x24, 
	0xE2, 0x22, 0x22, 0x70, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x44, 0x72, 0x33, 0x22, 0x22, 0x20, 0x00, 
	0x08, 0x22, 0x20, 0xDD, 0x17, 0x00, 0x04, 0x01, 0xDD, 0x11, 0x11, 0x14, 0xB0, 0xDD, 0x17, 0x00, 
	0x22, 0x22, 0x00, 0x00, 0x22, 0x22, 0x22, 0x32, 0x24, 0x42, 0xDD, 0x11, 0x22, 0x00, 0x00, 0x52, 
	0xDD, 0x04, 0x22, 0xDD, 0x06, 0x44, 0x72, 0x22, 0x22, 0x22, 0x20, 0x00, 0x02, 0xDD, 0x04, 0x22, 
	0x44, 0x22, 0x32, 0x22, 0x22, 0x20, 0x00, 0x02, 0x22, 0x20, 0xDD, 0x17, 0x00, 0x94, 0xDD, 0x12, 
	0x11, 0x10, 0x40, 0xDD, 0x17, 0x00, 0x22, 0x22, 0x00, 0x00, 0x22, 0x22, 0x22, 0x33, 0x24, 0x44, 
	0xDD, 0x11, 0x22, 0x00, 0x00, 0x72, 0xDD, 0x04, 0x22, 0xDD, 0x05, 0x44, 0x42, 0xDD, 0x04, 0x22, 
	0x20, 0x00, 0x02, 0x22, 0x22, 0x22, 0x24, 0x44, 0x23, 0x32, 0x22, 0x22, 0x20, 0x00, 0x02, 0x22, 
	0x20, 0xDD, 0x17, 0x00, 0x40, 0xDD, 0x12, 0x11, 0x10, 0x40, 0xDD, 0x11, 0x00, 0x72, 0xDD, 0x04, 
	0x22, 0x00, 0x62, 0x22, 0x00, 0x00, 0x62, 0x22, 0x22, 0x33, 0x22, 0x44, 0xA2, 0xDD, 0x10, 0x22, 
	0x00, 0x00, 0x52, 0xDD, 0x04, 0x22, 0xDD, 0x05, 0x44, 0x42, 0xDD, 0x04, 0x22, 0x20, 0x00, 0x02, 
	0x22, 0x22, 0x22, 0x74, 0x42, 0x23, 0x32, 0x22, 0x22, 0x80, 0x00, 0x02, 0x22, 0x50, 0x02, 0xDD, 
	0x04, 0x22, 0x20, 0xDD, 0x11, 0x00, 0x40, 0xDD, 0x12, 0x11, 0x1C, 0x40, 0xDD, 0x0F, 0x00, 0x07, 
	0xDD, 0x06, 0x22, 0x00, 0x02, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x73, 0x22, 0x44, 0x42, 0xDD, 
	0x10, 0x22, 0x00, 0x00, 0x62, 0xDD, 0x04, 0x22, 0x4E, 0x44, 0x44, 0x44, 0x74, 0xDD, 0x05, 0x22, 
	0x20, 0x00, 0x02, 0x22, 0x22, 0x22, 0x44, 0x42, 0x23, 0x72, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 
	0x00, 0x02, 0xDD, 0x05, 0x22, 0x27, 0xDD, 0x10, 0x00, 0x40, 0xDD, 0x13, 0x11, 0x94, 0xDD, 0x0E, 
	0x00, 0x07, 0xDD, 0x07, 0x22, 0x00, 0x02, 0x22, 0x20, 0x00, 0x02, 0x22, 0x22, 0x23, 0x32, 0x44, 
	0x44, 0x72, 0xDD, 0x0F, 0x22, 0x00, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x24, 0xB2, 0x44, 0xE7, 0x47, 
	0xDD, 0x05, 0x22, 0x70, 0x00, 0x02, 0x22, 0x22, 0x24, 0x44, 0x42, 0x33, 0x22, 0x22, 0x22, 0x00, 
	0x00, 0x72, 0x22, 0x00, 0x02, 0xDD, 0x07, 0x22, 0xDD, 0x0E, 0x00, 0x04, 0x91, 0xDD, 0x13, 0x11, 
	0x04, 0xDD, 0x0D, 0x00, 0x05, 0x22, 0x22, 0x22, 0x20, 0xDD, 0x05, 0x00, 0x02, 0x22, 0x20, 0x00, 
	0x07, 0x22, 0x22, 0x23, 0x32, 0x24, 0x44, 0x44, 0xDD, 0x0F, 0x22, 0x00, 0x00, 0x02, 0xDD, 0x05, 
	0x22, 0x24, 0x44, 0x4B, 0xDD, 0x06, 0x22, 0x00, 0x00, 0x52, 0x22, 0x2B, 0x44, 0x44, 0x22, 0x33, 
	0x22, 0x22, 0x26, 0x00, 0x00, 0x22, 0x22, 0xDD, 0x06, 0x00, 0x72, 0x22, 0x22, 0x22, 0xDD, 0x0D, 
	0x00, 0x04, 0x01, 0xDD, 0x13, 0x11, 0x14, 0x90, 0xDD, 0x0C, 0x00, 0x05, 0x22, 0x22, 0x00, 0x00, 
	0x00, 0x65, 0x55, 0x50, 0x00, 0x06, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x23, 0x3E, 0x24, 0x44, 
	0x44, 0x44, 0xDD, 0x0E, 0x22, 0x70, 0x00, 0x06, 0xDD, 0x05, 0x22, 0x24, 0x4B, 0x44, 0xDD, 0x06, 
	0x22, 0x00, 0x00, 0x22, 0x24, 0x44, 0x44, 0x44, 0x22, 0x33, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 
	0x25, 0x00, 0x00, 0x55, 0x55, 0x60, 0x00, 0x00, 0x02, 0x22, 0x25, 0xDD, 0x0D, 0x00, 0x04, 0x01, 
	0xDD, 0x13, 0x11, 0x19, 0x40, 0xDD, 0x0C, 0x00, 0x06, 0x22, 0x00, 0x00, 0x02, 0xDD, 0x04, 0x22, 
	0x20, 0x00, 0x22, 0x25, 0x00, 0x00, 0x22, 0x22, 0x22, 0x33, 0x24, 0xDD, 0x04, 0x44, 0x4A, 0xDD, 
	0x0C, 0xAA, 0xA0, 0x00, 0x00, 0xEA, 0xDD, 0x05, 0xAA, 0xAE, 0xEB, 0x2A, 0xDD, 0x04, 0xAA, 0xA0, 
	0x00, 0x00, 0xDD, 0x05, 0x44, 0x23, 0x32, 0x22, 0x22, 0x20, 0x00, 0x05, 0x22, 0x20, 0x00, 0xDD, 
	0x05, 0x22, 0x00, 0x00, 0x02, 0x25, 0xDD, 0x0D, 0x00, 0x49, 0xDD, 0x14, 0x11, 0x10, 0x40, 0xDD, 
	0x0C, 0x00, 0x06, 0x00, 0x00, 0x52, 0xDD, 0x05, 0x22, 0x20, 0x00, 0x22, 0x22, 0x00, 0x00, 0x22, 
	0x22, 0x22, 0x33, 0x2E, 0xDD, 0x11, 0x44, 0x40, 0x00, 0x00, 0x54, 0xDD, 0x04, 0x44, 0x42, 0x24, 
	0x42, 0x24, 0xDD, 0x04, 0x44, 0x20, 0x00, 0x04, 0xDD, 0x04, 0x44, 0x4E, 0x23, 0x32, 0x22, 0x22, 
	0x70, 0x00, 0x02, 0x22, 0x20, 0x00, 0xDD, 0x06, 0x22, 0x70, 0x00, 0x06, 0xDD, 0x0D, 0x00, 0x40, 
	0xDD, 0x15, 0x11, 0x40, 0xDD, 0x0E, 0x00, 0xDD, 0x07, 0x22, 0x20, 0x00, 0x22, 0x22, 0x00, 0x00, 
	0x02, 0x22, 0x22, 0x33, 0xF2, 0xDD, 0x09, 0x44, 0x4B, 0xDD, 0x07, 0x44, 0xB4, 0x00, 0x00, 0x06, 
	0xE4, 0x44, 0x44, 0x44, 0x42, 0x4E, 0x74, 0x24, 0x44, 0x44, 0x44, 0x42, 0x00, 0x00, 0x04, 0xDD, 
	0x04, 0x44, 0x42, 0x53, 0x32, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0xDD, 0x07, 0x22, 
	0x20, 0xDD, 0x0E, 0x00, 0x49, 0xDD, 0x15, 0x11, 0x99, 0xDD, 0x0D, 0x00, 0x02, 0x22, 0x22, 0x27, 
	0xDD, 0x06, 0x00, 0x62, 0x22, 0x80, 0x00, 0x02, 0x22, 0x22, 0x23, 0x32, 0xDD, 0x08, 0x44, 0x33, 
	0x33, 0x33, 0xB4, 0x44, 0x44, 0x44, 0x43, 0x33, 0x33, 0x30, 0x00, 0x00, 0x62, 0x44, 0x44, 0x44, 
	0x32, 0x44, 0x44, 0x23, 0x44, 0x44, 0x4E, 0x20, 0x00, 0x00, 0xDD, 0x05, 0x44, 0x42, 0x33, 0x22, 
	0x22, 0x22, 0x00, 0x00, 0x82, 0x22, 0x50, 0xDD, 0x05, 0x00, 0x05, 0x22, 0x22, 0x22, 0xDD, 0x0E, 
	0x00, 0x91, 0xDD, 0x15, 0x11, 0x04, 0xDD, 0x0D, 0x00, 0x02, 0x22, 0x26, 0x00, 0x05, 0xDD, 0x04, 
	0x22, 0x00, 0x02, 0x22, 0x20, 0x00, 0x02, 0x22, 0x22, 0x23, 0x32, 0x24, 0xDD, 0x06, 0x44, 0x43, 
	0xDD, 0x04, 0x33, 0x44, 0x44, 0x44, 0x33, 0x33, 0x33, 0x30, 0x00, 0x00, 0x00, 0x24, 0x44, 0x43, 
	0x35, 0xE4, 0x44, 0x53, 0x34, 0x44, 0x26, 0x00, 0x00, 0x0B, 0xDD, 0x05, 0x44, 0x72, 0x33, 0x22, 
	0x22, 0x22, 0x00, 0x00, 0x22, 0x22, 0x00, 0x07, 0x22, 0x22, 0x22, 0x25, 0x00, 0x08, 0x22, 0x22, 
	0xDD, 0x0D, 0x00, 0x04, 0x01, 0xDD, 0x15, 0x11, 0xC4, 0xDD, 0x0D, 0x00, 0x02, 0x20, 0x00, 0x52, 
	0xDD, 0x05, 0x22, 0x00, 0x02, 0x22, 0x20, 0x00, 0x08, 0x22, 0x22, 0x23, 0x32, 0x24, 0xDD, 0x06, 
	0x44, 0xDD, 0x05, 0x33, 0x44, 0x44, 0x43, 0xDD, 0x04, 0x33, 0x00, 0x00, 0x00, 0x02, 0x2A, 0x33, 
	0x3F, 0xAB, 0xA4, 0x53, 0x33, 0x22, 0x00, 0x00, 0x00, 0x04, 0xDD, 0x05, 0x44, 0x22, 0x33, 0x22, 
	0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x00, 0x02, 0xDD, 0x05, 0x22, 0x70, 0x00, 0x22, 0xDD, 0x0D, 
	0x00, 0x04, 0x01, 0xDD, 0x15, 0x11, 0x14, 0xDD, 0x0D, 0x00, 0x08, 0x00, 0xDD, 0x07, 0x22, 0x00, 
	0x02, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 0x33, 0x2A, 0xDD, 0x06, 0x44, 0xDD, 0x05, 0x33, 
	0x34, 0x44, 0x43, 0xDD, 0x04, 0x33, 0x30, 0x00, 0x00, 0x00, 0x05, 0x22, 0x5F, 0x44, 0x44, 0xE2, 
	0x22, 0xDD, 0x04, 0x00, 0xDD, 0x05, 0x44, 0x4B, 0x2F, 0x32, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 
	0x22, 0x00, 0x02, 0xDD, 0x06, 0x22, 0x20, 0xDD, 0x0E, 0x00, 0x04, 0xDD, 0x16, 0x11, 0x10, 0x40, 
	0xDD, 0x0D, 0x00, 0xDD, 0x04, 0x22, 0xDD, 0x05, 0x00, 0x05, 0x22, 0x22, 0x00, 0x00, 0x22, 0x22, 
	0x22, 0x33, 0x22, 0xDD, 0x06, 0x44, 0xDD, 0x05, 0x33, 0x34, 0x44, 0x43, 0xDD, 0x05, 0x33, 0xDD, 
	0x06, 0x00, 0x09, 0x90, 0xDD, 0x05, 0x00, 0xDD, 0x06, 0x44, 0x42, 0x23, 0x32, 0x22, 0x22, 0x20, 
	0x00, 0x07, 0x22, 0x27, 0xDD, 0x05, 0x00, 0x07, 0x22, 0x22, 0x22, 0x20, 0xDD, 0x0D, 0x00, 0x40, 
	0xDD, 0x16, 0x11, 0x10, 0x40, 0xDD, 0x0D, 0x00, 0x22, 0x22, 0x28, 0xDD, 0x07, 0x00, 0x22, 0x22, 
	0x00, 0x00, 0x22, 0x22, 0x22, 0x33, 0x32, 0x24, 0xDD, 0x05, 0x44, 0xDD, 0x05, 0x33, 0x34, 0x44, 
	0x43, 0xDD, 0x05, 0x33, 0x44, 0xDD, 0x0B, 0x00, 0x03, 0x33, 0x34, 0xDD, 0x04, 0x44, 0x22, 0xF3, 
	0x32, 0x22, 0x22, 0x20, 0x00, 0x02, 0x22, 0x20, 0xDD, 0x07, 0x00, 0x22, 0x22, 0x20, 0xDD, 0x0D, 
	0x00, 0x40, 0xDD, 0x17, 0x11, 0xB0, 0xDD, 0x0D, 0x00, 0x22, 0x20, 0x00, 0x05, 0xDD, 0x04, 0x22, 
	0x20, 0x00, 0x22, 0x22, 0x00, 0x00, 0x82, 0x22, 0x22, 0x23, 0x32, 0x2A, 0xDD, 0x05, 0x44, 0xDD, 
	0x05, 0x33, 0x34, 0x44, 0x43, 0xDD, 0x05, 0x33, 0x44, 0x4B, 0xDD, 0x09, 0x00, 0x03, 0x33, 0x33, 
	0x3B, 0xDD, 0x04, 0x44, 0x22, 0x33, 0x22, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0xDD, 
	0x04, 0x22, 0x27, 0x00, 0x00, 0x22, 0x20, 0xDD, 0x0D, 0x00, 0x4C, 0xDD, 0x17, 0x11, 0x04, 0xDD, 
	0x0D, 0x00, 0x20, 0x00, 0x02, 0xDD, 0x05, 0x22, 0x20, 0x00, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 
	0x22, 0x23, 0x3F, 0x22, 0xA4, 0xDD, 0x04, 0x44, 0xDD, 0x05, 0x33, 0xB4, 0x44, 0x43, 0xDD, 0x04, 
	0x33, 0x3B, 0x44, 0x44, 0x4B, 0xDD, 0x07, 0x00, 0x04, 0x43, 0x33, 0x33, 0x33, 0x44, 0x44, 0x44, 
	0xB2, 0x2F, 0x33, 0x22, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0xDD, 0x06, 0x22, 0x80, 
	0x00, 0x20, 0xDD, 0x0C, 0x00, 0x04, 0x01, 0xDD, 0x17, 0x11, 0x04, 0xDD, 0x0E, 0x00, 0xDD, 0x07, 
	0x22, 0x20, 0x00, 0x22, 0x22, 0x50, 0x00, 0x02, 0x22, 0x22, 0x23, 0x33, 0x22, 0x22, 0xDD, 0x04, 
	0x44, 0x43, 0xDD, 0x04, 0x33, 0x44, 0x44, 0x44, 0xDD, 0x04, 0x33, 0x34, 0xDD, 0x04, 0x44, 0xB3, 
	0x30, 0x00, 0x00, 0x33, 0x3B, 0x44, 0x43, 0x33, 0x33, 0x33, 0x44, 0x44, 0x42, 0x22, 0x23, 0x33, 
	0x22, 0x22, 0x22, 0x00, 0x00, 0x52, 0x22, 0x20, 0x00, 0xDD, 0x07, 0x22, 0x20, 0xDD, 0x0D, 0x00, 
	0x04, 0x01, 0xDD, 0x17, 0x11, 0x19, 0x90, 0xDD, 0x0C, 0x00, 0x02, 0x22, 0x22, 0x22, 0xDD, 0x06, 
	0x00, 0x72, 0x22, 0x20, 0x00, 0x06, 0x22, 0x22, 0x2F, 0x33, 0x32, 0x22, 0x24, 0xDD, 0x04, 0x44, 
	0x33, 0x33, 0x33, 0xB4, 0x44, 0x44, 0x44, 0x43, 0x33, 0x33, 0x3B, 0xDD, 0x05, 0x44, 0x43, 0xDD, 
	0x04, 0x33, 0x34, 0x44, 0x44, 0x33, 0x33, 0x34, 0x44, 0x44, 0x22, 0x22, 0x33, 0x33, 0x22, 0x22, 
	0x20, 0x00, 0x00, 0x22, 0x22, 0x20, 0xDD, 0x05, 0x00, 0x02, 0x22, 0x22, 0x22, 0xDD, 0x0D, 0x00, 
	0x9B, 0xC1, 0xDD, 0x17, 0x11, 0x10, 0x40, 0xDD, 0x0C, 0x00, 0x02, 0x22, 0x25, 0x00, 0x00, 0x65, 
	0x22, 0x55, 0x80, 0x00, 0x02, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 0x33, 0x35, 0x22, 0x22, 
	0x24, 0xDD, 0x04, 0x44, 0x4B, 0xDD, 0x07, 0x44, 0xB4, 0xDD, 0x07, 0x44, 0xB3, 0x33, 0x33, 0x33, 
	0x44, 0x44, 0x44, 0x43, 0x33, 0x44, 0x44, 0x22, 0x22, 0x27, 0x33, 0x32, 0x22, 0x22, 0x20, 0x00, 
	0x00, 0x22, 0x22, 0x00, 0x00, 0x05, 0x52, 0x25, 0x60, 0x00, 0x08, 0x22, 0x22, 0xDD, 0x0D, 0x00, 
	0x40, 0xDD, 0x18, 0x11, 0x10, 0x40, 0xDD, 0x0C, 0x00, 0x02, 0x26, 0x00, 0x07, 0xDD, 0x05, 0x22, 
	0x60, 0x02, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 0x33, 0x33, 0x22, 0x22, 0x22, 0x2E, 0xDD, 
	0x14, 0x44, 0xB3, 0x33, 0xDD, 0x06, 0x44, 0x4E, 0x22, 0x22, 0x22, 0x23, 0x33, 0x32, 0x22, 0x22, 
	0x20, 0x00, 0x00, 0x22, 0x22, 0x00, 0x62, 0xDD, 0x05, 0x22, 0x00, 0x08, 0x22, 0xDD, 0x0D, 0x00, 
	0x40, 0xDD, 0x19, 0x11, 0x90, 0xDD, 0x0E, 0x00, 0x52, 0xDD, 0x06, 0x22, 0x50, 0x02, 0x22, 0x22, 
	0x00, 0x00, 0x52, 0x22, 0x22, 0xF3, 0x33, 0x37, 0xDD, 0x23, 0x22, 0x33, 0x33, 0xF2, 0x22, 0x22, 
	0x80, 0x00, 0x02, 0x22, 0x22, 0x00, 0x52, 0xDD, 0x06, 0x22, 0x70, 0xDD, 0x0E, 0x00, 0x91, 0xDD, 
	0x19, 0x11, 0x04, 0xDD, 0x0D, 0x00, 0x72, 0x22, 0x22, 0x22, 0x26, 0x00, 0x86, 0x72, 0x20, 0x07, 
	0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x23, 0x33, 0x33, 0xDD, 0x22, 0x22, 0x23, 0x33, 0x33, 
	0x22, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x00, 0x52, 0x26, 0x60, 0x06, 0x72, 0x22, 0x22, 
	0x22, 0x20, 0xDD, 0x0C, 0x00, 0x04, 0x01, 0xDD, 0x19, 0x11, 0x04, 0xDD, 0x0D, 0x00, 0x22, 0x22, 
	0x22, 0xDD, 0x07, 0x00, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x23, 0x33, 0x33, 0x32, 0x22, 
	0x28, 0x00, 0x00, 0xDD, 0x18, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x33, 0x33, 0x33, 0x22, 0x22, 
	0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0xDD, 0x06, 0x00, 0x02, 0x22, 0x22, 0x20, 0xDD, 0x0C, 0x00, 
	0x04, 0x01, 0xDD, 0x19, 0x11, 0x14, 0xDD, 0x0D, 0x00, 0x52, 0x27, 0x00, 0x00, 0x06, 0x22, 0x22, 
	0x22, 0x56, 0x00, 0x22, 0x22, 0x70, 0x00, 0x02, 0x22, 0x22, 0x23, 0x33, 0x33, 0x3F, 0x22, 0x22, 
	0x00, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0xDD, 0x08, 0x00, 0x02, 0x22, 0x22, 
	0x00, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x2E, 0x33, 0x33, 0x33, 0x22, 0x22, 0x22, 
	0x00, 0x00, 0x52, 0x22, 0x20, 0x08, 0x52, 0x22, 0x22, 0x26, 0x00, 0x00, 0x07, 0x22, 0x20, 0xDD, 
	0x0C, 0x00, 0x04, 0xDD, 0x1A, 0x11, 0x10, 0x40, 0xDD, 0x0C, 0x00, 0x52, 0x00, 0x00, 0x52, 0xDD, 
	0x05, 0x22, 0x00, 0x22, 0x22, 0x20, 0x00, 0x07, 0x22, 0x22, 0x22, 0x33, 0x33, 0x33, 0x22, 0x22, 
	0x20, 0x00, 0x08, 0x22, 0x22, 0x00, 0x00, 0x07, 0x22, 0x22, 0x20, 0xDD, 0x07, 0x00, 0x22, 0x22, 
	0x22, 0x00, 0x00, 0x02, 0x22, 0x28, 0x00, 0x00, 0x22, 0x22, 0x23, 0x33, 0x33, 0x32, 0x22, 0x22, 
	0x25, 0x00, 0x00, 0x22, 0x22, 0x20, 0x02, 0xDD, 0x05, 0x22, 0x70, 0x00, 0x07, 0x20, 0xDD, 0x0C, 
	0x00, 0x40, 0xDD, 0x1A, 0x11, 0x10, 0x40, 0xDD, 0x0D, 0x00, 0x07, 0xDD, 0x07, 0x22, 0x00, 0x52, 
	0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 0x33, 0x33, 0x33, 0x32, 0x22, 0x28, 0x00, 0x00, 0x72, 
	0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x26, 0xDD, 0x06, 0x00, 0x08, 0x22, 0x22, 0x20, 0x00, 0x00, 
	0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x33, 0x33, 0x33, 0x32, 0x22, 0x22, 0x20, 0x00, 0x00, 
	0x22, 0x22, 0x70, 0x02, 0xDD, 0x06, 0x22, 0x27, 0xDD, 0x0E, 0x00, 0x40, 0xDD, 0x1B, 0x11, 0x99, 
	0xDD, 0x0D, 0x00, 0xDD, 0x04, 0x22, 0x00, 0x00, 0x00, 0x65, 0x00, 0x02, 0x22, 0x25, 0x00, 0x00, 
	0x22, 0x22, 0x22, 0xF3, 0x33, 0x33, 0x37, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0x00, 
	0x02, 0x22, 0x22, 0xDD, 0x06, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x00, 0x22, 0x22, 0x00, 0x00, 
	0x02, 0x22, 0x22, 0x33, 0x33, 0x33, 0x32, 0x22, 0x22, 0x20, 0x00, 0x06, 0x22, 0x22, 0x00, 0x05, 
	0x60, 0x00, 0x00, 0x05, 0x22, 0x22, 0x22, 0x20, 0xDD, 0x0C, 0x00, 0x09, 0x99, 0xDD, 0x1B, 0x11, 
	0x04, 0xDD, 0x0D, 0x00, 0x22, 0x22, 0x20, 0xDD, 0x06, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x22, 
	0x22, 0x22, 0x23, 0x33, 0x33, 0x33, 0x22, 0x22, 0x00, 0x00, 0x08, 0x22, 0x22, 0x00, 0x00, 0x02, 
	0x22, 0x22, 0x80, 0xDD, 0x05, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x25, 0x00, 0x00, 
	0x02, 0x22, 0x23, 0x33, 0x33, 0x33, 0x22, 0x22, 0x22, 0x70, 0x00, 0x02, 0x22, 0x22, 0xDD, 0x07, 
	0x00, 0x22, 0x22, 0x20, 0xDD, 0x0C, 0x00, 0x04, 0x01, 0xDD, 0x1B, 0x11, 0x04, 0xDD, 0x0D, 0x00, 
	0x22, 0x20, 0x00, 0x05, 0xDD, 0x04, 0x22, 0x20, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 
	0x23, 0x33, 0x33, 0x33, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x00, 0x00, 0x00, 0x22, 0x22, 
	0x20, 0xDD, 0x05, 0x00, 0x22, 0x22, 0x26, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 
	0x23, 0x33, 0x33, 0x33, 0x22, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x00, 0xDD, 0x04, 0x22, 
	0x25, 0x00, 0x00, 0x22, 0x20, 0xDD, 0x0C, 0x00, 0x04, 0x01, 0xDD, 0x1B, 0x11, 0x10, 0x40, 0xDD, 
	0x0C, 0x00, 0x70, 0x00, 0xDD, 0x06, 0x22, 0x20, 0x05, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 
	0x23, 0x33, 0x33, 0x33, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 
	0x20, 0xDD, 0x05, 0x00, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 
	0x23, 0x33, 0x33, 0x33, 0x22, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x25, 0x00, 0xDD, 0x06, 0x22, 
	0x20, 0x00, 0x50, 0xDD, 0x0C, 0x00, 0xB0, 0xC1, 0xDD, 0x1A, 0x11, 0x19, 0x50, 0x40, 0xDD, 0x0D, 
	0x00, 0xDD, 0x07, 0x22, 0x20, 0x00, 0x22, 0x22, 0x50, 0x00, 0x02, 0x22, 0x22, 0x2E, 0x33, 0x33, 
	0x33, 0x32, 0x22, 0x27, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0x00, 0x02, 0x22, 0x27, 0xDD, 0x04, 
	0x00, 0x08, 0x22, 0x22, 0x60, 0x00, 0x00, 0x22, 0x22, 0x00, 0x00, 0x05, 0x22, 0x22, 0x33, 0x33, 
	0x33, 0x3F, 0x22, 0x22, 0x22, 0x00, 0x00, 0x82, 0x22, 0x20, 0x00, 0x52, 0xDD, 0x06, 0x22, 0x20, 
	0xDD, 0x0D, 0x00, 0x40, 0x50, 0xDD, 0x1A, 0x11, 0x02, 0x60, 0x90, 0xDD, 0x0C, 0x00, 0x02, 0x22, 
	0x22, 0x22, 0xDD, 0x06, 0x00, 0x22, 0x22, 0x20, 0x00, 0x08, 0x22, 0x22, 0x22, 0x33, 0x33, 0x33, 
	0x32, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0x00, 0x02, 0x22, 0x22, 0xDD, 0x04, 0x00, 
	0x02, 0x22, 0x22, 0x00, 0x00, 0x00, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x33, 0x33, 0x33, 
	0x32, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x20, 0xDD, 0x05, 0x00, 0x02, 0x22, 0x22, 0x22, 
	0xDD, 0x0D, 0x00, 0x90, 0x62, 0x81, 0xDD, 0x18, 0x11, 0x17, 0x22, 0x00, 0x09, 0xDD, 0x0C, 0x00, 
	0x02, 0x22, 0x28, 0xDD, 0x07, 0x00, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 0x23, 0x33, 
	0x33, 0x32, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0xDD, 0x04, 
	0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x33, 0x33, 
	0x33, 0x52, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x20, 0xDD, 0x07, 0x00, 0x22, 0x22, 0xDD, 
	0x0E, 0x00, 0x02, 0x22, 0xDD, 0x18, 0x11, 0x22, 0x22, 0x00, 0x04, 0xDD, 0x0C, 0x00, 0x02, 0x28, 
	0x00, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x00, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 0x25, 
	0x33, 0x33, 0x37, 0x22, 0x22, 0x80, 0x00, 0x05, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0xDD, 
	0x04, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x27, 0x00, 0x00, 0x82, 0x22, 0x22, 0x33, 
	0x33, 0x3F, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x20, 0x02, 0xDD, 0x04, 0x22, 0x00, 
	0x00, 0x00, 0x22, 0xDD, 0x0C, 0x00, 0x04, 0x00, 0x02, 0x22, 0x2C, 0xDD, 0x16, 0x11, 0x52, 0x22, 
	0x50, 0x00, 0x09, 0xDD, 0x0E, 0x00, 0x06, 0xDD, 0x06, 0x22, 0x00, 0x52, 0x22, 0x22, 0x00, 0x00, 
	0x62, 0x22, 0x22, 0x22, 0x23, 0x33, 0x33, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x00, 0x00, 
	0x02, 0x22, 0x22, 0xDD, 0x04, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0x00, 
	0x22, 0x22, 0x23, 0x33, 0x33, 0xDD, 0x04, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x50, 0x02, 0xDD, 
	0x05, 0x22, 0x25, 0xDD, 0x0E, 0x00, 0x0B, 0x00, 0x00, 0x62, 0x22, 0x71, 0xDD, 0x14, 0x11, 0x12, 
	0x22, 0x20, 0x00, 0x00, 0x00, 0x40, 0xDD, 0x0C, 0x00, 0x06, 0xDD, 0x07, 0x22, 0x00, 0x02, 0x22, 
	0x22, 0x00, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x33, 0x33, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 
	0x00, 0x00, 0x05, 0x22, 0x22, 0xDD, 0x04, 0x00, 0x02, 0x22, 0x25, 0x00, 0x00, 0x02, 0x22, 0x20, 
	0x00, 0x00, 0x22, 0x22, 0x23, 0x33, 0x32, 0xDD, 0x04, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x00, 
	0x02, 0xDD, 0x06, 0x22, 0x27, 0xDD, 0x0D, 0x00, 0xB0, 0x00, 0x00, 0x00, 0x22, 0x22, 0xDD, 0x13, 
	0x11, 0x1C, 0x22, 0x26, 0xDD, 0x04, 0x00, 0x40, 0xDD, 0x0C, 0x00, 0x52, 0x22, 0x22, 0x22, 0x25, 
	0x66, 0x00, 0x00, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x2F, 0x33, 0x22, 
	0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x00, 0x00, 0x07, 0x22, 0x22, 0x60, 0x00, 0x00, 0x00, 0x02, 
	0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x23, 0x33, 0xDD, 0x05, 0x22, 
	0x00, 0x00, 0x02, 0x22, 0x22, 0xDD, 0x04, 0x00, 0x65, 0xDD, 0x04, 0x22, 0x20, 0xDD, 0x0C, 0x00, 
	0x40, 0x00, 0x00, 0x00, 0x08, 0x22, 0x2F, 0xDD, 0x04, 0x11, 0x1C, 0x00, 0xDD, 0x05, 0x11, 0x10, 
	0x00, 0x00, 0x00, 0x11, 0x11, 0x11, 0xF2, 0x22, 0xDD, 0x12, 0x00, 0x52, 0x22, 0x22, 0xDD, 0x06, 
	0x00, 0x02, 0x22, 0x22, 0x20, 0x00, 0x02, 0xDD, 0x08, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x00, 
	0x00, 0x02, 0x22, 0x22, 0xDD, 0x04, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x20, 0x00, 
	0x00, 0xDD, 0x09, 0x22, 0x00, 0x00, 0x72, 0x22, 0x22, 0xDD, 0x06, 0x00, 0x07, 0x22, 0x22, 0x20, 
	0x08, 0xDD, 0x10, 0x00, 0x02, 0x22, 0x71, 0x11, 0x11, 0x10, 0x00, 0x00, 0x00, 0x91, 0x11, 0x11, 
	0x10, 0x0B, 0x44, 0x44, 0x00, 0x00, 0x11, 0x12, 0x22, 0xDD, 0x06, 0x00, 0x04, 0xDD, 0x09, 0x00, 
	0x06, 0x22, 0x00, 0x62, 0x20, 0x00, 0x00, 0x00, 0x52, 0x22, 0x22, 0x20, 0x06, 0x22, 0x22, 0x20, 
	0x00, 0x06, 0xDD, 0x28, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x26, 0x00, 0x22, 0x22, 0x22, 0x50, 
	0x00, 0x00, 0x00, 0x22, 0x50, 0x02, 0x25, 0xDD, 0x09, 0x00, 0x04, 0xDD, 0x06, 0x00, 0x02, 0x22, 
	0x11, 0x10, 0x00, 0x04, 0x44, 0x4B, 0x00, 0x11, 0x11, 0x00, 0xDD, 0x04, 0x44, 0x00, 0x01, 0x22, 
	0x20, 0xDD, 0x10, 0x00, 0x22, 0x22, 0x00, 0x60, 0x00, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x20, 0x00, 
	0x22, 0x22, 0x20, 0x00, 0x00, 0xDD, 0x28, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x20, 0x00, 0xDD, 
	0x05, 0x22, 0x00, 0x00, 0x00, 0x60, 0x02, 0x22, 0x20, 0xDD, 0x08, 0x00, 0x09, 0xDD, 0x07, 0x00, 
	0x22, 0x2A, 0x00, 0x0B, 0x44, 0x44, 0x44, 0x40, 0x01, 0x10, 0x04, 0xDD, 0x04, 0x44, 0x90, 0x02, 
	0x26, 0xDD, 0x0F, 0x00, 0x08, 0x22, 0x22, 0x22, 0x00, 0x00, 0x00, 0x02, 0xDD, 0x05, 0x22, 0x20, 
	0x00, 0x22, 0x22, 0x26, 0x00, 0x00, 0xDD, 0x28, 0x22, 0x20, 0x00, 0x08, 0x22, 0x22, 0x20, 0x00, 
	0xDD, 0x06, 0x22, 0x80, 0x00, 0x00, 0x02, 0x22, 0x22, 0x28, 0xDD, 0x0F, 0x00, 0x08, 0x22, 0x00, 
	0x04, 0xDD, 0x04, 0x44, 0x00, 0x10, 0xDD, 0x05, 0x44, 0x40, 0x72, 0xDD, 0x08, 0x00, 0x10, 0x40, 
	0xDD, 0x05, 0x00, 0x82, 0xDD, 0x04, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x22, 0x25, 0x60, 0x00, 
	0x00, 0x00, 0x22, 0x22, 0x22, 0x00, 0x00, 0x72, 0xDD, 0x27, 0x22, 0x60, 0x00, 0x02, 0x22, 0x22, 
	0x20, 0x00, 0x00, 0x00, 0x65, 0xDD, 0x04, 0x22, 0x00, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x80, 0xDD, 
	0x05, 0x00, 0xB0, 0x10, 0xDD, 0x07, 0x00, 0x02, 0x20, 0xDD, 0x05, 0x44, 0x40, 0x10, 0xDD, 0x05, 
	0x44, 0x42, 0x20, 0xDD, 0x06, 0x00, 0x01, 0x11, 0x1C, 0x00, 0x00, 0xDD, 0x07, 0x22, 0x25, 0x00, 
	0x00, 0x00, 0x22, 0x22, 0x26, 0xDD, 0x06, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0xDD, 0x27, 
	0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x80, 0xDD, 0x05, 0x00, 0x08, 0x22, 0x22, 0x20, 0x00, 0x00, 
	0x06, 0xDD, 0x07, 0x22, 0x20, 0x00, 0x99, 0x11, 0x11, 0x90, 0xDD, 0x06, 0x00, 0x22, 0xDD, 0x05, 
	0x44, 0x40, 0x19, 0xDD, 0x05, 0x44, 0x42, 0x20, 0xDD, 0x05, 0x00, 0xDD, 0x04, 0x11, 0x09, 0x00, 
	0xDD, 0x06, 0x22, 0x28, 0x00, 0x07, 0x20, 0x00, 0x22, 0x27, 0x00, 0x00, 0x05, 0x22, 0x22, 0x22, 
	0x60, 0x02, 0x22, 0x22, 0x50, 0x00, 0x02, 0xDD, 0x27, 0x22, 0x00, 0x00, 0x52, 0x22, 0x22, 0x00, 
	0x62, 0x22, 0x22, 0x25, 0x00, 0x00, 0x05, 0x22, 0x20, 0x00, 0x22, 0x00, 0x08, 0xDD, 0x06, 0x22, 
	0x20, 0x00, 0x01, 0x11, 0x11, 0x11, 0x10, 0xDD, 0x05, 0x00, 0x22, 0xDD, 0x05, 0x44, 0x4B, 0x10, 
	0xDD, 0x05, 0x44, 0x42, 0x20, 0x00, 0x00, 0x00, 0x0C, 0xDD, 0x05, 0x11, 0x09, 0x00, 0xDD, 0x05, 
	0x22, 0x70, 0x00, 0x06, 0x22, 0x20, 0x00, 0x22, 0x00, 0x00, 0xDD, 0x05, 0x22, 0x60, 0x02, 0x22, 
	0x22, 0x20, 0x00, 0x02, 0xDD, 0x26, 0x22, 0x25, 0x00, 0x00, 0x22, 0x22, 0x22, 0x00, 0x02, 0xDD, 
	0x04, 0x22, 0x20, 0x00, 0x07, 0x20, 0x00, 0x22, 0x25, 0x00, 0x00, 0x52, 0xDD, 0x04, 0x22, 0x20, 
	0x0B, 0x01, 0xDD, 0x04, 0x11, 0x1C, 0xDD, 0x04, 0x00, 0x72, 0xDD, 0x05, 0x44, 0x40, 0x10, 0xDD, 
	0x05, 0x44, 0x42, 0x80, 0x00, 0x00, 0x01, 0xDD, 0x06, 0x11, 0xC0, 0xDD, 0x07, 0x00, 0x06, 0x22, 
	0x22, 0x20, 0x00, 0x80, 0x00, 0xDD, 0x06, 0x22, 0x00, 0x07, 0x22, 0x22, 0x20, 0x00, 0x00, 0xDD, 
	0x26, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x27, 0x00, 0x02, 0xDD, 0x05, 0x22, 0x28, 0x00, 0x00, 
	0x00, 0x22, 0x22, 0x25, 0xDD, 0x08, 0x00, 0x91, 0xDD, 0x06, 0x11, 0x00, 0x00, 0x00, 0x02, 0xDD, 
	0x05, 0x44, 0x40, 0x19, 0x94, 0xDD, 0x04, 0x44, 0x22, 0x00, 0x00, 0xC1, 0xDD, 0x07, 0x11, 0x10, 
	0x90, 0xDD, 0x05, 0x00, 0x82, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0xDD, 0x04, 0x22, 0x56, 0xDD, 
	0x04, 0x00, 0x22, 0x22, 0x25, 0x00, 0x00, 0xDD, 0x26, 0x22, 0x20, 0x00, 0x05, 0x22, 0x22, 0x20, 
	0x00, 0x00, 0x00, 0x08, 0x52, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0xDD, 0x04, 0x22, 0x60, 0xDD, 
	0x05, 0x00, 0x90, 0xDD, 0x08, 0x11, 0x10, 0x00, 0x02, 0x24, 0xDD, 0x04, 0x44, 0xB9, 0x11, 0xC0, 
	0x44, 0x44, 0x44, 0x42, 0x25, 0x01, 0xDD, 0x0A, 0x11, 0x00, 0x06, 0xDD, 0x07, 0x22, 0x26, 0x00, 
	0x00, 0x00, 0x22, 0x22, 0x70, 0xDD, 0x06, 0x00, 0x22, 0x22, 0x22, 0x00, 0x00, 0x52, 0xDD, 0x25, 
	0x22, 0x60, 0x00, 0x02, 0x22, 0x22, 0x20, 0xDD, 0x06, 0x00, 0x72, 0x22, 0x20, 0x00, 0x00, 0x08, 
	0xDD, 0x07, 0x22, 0x25, 0x00, 0x0C, 0xDD, 0x0A, 0x11, 0x08, 0x22, 0x44, 0x44, 0x44, 0x40, 0x31, 
	0x11, 0x11, 0x0B, 0x44, 0xB2, 0x23, 0x21, 0xDD, 0x0B, 0x11, 0x00, 0x06, 0xDD, 0x06, 0x22, 0x20, 
	0x00, 0x05, 0x20, 0x00, 0x22, 0x50, 0x00, 0x00, 0x00, 0x65, 0x55, 0x56, 0x00, 0x22, 0x22, 0x22, 
	0x00, 0x00, 0x02, 0xDD, 0x0D, 0x22, 0x25, 0xDD, 0x08, 0x00, 0x05, 0xDD, 0x0E, 0x22, 0x00, 0x00, 
	0x02, 0x22, 0x22, 0x20, 0x06, 0x55, 0x55, 0x60, 0x00, 0x00, 0x00, 0x22, 0x20, 0x00, 0x27, 0x00, 
	0x00, 0xDD, 0x06, 0x22, 0x25, 0x00, 0x01, 0xDD, 0x0B, 0x11, 0x23, 0x22, 0xA4, 0x4B, 0x0B, 0x11, 
	0x11, 0x11, 0x13, 0x3C, 0x33, 0x32, 0xDD, 0x0C, 0x11, 0x10, 0x06, 0xDD, 0x05, 0x22, 0x60, 0x00, 
	0x00, 0x22, 0x20, 0x00, 0x20, 0x00, 0x00, 0xDD, 0x05, 0x22, 0x00, 0x22, 0x22, 0x22, 0x20, 0x00, 
	0x02, 0xDD, 0x0B, 0x22, 0x60, 0xDD, 0x0D, 0x00, 0x62, 0xDD, 0x0B, 0x22, 0x00, 0x00, 0x22, 0x22, 
	0x22, 0x20, 0x02, 0xDD, 0x04, 0x22, 0x20, 0x00, 0x00, 0x20, 0x00, 0x22, 0x20, 0x00, 0x00, 0x82, 
	0xDD, 0x04, 0x22, 0x25, 0x00, 0xC1, 0xDD, 0x0B, 0x11, 0x12, 0x33, 0x3A, 0xB3, 0xC1, 0xDD, 0x04, 
	0x11, 0x33, 0x3E, 0xA1, 0xDD, 0x0C, 0x11, 0x10, 0xDD, 0x07, 0x00, 0x07, 0x22, 0x22, 0x20, 0x00, 
	0x00, 0x08, 0xDD, 0x06, 0x22, 0x00, 0x02, 0x22, 0x22, 0x20, 0x00, 0x02, 0xDD, 0x09, 0x22, 0x20, 
	0xDD, 0x11, 0x00, 0xDD, 0x0A, 0x22, 0x00, 0x00, 0x22, 0x22, 0x22, 0x00, 0x02, 0xDD, 0x05, 0x22, 
	0x28, 0x00, 0x00, 0x00, 0x22, 0x22, 0x27, 0xDD, 0x08, 0x00, 0xDD, 0x0D, 0x11, 0x17, 0x33, 0x31, 
	0xDD, 0x15, 0x11, 0xDD, 0x04, 0x00, 0x06, 0x52, 0xDD, 0x04, 0x22, 0x20, 0x00, 0x00, 0xDD, 0x04, 
	0x22, 0xDD, 0x04, 0x00, 0x02, 0x22, 0x22, 0x20, 0x00, 0x00, 0xDD, 0x08, 0x22, 0xDD, 0x07, 0x00, 
	0xDD, 0x06, 0x44, 0xDD, 0x07, 0x00, 0x02, 0xDD, 0x07, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 
	0xDD, 0x04, 0x00, 0x07, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0xDD, 0x05, 0x22, 0x56, 0xDD, 0x04, 
	0x00, 0x01, 0xDD, 0x25, 0x11, 0x00, 0x00, 0xDD, 0x07, 0x22, 0x20, 0x00, 0x20, 0x00, 0x22, 0x22, 
	0x20, 0xDD, 0x05, 0x00, 0x02, 0x22, 0x22, 0x27, 0x00, 0x00, 0xDD, 0x07, 0x22, 0xDD, 0x07, 0x00, 
	0x04, 0xDD, 0x06, 0x44, 0xDD, 0x08, 0x00, 0x02, 0xDD, 0x06, 0x22, 0x20, 0x00, 0x05, 0x22, 0x22, 
	0x22, 0xDD, 0x06, 0x00, 0x72, 0x22, 0x20, 0x00, 0x50, 0x00, 0xDD, 0x07, 0x22, 0x20, 0x00, 0x01, 
	0xDD, 0x25, 0x11, 0x00, 0x00, 0xDD, 0x06, 0x22, 0x20, 0x00, 0x52, 0x20, 0x00, 0x22, 0x20, 0xDD, 
	0x07, 0x00, 0x22, 0x22, 0x22, 0x00, 0x00, 0x72, 0xDD, 0x05, 0x22, 0x80, 0xDD, 0x07, 0x00, 0xDD, 
	0x07, 0x44, 0xDD, 0x09, 0x00, 0x02, 0xDD, 0x05, 0x22, 0x70, 0x00, 0x02, 0x22, 0x22, 0x20, 0xDD, 
	0x07, 0x00, 0x72, 0x20, 0x00, 0x52, 0x70, 0x00, 0x72, 0xDD, 0x05, 0x22, 0x20, 0x00, 0x01, 0xDD, 
	0x25, 0x11, 0x10, 0x00, 0xDD, 0x05, 0x22, 0x80, 0x00, 0x82, 0x22, 0x20, 0x00, 0x20, 0x00, 0x00, 
	0x05, 0xDD, 0x04, 0x22, 0x00, 0x22, 0x22, 0x22, 0x00, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x20, 0xDD, 
	0x07, 0x00, 0x04, 0xDD, 0x07, 0x44, 0xDD, 0x0A, 0x00, 0xDD, 0x05, 0x22, 0x00, 0x00, 0x02, 0x22, 
	0x22, 0x20, 0x02, 0x22, 0x22, 0x22, 0x25, 0x00, 0x00, 0x00, 0x20, 0x00, 0x52, 0x22, 0x80, 0x00, 
	0x02, 0xDD, 0x04, 0x22, 0x20, 0x00, 0xDD, 0x26, 0x11, 0x10, 0xDD, 0x07, 0x00, 0x72, 0x22, 0x22, 
	0x20, 0x00, 0x00, 0x00, 0x52, 0xDD, 0x05, 0x22, 0x00, 0x22, 0x22, 0x22, 0x20, 0x00, 0x02, 0xDD, 
	0x04, 0x22, 0xDD, 0x08, 0x00, 0xDD, 0x08, 0x44, 0xDD, 0x0A, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x00, 
	0x00, 0x72, 0x22, 0x22, 0x20, 0x02, 0xDD, 0x05, 0x22, 0x70, 0x00, 0x00, 0x00, 0x52, 0x22, 0x22, 
	0x70, 0xDD, 0x07, 0x00, 0xDD, 0x27, 0x11, 0x00, 0x00, 0xDD, 0x08, 0x22, 0x50, 0x00, 0x00, 0xDD, 
	0x07, 0x22, 0x00, 0x02, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 0x20, 0xDD, 0x0A, 0x00, 
	0x04, 0xDD, 0x05, 0x44, 0xDD, 0x0B, 0x00, 0x22, 0x22, 0x22, 0x28, 0x00, 0x00, 0x22, 0x22, 0x22, 
	0x00, 0x02, 0xDD, 0x06, 0x22, 0x28, 0x00, 0x00, 0x82, 0xDD, 0x07, 0x22, 0x20, 0x00, 0x01, 0xDD, 
	0x27, 0x11, 0xC0, 0x00, 0xDD, 0x07, 0x22, 0x80, 0x00, 0x00, 0x02, 0x22, 0x22, 0x25, 0xDD, 0x05, 
	0x00, 0x02, 0x22, 0x22, 0x26, 0x00, 0x00, 0x22, 0x22, 0x22, 0x80, 0xDD, 0x0A, 0x00, 0x04, 0xDD, 
	0x05, 0x44, 0xDD, 0x0B, 0x00, 0x02, 0x22, 0x22, 0x20, 0x00, 0x08, 0x22, 0x22, 0x22, 0xDD, 0x05, 
	0x00, 0x06, 0x22, 0x22, 0x22, 0x00, 0x00, 0x00, 0x02, 0xDD, 0x06, 0x22, 0x20, 0x00, 0x91, 0xDD, 
	0x27, 0x11, 0x10, 0x00, 0xDD, 0x06, 0x22, 0x00, 0x00, 0x07, 0x20, 0x02, 0x22, 0x28, 0xDD, 0x06, 
	0x00, 0x02, 0x22, 0x22, 0x22, 0x00, 0x00, 0x72, 0x22, 0x22, 0xDD, 0x0B, 0x00, 0x04, 0xDD, 0x04, 
	0x44, 0x4B, 0xDD, 0x0B, 0x00, 0x02, 0x22, 0x22, 0x20, 0x00, 0x02, 0x22, 0x22, 0x22, 0xDD, 0x07, 
	0x00, 0x22, 0x22, 0x00, 0x22, 0x00, 0x00, 0x02, 0xDD, 0x05, 0x22, 0x20, 0x00, 0xDD, 0x29, 0x11, 
	0x00, 0x52, 0x22, 0x22, 0x25, 0x00, 0x00, 0x00, 0x06, 0x22, 0x20, 0x02, 0x28, 0x00, 0x00, 0xDD, 
	0x04, 0x22, 0x20, 0x07, 0x22, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0xDD, 0x0B, 0x00, 0x04, 
	0xDD, 0x04, 0x44, 0x49, 0xDD, 0x0B, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 0x22, 
	0x00, 0xDD, 0x04, 0x22, 0x20, 0x00, 0x00, 0x22, 0x00, 0x22, 0x25, 0x00, 0x00, 0x00, 0x06, 0x22, 
	0x22, 0x22, 0x50, 0x09, 0xDD, 0x29, 0x11, 0xDD, 0x07, 0x00, 0x62, 0x22, 0x22, 0x20, 0x00, 0x00, 
	0x02, 0xDD, 0x05, 0x22, 0x20, 0x00, 0x22, 0x22, 0x22, 0x20, 0x00, 0x02, 0x22, 0x22, 0xDD, 0x0B, 
	0x00, 0x04, 0xDD, 0x04, 0x44, 0x40, 0xDD, 0x0B, 0x00, 0x02, 0x22, 0x22, 0x00, 0x00, 0x22, 0x22, 
	0x22, 0x20, 0x00, 0xDD, 0x06, 0x22, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x50, 0xDD, 0x06, 0x00, 
	0x01, 0xDD, 0x29, 0x11, 0xC0, 0x00, 0x02, 0xDD, 0x07, 0x22, 0x20, 0x00, 0x02, 0xDD, 0x04, 0x22, 
	0x25, 0x52, 0x20, 0x00, 0x22, 0x22, 0x22, 0x20, 0x00, 0x07, 0x22, 0x22, 0xDD, 0x0B, 0x00, 0x04, 
	0xDD, 0x04, 0x44, 0x40, 0xDD, 0x0B, 0x00, 0x02, 0x22, 0x27, 0x00, 0x00, 0x22, 0x22, 0x22, 0x20, 
	0x00, 0x22, 0x55, 0xDD, 0x05, 0x22, 0x00, 0x00, 0xDD, 0x08, 0x22, 0x00, 0x00, 0x91, 0xDD, 0x29, 
	0x11, 0x10, 0x00, 0x02, 0xDD, 0x06, 0x22, 0x20, 0x00, 0x60, 0x02, 0x22, 0x22, 0x50, 0xDD, 0x05, 
	0x00, 0x52, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x80, 0xDD, 0x0A, 0x00, 0x09, 0xDD, 0x04, 
	0x44, 0x40, 0xDD, 0x0B, 0x00, 0x02, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 0x22, 0x50, 0xDD, 0x05, 
	0x00, 0x62, 0x22, 0x22, 0x00, 0x00, 0x00, 0xDD, 0x07, 0x22, 0x00, 0x00, 0xDD, 0x2B, 0x11, 0x00, 
	0x02, 0xDD, 0x05, 0x22, 0x70, 0x00, 0x02, 0x20, 0x02, 0x22, 0x60, 0xDD, 0x06, 0x00, 0x02, 0x22, 
	0x22, 0x22, 0x00, 0x00, 0x22, 0x22, 0x20, 0xDD, 0x0B, 0x00, 0xDD, 0x04, 0x44, 0x40, 0xDD, 0x0B, 
	0x00, 0x22, 0x22, 0x20, 0x00, 0x05, 0x22, 0x22, 0x22, 0xDD, 0x07, 0x00, 0x82, 0x22, 0x00, 0x22, 
	0x00, 0x00, 0x52, 0xDD, 0x05, 0x22, 0x00, 0x0C, 0xDD, 0x2B, 0x11, 0x00, 0x00, 0x65, 0x55, 0x58, 
	0x00, 0x00, 0x00, 0x02, 0x22, 0x20, 0x02, 0xDD, 0x08, 0x00, 0x02, 0x22, 0x22, 0x22, 0x00, 0x00, 
	0x02, 0x22, 0x22, 0xDD, 0x0B, 0x00, 0xDD, 0x04, 0x44, 0x40, 0xDD, 0x0A, 0x00, 0x02, 0x22, 0x22, 
	0x00, 0x00, 0x02, 0x22, 0x22, 0x22, 0xDD, 0x08, 0x00, 0x02, 0x00, 0x22, 0x22, 0xDD, 0x04, 0x00, 
	0x55, 0x55, 0x68, 0x00, 0x01, 0xDD, 0x2B, 0x11, 0x10, 0xDD, 0x06, 0x00, 0x52, 0x22, 0x22, 0x20, 
	0x00, 0x00, 0x00, 0x03, 0x33, 0x33, 0x33, 0x00, 0x00, 0x08, 0x22, 0x22, 0x22, 0x60, 0x00, 0x02, 
	0x22, 0x22, 0x20, 0xDD, 0x0A, 0x00, 0xDD, 0x04, 0x44, 0x40, 0xDD, 0x0A, 0x00, 0x22, 0x22, 0x22, 
	0x00, 0x00, 0x82, 0x22, 0x22, 0x26, 0x00, 0x00, 0x03, 0x33, 0x33, 0x33, 0xDD, 0x04, 0x00, 0x22, 
	0x22, 0x22, 0x78, 0xDD, 0x06, 0x00, 0xDD, 0x2C, 0x11, 0x1C, 0x00, 0x06, 0xDD, 0x07, 0x22, 0x20, 
	0x00, 0x00, 0x03, 0xDD, 0x05, 0x33, 0x00, 0x00, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 
	0x22, 0x80, 0xDD, 0x09, 0x00, 0xDD, 0x04, 0x44, 0xB0, 0xDD, 0x09, 0x00, 0x02, 0x22, 0x22, 0x20, 
	0x00, 0x00, 0x22, 0x22, 0x22, 0x20, 0x00, 0x03, 0xDD, 0x05, 0x33, 0x00, 0x00, 0x00, 0xDD, 0x07, 
	0x22, 0x26, 0x00, 0x09, 0xDD, 0x2D, 0x11, 0x00, 0x06, 0xDD, 0x06, 0x22, 0x20, 0x00, 0x00, 0x03, 
	0xDD, 0x06, 0x33, 0x00, 0x00, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0xDD, 0x04, 0x22, 0xDD, 0x09, 
	0x00, 0xDD, 0x04, 0x44, 0xDD, 0x09, 0x00, 0x02, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0x22, 0x22, 
	0x22, 0x20, 0x00, 0x03, 0xDD, 0x06, 0x33, 0x00, 0x00, 0x00, 0xDD, 0x06, 0x22, 0x26, 0x00, 0x01, 
	0xDD, 0x2D, 0x11, 0x90, 0x06, 0xDD, 0x05, 0x22, 0x50, 0x00, 0x00, 0x00, 0xDD, 0x07, 0x33, 0x30, 
	0x00, 0x02, 0x22, 0x22, 0x22, 0x00, 0x00, 0xDD, 0x05, 0x22, 0xDD, 0x08, 0x00, 0xDD, 0x04, 0x44, 
	0xDD, 0x08, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x20, 0x00, 0x02, 0x22, 0x22, 0x22, 0x80, 0x00, 0xDD, 
	0x07, 0x33, 0x30, 0x00, 0x00, 0x00, 0x52, 0xDD, 0x04, 0x22, 0x26, 0x00, 0x01, 0xDD, 0x2D, 0x11, 
	0x10, 0xDD, 0x08, 0x00, 0x03, 0xDD, 0x08, 0x33, 0x30, 0x00, 0x02, 0x22, 0x22, 0x22, 0x00, 0x00, 
	0x02, 0xDD, 0x05, 0x22, 0x20, 0xDD, 0x11, 0x00, 0xDD, 0x06, 0x22, 0x00, 0x00, 0x02, 0x22, 0x22, 
	0x22, 0x00, 0x00, 0xDD, 0x09, 0x33, 0xDD, 0x09, 0x00, 0xDD, 0x2F, 0x11, 0xDD, 0x07, 0x00, 0x03, 
	0xDD, 0x09, 0x33, 0x30, 0x00, 0x02, 0x22, 0x22, 0x22, 0x20, 0x00, 0x08, 0xDD, 0x07, 0x22, 0x60, 
	0xDD, 0x0D, 0x00, 0x62, 0xDD, 0x06, 0x22, 0x20, 0x00, 0x00, 0x72, 0x22, 0x22, 0x22, 0x00, 0x00, 
	0xDD, 0x0A, 0x33, 0xDD, 0x07, 0x00, 0x01, 0xDD, 0x2F, 0x11, 0x10, 0xDD, 0x05, 0x00, 0x93, 0xDD, 
	0x0B, 0x33, 0x00, 0x08, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0x52, 0xDD, 0x08, 0x22, 0x25, 0xDD, 
	0x08, 0x00, 0x05, 0xDD, 0x09, 0x22, 0x50, 0x00, 0x00, 0x22, 0x22, 0x22, 0x26, 0x00, 0x03, 0xDD, 
	0x0B, 0x33, 0x30, 0xDD, 0x05, 0x00, 0xDD, 0x30, 0x11, 0x10, 0x00, 0x00, 0x00, 0xDD, 0x0E, 0x33, 
	0x00, 0x00, 0x22, 0x22, 0x22, 0x28, 0x00, 0x00, 0x05, 0xDD, 0x1A, 0x22, 0x25, 0x00, 0x00, 0x00, 
	0x22, 0x22, 0x22, 0x20, 0x00, 0x03, 0xDD, 0x0D, 0x33, 0x30, 0x00, 0x00, 0x00, 0xDD, 0x31, 0x11, 
	0x00, 0x00, 0x00, 0x03, 0xDD, 0x0D, 0x33, 0x90, 0x00, 0x72, 0x22, 0x22, 0x22, 0x00, 0x00, 0x00, 
	0x02, 0xDD, 0x19, 0x22, 0x00, 0x00, 0x00, 0x02, 0x22, 0x22, 0x22, 0x70, 0x00, 0x03, 0xDD, 0x0D, 
	0x33, 0x00, 0x00, 0x00, 0x01, 0xDD, 0x31, 0x11, 0x10, 0x00, 0x00, 0x03, 0xDD, 0x0D, 0x33, 0x30, 
	0x00, 0x02, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0x00, 0xDD, 0x18, 0x22, 0x50, 0x00, 0x00, 0x00, 
	0x72, 0x22, 0x22, 0x22, 0x00, 0x00, 0xDD, 0x0E, 0x33, 0x00, 0x00, 0x00, 0xDD, 0x33, 0x11, 0x00, 
	0x00, 0x00, 0xDD, 0x0D, 0x33, 0x30, 0x00, 0x02, 0x22, 0x22, 0x22, 0x20, 0xDD, 0x04, 0x00, 0x72, 
	0xDD, 0x15, 0x22, 0x80, 0xDD, 0x04, 0x00, 0xDD, 0x04, 0x22, 0x00, 0x00, 0xDD, 0x0D, 0x33, 0x30, 
	0x00, 0x00, 0x01, 0xDD, 0x33, 0x11, 0x10, 0x00, 0x00, 0x03, 0xDD, 0x0D, 0x33, 0x00, 0x00, 0xDD, 
	0x04, 0x22, 0xDD, 0x1E, 0x00, 0x02, 0x22, 0x22, 0x22, 0x28, 0x00, 0x03, 0xDD, 0x0D, 0x33, 0x00, 
	0x00, 0x00, 0xDD, 0x35, 0x11, 0x00, 0x00, 0x00, 0xDD, 0x0D, 0x33, 0x00, 0x00, 0xDD, 0x04, 0x22, 
	0xDD, 0x1E, 0x00, 0x02, 0x22, 0x22, 0x22, 0x20, 0x00, 0x03, 0xDD, 0x0C, 0x33, 0x30, 0x00, 0x00, 
	0x01, 0xDD, 0x35, 0x11, 0x19, 0x00, 0x00, 0x00, 0xDD, 0x0C, 0x33, 0x00, 0x00, 0x62, 0x22, 0x22, 
	0x22, 0x20, 0xDD, 0x1D, 0x00, 0xDD, 0x04, 0x22, 0x50, 0x00, 0x03, 0xDD, 0x0B, 0x33, 0x30, 0x00, 
	0x00, 0x00, 0xDD, 0x37, 0x11, 0x10, 0x00, 0x00, 0x00, 0xDD, 0x0B, 0x33, 0x30, 0x00, 0x02, 0xDD, 
	0x04, 0x22, 0xDD, 0x1C, 0x00, 0x02, 0xDD, 0x04, 0x22, 0x00, 0x00, 0xDD, 0x0B, 0x33, 0x30, 0x00, 
	0x00, 0x00, 0xDD, 0x39, 0x11, 0x00, 0x00, 0x00, 0x03, 0xDD, 0x0A, 0x33, 0x30, 0x00, 0x02, 0xDD, 
	0x04, 0x22, 0x70, 0xDD, 0x1B, 0x00, 0x52, 0xDD, 0x04, 0x22, 0x00, 0x00, 0xDD, 0x0B, 0x33, 0x00, 
	0x00, 0x00, 0x01, 0xDD, 0x3A, 0x11, 0xDD, 0x04, 0x00, 0x03, 0xDD, 0x08, 0x33, 0x30, 0x00, 0x05, 
	0xDD, 0x04, 0x22, 0x29, 0xDD, 0x04, 0xCC, 0xDD, 0x15, 0x11, 0x1C, 0xCC, 0xDD, 0x04, 0x22, 0x27, 
	0x00, 0x00, 0xDD, 0x09, 0x33, 0xDD, 0x04, 0x00, 0x09, 0xDD, 0x3C, 0x11, 0xDD, 0x0F, 0x00, 0x22, 
	0x22, 0x22, 0x2C, 0xDD, 0x1D, 0x11, 0x22, 0x22, 0x22, 0x20, 0xDD, 0x0E, 0x00, 0x01, 0xDD, 0x3E, 
	0x11, 0xDD, 0x0E, 0x00, 0x72, 0x22, 0x22, 0xDD, 0x1E, 0x11, 0x12, 0x22, 0x22, 0x20, 0xDD, 0x0D, 
	0x00, 0x01, 0xDD, 0x40, 0x11, 0xC0, 0xDD, 0x0C, 0x00, 0x02, 0x22, 0xDD, 0x20, 0x11, 0x12, 0x22, 
	0xDD, 0x0D, 0x00, 0xC1, 0xDD, 0x43, 0x11, 0xDD, 0x0B, 0x00, 0x02, 0xDD, 0x22, 0x11, 0x12, 0xDD, 
	0x0B, 0x00, 0x01, 0xDD, 0x22, 0x11, 0xDD, 0xDD, 0xDD
};

static INT32 DrvGfxDecode()
{
	INT32 Plane[1] = { 0 };
	INT32 XOffs[16] = { 0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7 /*STEP8(0,1)*/ };
	INT32 YOffs[16] = { 0,0,8,8,16,16,24,24,32,32,40,40,48,48,56,56 /*STEP8(0,8)*/ };

	UINT8 *tmp = (UINT8*)BurnMalloc(0x200);
	if (tmp == NULL) {
		return 1;
	}

	memcpy (tmp, DrvGfxROM, 0x200);

	GfxDecode(0x040, 1, 8*2, 8*2, Plane, XOffs, YOffs, 0x040, tmp, DrvGfxROM);

	BurnFree(tmp);

	return 0;
}

static void DrvExpandCarData() // very basic, very custom, rle compression
{
	INT32 i, j = 0, k = 0;
	while (1)
	{
		if (cardata[j] == 0xdd) {
			if (cardata[j+1] == 0xdd && cardata[j+2] == 0xdd) break;
			j++;
			INT32 size = cardata[j++];
			for (i = 0; i < size; i++) {
				DrvCarBitmap[k++] = cardata[j];
			}
			j++;
		} else {
			DrvCarBitmap[k++] = cardata[j++];
		}
	}
	
#if 0 // compress
	int i,j,k,r;
	for (j = 0, r = 0; j < (110*(256/2)); j++)
	{
		for (k = 1; k < 256; k++)
		{
			if (cardata[j+k] == cardata[j]) {
				continue;
			} else {
				break;
			}
		}
		if (k > 3) {
			output[r++] = 0xdd;
			output[r++] = k;
			output[r++] = cardata[j];
			j+=k-1;
		} else {
			output[r++] = cardata[j];
		}
	}
		
	output[r+0] = 0xdd;
	output[r+1] = 0xdd;
	output[r+2] = 0xdd;
#endif
}

static INT32 DrvInit()
{
	BurnSetRefreshRate(57.00);

	BurnAllocMemIndex();

	{
		if (BurnLoadRom(DrvM6502ROM + 0x0000,  0, 1)) return 1;
		if (BurnLoadRom(DrvM6502ROM + 0x0800,  1, 1)) return 1;

		if (BurnLoadRom(DrvGfxROM   + 0x0000,  2, 1)) return 1;

		DrvGfxDecode();
		DrvExpandCarData();
	}

	M6502Init(0, TYPE_M6502);
	M6502Open(0);
	M6502MapMemory(DrvM6502RAM,		0x0000, 0x00ff, MAP_RAM);
	M6502MapMemory(DrvM6502RAM,		0x0100, 0x01ff, MAP_RAM);
	M6502MapMemory(DrvM6502ROM,		0x9000, 0x9fff, MAP_ROM);
	M6502MapMemory(DrvM6502ROM + 0x0f00,	0xff00, 0xffff, MAP_ROM); // vectors
	M6502SetWriteHandler(nitedrvr_write);
	M6502SetReadHandler(nitedrvr_read);
	M6502Close();

	BurnWatchdogInit(DrvDoReset, 180); // ?

	GenericTilesInit();
	GenericTilemapInit(0, TILEMAP_SCAN_ROWS, bg_map_callback, 16, 16, 32, 4);
	GenericTilemapSetGfx(0, DrvGfxROM, 1, 16, 16, 0x1000*4, 0, 0);

	BurnSampleInit(0);
	BurnSampleSetAllRoutesAllSamples(1.00, BURN_SND_ROUTE_BOTH);
	BurnSampleSetBuffered(M6502TotalCycles, 1008000);

	DrvDoReset(1);

	return 0;
}

static INT32 DrvExit()
{
	GenericTilesExit();

	M6502Exit();

	BurnSampleExit();

	BurnFreeMemIndex();

	return 0;
}

static UINT8 paldata[16*3] = {
	0x08, 0x7F, 0xC8, 
	0xFF, 0x00, 0xFF, 
	0x00, 0x00, 0x00, 
	0x4A, 0x96, 0xA6, 
	0xAB, 0xA8, 0xB6, 
	0x04, 0x2E, 0x48, 
	0x04, 0x51, 0x7F, 
	0x08, 0x16, 0x20, 
	0x06, 0x6A, 0xA7, 
	0x46, 0x8A, 0xC8, 
	0x77, 0x68, 0x7D, 
	0x89, 0x9D, 0xB4, 
	0xAA, 0x66, 0xE7, 
	0xFF, 0x00, 0xFF, 
	0x3C, 0x3E, 0x46, 
	0x31, 0x63, 0x6E
};

static void DrvPaletteUpdate()
{
	// first two colors (b & w) palette updated in routines
	for (INT32 i = 2; i < 18; i++)
	{
		UINT8 *p = paldata + (i - 2) * 3;
		DrvPalette[i] = BurnHighCol(p[2], p[1], p[0], 0);
	}
}

static void draw_car() // bitmap image is 256*110
{
	for (INT32 y = 0; y < 110; y++)
	{
		UINT16 *dst = pTransDraw + (((nScreenHeight - 110) + (109-y)) * nScreenWidth) + ((nScreenWidth / 2) - (256 / 2));
		UINT8 *src = DrvCarBitmap + (y * (256 / 2));

		for (INT32 x = 0; x < 256; x++)
		{
			INT32 pxl = ((src[x/2] >> ((x & 1) * 4)) & 15);
			if (pxl != 1) dst[x^1] = 2 + pxl;
		}
	}
}

static void draw_roadway()
{
	for (INT32 roadway = 0; roadway < 16; roadway++)
	{
		INT32 bx = DrvHVCRAM[roadway];
		INT32 by = DrvHVCRAM[roadway + 16];
		INT32 ex = bx + ((DrvHVCRAM[roadway + 32] & 0xf0) >> 4);
		INT32 ey = by + (16 - (DrvHVCRAM[roadway + 32] & 0x0f));

		by *= 2;
		ey *= 2;
		bx *= 2;
		ex *= 2;
		
		if (ey > nScreenHeight) ey = nScreenHeight;
		if (ex > nScreenWidth) ex = nScreenWidth;

		for (INT32 y = by; y < ey; y++)
		{
			UINT16 *dst = pTransDraw + (y * nScreenWidth);

			for (INT32 x = bx; x < ex; x++)
			{
				dst[x] = 1;
			}
		}
	}
}

static INT32 DrvDraw()
{
	if (DrvRecalc) {
		DrvPaletteUpdate();
	}
	
	BurnTransferClear();
	
	GenericTilesSetClip(-1, -1, -1, 31*2);
	GenericTilemapDraw(0, pTransDraw, 0);
	GenericTilesClearClip();

	draw_roadway();
	
	draw_car();

	BurnTransferCopy(DrvPalette);

	return 0;
}

static void crash_toggle_callback()
{
	if (crash_en && crash_data_en)
	{
		crash_data--;

		if (!crash_data)
			crash_data_en = 0;    // Done counting?

		DrvPalette[0 ^ (crash_data & 0x01)] = 0;
		DrvPalette[1 ^ (crash_data & 0x01)] = ~0;
	}
}

static INT32 DrvFrame()
{
	BurnWatchdogUpdate();

	if (DrvReset) {
		DrvDoReset(1);
	}

	M6502NewFrame();

	{
		DrvInputs[0] = 0;
		DrvInputs[1] = 0;

		for (INT32 i = 0; i < 8; i++) {
			DrvInputs[0] ^= (DrvJoy1[i] & 1) << i;
			DrvInputs[1] ^= (DrvJoy2f[i] & 1) << i;
		}
	}
	INT32 nInterleave = 128; // 256/2
	INT32 nCyclesTotal[1] = { 1008000 / 57 };
	INT32 nCyclesDone[1] = { 0 };

	M6502Open(0);
	vblank = 0;

	for (INT32 i = 0; i < nInterleave; i++)
	{
		CPU_RUN(0, M6502);

		if (i == 240/2) {
			vblank = 1;
			M6502SetIRQLine(0, CPU_IRQSTATUS_HOLD);
		}
	}

	M6502Close();

	if (pBurnSoundOut) {
		BurnSampleRender(pBurnSoundOut, nBurnSoundLen);

		if (sound_disable) {
			BurnSoundClear();
		}
	}

	crash_timer++;
	if (crash_timer == 7) {
		crash_toggle_callback();
		crash_timer = 0;
	}

	if (pBurnDraw) {
		DrvDraw();
	}

	return 0;
}

static INT32 DrvScan(INT32 nAction, INT32 *pnMin)
{
	struct BurnArea ba;

	if (pnMin) {
		*pnMin = 0x029702;
	}

	if (nAction & ACB_VOLATILE) {
		memset(&ba, 0, sizeof(ba));

		ba.Data	  = AllRam;
		ba.nLen	  = RamEnd - AllRam;
		ba.szName = "All Ram";
		BurnAcb(&ba);

		M6502Scan(nAction);
		BurnWatchdogScan(nAction);;

		SCAN_VAR(crash_en);
		SCAN_VAR(crash_data_en);
		SCAN_VAR(crash_data);
		SCAN_VAR(crash_timer);
		SCAN_VAR(steering_val);
		SCAN_VAR(last_steering_val);
		SCAN_VAR(steering_buf);
		SCAN_VAR(ac_line);
		SCAN_VAR(m_gear);
		SCAN_VAR(m_track);
		SCAN_VAR(sound_disable);
		SCAN_VAR(last);
	}

	return 0;
}

static struct BurnSampleInfo nitedrvrSampleDesc[] = {
#if !defined ROM_VERIFY
	{ "engine", 		SAMPLE_AUTOLOOP },
	{ "screech1", 		SAMPLE_NOLOOP },
	{ "screech2", 		SAMPLE_NOLOOP },
	{ "crash", 		    SAMPLE_NOLOOP },
#endif
	{ "", 0 }
};

STD_SAMPLE_PICK(nitedrvr)
STD_SAMPLE_FN(nitedrvr)


// Night Driver

static struct BurnRomInfo nitedrvrRomDesc[] = {
	{ "006569-01.d2",	0x0800, 0x7afa7542, 1 | BRF_PRG | BRF_ESS }, //  0 M6502 Code
	{ "006570-01.f2",	0x0800, 0xbf5d77b1, 1 | BRF_PRG | BRF_ESS }, //  1

	{ "006568-01.p2",	0x0200, 0xf80d8889, 2 | BRF_GRA },           //  2 Tile Graphics

	{ "006559-01.h7",	0x0100, 0x5a8d0e42, 0 | BRF_OPT },           //  3 Sync PROM
};

STD_ROM_PICK(nitedrvr)
STD_ROM_FN(nitedrvr)

struct BurnDriver BurnDrvNitedrvr = {
	"nitedrvr", NULL, NULL, "nitedrvr", "1976",
	"Night Driver\0", NULL, "Atari", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HISCORE_SUPPORTED, 2, HARDWARE_MISC_PRE90S, GBF_RACING, 0,
	NULL, nitedrvrRomInfo, nitedrvrRomName, NULL, NULL, nitedrvrSampleInfo, nitedrvrSampleName, NitedrvrInputInfo, NitedrvrDIPInfo,
	DrvInit, DrvExit, DrvFrame, DrvDraw, DrvScan, &DrvRecalc, 0x12,
	512, 512, 4, 3
};
