/**
 * Copyright (c) 2010-2017, Grill Balázs, IncQueryLabs
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.testing.core.coverage;

import com.google.common.base.Supplier;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import java.util.Collection;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.eclipse.viatra.query.runtime.api.AdvancedViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.api.scope.QueryScope;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.runtime.matchers.backend.IQueryBackend;
import org.eclipse.viatra.query.runtime.matchers.psystem.PBody;
import org.eclipse.viatra.query.runtime.matchers.psystem.PConstraint;
import org.eclipse.viatra.query.runtime.matchers.psystem.PTraceable;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQueries;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.runtime.matchers.psystem.rewriters.IPTraceableTraceProvider;
import org.eclipse.viatra.query.runtime.rete.matcher.ReteBackendFactory;
import org.eclipse.viatra.query.runtime.rete.matcher.ReteEngine;
import org.eclipse.viatra.query.runtime.rete.network.Node;
import org.eclipse.viatra.query.runtime.rete.traceability.CompiledQuery;
import org.eclipse.viatra.query.runtime.rete.traceability.CompiledSubPlan;
import org.eclipse.viatra.query.runtime.rete.traceability.RecipeTraceInfo;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This utility class can determine trace connection between Nodes in a Rete network and elements in a PQuery model.
 * @since 1.6
 */
@SuppressWarnings("all")
public class ReteNetworkTrace {
  private final ReteEngine reteEngine;

  private final EMFScope scope;

  private final Multimap<PTraceable, Node> traceableToNodeMap = Multimaps.<PTraceable, Node>newSetMultimap(CollectionLiterals.<PTraceable, Collection<Node>>newHashMap(), ((Supplier<Set<Node>>) () -> {
    return CollectionLiterals.<Node>newHashSet();
  }));

  private final IPTraceableTraceProvider traceProvider;

  private void updateMapWithCanonicalTraceable(final PTraceable derivedTraceable, final Node node, final IPTraceableTraceProvider traceProvider, final Multimap<PTraceable, Node> traceableToNodeMap) {
    final Consumer<PTraceable> _function = (PTraceable traceable) -> {
      traceableToNodeMap.put(traceable, node);
    };
    traceProvider.getCanonicalTraceables(derivedTraceable).forEach(_function);
  }

  /**
   * @throws ViatraQueryRuntimeException
   */
  public ReteNetworkTrace(final ViatraQueryMatcher<?> matcher, final IPTraceableTraceProvider traceProvider) {
    ViatraQueryEngine _engine = matcher.getEngine();
    IQueryBackend _queryBackend = ((AdvancedViatraQueryEngine) _engine).getQueryBackend(
      ReteBackendFactory.INSTANCE);
    this.reteEngine = ((ReteEngine) _queryBackend);
    QueryScope _scope = matcher.getEngine().getScope();
    this.scope = ((EMFScope) _scope);
    this.traceProvider = traceProvider;
    final Consumer<RecipeTraceInfo> _function = (RecipeTraceInfo recipeTrace) -> {
      final Node node = recipeTrace.getNode();
      boolean _matched = false;
      if (recipeTrace instanceof CompiledSubPlan) {
        _matched=true;
        final Consumer<PConstraint> _function_1 = (PConstraint it) -> {
          this.updateMapWithCanonicalTraceable(it, node, traceProvider, this.traceableToNodeMap);
        };
        ((CompiledSubPlan)recipeTrace).getSubPlan().getAllEnforcedConstraints().forEach(_function_1);
      }
      if (!_matched) {
        if (recipeTrace instanceof CompiledQuery) {
          _matched=true;
          this.updateMapWithCanonicalTraceable(((CompiledQuery)recipeTrace).getQuery(), node, traceProvider, this.traceableToNodeMap);
          final BiConsumer<PBody, RecipeTraceInfo> _function_1 = (PBody derivedBody, RecipeTraceInfo parentRecipeTrace) -> {
            this.updateMapWithCanonicalTraceable(derivedBody, parentRecipeTrace.getNode(), traceProvider, this.traceableToNodeMap);
          };
          ((CompiledQuery)recipeTrace).getParentRecipeTracesPerBody().forEach(_function_1);
        }
      }
    };
    this.reteEngine.getReteNet().getRecipeTraces().forEach(_function);
  }

  /**
   * Find all nodes in the Rete network which originate from the given {@link PTraceable}
   */
  public Iterable<Node> findNodes(final PTraceable traceable) {
    return this.traceableToNodeMap.get(traceable);
  }

  /**
   * Extract the coverage of a PQuery based on a Rete coverage
   */
  public CoverageInfo<PTraceable> traceCoverage(final PQuery pQuery, final CoverageInfo<Node> reteCoverage) {
    CoverageInfo<PTraceable> _xblockexpression = null;
    {
      final CoverageInfo<PTraceable> coverage = new CoverageInfo<PTraceable>();
      final Consumer<PTraceable> _function = (PTraceable traceable) -> {
        final Function1<Node, CoverageState> _function_1 = (Node it) -> {
          return reteCoverage.get(CoverageContext.<Node>create(it, this.scope));
        };
        final Function2<CoverageState, CoverageState, CoverageState> _function_2 = (CoverageState r, CoverageState t) -> {
          return r.best(t);
        };
        final CoverageState coverageBasedOnTracedReteNodes = IterableExtensions.<CoverageState, CoverageState>fold(IterableExtensions.<Node, CoverageState>map(this.findNodes(traceable), _function_1), CoverageState.NOT_REPRESENTED_UNKNOWN_REASON, _function_2);
        CoverageState _xifexpression = null;
        if ((Objects.equals(coverageBasedOnTracedReteNodes, CoverageState.NOT_REPRESENTED_UNKNOWN_REASON) && this.hasRemovalReason(traceable))) {
          _xifexpression = CoverageState.NOT_REPRESENTED;
        } else {
          _xifexpression = coverageBasedOnTracedReteNodes;
        }
        final CoverageState amendedCoverage = _xifexpression;
        coverage.put(CoverageContext.<PTraceable>create(traceable, this.scope), amendedCoverage);
      };
      PQueries.getTraceables(pQuery).forEach(_function);
      _xblockexpression = coverage;
    }
    return _xblockexpression;
  }

  private boolean hasRemovalReason(final PTraceable canonical) {
    return this.traceProvider.isRemoved(canonical);
  }

  /**
   * Extract the coverage of a ViatraQueryMatcher based on a Rete coverage
   */
  public CoverageInfo<PTraceable> traceCoverage(final ViatraQueryMatcher<?> matcher, final CoverageInfo<Node> reteCoverage) {
    return this.traceCoverage(matcher.getSpecification().getInternalQueryRepresentation(), reteCoverage);
  }
}
