/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.api.table;

import java.sql.Timestamp;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.WeekFields;
import java.util.Arrays;
import oracle.kv.impl.api.table.TimestampDefImpl;
import oracle.kv.impl.api.table.TimestampValueImpl;

public class TimestampUtils {
    private static final int MAX_PRECISION = 9;
    private static final ZoneId UTCZone = ZoneId.of(ZoneOffset.UTC.getId());
    private static final int YEAR_POS = 0;
    private static final int YEAR_BITS = 14;
    private static final int MONTH_POS = 14;
    private static final int MONTH_BITS = 4;
    private static final int DAY_POS = 18;
    private static final int DAY_BITS = 5;
    private static final int HOUR_POS = 23;
    private static final int HOUR_BITS = 5;
    private static final int MINUTE_POS = 28;
    private static final int MINUTE_BITS = 6;
    private static final int SECOND_POS = 34;
    private static final int SECOND_BITS = 6;
    private static final int NANO_POS = 40;
    private static final int YEAR_EXCESS = 6384;
    private static final int NO_FRACSECOND_BYTES = 5;
    private static final char[] compSeparators = new char[]{'-', '-', 'T', ':', ':', '.'};
    private static final String[] compNames = new String[]{"year", "month", "day", "hour", "minute", "second", "fractional second"};
    private static final int[] nFracSecbits = new int[]{0, 4, 7, 10, 14, 17, 20, 24, 27, 30};

    static byte[] toBytes(Timestamp value, int precision) {
        ZonedDateTime zdt = TimestampUtils.toUTCDateTime(value);
        int fracSeconds = TimestampUtils.fracSecondToPrecision(zdt.getNano(), precision);
        return TimestampUtils.toBytes(zdt.getYear(), zdt.getMonthValue(), zdt.getDayOfMonth(), zdt.getHour(), zdt.getMinute(), zdt.getSecond(), fracSeconds, precision);
    }

    public static byte[] toBytes(int year, int month, int day, int hour, int minute, int second, int fracSeconds, int precision) {
        int nbytes;
        byte[] bytes = new byte[TimestampUtils.getNumBytes(precision)];
        TimestampUtils.writeBits(bytes, year + 6384, 0, 14);
        TimestampUtils.writeBits(bytes, month, 14, 4);
        int pos = TimestampUtils.writeBits(bytes, day, 18, 5);
        if (hour > 0) {
            pos = TimestampUtils.writeBits(bytes, hour, 23, 5);
        }
        if (minute > 0) {
            pos = TimestampUtils.writeBits(bytes, minute, 28, 6);
        }
        if (second > 0) {
            pos = TimestampUtils.writeBits(bytes, second, 34, 6);
        }
        if (fracSeconds > 0) {
            pos = TimestampUtils.writeBits(bytes, fracSeconds, 40, TimestampUtils.getFracSecondBits(precision));
        }
        return (nbytes = pos / 8 + 1) < bytes.length ? Arrays.copyOf(bytes, nbytes) : bytes;
    }

    static int fracSecondToPrecision(int nanos, int precision) {
        if (precision == 0) {
            return 0;
        }
        if (precision == 9) {
            return nanos;
        }
        return nanos / (int)Math.pow(10.0, 9 - precision);
    }

    static Timestamp fromBytes(byte[] bytes, int precision) {
        int[] comps = TimestampUtils.extractFromBytes(bytes, precision);
        return TimestampUtils.createTimestamp(comps);
    }

    private static int[] extractFromBytes(byte[] bytes, int precision) {
        int[] comps = new int[]{TimestampUtils.getYear(bytes), TimestampUtils.getMonth(bytes), TimestampUtils.getDay(bytes), TimestampUtils.getHour(bytes), TimestampUtils.getMinute(bytes), TimestampUtils.getSecond(bytes), TimestampUtils.getNano(bytes, precision)};
        return comps;
    }

    static int getYear(byte[] bytes) {
        return TimestampUtils.readBits(bytes, 0, 14) - 6384;
    }

    static int getMonth(byte[] bytes) {
        return TimestampUtils.readBits(bytes, 14, 4);
    }

    static int getDay(byte[] bytes) {
        return TimestampUtils.readBits(bytes, 18, 5);
    }

    static int getHour(byte[] bytes) {
        if (23 < bytes.length * 8) {
            return TimestampUtils.readBits(bytes, 23, 5);
        }
        return 0;
    }

    static int getMinute(byte[] bytes) {
        if (28 < bytes.length * 8) {
            return TimestampUtils.readBits(bytes, 28, 6);
        }
        return 0;
    }

    static int getSecond(byte[] bytes) {
        if (34 < bytes.length * 8) {
            return TimestampUtils.readBits(bytes, 34, 6);
        }
        return 0;
    }

    static int getNano(byte[] bytes, int precision) {
        int fracSecond = TimestampUtils.getFracSecond(bytes, precision);
        if (fracSecond > 0 && precision < 9) {
            fracSecond *= (int)Math.pow(10.0, 9 - precision);
        }
        return fracSecond;
    }

    static int getFracSecond(byte[] bytes, int precision) {
        if (40 < bytes.length * 8) {
            int num = TimestampUtils.getFracSecondBits(precision);
            return TimestampUtils.readBits(bytes, 40, num);
        }
        return 0;
    }

    public static int getWeekOfYear(Timestamp ts) {
        return TimestampUtils.toUTCDateTime(ts).get(WeekFields.SUNDAY_START.weekOfYear());
    }

    public static int getISOWeekOfYear(Timestamp ts) {
        return TimestampUtils.toUTCDateTime(ts).get(WeekFields.ISO.weekOfYear());
    }

    static int compareBytes(byte[] bs1, byte[] bs2) {
        int size = Math.min(bs1.length, bs2.length);
        int ret = TimestampUtils.compare(bs1, bs2, size);
        if (ret != 0) {
            return ret;
        }
        ret = bs1.length - bs2.length;
        if (ret != 0) {
            byte[] bs = ret > 0 ? bs1 : bs2;
            for (int i = size; i < bs.length; ++i) {
                if (bs[i] == 0) continue;
                return ret;
            }
        }
        return 0;
    }

    static int compareBytes(byte[] bs1, int precision1, byte[] bs2, int precision2) {
        if (precision1 == precision2 || bs1.length <= 5 || bs2.length <= 5) {
            return TimestampUtils.compareBytes(bs1, bs2);
        }
        assert (bs1.length > 5);
        assert (bs2.length > 5);
        int ret = TimestampUtils.compare(bs1, bs2, 5);
        if (ret != 0) {
            return ret;
        }
        int fs1 = TimestampUtils.getFracSecond(bs1, precision1);
        int fs2 = TimestampUtils.getFracSecond(bs2, precision2);
        int base = (int)Math.pow(10.0, Math.abs(precision1 - precision2));
        return precision1 < precision2 ? fs1 * base - fs2 : -1 * (fs2 * base - fs1);
    }

    private static int compare(byte[] bs1, byte[] bs2, int len) {
        assert (bs1.length >= len && bs2.length >= len);
        for (int i = 0; i < len; ++i) {
            byte b1 = bs1[i];
            byte b2 = bs2[i];
            if (b1 == b2) continue;
            return (b1 & 0xFF) - (b2 & 0xFF);
        }
        return 0;
    }

    static int getNumBytes(int precision) {
        return 5 + (TimestampUtils.getFracSecondBits(precision) + 7) / 8;
    }

    private static int getFracSecondBits(int precision) {
        return nFracSecbits[precision];
    }

    static Timestamp parseString(String timestampString, String pattern, boolean withZoneUTC) {
        return TimestampUtils.parseString(timestampString, pattern, withZoneUTC, pattern == null);
    }

    public static Timestamp parseString(String timestampString) {
        return TimestampUtils.parseString(timestampString, null, true);
    }

    private static Timestamp parseString(String timestampString, String pattern, boolean withZoneUTC, boolean optionalFracSecond) {
        boolean optionalZoneOffset = false;
        if ((pattern == null || pattern.equals("uuuu-MM-dd['T'HH:mm:ss]")) && withZoneUTC) {
            String tsStr = TimestampUtils.trimUTCZoneOffset(timestampString);
            if (tsStr != null) {
                return TimestampUtils.parseWithDefaultPattern(tsStr);
            }
            optionalZoneOffset = true;
        }
        String fmt = pattern == null ? "uuuu-MM-dd['T'HH:mm:ss]" : pattern;
        try {
            boolean hasOffset;
            DateTimeFormatter dtf = TimestampUtils.getDateTimeFormatter(fmt, withZoneUTC, optionalFracSecond, optionalZoneOffset, 0);
            TemporalAccessor ta = dtf.parse(timestampString);
            if (!(ta.isSupported(ChronoField.YEAR) && ta.isSupported(ChronoField.MONTH_OF_YEAR) && ta.isSupported(ChronoField.DAY_OF_MONTH))) {
                throw new IllegalArgumentException("The timestamp string must contain year, month and day");
            }
            boolean bl = hasOffset = ta.isSupported(ChronoField.OFFSET_SECONDS) && ta.get(ChronoField.OFFSET_SECONDS) != 0;
            Instant instant = ta.isSupported(ChronoField.HOUR_OF_DAY) ? (hasOffset ? OffsetDateTime.from(ta).toInstant() : Instant.from(ta)) : LocalDate.from(ta).atStartOfDay(hasOffset ? ZoneOffset.from(ta) : UTCZone).toInstant();
            return TimestampUtils.toTimestamp(instant);
        }
        catch (IllegalArgumentException iae) {
            throw new IllegalArgumentException("Failed to parse the date string '" + timestampString + "' with the pattern: " + fmt + ": " + iae.getMessage(), iae);
        }
        catch (DateTimeParseException dtpe) {
            throw new IllegalArgumentException("Failed to parse the date string '" + timestampString + "' with the pattern: " + fmt + ": " + dtpe.getMessage(), dtpe);
        }
        catch (DateTimeException dte) {
            throw new IllegalArgumentException("Failed to parse the date string '" + timestampString + "' with the pattern: " + fmt + ": " + dte.getMessage(), dte);
        }
    }

    private static String trimUTCZoneOffset(String ts) {
        if (ts.endsWith("Z")) {
            return ts.substring(0, ts.length() - 1);
        }
        if (ts.endsWith("+00:00")) {
            return ts.substring(0, ts.length() - 6);
        }
        if (!TimestampUtils.hasSignOfZoneOffset(ts)) {
            return ts;
        }
        return null;
    }

    private static boolean hasSignOfZoneOffset(String ts) {
        if (ts.indexOf(43) > 0) {
            return true;
        }
        int pos = 0;
        for (int i = 0; i < 3; ++i) {
            if ((pos = ts.indexOf(45, pos + 1)) >= 0) continue;
            return false;
        }
        return true;
    }

    private static Timestamp parseWithDefaultPattern(String ts) {
        int i;
        if (ts.isEmpty()) {
            TimestampUtils.raiseParseError(ts, "");
        }
        int[] comps = new int[7];
        int comp = 0;
        int val = 0;
        int ndigits = 0;
        int len = ts.length();
        boolean isBC = ts.charAt(0) == '-';
        int n = i = isBC ? 1 : 0;
        while (i < len) {
            char ch = ts.charAt(i);
            if (comp < 6) {
                switch (ch) {
                    case '0': 
                    case '1': 
                    case '2': 
                    case '3': 
                    case '4': 
                    case '5': 
                    case '6': 
                    case '7': 
                    case '8': 
                    case '9': {
                        val = val * 10 + (ch - 48);
                        ++ndigits;
                        break;
                    }
                    default: {
                        if (ch == compSeparators[comp]) {
                            TimestampUtils.checkAndSetValue(comps, comp, val, ndigits, ts);
                            ++comp;
                            val = 0;
                            ndigits = 0;
                            break;
                        }
                        TimestampUtils.raiseParseError(ts, "invalid character '" + ch + "' while parsing component " + compNames[comp]);
                        break;
                    }
                }
            } else {
                assert (comp == 6);
                switch (ch) {
                    case '0': 
                    case '1': 
                    case '2': 
                    case '3': 
                    case '4': 
                    case '5': 
                    case '6': 
                    case '7': 
                    case '8': 
                    case '9': {
                        val = val * 10 + (ch - 48);
                        ++ndigits;
                        break;
                    }
                    default: {
                        TimestampUtils.raiseParseError(ts, "invalid character '" + ch + "' while parsing component " + compNames[comp]);
                    }
                }
            }
            ++i;
        }
        TimestampUtils.checkAndSetValue(comps, comp, val, ndigits, ts);
        if (comp < 2) {
            TimestampUtils.raiseParseError(ts, "the timestamp string must have at least the 3 date components");
        }
        if (comp == 6 && comps[6] > 0) {
            if (ndigits > 9) {
                TimestampUtils.raiseParseError(ts, "the fractional-seconds part contains more than 9 digits");
            } else if (ndigits < 9) {
                comps[6] = comps[6] * (int)Math.pow(10.0, 9 - ndigits);
            }
        }
        if (isBC) {
            comps[0] = -comps[0];
        }
        return TimestampUtils.createTimestamp(comps);
    }

    private static void checkAndSetValue(int[] comps, int comp, int value, int ndigits, String ts) {
        if (ndigits == 0) {
            TimestampUtils.raiseParseError(ts, "component " + compNames[comp] + "has 0 digits");
        }
        comps[comp] = value;
    }

    private static void raiseParseError(String ts, String err) {
        String errMsg = "Failed to parse the timestamp string '" + ts + "' with the pattern: " + "uuuu-MM-dd['T'HH:mm:ss]" + ": ";
        throw new IllegalArgumentException(errMsg + err);
    }

    static String formatString(TimestampValueImpl value, String pattern, boolean withZoneUTC) {
        if ((pattern == null || pattern.equals("uuuu-MM-dd['T'HH:mm:ss]")) && withZoneUTC) {
            return TimestampUtils.stringFromBytes(value.getBytes(), value.getDefinition().getPrecision());
        }
        return TimestampUtils.formatString(value.get(), pattern, withZoneUTC, false, pattern == null ? value.getDefinition().getPrecision() : 0);
    }

    private static String stringFromBytes(byte[] bytes, int precision) {
        int[] comps = TimestampUtils.extractFromBytes(bytes, precision);
        StringBuilder sb = new StringBuilder("%04d-%02d-%02dT%02d:%02d:%02d");
        if (precision > 0) {
            sb.append(".");
            sb.append("%0");
            sb.append(precision);
            sb.append("d");
            int fs = comps[6] / (int)Math.pow(10.0, 9 - precision);
            return String.format(sb.toString(), comps[0], comps[1], comps[2], comps[3], comps[4], comps[5], fs);
        }
        return String.format(sb.toString(), comps[0], comps[1], comps[2], comps[3], comps[4], comps[5]);
    }

    public static String formatString(Timestamp timestamp, String pattern, boolean withZoneUTC) {
        return TimestampUtils.formatString(timestamp, pattern, withZoneUTC, false, 0);
    }

    static String formatString(Timestamp timestamp, String pattern, boolean withZoneUTC, boolean optionalFracSecond, int nFracSecond) {
        String fmt = pattern == null ? "uuuu-MM-dd['T'HH:mm:ss]" : pattern;
        try {
            ZonedDateTime zdt = TimestampUtils.toUTCDateTime(timestamp);
            return zdt.format(TimestampUtils.getDateTimeFormatter(fmt, withZoneUTC, optionalFracSecond, false, nFracSecond));
        }
        catch (IllegalArgumentException iae) {
            throw new IllegalArgumentException("Failed to format the timestamp with pattern '" + fmt + "': " + iae.getMessage(), iae);
        }
        catch (DateTimeException dte) {
            throw new IllegalArgumentException("Failed to format the timestamp with pattern '" + fmt + "': " + dte.getMessage(), dte);
        }
    }

    static String formatString(Timestamp value) {
        return TimestampUtils.formatString(value, null, true, true, 0);
    }

    static Timestamp roundToPrecision(Timestamp timestamp, int precision) {
        Timestamp ts;
        if (precision == 9 || timestamp.getNanos() == 0) {
            return timestamp;
        }
        long seconds = TimestampUtils.getSeconds(timestamp);
        int nanos = TimestampUtils.getNanosOfSecond(timestamp);
        double base = Math.pow(10.0, 9 - precision);
        if ((nanos = (int)((double)Math.round((double)nanos / base) * base)) == (int)Math.pow(10.0, 9.0)) {
            ++seconds;
            nanos = 0;
        }
        if ((ts = TimestampUtils.createTimestamp(seconds, nanos)).compareTo(TimestampDefImpl.MAX_VALUE) > 0) {
            ts = (Timestamp)TimestampDefImpl.MAX_VALUE.clone();
            nanos = (int)((double)((int)((double)ts.getNanos() / base)) * base);
            ts.setNanos((int)((double)((int)((double)ts.getNanos() / base)) * base));
        }
        return ts;
    }

    static long toMilliseconds(Timestamp timestamp) {
        return timestamp.getTime();
    }

    public static long getSeconds(Timestamp timestamp) {
        long ms = timestamp.getTime();
        return ms > 0L ? ms / 1000L : (ms - 999L) / 1000L;
    }

    public static int getNanosOfSecond(Timestamp timestamp) {
        return timestamp.getNanos();
    }

    static Timestamp minusNanos(Timestamp base, long nanosToSubtract) {
        return TimestampUtils.toTimestamp(TimestampUtils.toInstant(base).minusNanos(nanosToSubtract));
    }

    static Timestamp plusNanos(Timestamp base, long nanosToAdd) {
        return TimestampUtils.toTimestamp(TimestampUtils.toInstant(base).plusNanos(nanosToAdd));
    }

    static Timestamp minusMillis(Timestamp base, long millisToSubtract) {
        return TimestampUtils.toTimestamp(TimestampUtils.toInstant(base).minusMillis(millisToSubtract));
    }

    static Timestamp plusMillis(Timestamp base, long millisToAdd) {
        return TimestampUtils.toTimestamp(TimestampUtils.toInstant(base).plusMillis(millisToAdd));
    }

    public static Timestamp createTimestamp(long seconds, int nanosOfSecond) {
        Timestamp ts = new Timestamp(seconds * 1000L);
        ts.setNanos(nanosOfSecond);
        return ts;
    }

    public static Timestamp createTimestamp(int[] comps) {
        if (comps.length < 3) {
            throw new IllegalArgumentException("Invalid timestamp components, it should contain at least 3 components: year, month and day, but only " + comps.length);
        }
        if (comps.length > 7) {
            throw new IllegalArgumentException("Invalid timestamp components, it should contain at most 7 components: year, month, day, hour, minute, second and nanosecond, but has " + comps.length + " components");
        }
        int num = comps.length;
        for (int i = 0; i < num; ++i) {
            TimestampUtils.validateComponent(i, comps[i]);
        }
        try {
            ZonedDateTime zdt = ZonedDateTime.of(comps[0], comps[1], comps[2], num > 3 ? comps[3] : 0, num > 4 ? comps[4] : 0, num > 5 ? comps[5] : 0, num > 6 ? comps[6] : 0, UTCZone);
            return TimestampUtils.toTimestamp(zdt.toInstant());
        }
        catch (DateTimeException dte) {
            throw new IllegalArgumentException("Invalid timestamp components: " + dte.getMessage());
        }
    }

    public static void validateComponent(int index, int value) {
        switch (index) {
            case 0: {
                if (value >= TimestampDefImpl.MIN_YEAR && value <= TimestampDefImpl.MAX_YEAR) break;
                throw new IllegalArgumentException("Invalid year, it should be in range from " + TimestampDefImpl.MIN_YEAR + " to " + TimestampDefImpl.MAX_YEAR + ": " + value);
            }
            case 1: {
                if (value >= 1 && value <= 12) break;
                throw new IllegalArgumentException("Invalid month, it should be in range from 1 to 12: " + value);
            }
            case 2: {
                if (value >= 1 && value <= 31) break;
                throw new IllegalArgumentException("Invalid day, it should be in range from 1 to 31: " + value);
            }
            case 3: {
                if (value >= 0 && value <= 23) break;
                throw new IllegalArgumentException("Invalid hour, it should be in range from 0 to 23: " + value);
            }
            case 4: {
                if (value >= 0 && value <= 59) break;
                throw new IllegalArgumentException("Invalid minute, it should be in range from 0 to 59: " + value);
            }
            case 5: {
                if (value >= 0 && value <= 59) break;
                throw new IllegalArgumentException("Invalid second, it should be in range from 0 to 59: " + value);
            }
            case 6: {
                if (value >= 0 && value <= 999999999) break;
                throw new IllegalArgumentException("Invalid second, it should be in range from 0 to 999999999: " + value);
            }
        }
    }

    private static Instant toInstant(Timestamp timestamp) {
        return Instant.ofEpochSecond(TimestampUtils.getSeconds(timestamp), TimestampUtils.getNanosOfSecond(timestamp));
    }

    private static ZonedDateTime toUTCDateTime(Timestamp timestamp) {
        return TimestampUtils.toInstant(timestamp).atZone(UTCZone);
    }

    private static Timestamp toTimestamp(Instant instant) {
        return TimestampUtils.createTimestamp(instant.getEpochSecond(), instant.getNano());
    }

    private static int writeBits(byte[] bytes, int value, int pos, int len) {
        assert (value > 0 && pos + len <= bytes.length * 8);
        int ind = pos / 8;
        int lastPos = 0;
        for (int i = 0; i < len; ++i) {
            int bi = (pos + i) % 8;
            if ((value & 1 << len - i - 1) != 0) {
                int n = ind;
                bytes[n] = (byte)(bytes[n] | 1 << 7 - bi);
                lastPos = pos + i;
            }
            if (bi != 7) continue;
            ++ind;
        }
        return lastPos;
    }

    private static int readBits(byte[] bytes, int pos, int len) {
        int value = 0;
        int ind = pos / 8;
        for (int i = 0; i < len && ind < bytes.length; ++i) {
            int bi = (i + pos) % 8;
            if ((bytes[ind] & 1 << 7 - bi) != 0) {
                value |= 1 << len - 1 - i;
            }
            if (bi != 7) continue;
            ++ind;
        }
        return value;
    }

    private static DateTimeFormatter getDateTimeFormatter(String pattern, boolean withZoneUTC, boolean optionalFracSecond, boolean optionalOffsetId, int nFracSecond) {
        DateTimeFormatterBuilder dtfb = new DateTimeFormatterBuilder();
        dtfb.appendPattern(pattern);
        if (optionalFracSecond) {
            dtfb.optionalStart();
            dtfb.appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true);
            dtfb.optionalEnd();
        } else if (nFracSecond > 0) {
            dtfb.appendFraction(ChronoField.NANO_OF_SECOND, nFracSecond, nFracSecond, true);
        }
        if (optionalOffsetId) {
            dtfb.optionalStart();
            dtfb.appendOffset("+HH:MM", "Z");
            dtfb.optionalEnd();
        }
        return dtfb.toFormatter().withZone(withZoneUTC ? UTCZone : ZoneId.systemDefault());
    }
}

