"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert = require("assert");
const path_1 = require("path");
const paths_1 = require("../src/languageservice/utils/paths");
const vscode_uri_1 = require("vscode-uri");
class TestWorkspace {
    constructor(workspaceFolders, workspaceRoot) {
        this.folders = workspaceFolders;
        this.root = vscode_uri_1.URI.parse(workspaceRoot);
    }
    resolve(relPath) {
        return paths_1.relativeToAbsolutePath(this.folders, this.root, relPath);
    }
}
const ws1 = new TestWorkspace([
    {
        uri: 'file:///home/aFolder/',
        name: 'aFolder'
    }
], 'file:///home/aFolder/');
const ws2 = new TestWorkspace([
    {
        uri: 'file:///usr/testuser/projects/workspace/folder-1/',
        name: 'folder-1'
    },
    {
        uri: 'file:///usr/testuser/projects/workspace/folder-2/',
        name: 'folder-2'
    },
    {
        uri: 'file:///usr/testuser/projects/workspace/folder-3/',
        name: 'folder-3'
    }
], 'file:///usr/testuser/projects/workspace/');
const ws3 = new TestWorkspace([
    {
        uri: 'file:///c%3A/Users/testuser/dev/carrots',
        name: 'carrots'
    },
    {
        uri: 'file:///c%3A/Users/testuser/dev/potatoes',
        name: 'potatoes'
    }
], 'file:///c%3A/Users/testuser/dev/potatoes');
const checkBadPath = (path) => {
    it('Rejects "' + path + '"', () => {
        assert(!paths_1.isRelativePath(path));
    });
};
const checkGoodPath = (path, expect1, expect2, expect3) => {
    describe('Relative path = "' + path + '"', () => {
        it('Recognises relative path', () => {
            assert(paths_1.isRelativePath(path));
        });
        it('Resolves relative path in single-root workspace', () => {
            assert.equal(ws1.resolve(path), expect1);
        });
        it('Resolves relative path in multi-root workspace', () => {
            assert.equal(ws2.resolve(path), expect2);
        });
        it('Resolves relative path in multi-root nested workspace', () => {
            assert.equal(ws3.resolve(path), expect3);
        });
    });
};
suite('File path tests', () => {
    describe('Recognises not relative paths', () => {
        checkBadPath(path_1.join('/', 'file.json'));
        checkBadPath(path_1.join('/', 'absolutepath', 'file.json.'));
        checkBadPath(path_1.join('//', 'notrelativepath', 'file.json'));
        checkBadPath(path_1.join('C:', 'notrelativepath', 'file.json'));
        checkBadPath(path_1.join('directory.json', '/'));
        checkBadPath(path_1.join('.', 'dir', 'subdir', '/'));
    });
    describe('Recognises and correctly resolves relative paths', () => {
        checkGoodPath('file.json', 'file:///home/aFolder/file.json', 'file:///usr/testuser/projects/workspace/file.json', 'file:///c%3A/Users/testuser/dev/potatoes/file.json');
        checkGoodPath('file.long.extension.json', 'file:///home/aFolder/file.long.extension.json', 'file:///usr/testuser/projects/workspace/file.long.extension.json', 'file:///c%3A/Users/testuser/dev/potatoes/file.long.extension.json');
        checkGoodPath(path_1.join('.', 'file.json'), 'file:///home/aFolder/file.json', 'file:///usr/testuser/projects/workspace/file.json', 'file:///c%3A/Users/testuser/dev/potatoes/file.json');
        checkGoodPath(path_1.join('.', 'folder', 'file.json'), 'file:///home/aFolder/folder/file.json', 'file:///usr/testuser/projects/workspace/folder/file.json', 'file:///c%3A/Users/testuser/dev/potatoes/folder/file.json');
        checkGoodPath(path_1.join('.', 'long', 'path', 'to', 'file.json'), 'file:///home/aFolder/long/path/to/file.json', 'file:///usr/testuser/projects/workspace/long/path/to/file.json', 'file:///c%3A/Users/testuser/dev/potatoes/long/path/to/file.json');
        checkGoodPath(path_1.join('..', 'file.json'), 'file:///home/file.json', 'file:///usr/testuser/projects/file.json', 'file:///c%3A/Users/testuser/dev/file.json');
        checkGoodPath(path_1.join('.', 'relativepath', '..', 'file.json'), 'file:///home/aFolder/file.json', 'file:///usr/testuser/projects/workspace/file.json', 'file:///c%3A/Users/testuser/dev/potatoes/file.json');
        checkGoodPath(path_1.join('..', '..', 'relative', 'path', 'file.json'), 'file:///relative/path/file.json', 'file:///usr/testuser/relative/path/file.json', 'file:///c%3A/Users/testuser/relative/path/file.json');
        checkGoodPath(path_1.join('..', '..', 'relative', '@path', 'file.json'), 'file:///relative/%40path/file.json', 'file:///usr/testuser/relative/%40path/file.json', 'file:///c%3A/Users/testuser/relative/%40path/file.json');
        describe('Relative path = a workspace folder', () => {
            const path1 = path_1.join('aFolder', 'file.json');
            const path2 = path_1.join('folder-2', 'file.json');
            const path3 = path_1.join('carrots', 'file.json');
            it('Recognises relative path "' + path1 + '"', () => {
                assert(paths_1.isRelativePath(path1));
            });
            it('Resolves "' + path1 + '" in single-root workspace', () => {
                assert.equal(ws1.resolve(path1), 'file:///home/aFolder/file.json');
            });
            it('Resolves "' + path2 + '" in multi-root workspace', () => {
                assert.equal(ws2.resolve(path2), 'file:///usr/testuser/projects/workspace/folder-2/file.json');
            });
            it('Resolves "' + path3 + '" in multi-root nested workspace', () => {
                assert.equal(ws3.resolve(path3), 'file:///c%3A/Users/testuser/dev/carrots/file.json');
            });
        });
        describe('Path with mixed delimiters (Windows only)', () => {
            const path = 'some/strange\\but/functional\\path\\file.json';
            it('Recognises relative path "' + path + '"', function () {
                if (process.platform !== 'win32') {
                    this.skip();
                }
                else {
                    assert(paths_1.isRelativePath(path));
                }
            });
            it('Resolves "' + path + '" in single-root workspace', function () {
                if (process.platform !== 'win32') {
                    this.skip();
                }
                else {
                    assert.equal(ws1.resolve(path), 'file:///home/aFolder/some/strange/but/functional/path/file.json');
                }
            });
            it('Resolves "' + path + '" in multi-root workspace', function () {
                if (process.platform !== 'win32') {
                    this.skip();
                }
                else {
                    assert.equal(ws2.resolve(path), 'file:///usr/testuser/projects/workspace/some/strange/but/functional/path/file.json');
                }
            });
            it('Resolves "' + path + '" in multi-root nested workspace', function () {
                if (process.platform !== 'win32') {
                    this.skip();
                }
                else {
                    assert.equal(ws3.resolve(path), 'file:///c%3A/Users/testuser/dev/potatoes/some/strange/but/functional/path/file.json');
                }
            });
        });
    });
});
//# sourceMappingURL=paths.test.js.map