"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const typescript_1 = __importDefault(require("typescript"));
const eslint_config_provider_1 = require("./eslint-config-provider");
const fixtureDirPath = path_1.default.resolve(__dirname, "../test-fixtures/eslnt-config-provider");
function readFile(path, encoding) {
    return fs_1.default.readFileSync(path, { encoding: encoding });
}
class FileUpdater {
    constructor() {
        this.callbackList = [];
    }
    init(path) {
        this.path = path;
        this.callbackList = [];
        this.originalContent = readFile(path, "utf8");
    }
    update(replace) {
        if (this.originalContent) {
            const newContent = replace(this.originalContent);
            fs_1.default.writeFileSync(this.path, newContent, "utf8");
            this.callbackList
                .filter(({ path }) => this.path === path)
                .forEach(({ callback }) => {
                callback(this.path, typescript_1.default.FileWatcherEventKind.Changed);
            });
        }
    }
    reset() {
        if (this.originalContent) {
            fs_1.default.writeFileSync(this.path, this.originalContent, "utf8");
            delete this.originalContent;
        }
        delete this.path;
        this.callbackList = [];
    }
    createWatcher() {
        return {
            watchFile: (path, callback) => {
                this.callbackList.push({ path, callback });
                return {
                    close: () => { },
                };
            },
        };
    }
}
describe("ESLintConfigProvider", () => {
    describe("#getConfigForFile", () => {
        const fileUpdater = new FileUpdater();
        afterEach(() => fileUpdater.reset());
        it("should load base config", () => {
            const prjDir = path_1.default.resolve(fixtureDirPath, "prj-with-eslintrc-yml");
            const provider = new eslint_config_provider_1.ESLintConfigProvider({
                directoriesToWatch: [prjDir],
                host: {
                    readFile,
                    watchFile: fileUpdater.createWatcher().watchFile,
                },
            });
            const config = provider.getConfigForFile(path_1.default.resolve(prjDir, "main.ts"));
            expect(config.rules).toEqual({ curly: ["error"] });
        });
        it("should reload base config after changes eslintrc.yml file", () => {
            const prjDir = path_1.default.resolve(fixtureDirPath, "prj-with-eslintrc-yml");
            fileUpdater.init(path_1.default.resolve(prjDir, ".eslintrc.yml"));
            const provider = new eslint_config_provider_1.ESLintConfigProvider({
                directoriesToWatch: [prjDir],
                host: {
                    readFile,
                    watchFile: fileUpdater.createWatcher().watchFile,
                },
            });
            provider.getConfigForFile(path_1.default.resolve(prjDir, "main.ts"));
            fileUpdater.update(s => s.replace("# TO_BE_ENABLED_AFTER ", ""));
            const configAfterChanges = provider.getConfigForFile(path_1.default.resolve(prjDir, "main.ts"));
            expect(configAfterChanges.rules).toEqual({ curly: ["error"], semi: ["error"] });
        });
        // FIXME
        // Using Node.js REPL, CascadingConfigArrayFactory#clearCache and relaod .eslintrc.js returns the updated config object.
        // But the reloading does not work well under thw folloiwng test case...
        it.skip("should reload base config after changes eslintrc.* file", () => {
            const prjDir = path_1.default.resolve(fixtureDirPath, "prj-with-eslintrc");
            fileUpdater.init(path_1.default.resolve(prjDir, ".eslintrc.js"));
            const provider = new eslint_config_provider_1.ESLintConfigProvider({
                directoriesToWatch: [prjDir],
                host: {
                    readFile,
                    watchFile: fileUpdater.createWatcher().watchFile,
                },
            });
            provider.getConfigForFile(path_1.default.resolve(prjDir, "main.ts"));
            fileUpdater.update(s => s.replace("// TO_BE_ENABLED_AFTER ", ""));
            const configAfterChanges = provider.getConfigForFile(path_1.default.resolve(prjDir, "main.ts"));
            expect(configAfterChanges.rules).toEqual({ curly: ["error"], semi: ["error"] });
        });
    });
});
//# sourceMappingURL=eslint-config-provider.test.js.map