/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: RDFXMLResourceImpl.java,v 1.3 2007/03/18 08:39:02 lzhang Exp $
 */
package org.eclipse.eodm.rdf.resource;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLHandler;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.resource.parser.exception.ParserException;
import org.eclipse.eodm.rdf.resource.parser.impl.RDFXMLLoader;

/**
 * RDFXMLResourceImpl is responsible for storing and loading EODM-Based RDF
 * meomory model to/from a RDF/XML resource that conforms to the <a
 * href="http://www.w3.org/TR/2004/REC-rdf-syntax-grammar-20040210/">W3C RDF/XML
 * Syntax Specificatoin (W3C Recommendation, 10 February 2004) </a>.
 * 
 * @see <a href="http://www.w3.org/TR/2004/REC-rdf-syntax-grammar-20040210/">W3C
 *      RDF/XML Syntax Specificatoin (W3C Recommendation, 10 February 2004) </a>
 * 
 */
public class RDFXMLResourceImpl extends ResourceImpl implements RDFXMLResource {

    protected String defaultEncoding = null;

    /**
     * Construct a RDFXMLResource object based on its URI.
     * 
     * @param uri
     *            the URI of the RDFXMLResource
     */
    public RDFXMLResourceImpl(URI uri) {
        super(uri);
    }

    /**
     * Get the encoding to use from a options map and input steam.
     * 
     * @param options
     *            the map that contains all the user-specified resource options
     * @return the name of the encoding specified in the options map or
     *         <code>null</code> if not specified.
     */
    protected String getEncodingOption(Map options) {
        if (options == null)
            return null;
        Object enc = options.get(RDFXMLResource.OPTION_ENCODING);
        if (enc != null)
            if (enc instanceof String)
                return (String) enc;
        return null;
    }

    /**
     * Read a buffer for guessing encoding.
     * 
     * @param is
     *            the input stream that supports mark
     * @return a buffer of bytes of the input stream
     * @throws IOException
     */
    protected byte[] readGuessBuffer(InputStream is) throws IOException {
        int BUFFER_SIZE = 200;

        if (is.available() == 0) {
            return new byte[0];
        }

        byte[] buffer = new byte[BUFFER_SIZE];
        is.mark(BUFFER_SIZE);
        int bytesRead = is.read(buffer, 0, BUFFER_SIZE);
        int totalBytesRead = bytesRead;

        while (bytesRead != -1 && (totalBytesRead < BUFFER_SIZE)) {
            bytesRead = is.read(buffer, totalBytesRead, BUFFER_SIZE
                                                        - totalBytesRead);

            if (bytesRead != -1)
                totalBytesRead += bytesRead;
        }

        if (totalBytesRead < BUFFER_SIZE) {
            byte[] smallerBuffer = new byte[totalBytesRead];
            System.arraycopy(buffer, 0, smallerBuffer, 0, totalBytesRead);
            smallerBuffer = buffer;
        }

        is.reset();
        return buffer;
    }

    /**
     * Guess the encoding of the input stream. Note that the input stream must
     * support mark.
     * 
     * @param is
     *            the input stream
     * @return the guessed encoding or null
     * @throws IOException
     */
    protected String guessEncoding(InputStream is) throws IOException {
        if (!is.markSupported())
            return null;
        return XMLHandler.getXMLEncoding(readGuessBuffer(is));
    }

    /**
     * Acutally load a RDF/XML resource from an input stream given a set of
     * options.
     * 
     * @param inputStream
     *            the {@link java.io.InputStream InputStream}to load a RDF/XML
     *            content from
     * @param options
     *            the map of options used for the loading, e.g. the encoding
     *            option.
     * @throws java.io.IOException
     *             If an I/O error occurs in the operation with the input stream
     *             or the RDF/XML resource does not conforms to the W3C RDF/XML
     *             Syntax specification. The
     *             {@linkplain java.lang.Throwable#getCause() cause}of the
     *             IOException is a
     *             {@link org.eclipse.eodm.rdfs.util.exception.ParserException 
     *             ParserException}
     */
    protected void doLoad(InputStream inputStream, Map options)
            throws IOException {
        String enc = getEncodingOption(options);
        if (enc == null) {
            if (!inputStream.markSupported())
                inputStream = new BufferedInputStream(inputStream);
            enc = guessEncoding(inputStream);
        }
        defaultEncoding = enc; // remember this encoding for save

        RDFXMLLoader.setDefaultBaseURI(getURI().toString());

        try {
            Document document = null;
            if (enc != null)
                document = RDFXMLLoader.loadFromStream(inputStream, enc);
            else
                document = RDFXMLLoader.loadFromStream(inputStream);

            if (document != null)
                getContents().add(document);

        } catch (ParserException pe) {
            IOException ioEx = new IOException(
                    "Exception in loading from RDF/XML stream");
            ioEx.initCause(pe);
            throw ioEx;
        }
    }

    /**
     * Acutally save a RDF/XML resource to an output stream given a set of
     * options.
     * 
     * @param outputStream
     *            the {@link java.io.OutputStream OutputStream}to save a
     *            RDF/XML content to.
     * @param options
     *            the map of options used for the saving, e.g. the encoding
     *            option.
     * @throws java.io.IOException
     *             If an I/O error occurs in the operation with the output
     *             stream.
     */
    protected void doSave(OutputStream outputStream, Map options)
            throws IOException {
        String enc = getEncodingOption(options);
        if (enc == null)
            enc = defaultEncoding;

        // Find the first document in the contents of the resource
        Document document = null;
        for (Iterator iterator = getContents().iterator(); iterator.hasNext();) {
            Object obj = iterator.next();
            if (obj instanceof Document) {
                document = (Document) obj;
                break;
            }
        }

        if (document != null) {
            if (enc != null)
                RDFXMLSaver.saveToStream(document, outputStream, enc);
            else
                RDFXMLSaver.saveToStream(document, outputStream);
        }
    }

}