/**
 * Copyright (c) 2015 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 */
package org.eclipse.papyrus.dev.assistants.codegen.generator;

import java.util.ArrayList;
import java.util.HashMap;
import javax.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.dev.assistants.codegen.generator.ModelingAssistantProviderRule;
import org.eclipse.papyrus.infra.filters.Filter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ConnectionAssistant;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ElementTypeFilter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.PopupAssistant;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeSetConfiguration;
import org.eclipse.papyrus.uml.profile.assistants.generator.FiltersUtil;
import org.eclipse.papyrus.uml.profile.types.generator.UMLElementTypes;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Element Type to Assistant transformation rule.
 */
@SuppressWarnings("all")
public class ElementTypeToAssistantRule {
  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;
  
  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;
  
  @Inject
  @Extension
  private FiltersUtil _filtersUtil;
  
  @Inject
  @Extension
  private ModelingAssistantProviderRule _modelingAssistantProviderRule;
  
  public PopupAssistant toPopupAssistant(final ElementTypeConfiguration type) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(type);
    final PopupAssistant _result;
    synchronized (_createCache_toPopupAssistant) {
      if (_createCache_toPopupAssistant.containsKey(_cacheKey)) {
        return _createCache_toPopupAssistant.get(_cacheKey);
      }
      PopupAssistant _createPopupAssistant = ElementTypeToAssistantRule.assistantFactory.createPopupAssistant();
      _result = _createPopupAssistant;
      _createCache_toPopupAssistant.put(_cacheKey, _result);
    }
    _init_toPopupAssistant(_result, type);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, PopupAssistant> _createCache_toPopupAssistant = CollectionLiterals.newHashMap();
  
  private void _init_toPopupAssistant(final PopupAssistant it, final ElementTypeConfiguration type) {
    it.setElementTypeID(type.getIdentifier());
    it.setOwnedFilter(this._filtersUtil.reduce(this.createPossibleOwnersFilter(type)));
  }
  
  private Filter createPossibleOwnersFilter(final ElementTypeConfiguration type) {
    final Function2<Filter, ElementTypeConfiguration, Filter> _function = new Function2<Filter, ElementTypeConfiguration, Filter>() {
      @Override
      public Filter apply(final Filter filter, final ElementTypeConfiguration elementType) {
        Filter _xifexpression = null;
        boolean _canContainType = ElementTypeToAssistantRule.this._uMLElementTypes.canContainType(elementType, type);
        if (_canContainType) {
          ElementTypeFilter _filter = ElementTypeToAssistantRule.this.toFilter(elementType);
          _xifexpression = ElementTypeToAssistantRule.this._filtersUtil.operator_or(filter, _filter);
        } else {
          _xifexpression = filter;
        }
        return _xifexpression;
      }
    };
    return IterableExtensions.<ElementTypeConfiguration, Filter>fold(this._uMLElementTypes.getBaseElementTypes(), null, _function);
  }
  
  public ConnectionAssistant toConnectionAssistant(final ElementTypeConfiguration type) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(type);
    final ConnectionAssistant _result;
    synchronized (_createCache_toConnectionAssistant) {
      if (_createCache_toConnectionAssistant.containsKey(_cacheKey)) {
        return _createCache_toConnectionAssistant.get(_cacheKey);
      }
      ConnectionAssistant _createConnectionAssistant = ElementTypeToAssistantRule.assistantFactory.createConnectionAssistant();
      _result = _createConnectionAssistant;
      _createCache_toConnectionAssistant.put(_cacheKey, _result);
    }
    _init_toConnectionAssistant(_result, type);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ConnectionAssistant> _createCache_toConnectionAssistant = CollectionLiterals.newHashMap();
  
  private void _init_toConnectionAssistant(final ConnectionAssistant it, final ElementTypeConfiguration type) {
    it.setElementTypeID(type.getIdentifier());
    it.setOwnedSourceFilter(this._filtersUtil.reduce(this.createPossibleSourcesFilter(type)));
    it.setOwnedTargetFilter(this._filtersUtil.reduce(this.createPossibleTargetsFilter(type)));
  }
  
  public ElementTypeSetConfiguration owningSet(final ElementTypeConfiguration elementType) {
    EObject _eContainer = elementType.eContainer();
    return ((ElementTypeSetConfiguration) _eContainer);
  }
  
  public ElementTypeFilter toFilter(final ElementTypeConfiguration elementType) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(elementType);
    final ElementTypeFilter _result;
    synchronized (_createCache_toFilter) {
      if (_createCache_toFilter.containsKey(_cacheKey)) {
        return _createCache_toFilter.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = ElementTypeToAssistantRule.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_toFilter.put(_cacheKey, _result);
    }
    _init_toFilter(_result, elementType);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_toFilter = CollectionLiterals.newHashMap();
  
  private void _init_toFilter(final ElementTypeFilter it, final ElementTypeConfiguration elementType) {
    it.setElementTypeID(elementType.getIdentifier());
    it.setName(elementType.getName());
    this._modelingAssistantProviderRule.toModelingAssistantProvider(this.owningSet(elementType)).getOwnedFilters().add(it);
  }
  
  private Filter createPossibleSourcesFilter(final ElementTypeConfiguration type) {
    final Function1<ElementTypeConfiguration, Boolean> _function = new Function1<ElementTypeConfiguration, Boolean>() {
      @Override
      public Boolean apply(final ElementTypeConfiguration it) {
        boolean _isRelationship = ElementTypeToAssistantRule.this._uMLElementTypes.isRelationship(it);
        return Boolean.valueOf((!_isRelationship));
      }
    };
    final Function2<Filter, ElementTypeConfiguration, Filter> _function_1 = new Function2<Filter, ElementTypeConfiguration, Filter>() {
      @Override
      public Filter apply(final Filter filter, final ElementTypeConfiguration elementType) {
        Filter _xifexpression = null;
        boolean _canSourceToType = ElementTypeToAssistantRule.this._uMLElementTypes.canSourceToType(elementType, type);
        if (_canSourceToType) {
          ElementTypeFilter _filter = ElementTypeToAssistantRule.this.toFilter(elementType);
          _xifexpression = ElementTypeToAssistantRule.this._filtersUtil.operator_or(filter, _filter);
        } else {
          _xifexpression = filter;
        }
        return _xifexpression;
      }
    };
    return IterableExtensions.<ElementTypeConfiguration, Filter>fold(IterableExtensions.<ElementTypeConfiguration>filter(this._uMLElementTypes.getBaseElementTypes(), _function), null, _function_1);
  }
  
  private Filter createPossibleTargetsFilter(final ElementTypeConfiguration type) {
    final Function1<ElementTypeConfiguration, Boolean> _function = new Function1<ElementTypeConfiguration, Boolean>() {
      @Override
      public Boolean apply(final ElementTypeConfiguration it) {
        boolean _isRelationship = ElementTypeToAssistantRule.this._uMLElementTypes.isRelationship(it);
        return Boolean.valueOf((!_isRelationship));
      }
    };
    final Function2<Filter, ElementTypeConfiguration, Filter> _function_1 = new Function2<Filter, ElementTypeConfiguration, Filter>() {
      @Override
      public Filter apply(final Filter filter, final ElementTypeConfiguration elementType) {
        Filter _xifexpression = null;
        boolean _canTargetFromType = ElementTypeToAssistantRule.this._uMLElementTypes.canTargetFromType(elementType, type);
        if (_canTargetFromType) {
          ElementTypeFilter _filter = ElementTypeToAssistantRule.this.toFilter(elementType);
          _xifexpression = ElementTypeToAssistantRule.this._filtersUtil.operator_or(filter, _filter);
        } else {
          _xifexpression = filter;
        }
        return _xifexpression;
      }
    };
    return IterableExtensions.<ElementTypeConfiguration, Filter>fold(IterableExtensions.<ElementTypeConfiguration>filter(this._uMLElementTypes.getBaseElementTypes(), _function), null, _function_1);
  }
}
