/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.seata.sqlparser.druid;

import com.alibaba.druid.sql.SQLUtils;
import com.alibaba.druid.sql.ast.SQLStatement;
import com.alibaba.druid.sql.ast.statement.SQLDeleteStatement;
import com.alibaba.druid.sql.ast.statement.SQLInsertStatement;
import com.alibaba.druid.sql.ast.statement.SQLReplaceStatement;
import com.alibaba.druid.sql.ast.statement.SQLSelectStatement;
import com.alibaba.druid.sql.ast.statement.SQLUpdateStatement;
import org.apache.seata.common.exception.NotSupportYetException;
import org.apache.seata.common.util.CollectionUtils;
import org.apache.seata.sqlparser.SQLRecognizer;
import org.apache.seata.sqlparser.SQLRecognizerFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * DruidSQLRecognizerFactoryImpl
 *
 */
class DruidSQLRecognizerFactoryImpl implements SQLRecognizerFactory {
    @Override
    public List<SQLRecognizer> create(String sql, String dbType) {
        List<SQLStatement> sqlStatements;
        try {
            sqlStatements = SQLUtils.parseStatements(sql, DruidDbTypeAdapter.getAdaptiveDbType(dbType));
        } catch (RuntimeException e) {
            if (isParserException(e)) {
                throw new NotSupportYetException(
                        "not support the sql syntax: " + sql
                                + "\nplease see the doc about SQL restrictions https://seata.apache.org/zh-cn/docs/user/sqlreference/dml",
                        e);
            }
            throw e;
        }

        if (CollectionUtils.isEmpty(sqlStatements)) {
            throw new UnsupportedOperationException("Unsupported SQL: " + sql);
        }
        if (sqlStatements.size() > 1
                && !(sqlStatements.stream().allMatch(statement -> statement instanceof SQLUpdateStatement)
                        || sqlStatements.stream().allMatch(statement -> statement instanceof SQLDeleteStatement))) {
            throw new UnsupportedOperationException("ONLY SUPPORT SAME TYPE (UPDATE OR DELETE) MULTI SQL -" + sql);
        }
        List<SQLRecognizer> recognizers = null;
        SQLRecognizer recognizer = null;
        for (SQLStatement sqlStatement : sqlStatements) {
            SQLOperateRecognizerHolder recognizerHolder =
                    SQLOperateRecognizerHolderFactory.getSQLRecognizerHolder(dbType.toLowerCase());
            if (sqlStatement instanceof SQLInsertStatement) {
                recognizer = recognizerHolder.getInsertRecognizer(sql, sqlStatement);
            } else if (sqlStatement instanceof SQLUpdateStatement) {
                recognizer = recognizerHolder.getUpdateRecognizer(sql, sqlStatement);
            } else if (sqlStatement instanceof SQLDeleteStatement) {
                recognizer = recognizerHolder.getDeleteRecognizer(sql, sqlStatement);
            } else if (sqlStatement instanceof SQLSelectStatement) {
                recognizer = recognizerHolder.getSelectForUpdateRecognizer(sql, sqlStatement);
            }

            // When recognizer is null, it indicates that recognizerHolder cannot allocate unsupported syntax, like
            // merge and replace
            if (sqlStatement instanceof SQLReplaceStatement) {
                // just like:replace into t (id,dr) values (1,'2'), (2,'3')
                throw new NotSupportYetException(
                        "not support the sql syntax with ReplaceStatement:" + sqlStatement
                                + "\nplease see the doc about SQL restrictions https://seata.apache.org/zh-cn/docs/user/sqlreference/dml");
            }

            if (recognizer != null && recognizer.isSqlSyntaxSupports()) {
                if (recognizers == null) {
                    recognizers = new ArrayList<>();
                }
                recognizers.add(recognizer);
            }
        }
        return recognizers;
    }

    /**
     * Check if the exception is a Druid ParserException
     * Use class name comparison to avoid directly referencing the ParserException class
     */
    private boolean isParserException(Throwable e) {
        if (e == null) {
            return false;
        }
        String className = e.getClass().getName();
        return "com.alibaba.druid.sql.parser.ParserException".equals(className);
    }
}
