/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.backup.impl;

import com.google.errorprone.annotations.RestrictedApi;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.backup.BackupInfo;
import org.apache.hadoop.hbase.backup.BackupType;
import org.apache.hadoop.hbase.backup.HBackupFileSystem;
import org.apache.hadoop.hbase.backup.impl.BackupException;
import org.apache.hadoop.hbase.backup.util.BackupUtils;
import org.apache.hadoop.hbase.shaded.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.shaded.protobuf.generated.BackupProtos;
import org.apache.hadoop.hbase.shaded.protobuf.generated.HBaseProtos;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class BackupManifest {
    private static final Logger LOG = LoggerFactory.getLogger(BackupManifest.class);
    public static final String MANIFEST_FILE_NAME = ".backup.manifest";
    private BackupImage backupImage;

    public BackupManifest(BackupInfo backup) {
        BackupImage.Builder builder = BackupImage.newBuilder();
        this.backupImage = builder.withBackupId(backup.getBackupId()).withType(backup.getType()).withRootDir(backup.getBackupRootDir()).withTableList(backup.getTableNames()).withStartTime(backup.getStartTs()).withCompleteTime(backup.getCompleteTs()).build();
    }

    public BackupManifest(BackupInfo backup, TableName table) {
        ArrayList<TableName> tables = new ArrayList<TableName>();
        tables.add(table);
        BackupImage.Builder builder = BackupImage.newBuilder();
        this.backupImage = builder.withBackupId(backup.getBackupId()).withType(backup.getType()).withRootDir(backup.getBackupRootDir()).withTableList(tables).withStartTime(backup.getStartTs()).withCompleteTime(backup.getCompleteTs()).build();
    }

    public BackupManifest(Configuration conf, Path backupPath) throws IOException {
        this(backupPath.getFileSystem(conf), backupPath);
    }

    public BackupManifest(FileSystem fs, Path backupPath) throws BackupException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Loading manifest from: " + backupPath.toString());
        }
        try {
            FileStatus[] subFiles = BackupUtils.listStatus(fs, backupPath, null);
            if (subFiles == null) {
                String string = backupPath.toString() + " does not exist";
                LOG.error(string);
                throw new IOException(string);
            }
            for (FileStatus subFile : subFiles) {
                if (!subFile.getPath().getName().equals(MANIFEST_FILE_NAME)) continue;
                long len = subFile.getLen();
                byte[] pbBytes = new byte[(int)len];
                try (FSDataInputStream in = fs.open(subFile.getPath());){
                    in.readFully(pbBytes);
                }
                catch (IOException e) {
                    throw new BackupException(e.getMessage());
                }
                BackupProtos.BackupImage proto = null;
                try {
                    proto = BackupProtos.BackupImage.parseFrom((byte[])pbBytes);
                }
                catch (Exception e) {
                    throw new BackupException(e);
                }
                this.backupImage = BackupManifest.hydrateRootDir(BackupImage.fromProto(proto), backupPath);
                LOG.debug("Loaded manifest instance from manifest file: " + BackupUtils.getPath(subFile.getPath()));
                return;
            }
            String string = "No manifest file found in: " + backupPath.toString();
            throw new IOException(string);
        }
        catch (IOException e) {
            throw new BackupException(e.getMessage());
        }
    }

    @RestrictedApi(explanation="Should only be called internally or in tests", link="", allowedOnPath="(.*/src/test/.*|.*/org/apache/hadoop/hbase/backup/impl/BackupManifest.java)")
    public static BackupImage hydrateRootDir(BackupImage backupImage, Path backupPath) throws IOException {
        String providedRootDir = HBackupFileSystem.getRootDirFromBackupPath(backupPath, backupImage.backupId).toString();
        backupImage.setRootDir(providedRootDir);
        for (BackupImage ancestor : backupImage.getAncestors()) {
            ancestor.setRootDir(providedRootDir);
        }
        return backupImage;
    }

    public BackupType getType() {
        return this.backupImage.getType();
    }

    public List<TableName> getTableList() {
        return this.backupImage.getTableNames();
    }

    public void store(Configuration conf) throws BackupException {
        byte[] data = this.backupImage.toProto().toByteArray();
        Path manifestFilePath = new Path(HBackupFileSystem.getBackupPath(this.backupImage.getRootDir(), this.backupImage.getBackupId()), MANIFEST_FILE_NAME);
        try (FSDataOutputStream out = manifestFilePath.getFileSystem(conf).create(manifestFilePath, true);){
            out.write(data);
        }
        catch (IOException e) {
            throw new BackupException(e.getMessage());
        }
        LOG.info("Manifest file stored to " + manifestFilePath);
    }

    public BackupImage getBackupImage() {
        return this.backupImage;
    }

    public void addDependentImage(BackupImage image) {
        this.backupImage.addAncestor(image);
    }

    public void setIncrTimestampMap(Map<TableName, Map<String, Long>> incrTimestampMap) {
        this.backupImage.setIncrTimeRanges(incrTimestampMap);
    }

    public Map<TableName, Map<String, Long>> getIncrTimestampMap() {
        return this.backupImage.getIncrTimeRanges();
    }

    public ArrayList<BackupImage> getRestoreDependentList(boolean reverse) {
        TreeMap<Long, BackupImage> restoreImages = new TreeMap<Long, BackupImage>();
        restoreImages.put(this.backupImage.startTs, this.backupImage);
        for (BackupImage image : this.backupImage.getAncestors()) {
            restoreImages.put(image.startTs, image);
        }
        return new ArrayList<BackupImage>(reverse ? restoreImages.descendingMap().values() : restoreImages.values());
    }

    public ArrayList<BackupImage> getDependentListByTable(TableName table) {
        ArrayList<BackupImage> tableImageList = new ArrayList<BackupImage>();
        ArrayList<BackupImage> imageList = this.getRestoreDependentList(true);
        for (BackupImage image : imageList) {
            if (!image.hasTable(table)) continue;
            tableImageList.add(image);
            if (image.getType() != BackupType.FULL) continue;
            break;
        }
        Collections.reverse(tableImageList);
        return tableImageList;
    }

    public BackupInfo toBackupInfo() {
        BackupInfo info = new BackupInfo();
        info.setType(this.backupImage.getType());
        List<TableName> list = this.backupImage.getTableNames();
        TableName[] tables = new TableName[list.size()];
        info.addTables(list.toArray(tables));
        info.setBackupId(this.backupImage.getBackupId());
        info.setStartTs(this.backupImage.getStartTs());
        info.setBackupRootDir(this.backupImage.getRootDir());
        if (this.backupImage.getType() == BackupType.INCREMENTAL) {
            info.setHLogTargetDir(BackupUtils.getLogBackupDir(this.backupImage.getRootDir(), this.backupImage.getBackupId()));
        }
        return info;
    }

    public static class BackupImage
    implements Comparable<BackupImage> {
        private String backupId;
        private BackupType type;
        private String rootDir;
        private List<TableName> tableList;
        private long startTs;
        private long completeTs;
        private ArrayList<BackupImage> ancestors;
        private Map<TableName, Map<String, Long>> incrTimeRanges;

        static Builder newBuilder() {
            return new Builder();
        }

        public BackupImage() {
        }

        private BackupImage(String backupId, BackupType type, String rootDir, List<TableName> tableList, long startTs, long completeTs) {
            this.backupId = backupId;
            this.type = type;
            this.rootDir = rootDir;
            this.tableList = tableList;
            this.startTs = startTs;
            this.completeTs = completeTs;
        }

        static BackupImage fromProto(BackupProtos.BackupImage im) {
            String backupId = im.getBackupId();
            String rootDir = im.getBackupRootDir();
            long startTs = im.getStartTs();
            long completeTs = im.getCompleteTs();
            List tableListList = im.getTableListList();
            ArrayList<TableName> tableList = new ArrayList<TableName>();
            for (HBaseProtos.TableName tn : tableListList) {
                tableList.add(ProtobufUtil.toTableName((HBaseProtos.TableName)tn));
            }
            List ancestorList = im.getAncestorsList();
            BackupType type = im.getBackupType() == BackupProtos.BackupType.FULL ? BackupType.FULL : BackupType.INCREMENTAL;
            BackupImage image = new BackupImage(backupId, type, rootDir, tableList, startTs, completeTs);
            for (BackupProtos.BackupImage img : ancestorList) {
                image.addAncestor(BackupImage.fromProto(img));
            }
            image.setIncrTimeRanges(BackupImage.loadIncrementalTimestampMap(im));
            return image;
        }

        BackupProtos.BackupImage toProto() {
            BackupProtos.BackupImage.Builder builder = BackupProtos.BackupImage.newBuilder();
            builder.setBackupId(this.backupId);
            builder.setCompleteTs(this.completeTs);
            builder.setStartTs(this.startTs);
            if (this.type == BackupType.FULL) {
                builder.setBackupType(BackupProtos.BackupType.FULL);
            } else {
                builder.setBackupType(BackupProtos.BackupType.INCREMENTAL);
            }
            for (TableName name : this.tableList) {
                builder.addTableList(ProtobufUtil.toProtoTableName((TableName)name));
            }
            if (this.ancestors != null) {
                for (BackupImage im : this.ancestors) {
                    builder.addAncestors(im.toProto());
                }
            }
            this.setIncrementalTimestampMap(builder);
            return builder.build();
        }

        private static Map<TableName, Map<String, Long>> loadIncrementalTimestampMap(BackupProtos.BackupImage proto) {
            List list = proto.getTstMapList();
            HashMap<TableName, Map<String, Long>> incrTimeRanges = new HashMap<TableName, Map<String, Long>>();
            if (list == null || list.size() == 0) {
                return incrTimeRanges;
            }
            for (BackupProtos.TableServerTimestamp tst : list) {
                TableName tn = ProtobufUtil.toTableName((HBaseProtos.TableName)tst.getTableName());
                HashMap<String, Long> map = (HashMap<String, Long>)incrTimeRanges.get(tn);
                if (map == null) {
                    map = new HashMap<String, Long>();
                    incrTimeRanges.put(tn, map);
                }
                List listSt = tst.getServerTimestampList();
                for (BackupProtos.ServerTimestamp stm : listSt) {
                    ServerName sn = ProtobufUtil.toServerName((HBaseProtos.ServerName)stm.getServerName());
                    map.put(sn.getHostname() + ":" + sn.getPort(), stm.getTimestamp());
                }
            }
            return incrTimeRanges;
        }

        private void setIncrementalTimestampMap(BackupProtos.BackupImage.Builder builder) {
            if (this.incrTimeRanges == null) {
                return;
            }
            for (Map.Entry<TableName, Map<String, Long>> entry : this.incrTimeRanges.entrySet()) {
                TableName key = entry.getKey();
                Map<String, Long> value = entry.getValue();
                BackupProtos.TableServerTimestamp.Builder tstBuilder = BackupProtos.TableServerTimestamp.newBuilder();
                tstBuilder.setTableName(ProtobufUtil.toProtoTableName((TableName)key));
                for (Map.Entry<String, Long> entry2 : value.entrySet()) {
                    String s = entry2.getKey();
                    BackupProtos.ServerTimestamp.Builder stBuilder = BackupProtos.ServerTimestamp.newBuilder();
                    HBaseProtos.ServerName.Builder snBuilder = HBaseProtos.ServerName.newBuilder();
                    ServerName sn = ServerName.parseServerName((String)s);
                    snBuilder.setHostName(sn.getHostname());
                    snBuilder.setPort(sn.getPort());
                    stBuilder.setServerName(snBuilder.build());
                    stBuilder.setTimestamp(entry2.getValue().longValue());
                    tstBuilder.addServerTimestamp(stBuilder.build());
                }
                builder.addTstMap(tstBuilder.build());
            }
        }

        public String getBackupId() {
            return this.backupId;
        }

        private void setBackupId(String backupId) {
            this.backupId = backupId;
        }

        public BackupType getType() {
            return this.type;
        }

        private void setType(BackupType type) {
            this.type = type;
        }

        public String getRootDir() {
            return this.rootDir;
        }

        private void setRootDir(String rootDir) {
            this.rootDir = rootDir;
        }

        public List<TableName> getTableNames() {
            return this.tableList;
        }

        private void setTableList(List<TableName> tableList) {
            this.tableList = tableList;
        }

        public long getStartTs() {
            return this.startTs;
        }

        private void setStartTs(long startTs) {
            this.startTs = startTs;
        }

        public long getCompleteTs() {
            return this.completeTs;
        }

        private void setCompleteTs(long completeTs) {
            this.completeTs = completeTs;
        }

        public ArrayList<BackupImage> getAncestors() {
            if (this.ancestors == null) {
                this.ancestors = new ArrayList();
            }
            return this.ancestors;
        }

        public void removeAncestors(List<String> backupIds) {
            ArrayList<BackupImage> toRemove = new ArrayList<BackupImage>();
            for (BackupImage im : this.ancestors) {
                if (!backupIds.contains(im.getBackupId())) continue;
                toRemove.add(im);
            }
            this.ancestors.removeAll(toRemove);
        }

        private void addAncestor(BackupImage backupImage) {
            this.getAncestors().add(backupImage);
        }

        public boolean hasAncestor(String token) {
            for (BackupImage image : this.getAncestors()) {
                if (!image.getBackupId().equals(token)) continue;
                return true;
            }
            return false;
        }

        public boolean hasTable(TableName table) {
            return this.tableList.contains(table);
        }

        @Override
        public int compareTo(BackupImage other) {
            String name2;
            String thisBackupId = this.getBackupId();
            String otherBackupId = other.getBackupId();
            int index1 = thisBackupId.lastIndexOf("_");
            int index2 = otherBackupId.lastIndexOf("_");
            String name1 = thisBackupId.substring(0, index1);
            if (name1.equals(name2 = otherBackupId.substring(0, index2))) {
                Long thisTS = Long.valueOf(thisBackupId.substring(index1 + 1));
                Long otherTS = Long.valueOf(otherBackupId.substring(index2 + 1));
                return thisTS.compareTo(otherTS);
            }
            return name1.compareTo(name2);
        }

        public boolean equals(Object obj) {
            if (obj instanceof BackupImage) {
                return this.compareTo((BackupImage)obj) == 0;
            }
            return false;
        }

        public int hashCode() {
            int hash = 33 * this.getBackupId().hashCode() + this.type.hashCode();
            hash = 33 * hash + this.rootDir.hashCode();
            hash = 33 * hash + Long.valueOf(this.startTs).hashCode();
            hash = 33 * hash + Long.valueOf(this.completeTs).hashCode();
            for (TableName table : this.tableList) {
                hash = 33 * hash + table.hashCode();
            }
            return hash;
        }

        public Map<TableName, Map<String, Long>> getIncrTimeRanges() {
            return this.incrTimeRanges;
        }

        private void setIncrTimeRanges(Map<TableName, Map<String, Long>> incrTimeRanges) {
            this.incrTimeRanges = incrTimeRanges;
        }

        static class Builder {
            BackupImage image = new BackupImage();

            Builder() {
            }

            Builder withBackupId(String backupId) {
                this.image.setBackupId(backupId);
                return this;
            }

            Builder withType(BackupType type) {
                this.image.setType(type);
                return this;
            }

            Builder withRootDir(String rootDir) {
                this.image.setRootDir(rootDir);
                return this;
            }

            Builder withTableList(List<TableName> tableList) {
                this.image.setTableList(tableList);
                return this;
            }

            Builder withStartTime(long startTime) {
                this.image.setStartTs(startTime);
                return this;
            }

            Builder withCompleteTime(long completeTime) {
                this.image.setCompleteTs(completeTime);
                return this;
            }

            BackupImage build() {
                return this.image;
            }
        }
    }
}

