/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.accumulo.manager.tableOps.clone;

import java.util.EnumSet;

import org.apache.accumulo.core.fate.Repo;
import org.apache.accumulo.core.manager.state.tables.TableState;
import org.apache.accumulo.manager.Manager;
import org.apache.accumulo.manager.tableOps.ManagerRepo;
import org.apache.accumulo.manager.tableOps.Utils;
import org.slf4j.LoggerFactory;

class FinishCloneTable extends ManagerRepo {

  private static final long serialVersionUID = 1L;
  private final CloneInfo cloneInfo;

  public FinishCloneTable(CloneInfo cloneInfo) {
    this.cloneInfo = cloneInfo;
  }

  @Override
  public long isReady(long tid, Manager environment) {
    return 0;
  }

  @Override
  public Repo<Manager> call(long tid, Manager environment) {
    // directories are intentionally not created.... this is done because directories should be
    // unique
    // because they occupy a different namespace than normal tablet directories... also some clones
    // may never create files.. therefore there is no need to consume namenode space w/ directories
    // that are not used... tablet will create directories as needed

    final EnumSet<TableState> expectedCurrStates = EnumSet.of(TableState.NEW);
    if (cloneInfo.isKeepOffline()) {
      environment.getTableManager().transitionTableState(cloneInfo.getTableId(), TableState.OFFLINE,
          expectedCurrStates);
    } else {
      environment.getTableManager().transitionTableState(cloneInfo.getTableId(), TableState.ONLINE,
          expectedCurrStates);
    }
    Utils.unreserveTable(environment, cloneInfo.getTableId(), tid, true);
    Utils.unreserveNamespace(environment, cloneInfo.getNamespaceId(), tid, false);
    Utils.unreserveTable(environment, cloneInfo.getSrcTableId(), tid, false);

    environment.getEventCoordinator().event("Cloned table %s from %s", cloneInfo.getTableName(),
        cloneInfo.getSrcTableId());

    LoggerFactory.getLogger(FinishCloneTable.class)
        .debug("Cloned table " + cloneInfo.getSrcTableId() + " " + cloneInfo.getTableId() + " "
            + cloneInfo.getTableName());

    return null;
  }

  @Override
  public void undo(long tid, Manager environment) {}

}
