---
authors: "thorn0"
title: "Prettier 2.0 “2020”"
---

Better defaults, a better CLI and better heuristics. Oh, and TypeScript 3.8.

After a long and careful consideration, we decided to change the default values for the `trailingComma`, `arrowParens`, and `endOfLine` options. We made the CLI more intuitive. And we've finally dropped support for Node versions older than 10, which had grown to become a huge maintenance hassle and an obstacle for contributors. Read below in details.

<!-- truncate -->

## Highlights

### JavaScript

#### Improved method chain breaking heuristic ([#6685](https://github.com/prettier/prettier/pull/6685) by [@mmkal](https://github.com/mmkal))

Previously, any method call chain of length three or longer would be automatically broken into multiple lines. The new heuristic is based on the complexity of the call arguments in the chain, rather than simply on the chain's length. Now, if chained method calls have arguments which aren't easy to understand at a glance (e.g. functions or deeply-nested objects), the chain is broken. Otherwise, they're allowed to remain on one line. See prior issues [#3197](https://github.com/prettier/prettier/issues/3107), [#4765](https://github.com/prettier/prettier/pull/4765), [#1565](https://github.com/prettier/prettier/issues/1565) and [#4125](https://github.com/prettier/prettier/pull/4125) for context and examples.

To get best results, make sure your value for the [`printWidth` option](/docs/options#print-width) isn't too high.

<!-- prettier-ignore -->
```javascript
// Prettier 1.19
if (
  foo
    .one()
    .two()
    .three() ===
  bar
    .four()
    .five()
    .six()
) {
  // ...
}

// Prettier 2.0
if (foo.one().two().three() === bar.four().five().six()) {
  // ...
}
```

#### Ultimate fix for Closure-style type casts ([#7791](https://github.com/prettier/prettier/pull/7791) by [@thorn0](https://github.com/thorn0), [#7011](https://github.com/prettier/prettier/pull/7011) by [@evilebottnawi](https://github.com/evilebottnawi))

Prettier has been trying not to corrupt these JSDoc type assertions since [v1.6.0](https://prettier.io/blog/2017/08/29/1.6.0.html#handle-closure-compiler-type-cast-syntax-correctly-2484httpsgithubcomprettierprettierpull2484-by-yangsuhttpsgithubcomyangsu), with mixed results. As type checking based on JSDoc becomes increasingly common, we've been getting new bug reports about this syntax. The bugs were tricky because the required parentheses around the expression weren't part of the AST, so Prettier didn't have a good way to detect their presence.

Finally, we used the `createParenthesizedExpressions` option of the Babel parser to represent parentheses in the AST using non-standard nodes. This helped fix all the reported bugs.

Consequently, Prettier won't recognize JSDoc type casts if the `flow` or `typescript` parser is used, but this is reasonable as this syntax makes little sense in Flow and TypeScript files.

<!-- prettier-ignore -->
```jsx
// Input
const nestedAssertions = /** @type {MyType} */
  (/** @type {unknown} */
      (x));

// Prettier 1.19
const nestedAssertions /** @type {MyType} */ /** @type {unknown} */ = x;

// Prettier 2.0
const nestedAssertions = /** @type {MyType} */ (/** @type {unknown} */ (x));
```

Reference documentation for this syntax: [Closure Compiler](https://github.com/google/closure-compiler/wiki/Annotating-JavaScript-for-the-Closure-Compiler#type-type), [TypeScript (with `--checkJs`)](https://www.typescriptlang.org/docs/handbook/type-checking-javascript-files.html#casts).

### TypeScript

#### TypeScript 3.8 ([#7631](https://github.com/prettier/prettier/pull/7631) by [@thorn0](https://github.com/thorn0), [#7764](https://github.com/prettier/prettier/pull/7764) by [@sosukesuzuki](https://github.com/sosukesuzuki), [#7804](https://github.com/prettier/prettier/pull/7804) by [@sosukesuzuki](https://github.com/sosukesuzuki))

Prettier now supports the new syntax added in TypeScript 3.8:

- [Type-Only Imports and Exports](https://devblogs.microsoft.com/typescript/announcing-typescript-3-8/#type-only-imports-exports)
- [ECMAScript Private Fields](https://devblogs.microsoft.com/typescript/announcing-typescript-3-8/#ecmascript-private-fields)
- [`export * as ns`](https://devblogs.microsoft.com/typescript/announcing-typescript-3-8/#export-star-as-namespace-syntax)

### CLI

#### Test whether passed globs are names of existing files before treating them as globs ([#7587](https://github.com/prettier/prettier/pull/7587) by [@fisker](https://github.com/fisker))

Since file names in Linux can contain almost any characters, strings like `foo*.js` and `[bar].css` are valid file names. Previously, if the user needed to format a file named `[bar].css`, a glob escaping syntax had to be used: `prettier "\[bar].css"` (this one doesn't work on Windows, where backslashes are treated as path separators) or `prettier "[[]bar].css"`. Because of this, important use cases were broken. E.g. [lint-staged](https://github.com/okonet/lint-staged) passes absolute paths and knows nothing about the escaping syntax. Now, when Prettier CLI gets a glob, it first tries to resolve it as a literal file name.

#### Expand directories, including `.` ([#7660](https://github.com/prettier/prettier/pull/7660) by [@thorn0](https://github.com/thorn0))

It's finally possible to run `prettier --write .` to format all supported files in the current directory and its subdirectories.
Directory names can be mixed with file names and glob patterns (e.g. `prettier src "test/*.spec.js" foo.js`).

Also, the order in which files are processed has changed. Previously, all the files were sorted alphabetically before
formatting. Now, their order corresponds to the order of the specified paths. For each path, the list of resolved
files is sorted, but the full sorting of the resulting combined list isn't done anymore.

There are also changes in how Prettier CLI reports errors if passed patterns don't match any files. Previously, Prettier CLI
printed a "No matching files" error if it couldn't find any files at all—for all the patterns together, not for an individual pattern.
In Prettier 2.0, the CLI also prints such errors for individual patterns.

## Breaking changes

### API

#### Fix config overrides pattern matching to include dot files ([#5813](https://github.com/prettier/prettier/pull/5813) by [@chrisblossom](https://github.com/chrisblossom))

Previously, [configuration overrides](/docs/configuration#configuration-overrides) weren't applied to files whose name had a leading dot.

#### Drop support for Node versions older than 10 ([#6908](https://github.com/prettier/prettier/pull/6908) by [@fisker](https://github.com/fisker))

The minimal required Node version now is 10.13.0. For our contributors, this means there is no need anymore to jump through hoops to make tests pass on Node 4.

#### Change default value for `trailingComma` to `es5` ([#6963](https://github.com/prettier/prettier/pull/6963) by [@fisker](https://github.com/fisker))

Before version 2.0, Prettier was avoiding trailing commas by default where possible.
This made the resulting JavaScript compatible with now very old environments such as IE8, but implied [some missed opportunities](https://medium.com/@nikgraf/why-you-should-enforce-dangling-commas-for-multiline-statements-d034c98e36f8).

Prettier has included an [option to configure trailing commas](https://prettier.io/docs/options#trailing-commas) since its early days, and an initiative to change the default value has been out there [for over three years](https://github.com/prettier/prettier/issues/68).
Finally, the default value becomes `es5` instead of `none` in Prettier v2.0.

If the old behavior is still preferred, please configure Prettier with `{ "trailingComma": "none" }`.
There is a possibility that the default value will change to `all` (meaning even more trailing commas) in a future major version of Prettier as the JavaScript ecosystem further matures.

#### Plugin API: changes in `prettier.util` ([#6993](https://github.com/prettier/prettier/pull/6993) by [@fisker](https://github.com/fisker))

- `prettier.util.mapDoc` has been removed.<br />
  Use `prettier.doc.utils.mapDoc` instead.
- `prettier.util.isNextLineEmpty` has been updated.<br />
  Use `isNextLineEmpty(text, node, locEnd)` instead of `isNextLineEmpty(text, node, options)`.
- `prettier.util.isPreviousLineEmpty` has been updated.<br />
  Use `isPreviousLineEmpty(text, node, locStart)` instead of `isPreviousLineEmpty(text, node, options)`.
- `prettier.util.getNextNonSpaceNonCommentCharacterIndex` has been updated.<br />
  Use `getNextNonSpaceNonCommentCharacterIndex(text, node, locEnd)` instead of `getNextNonSpaceNonCommentCharacterIndex(text, node, options)`.

#### Change default value for `arrowParens` to `always` ([#7430](https://github.com/prettier/prettier/pull/7430) by [@kachkaev](https://github.com/kachkaev))

[Since version 1.9](https://prettier.io/blog/2017/12/05/1.9.0.html#option-to-add-parens-in-arrow-function-arguments-3324httpsgithubcomprettierprettierpull3324-by-rattrayalexhttpsgithubcomrattrayalex-and-suchipihttpsgithubcomsuchipi), Prettier has had an [option](https://prettier.io/docs/options#arrow-function-parentheses) to always wrap arrow function arguments with parentheses.
In version 2.0, this behavior has become the default.

<!-- prettier-ignore -->
```js
// Input
const fn = (x) => y => x + y;

// Prettier 1.19
const fn = x => y => x + y;

// Prettier 2.0
const fn = (x) => (y) => x + y;
```

At first glance, avoiding parentheses in the isolated example above may look like a better choice because it results in less visual noise.
However, when Prettier removes parentheses, it becomes harder to add type annotations, extra arguments, default values, or a variety of [other things](https://twitter.com/ManuelBieh/status/1181880524954050560).
Consistent use of parentheses provides a better developer experience when editing real codebases, which justifies the change.

You are encouraged to use the new default value, but if the old behavior is still preferred, please configure Prettier with `{ "arrowParens": "avoid" }`.

#### Change default value for `endOfLine` to `lf` ([#7435](https://github.com/prettier/prettier/pull/7435) by [@kachkaev](https://github.com/kachkaev))

Early versions of Prettier were formatting all files with the \*nix flavor of [line endings](https://en.wikipedia.org/wiki/Newline) (`\n`, also known as `LF`).
This behavior was changed in [#472](https://github.com/prettier/prettier/pull/472), which allowed preserving Windows line endings (`\r\n`, also known as `CRLF`).

[Since Prettier version 1.15](https://prettier.io/blog/2018/11/07/1.15.0.html#add-an-option-to-enforce-line-endings-5327-by-kachkaev), the flavor of line endings has been configurable via the `endOfLine` option.
The default value was set to `auto` for backwards compatibility, which meant that Prettier preserved whichever flavor of line endings was already present in a given file.

That meant Prettier converted mixed line endings within one file to what was found at the end of the first line.
However, line endings in separate files could still remain inconsistent.
Besides, contributors on different operating systems could accidentally change line endings in previously committed files and this would be fine with Prettier.
Doing so would produce a large `git diff` and thus make the line-by-line history for a file (`git blame`) harder to explore.

You are encouraged to use the new default value for `endOfLine`, which is now `lf`.
It may be also worth [checking the option docs](https://prettier.io/docs/options#end-of-line) to ensure your project repository is configured correctly.
This will help you avoid a mix of line endings in the repo and a broken `git blame`.
If the old behavior is still preferred, please configure Prettier with `{ "endOfLine": "auto" }`.

If you use Travis CI, be aware of [the `autocrlf` option](https://docs.travis-ci.com/user/customizing-the-build#git-end-of-line-conversion-control) in `.travis.yml`.

#### Cache plugin search results ([#7485](https://github.com/prettier/prettier/pull/7485) by [@fisker](https://github.com/fisker))

Previously, Prettier searched the file system for plugins on every `prettier.format` call. Now, search results are cached. The cache can be cleared by calling `prettier.clearConfigCache()`.

#### Remove deprecated options and option values ([#6993](https://github.com/prettier/prettier/pull/6993), [#7511](https://github.com/prettier/prettier/pull/7511), [#7533](https://github.com/prettier/prettier/pull/7533), [#7535](https://github.com/prettier/prettier/pull/7535), [#7536](https://github.com/prettier/prettier/pull/7536) by [@fisker](https://github.com/fisker))

- Options:
  - `useFlowParser` (`--flow-parser` in CLI) has been deprecated [since `v0.0.10`](https://github.com/prettier/prettier/blob/master/CHANGELOG.md#0010).
- Option values:
  - `parser`: `babylon` (renamed to `babel` in v1.16.0), `postcss` (renamed to `css` in v1.7.1), `typescript-eslint` (an old alias for `typescript`)
  - `proseWrap`: `true` (renamed to `always` in v1.9.0), `false` (renamed to `never` in v1.9.0)
  - `trailingComma`: `true` (renamed to `es5` in v0.19.0), `false` (renamed to `none` in v0.19.0)

#### Remove the `version` parameter of `prettier.getSupportInfo` ([#7620](https://github.com/prettier/prettier/pull/7620) by [@thorn0](https://github.com/thorn0))

Since Prettier 1.8.0, it was possible to pass a version number to `prettier.getSupportInfo` to get information on the languages, options, etc. supported by previous versions. This interesting but apparently not very useful API kept causing maintenance problems and has been removed in Prettier 2.0.0.

## Other changes

### JavaScript

#### Always add a space after the `function` keyword ([#3903](https://github.com/prettier/prettier/pull/3903) by [@j-f1](https://github.com/j-f1), [@josephfrazier](https://github.com/josephfrazier), [@sosukesuzuki](https://github.com/sosukesuzuki), [@thorn0](https://github.com/thorn0); [#7516](https://github.com/prettier/prettier/pull/7516) by [@bakkot](https://github.com/bakkot))

Previously, a space would be added after the `function` keyword in function declarations, but not in function expressions. Now, for consistency, a space is always added after the `function` keyword. The only exception is generator declarations where `function*` is treated as a whole word.

<!-- prettier-ignore -->
```ts
// Prettier 1.19
const identity = function(value) {
  return value;
};
function identity(value) {
  return value;
}
const f = function<T>(value: T) {};
const g = function*() {};

// Prettier 2.0
const identity = function (value) {
  return value;
};
function identity(value) {
  return value;
}
const f = function <T>(value: T) {};
const g = function* () {};
```

#### Fix unstable formatting of labeled statements with comments ([#6984](https://github.com/prettier/prettier/pull/6984) by [@clement26695](https://github.com/clement26695))

<!-- prettier-ignore -->
```jsx
// Input
loop1:
//test
const i = 3;

// Prettier 1.19 (first output)
loop1: //test
const i = 3;

// Prettier 1.19 (second output)
//test
loop1: const i = 3;

// Prettier 2.0 (first and second outputs)
//test
loop1: const i = 3;
```

#### Fix formatting of logical, binary and sequence expressions in template literals ([#7010](https://github.com/prettier/prettier/pull/7010) by [@evilebottnawi](https://github.com/evilebottnawi))

<!-- prettier-ignore -->
```js
// Input
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${foo || bar}`;
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${foo | bar}`;
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${(foo, bar)}`;

// Prettier 1.19
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${foo ||
  bar}`;
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${foo |
  bar}`;
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${(foo,
bar)}`;

// Prettier 2.0
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${
  foo || bar
}`;
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${
  foo | bar
}`;
`111111111 222222222 333333333 444444444 555555555 666666666 777777777 ${
  (foo, bar)
}`;
```

#### Fix unstable formatting of logical expressions ([#7026](https://github.com/prettier/prettier/pull/7026) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```jsx
// Input
const averredBathersBoxroomBuggyNurl =
  bifornCringerMoshedPerplexSawder === 1 ||
  (askTrovenaBeenaDependsRowans === 2 || glimseGlyphsHazardNoopsTieTie === 3);

// Prettier 1.19 (first output)
const averredBathersBoxroomBuggyNurl =
  bifornCringerMoshedPerplexSawder === 1 ||
  askTrovenaBeenaDependsRowans === 2 || glimseGlyphsHazardNoopsTieTie === 3;

// Prettier 1.19 (second output)
const averredBathersBoxroomBuggyNurl =
  bifornCringerMoshedPerplexSawder === 1 ||
  askTrovenaBeenaDependsRowans === 2 ||
  glimseGlyphsHazardNoopsTieTie === 3;

// Prettier 2.0 (first and second outputs)
const averredBathersBoxroomBuggyNurl =
  bifornCringerMoshedPerplexSawder === 1 ||
  askTrovenaBeenaDependsRowans === 2 ||
  glimseGlyphsHazardNoopsTieTie === 3;
```

#### Format `throw` like `return` ([#7070](https://github.com/prettier/prettier/pull/7070) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```js
// Input
function foo() {
  throw this.hasPlugin("dynamicImports") && this.lookahead().type === tt.parenLeft.right;
}

// Prettier 1.19
function foo() {
  throw this.hasPlugin("dynamicImports") &&
    this.lookahead().type === tt.parenLeft.right;
}


// Prettier 2.0
function foo() {
  throw (
    this.hasPlugin("dynamicImports") &&
    this.lookahead().type === tt.parenLeft.right
  );
}
```

#### Fix indentation in ternaries nested in conditions of other ternaries ([#7087](https://github.com/prettier/prettier/pull/7087) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```jsx
// Input
const foo = (number % 10 >= 2 && (number % 100 < 10 || number % 100 >= 20) ?
kochabCooieGameOnOboleUnweave : annularCooeedSplicesWalksWayWay)
? anodyneCondosMalateOverateRetinol : averredBathersBoxroomBuggyNurl;

// Prettier 1.19
const foo = (number % 10 >= 2 && (number % 100 < 10 || number % 100 >= 20)
? kochabCooieGameOnOboleUnweave
: annularCooeedSplicesWalksWayWay)
  ? anodyneCondosMalateOverateRetinol
  : averredBathersBoxroomBuggyNurl;

// Prettier 2.0
const foo = (
  number % 10 >= 2 && (number % 100 < 10 || number % 100 >= 20)
    ? kochabCooieGameOnOboleUnweave
    : annularCooeedSplicesWalksWayWay
)
  ? anodyneCondosMalateOverateRetinol
  : averredBathersBoxroomBuggyNurl;
```

#### Tweak function composition logic for decorators ([#7138](https://github.com/prettier/prettier/pull/7138) by [@brainkim](https://github.com/brainkim))

Because decorators modify the line following, splitting a decorator call’s
arguments onto multiple lines can obscure the relationship between the
decorator and its intended target, producing less-readable code. Therefore, the
function composition logic introduced in [#6033](https://github.com/prettier/prettier/pull/6033) has been changed to exclude
decorator calls.

<!-- prettier-ignore -->
```jsx
// Input
export class Item {
  @OneToOne(() => Thing, x => x.item)
  thing!: Thing;
}

// Prettier 1.19
export class Item {
  @OneToOne(
    () => Thing,
    x => x.item,
  )
  thing!: Thing;
}

// Prettier 2.0
export class Item {
  @OneToOne(() => Thing, x => x.item)
  thing!: Thing;
}
```

#### Correct semicolon placement in empty `return` statement with comment ([#7140](https://github.com/prettier/prettier/pull/7140) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```js
// Input
return // comment
;

// Prettier 1.19
return // comment;

// Prettier 2.0
return; // comment
```

#### Respect whitespace significance in HTML template literals ([#7208](https://github.com/prettier/prettier/pull/7208) by [@saschanaz](https://github.com/saschanaz))

Prettier had been adding newlines for every HTML template string, which could lead to unexpected whitespace in rendered HTML.
This doesn't happen anymore unless `--html-whitespace-sensitivity ignore` option is given.

<!-- prettier-ignore -->
```jsx
// Input
html`<div>`;
html`  <span>TEXT</span>  `;

// Prettier 1.19
html`
  <div></div>
`;
html`
  <span>TEXT</span>
`;

// Prettier 2.0
html`<div></div>`;
html` <span>TEXT</span> `;
```

#### Remove unnecessary parens when yielding JSX ([#7367](https://github.com/prettier/prettier/pull/7367) by [@cola119](https://github.com/cola119))

<!-- prettier-ignore -->
```jsx
// Input
function* f() {
  yield <div>generator</div>;
}

// Prettier 1.19
function* f() {
  yield (<div>generator</div>);
}

// Prettier 2.0
function* f() {
  yield <div>generator</div>;
}
```

#### Keep parentheses around comma expressions in default export declarations ([#7491](https://github.com/prettier/prettier/pull/7491) by [@fisker](https://github.com/fisker))

Omitting these parentheses makes the code invalid.

<!-- prettier-ignore -->
```jsx
// Input
export default (1, 2);

// Prettier 1.19
export default 1, 2;

// Prettier 2.0
export default (1, 2);
```

#### Fix edge cases with parentheses around optional chaining ([#7500](https://github.com/prettier/prettier/pull/7500) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```jsx
// Input
(foo?.bar)();
new (foo?.bar)();

// Prettier 1.19
foo?.bar();
new foo?.bar();

// Prettier 2.0
(foo?.bar)();
new (foo?.bar)();
```

#### Don't wrap `undefined` in parentheses in conditional expressions within JSX ([#7504](https://github.com/prettier/prettier/pull/7504) by [@fisker](https://github.com/fisker))

Previously, parentheses were added around any expression except `null`. Now, `undefined` is excluded too.

<!-- prettier-ignore -->
```js
// Input
foo ? <span>loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong jsx</span> :
undefined
foo ? <span>loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong jsx</span> :
null

// Prettier 1.19
foo ? (
  <span>
    loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong jsx
  </span>
) : (
  undefined
);
foo ? (
  <span>
    loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong jsx
  </span>
) : null;

// Prettier 2.0
foo ? (
  <span>
    loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong jsx
  </span>
) : undefined;
foo ? (
  <span>
    loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong jsx
  </span>
) : null;
```

#### Keep comments position for assignments/variables ([#7709](https://github.com/prettier/prettier/pull/7709) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```jsx
// Input
const foo = /* comments */
  bar;

// Prettier 1.19
const foo /* comments */ = bar;

// Prettier 2.0
const foo = /* comments */ bar;
```

### TypeScript

#### Babel as an alternative parser for TypeScript ([#6400](https://github.com/prettier/prettier/pull/6400) by [@JounQin](https://github.com/JounQin) & [@thorn0](https://github.com/thorn0))

A new value for the `parser` option has been added: `babel-ts`, which makes use of Babel’s TypeScript plugin. The `babel-ts` parser supports JavaScript features not yet supported by TypeScript (ECMAScript proposals, e.g. [private methods and accessors](https://github.com/tc39/proposal-private-methods)), but it's less permissive when it comes to error recovery and less battle-tested than the `typescript` parser. While Babel’s TypeScript plugin is quite mature, ASTs produced by the two parsers aren't 100% compatible. We tried to take the discrepancies into account, but there are most likely still cases where code gets formatted differently or even incorrectly. We call upon you, our users, to help us find such cases. If you see them, please raise issues. In the long run, this will help with [unifying the AST format](https://github.com/typescript-eslint/typescript-eslint/tree/master/packages/typescript-estree#ast-alignment-tests) in future versions of the parsers and thus contribute to a better, more solid JavaScript parser ecosystem.

#### Fix formatting of complex types in arrow function return type annotations ([#6901](https://github.com/prettier/prettier/pull/6901) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```ts
// Input
export const getVehicleDescriptor = async (
  vehicleId: string
): Promise<
  Collections.Parts.PrintedCircuitBoardAssembly["attributes"] & undefined
> => {};

// Prettier 1.19
export const getVehicleDescriptor = async (
  vehicleId: string
): Promise<Collections.Parts.PrintedCircuitBoardAssembly["attributes"] &
  undefined> => {};

// Prettier 2.0
export const getVehicleDescriptor = async (
  vehicleId: string
): Promise<
  Collections.Parts.PrintedCircuitBoardAssembly["attributes"] & undefined
> => {};
```

#### Print JSDoc-only types as is instead of throwing errors ([#7020](https://github.com/prettier/prettier/pull/7020) by [@thorn0](https://github.com/thorn0))

Another fix related to error recovery. Should come in handy to those who migrate from Flow to TypeScript.

<!-- prettier-ignore -->
```jsx
// Input
function fromFlow(arg: ?Maybe) {}

// Prettier 1.19
Error: unknown type: "TSJSDocNullableType"

// Prettier 2.0
function fromFlow(arg: ?Maybe) {}
```

#### Don't print trailing commas after rest elements in tuples ([#7075](https://github.com/prettier/prettier/pull/7075) by [@thorn0](https://github.com/thorn0))

- A rest element is always the last element of a tuple type. Nothing can be added after it.
- While TS is okay with this comma, [Babel doesn't parse it](https://github.com/babel/babel/issues/10769)
- In function parameters and array destructuring, such a comma is a syntax error. Keeping it in tuples is inconsistent.

<!-- prettier-ignore -->
```jsx
// Input
type ValidateArgs = [
	{
		[key: string]: any;
	},
	string,
	...string[],
];

// Prettier 1.19
type ValidateArgs = [
  {
    [key: string]: any;
  },
  string,
  ...string[],
];

// Prettier 2.0
type ValidateArgs = [
  {
    [key: string]: any;
  },
  string,
  ...string[]
];
```

#### Fix indentation of arrow functions in variable declarations followed by comments ([#7094](https://github.com/prettier/prettier/pull/7094) by [@sosukesuzuki](https://github.com/sosukesuzuki))

This could happen with code written in the no-semicolon style if the statement after the variable declaration was prefixed with a semicolon to avoid ASI issues.

<!-- prettier-ignore -->
```ts
// Input
const foo = () => {
  return
}

// foo
;[1,2,3].forEach(bar)

// Prettier 1.19
const foo = () => {
    return;
  };

  // foo
[1, 2, 3].forEach(bar);

// Prettier 2.0
const foo = () => {
  return;
};

// foo
[1, 2, 3].forEach(bar);
```

#### Fix printing of comments in function types ([#7104](https://github.com/prettier/prettier/pull/7104) by [@thorn0](https://github.com/thorn0))

<!-- prettier-ignore -->
```ts
// Input
type f1 = (
  currentRequest: {a: number},
  // TODO this is a very very very very long comment that makes it go > 80 columns
) => number;

// Prettier 1.19
type f1 = (currentRequest: {
  a: number;
}) => // TODO this is a very very very very long comment that makes it go > 80 columns
number;

// Prettier 2.0
type f1 = (
  currentRequest: { a: number }
  // TODO this is a very very very very long comment that makes it go > 80 columns
) => number;
```

#### Fix formatting of comments for function-like nodes ([#7144](https://github.com/prettier/prettier/pull/7144) by [@armano2](https://github.com/armano2))

<!-- prettier-ignore -->
```ts
// Input
interface foo1 {
  bar1/* foo */ (/* baz */) // bat
  bar2/* foo */ ? /* bar */ (/* baz */) /* bat */;
  bar3/* foo */ (/* baz */) /* bat */
  bar4/* foo */ ? /* bar */ (bar: /* baz */ string): /* bat */ string;
  /* foo */ (/* bar */): /* baz */ string;
  /* foo */ (bar: /* bar */ string): /* baz */ string;
  /* foo */ new /* bar */  (a: /* baz */ string): /* bat */ string
  /* foo */ new /* bar */ (/* baz */): /* bat */ string
}

type foo7 = /* foo */ (/* bar */) /* baz */ => void
type foo8 = /* foo */ (a: /* bar */ string) /* baz */ => void
let foo9: new /* foo */ (/* bar */) /* baz */ => string;
let foo10: new /* foo */ (a: /* bar */ string) /* baz */ => string;

// Prettier 1.19
interface foo1 {
  bar1 /* foo */ /* baz */(); // bat
  bar2 /* foo */ /* bar */ /* baz */ /* bat */?();
  bar3 /* foo */ /* baz */() /* bat */;
  bar4 /* foo */?(/* bar */ bar: /* baz */ string): /* bat */ string;
  /* foo */ (): /* bar */ /* baz */ string;
  /* foo */ (bar: /* bar */ string): /* baz */ string;
  /* foo */ new (/* bar */ a: /* baz */ string): /* bat */ string;
  /* foo */ new (): /* bar */ /* baz */ /* bat */ string;
}

type foo7 = /* foo */ () => /* bar */ /* baz */ void;
type foo8 = /* foo */ (a: /* bar */ string) => /* baz */ void;
let foo9: new () => /* foo */ /* bar */ /* baz */ string;
let foo10: new (/* foo */ a: /* bar */ string) => /* baz */ string;

// Prettier 2.0
interface foo1 {
  bar1 /* foo */(/* baz */); // bat
  bar2 /* foo */ /* bar */?(/* baz */) /* bat */;
  bar3 /* foo */(/* baz */) /* bat */;
  bar4 /* foo */?(/* bar */ bar: /* baz */ string): /* bat */ string;
  /* foo */ (/* bar */): /* baz */ string;
  /* foo */ (bar: /* bar */ string): /* baz */ string;
  /* foo */ new (/* bar */ a: /* baz */ string): /* bat */ string;
  /* foo */ new (/* baz */): /* bar */ /* bat */ string;
}

type foo7 = /* foo */ (/* bar */) => /* baz */ void;
type foo8 = /* foo */ (a: /* bar */ string) => /* baz */ void;
let foo9: new (/* bar */) => /* foo */ /* baz */ string;
let foo10: new (/* foo */ a: /* bar */ string) => /* baz */ string;
```

<!-- prettier-ignore -->
```ts
// Input
abstract class Test {
  abstract foo12 /* foo */ (a: /* bar */ string): /* baz */ void
  abstract foo13 /* foo */ (/* bar */) /* baz */
}

// Prettier 1.19
Error: Comment "foo" was not printed. Please report this error!

// Prettier 2.0
abstract class Test {
  abstract foo12 /* foo */(a: /* bar */ string): /* baz */ void;
  abstract foo13 /* foo */(/* bar */); /* baz */
}
```

#### Fix printing of mapped types with the template type omitted ([#7221](https://github.com/prettier/prettier/pull/7221) by [@cola119](https://github.com/cola119))

<!-- prettier-ignore -->
```ts
// Input
type A = { [key in B] };

// Prettier 1.19
type A = { [key in B]: };

// Prettier 2.0
type A = { [key in B] };
```

#### Fix edge cases of printing index signatures ([#7228](https://github.com/prettier/prettier/pull/7228) by [@cola119](https://github.com/cola119))

Even though index signatures without type annotations or with multiple parameters aren't valid TypeScript, the TypeScript parser supports this syntax. In line with the [previous error recovery efforts](https://prettier.io/blog/2019/11/09/1.19.0.html#enable-formatting-even-if-there-are-parse-errors-in-some-cases-6816-by-thorn0-and-the-babel-team), Prettier now makes sure its output still can be parsed in these cases. Previous versions produced unparsable code.

<!-- prettier-ignore -->
```ts
// Input
type A = { [key: string] };
type B = { [a: string, b: string]: string; };

// Prettier 1.19
type A = { [key: string]:  };
type B = { [a: stringb: string]: string };

// Prettier 2.0
type A = { [key: string] };
type B = { [a: string, b: string]: string };
```

#### Fix printing of comments in empty type parameters ([#7729](https://github.com/prettier/prettier/pull/7729) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```ts
// Input
const a: T</* comment */> = 1;

// Prettier 1.19
Error: Comment "comment" was not printed. Please report this error!

// Prettier 2.0
const a: T</* comment */> = 1;
```

### Flow

#### Add support for `symbol` ([#7472](https://github.com/prettier/prettier/pull/7472) by [@fisker](https://github.com/fisker))

A new AST node type was introduced in flow@0.114.0, now it's recognized.

<!-- prettier-ignore -->
```jsx
// Input
const x: symbol = Symbol();

// Prettier after updating Flow, but without this fix
Error: unknown type: "SymbolTypeAnnotation"

// Prettier 2.0
const x: symbol = Symbol();
```

#### Add support for decorators ([#7482](https://github.com/prettier/prettier/pull/7482) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```jsx
// Input
/* @flow */
@decorator4
class Foo {
  @decorator1
  method1() {}

  @decorator2
  @decorator3
  method2() {}
}

// Prettier 1.19
SyntaxError: Unexpected token `@`, expected the token `class` (2:1)

// Prettier 2.0
/* @flow */
@decorator4
class Foo {
  @decorator1
  method1() {}

  @decorator2
  @decorator3
  method2() {}
}
```

#### Fix private class field declarations ([#7484](https://github.com/prettier/prettier/pull/7484) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```js
// Input
class Foo {
  #privateProp: number;
}

// Prettier 1.19
class Foo {
  privateProp: number;
}

// Prettier 2.0
class Foo {
  #privateProp: number;
}
```

### CSS

#### Don't lowercase element names in CSS selectors ([#6947](https://github.com/prettier/prettier/pull/6947) by [@ark120202](https://github.com/ark120202))

Previously, Prettier already preserved casing of unknown element names, but it did lowercase names of HTML elements.
This caused issues when CSS was applied to a case-sensitive document and there was an element with the same name as in HTML, which is the case in NativeScript.
Prettier now always preserves original casing.

<!-- prettier-ignore -->
```css
/* Input */
Label {
}

/* Prettier 1.19 */
label {
}

/* Prettier 2.0 */
Label {
}
```

### SCSS

#### Don't add extra comma after last comment in map ([#6918](https://github.com/prettier/prettier/pull/6918) by [@fisker](https://github.com/fisker))

Previously, when `trailingComma` set to `es5`, an extra comma was added after last comment in an SCSS map.

<!-- prettier-ignore -->
```scss
// Input
$my-map: (
  'foo': 1, // Comment
  'bar': 2, // Comment
);

// Prettier 1.19
$my-map: (
  "foo": 1,
  // Comment
    "bar": 2,
  // Comment,
);

// Prettier 2.0
$my-map: (
  "foo": 1,
  // Comment
    "bar": 2,
  // Comment
);
```

#### Fix whitespace in SCSS concatenation ([#7211](https://github.com/prettier/prettier/pull/7211) by [@sasurau4](https://github.com/sasurau4))

<!-- prettier-ignore -->
```scss
// Input
a {
  background-image: url($test-path + 'static/test.jpg');
}

// Prettier 1.19
a {
  background-image: url($test-path+"static/test.jpg");
}

// Prettier 2.0
a {
  background-image: url($test-path + "static/test.jpg");
}
```

### Less

#### Fix multiple long-standing issues by updating `postcss-less` ([#6981](https://github.com/prettier/prettier/pull/6981) by [@fisker](https://github.com/fisker), [#7021](https://github.com/prettier/prettier/pull/7021) by [@evilebottnawi](https://github.com/evilebottnawi), [@thorn0](https://github.com/thorn0))

- [`each`](http://lesscss.org/functions/#list-functions-each) is supported now ([#5653](https://github.com/prettier/prettier/issues/5653)).
- `!important` was being moved out of mixin call parameters ([#3544](https://github.com/prettier/prettier/issues/3544)).
- Comments in rulesets passed to mixin calls caused duplicate semicolons ([#3096](https://github.com/prettier/prettier/issues/3096)).
- `::before` was broken in mixin call parameters ([#5791](https://github.com/prettier/prettier/issues/5791)).

### HTML

#### Comments in `pre` tags caused bad formatting of following closing tag ([#5959](https://github.com/prettier/prettier/pull/5959) by [@selvazhagan](https://github.com/selvazhagan))

<!-- prettier-ignore -->
```html
<!-- Input -->
<details>
<pre>
    <!-- TEST -->
  </pre></details>

<!-- Prettier 1.19 -->
<details>
  <pre>
    <!-- TEST -->
  </pre></details</details
>

<!-- Prettier 2.0 -->
<details>
  <pre>
    <!-- TEST -->
  </pre>
</details>
```

#### Don't treat colons as namespace prefix delimiters in tag names ([#7273](https://github.com/prettier/prettier/pull/7273) by [@fisker](https://github.com/fisker))

In HTML5, colons don't have any special meaning in tag names. Previously, Prettier treated them the XML way, as namespace prefix delimiters, but no more. In practice, this means that tags whose ancestors have colons in their names are now treated as usual HTML tags: if they're known standard tags, their names can be lowercased and assumptions can be made about their whitespace sensitivity; custom elements whose names are unknown to Prettier retain the casing of their names and, if `--html-whitespace-sensitivity` is set to `css`, are treated as inline.

<!-- prettier-ignore -->
```html
<!-- Input -->
<with:colon>
<div> looooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog block </div>
<DIV> block </DIV><DIV> block </DIV> <DIV> block </DIV><div> block </div><div> block </div>
<pre> pre pr
e</pre>
<textarea> pre-wrap pr
e-wrap </textarea>
<span> looooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog inline </span>
<span> inline </span><span> inline </span> <span> inline </span><span> inline </span>
</with:colon>

<!-- Prettier 1.19 -->
<with:colon>
  <div>
    looooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog block
  </div>
  <DIV> block </DIV><DIV> block </DIV> <DIV> block </DIV><div> block </div
  ><div> block </div>
  <pre> pre pr e</pre>
  <textarea> pre-wrap pr e-wrap </textarea>
  <span>
    looooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog inline
  </span>
  <span> inline </span><span> inline </span> <span> inline </span
  ><span> inline </span>
</with:colon>

<!-- Prettier 2.0 -->
<with:colon>
  <div>
    looooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog block
  </div>
  <div>block</div>
  <div>block</div>
  <div>block</div>
  <div>block</div>
  <div>block</div>
  <pre>
 pre pr
e</pre
  >
  <textarea>
 pre-wrap pr
e-wrap </textarea
  >
  <span>
    looooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog inline
  </span>
  <span> inline </span><span> inline </span> <span> inline </span
  ><span> inline </span>
</with:colon>
```

#### Do not throw on broken HTML ([#7293](https://github.com/prettier/prettier/pull/7293) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```html
<!-- Input -->
<div><span>
<

<!-- Prettier 1.19 -->
TypeError: Cannot read property 'start' of null

<!-- Prettier 2.0 -->
<div><span> < </span></div>
```

#### Fix srcset parse error ([#7295](https://github.com/prettier/prettier/pull/7295) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```html
<!-- Input -->
<img

srcset="
             /media/examples/surfer-240-200.jpg
">

<!-- Prettier 1.19 -->
Error: Mixed descriptor in srcset is not supported

<!-- Prettier 2.0 -->
<img srcset="/media/examples/surfer-240-200.jpg" />
```

#### Fix error thrown on unclosed tag in `pre` tag ([#7392](https://github.com/prettier/prettier/pull/7392) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```html
<!-- Input -->
<pre><br /></pre>
<pre><hr></pre>

<!-- Prettier 1.19 -->
TypeError: Cannot read property 'start' of null

<!-- Prettier 2.0 -->
<pre><br /></pre>
<pre><hr /></pre>
```

#### Fix inconsistent formatting of self-closing tags ([#7395](https://github.com/prettier/prettier/pull/7395) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```html
<!-- Input -->
<span><input type="checkbox"/> </span>
<span><span><input type="checkbox"/></span></span>
<span><input type="checkbox"/></span>

<!-- Prettier 1.19 -->
<span><input type="checkbox" /> </span>
<span
  ><span><input type="checkbox"/></span
></span>
<span><input type="checkbox"/></span>

<!-- Prettier 2.0 -->
<span><input type="checkbox" /> </span>
<span
  ><span><input type="checkbox" /></span
></span>
<span><input type="checkbox" /></span>
```

#### Fix empty lines unexpectedly added after `table` tags ([#7461](https://github.com/prettier/prettier/pull/7461) by [@ikatyang](https://github.com/ikatyang))

<!-- prettier-ignore -->
```html
<!-- Input -->
<table><tr>
</tr>
</table><div>Should not insert empty line before this div</div>

<!-- Prettier 1.19 -->
<table>
  <tr></tr>
</table>

<div>Should not insert empty line before this div</div>

<!-- Prettier 2.0 -->
<table>
  <tr></tr>
</table>
<div>Should not insert empty line before this div</div>
```

#### Format value of HTML `class` attribute ([#7555](https://github.com/prettier/prettier/pull/7555) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```html
<!-- Input -->
<div class="  foo
bar baz"></div>
<div class="
another element with so many classes
even can not fit one line
really a lot and lot of classes
"></div>

<!-- Prettier 1.19 -->
<div
  class="  foo
bar baz"
></div>
<div
  class="
another element with so many classes
even can not fit one line
really a lot and lot of classes
"
></div>

<!-- Prettier 2.0 -->
<div class="foo bar baz"></div>
<div
  class="another element with so many classes even can not fit one line really a lot and lot of classes"
></div>
```

#### Format value of HTML `style` attribute ([#7556](https://github.com/prettier/prettier/pull/7556) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```html
<!-- Input -->
<div style="   color : red;
            display    :inline ">
  </div>
<div style="   color : red;
            display    :inline; height: auto;
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0; ">
  </div>

<!-- Prettier 1.19 -->
<div
  style="   color : red;
            display    :inline "
></div>
<div
  style="   color : red;
            display    :inline; height: auto;
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0; "
></div>


<!-- Prettier 2.0 -->
<div style="color: red; display: inline;"></div>
<div
  style="
    color: red;
    display: inline;
    height: auto;
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
  "
></div>
```

#### Support `<!-- prettier ignore -->` for text ([#7654](https://github.com/prettier/prettier/pull/7654) by [@graemeworthy](https://github.com/graemeworthy))

Previously, it worked only for tags. Useful for protecting various macros and pre-processor commands from being corrupted by formatting.

<!-- prettier-ignore -->
```html
<!-- Input -->
<!-- prettier-ignore -->
A super long string that has been marked as ignore because it was probably generated by some script.

<!-- Prettier 1.19 -->
<!-- prettier-ignore -->
A super long string that has been marked as ignore because it was probably
generated by some script.

<!-- Prettier 2.0 -->
<!-- prettier-ignore -->
A super long string that has been marked as ignore because it was probably generated by some script.
```

<!-- prettier-ignore -->
```html
<!-- Input -->
<!-- prettier-ignore -->
| Dogs | Cats | Weasels | Bats | Pigs | Mice | Hedgehogs | Capybaras | Rats | Tigers |
| ---- | ---- | ------- | ---- | ---- | ---- | --------- | --------- | ---- | ------ |
|    1 |    1 |       0 |    0 |    1 |    1 |         5 |        16 |    4 |      0 |

<!-- Prettier 1.19 -->
<!-- prettier-ignore -->
| Dogs | Cats | Weasels | Bats | Pigs | Mice | Hedgehogs | Capybaras | Rats |
Tigers | | ---- | ---- | ------- | ---- | ---- | ---- | --------- | --------- |
---- | ------ | | 1 | 1 | 0 | 0 | 1 | 1 | 5 | 16 | 4 | 0 |

<!-- Prettier 2.0 -->
<!-- prettier-ignore -->
| Dogs | Cats | Weasels | Bats | Pigs | Mice | Hedgehogs | Capybaras | Rats | Tigers |
| ---- | ---- | ------- | ---- | ---- | ---- | --------- | --------- | ---- | ------ |
|    1 |    1 |       0 |    0 |    1 |    1 |         5 |        16 |    4 |      0 |
```

### Vue

#### Format vue SFC containing JSX script ([#7180](https://github.com/prettier/prettier/pull/7180) by [@sosukesuzuki](https://github.com/sosukesuzuki))

<!-- prettier-ignore -->
```html
<!-- Input -->
<script lang="jsx">
export default {
  data: () => ({
    message: 'hello with jsx'
  }),
  render(h) {



    return <div>{this.message}</div>
  }
}
</script>

<!-- Prettier 1.19 -->
<script lang="jsx">
export default {
  data: () => ({
    message: 'hello with jsx'
  }),
  render(h) {



    return <div>{this.message}</div>
  }
}
</script>

<!-- Prettier 2.0 -->
<script lang="jsx">
export default {
  data: () => ({
    message: "hello with jsx"
  }),
  render(h) {
    return <div>{this.message}</div>;
  }
};
</script>
```

#### Don't print sole string literals in attributes on a new line ([#7479](https://github.com/prettier/prettier/pull/7479) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```vue
<!-- Input -->
<template>
<MyComponent
  :attr1="`loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong ${template} literal value`"
  :attr2="'loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong string literal value'"/>
</template>

<!-- Prettier 1.19 -->
<template>
  <MyComponent
    :attr1="
      `loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong ${template} literal value`
    "
    :attr2="
      'loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong string literal value'
    "
  />
</template>

<!-- Prettier 2.0 -->
<template>
  <MyComponent
    :attr1="`loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong ${template} literal value`"
    :attr2="'loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooong string literal value'"
  />
</template>
```

#### Fix indentation of Vue expressions ([#7781](https://github.com/prettier/prettier/pull/7781) by [@fisker](https://github.com/fisker))

<!-- prettier-ignore -->
```vue
<!-- Input -->
<template>
<MyComponent v-if="
long_long_long_long_long_long_long_condition_1 && long_long_long_long_long_long_long_condition_2 &&
long_long_long_long_long_long_long_condition_3 &&
long_long_long_long_long_long_long_condition_4
"
:attr="
`loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog string 1` +
`loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog string 2`
"
/>
</template>

<!-- Prettier 1.19 -->
<template>
  <MyComponent
    v-if="
      long_long_long_long_long_long_long_condition_1 &&
        long_long_long_long_long_long_long_condition_2 &&
        long_long_long_long_long_long_long_condition_3 &&
        long_long_long_long_long_long_long_condition_4
    "
    :attr="
      `loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog string 1` +
        `loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog string 2`
    "
  />
</template>

<!-- Prettier 2.0 -->
<template>
  <MyComponent
    v-if="
      long_long_long_long_long_long_long_condition_1 &&
      long_long_long_long_long_long_long_condition_2 &&
      long_long_long_long_long_long_long_condition_3 &&
      long_long_long_long_long_long_long_condition_4
    "
    :attr="
      `loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog string 1` +
      `loooooooooooooooooooooooooooooooooooooooooooooooooooooooooooog string 2`
    "
  />
</template>

```

### Angular

#### Unofficial rudimentary support for some of the most used directives of AngularJS 1.x ([#6869](https://github.com/prettier/prettier/pull/6869) by [@thorn0](https://github.com/thorn0))

While there are some syntax incompatibilities (one-time bindings and the precedence of `|`) between the expression languages of the old AngularJS and the new Angular, overall the two languages are compatible enough for legacy and hybrid AngularJS-based projects to be able to benefit from using Prettier. Previously, when Prettier formatted AngularJS templates using the Angular parser, it formatted expressions only in interpolations. Now, some of the most used AngularJS directives are formatted too, namely: `ng-if`, `ng-show`, `ng-hide`, `ng-class`, `ng-style`.

<!-- prettier-ignore -->
```html
<!-- Input -->
<div ng-if="$ctrl .shouldShowWarning&&!$ctrl.loading  ">Warning!</div>

<!-- Prettier 1.19 -->
<div ng-if="$ctrl .shouldShowWarning&&!$ctrl.loading  ">Warning!</div>

<!-- Prettier 2.0 -->
<div ng-if="$ctrl.shouldShowWarning && !$ctrl.loading">Warning!</div>
```

#### Fix formatting of i18n attributes ([#7371](https://github.com/prettier/prettier/pull/7371) by [@thorn0](https://github.com/thorn0))

Prettier 1.19 [added](https://prettier.io/blog/2019/11/09/1.19.0.html#add-formatting-for-i18n-attributes-6695-by-voithos)
support for formatting [i18n attributes](https://angular.io/guide/i18n), but putting the closing quote mark on a new line
broke [custom ids](https://angular.io/guide/i18n#set-a-custom-id-for-persistence-and-maintenance). This is fixed now.

<!-- prettier-ignore -->
```html
<!-- Input -->
<div i18n-prop="Special-property|This is a special property with much important information@@MyTextId"
prop="My Text"></div>

<!-- Prettier 1.19 -->
<div
  i18n-prop="
    Special-property|This is a special property with much important
    information@@MyTextId
  "
  prop="My Text"
></div>

<!-- Prettier 2.0 -->
<div
  i18n-prop="
    Special-property|This is a special property with much important
    information@@MyTextId"
  prop="My Text"
></div>
```

### Handlebars (alpha)

#### Fix superfluous line breaks in `ConcatStatement` ([#7051](https://github.com/prettier/prettier/pull/7051) by [@dcyriller](https://github.com/dcyriller))

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
<a href="a-very-long-href-from-a-third-party-marketing-platform{{id}}longer-than-eighty-chars">Link</a>

{{!-- Prettier 1.19 --}}
<a
  href="a-very-long-href-from-a-third-party-marketing-platform
    {{id}}
    longer-than-eighty-chars"
>
  Link
</a>

{{!-- Prettier 2.0 --}}
<a
  href="a-very-long-href-from-a-third-party-marketing-platform{{id}}longer-than-eighty-chars"
>
  Link
</a>
```

and

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
<div class="hello {{if goodbye true}} {{if goodbye false}} {{if goodbye true}} {{if goodbye false}} {{if goodbye true}}">
  Hello
</div>

{{!-- Prettier 1.19 --}}
<div
  class="hello
    {{if goodbye true}}

    {{if goodbye false}}

    {{if goodbye true}}

    {{if goodbye false}}

    {{if goodbye true}}"
>
  Hello
</div>

{{!-- Prettier 2.0 --}}
<div
  class="hello {{if goodbye true}} {{if goodbye false}} {{if goodbye true}} {{if
      goodbye
      false
    }} {{if goodbye true}}"
>
  Hello
</div>
```

#### Fix a falling mustache issue ([#7052](https://github.com/prettier/prettier/pull/7052) by [@dcyriller](https://github.com/dcyriller))

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
<GlimmerComponent @progress={{aPropertyEndingAfterEightiethColumnToHighlightAWeirdClosingParenIssue}} />

{{!-- Prettier 1.19 --}}
<GlimmerComponent
  @progress={{aPropertyEndingAfterEightiethColumnToHighlightAWeirdClosingParenIssue
  }}
/>

{{!-- Prettier 2.0 --}}
<GlimmerComponent
  @progress={{aPropertyEndingAfterEightiethColumnToHighlightAWeirdClosingParenIssue}}
/>
```

#### Improve `MustacheStatement` printing ([#7157](https://github.com/prettier/prettier/pull/7157) by [@dcyriller](https://github.com/dcyriller))

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
<p>Hi here is your name, as it will be displayed  {{firstName}}  {{lastName}}   , welcome!</p>

{{!-- Prettier 1.19 --}}
<p>
  Hi here is your name, as it will be displayed {{firstName}} {{lastName
  }} , welcome!
</p>

{{!-- Prettier 2.0 --}}
<p>
  Hi here is your name, as it will be displayed {{firstName}} {{
    lastName
  }} , welcome!
</p>
```

#### Add support for `prettier-ignore` ([#7275](https://github.com/prettier/prettier/pull/7275) by [@chadian](https://github.com/chadian))

<!-- prettier-ignore -->
```hbs
{{! Input }}
{{! prettier-ignore }}
<div>
  "hello! my parent was ignored"
  {{#my-crazy-component     "shall"     be="preserved"}}
    <This

      is="preserved"
    />
  {{/my-crazy-component}}
</div>

{{#a-normal-component    isRestoredTo   = "order"    }}
  <ThisWillBeNormal backTo    =   "normal" />
{{/a-normal-component}}

{{! Prettier 1.19 }}
{{! prettier-ignore }}
<div>
  "hello! my parent was ignored"
  {{#my-crazy-component "shall" be="preserved"}}
    <This is="preserved" />
  {{/my-crazy-component}}
</div>

{{#a-normal-component isRestoredTo="order"}}
  <ThisWillBeNormal backTo="normal" />
{{/a-normal-component}}

{{! Prettier 2.0 }}
{{! prettier-ignore }}
<div>
  "hello! my parent was ignored"
  {{#my-crazy-component     "shall"     be="preserved"}}
    <This

      is="preserved"
    />
  {{/my-crazy-component}}
</div>

{{#a-normal-component isRestoredTo='order'}}
  <ThisWillBeNormal backTo='normal' />
{{/a-normal-component}}
```

#### Support printing inline Handlebars in HTML ([#7306](https://github.com/prettier/prettier/pull/7306) by [@dcyriller](https://github.com/dcyriller))

<!-- prettier-ignore -->
```
<!-- Input -->
<script type="text/x-handlebars-template">
  {{component arg1='hey' arg2=(helper this.arg7 this.arg4) arg3=anotherone arg6=this.arg8}}
</script>

<!-- Prettier 1.19 -->
<script type="text/x-handlebars-template">
  {{component arg1='hey' arg2=(helper this.arg7 this.arg4) arg3=anotherone arg6=this.arg8}}
</script>

<!-- Prettier 2.0 -->
<script type="text/x-handlebars-template">
  {{component
    arg1='hey'
    arg2=(helper this.arg7 this.arg4)
    arg3=anotherone
    arg6=this.arg8
  }}
</script>
```

#### Fix stripped value from AttrNode ([#7552](https://github.com/prettier/prettier/pull/7552) by [@bantic](https://github.com/bantic))

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
<ul class="abc
       def">
</ul>

{{!-- Prettier 1.19 --}}
<ul class></ul>

{{!-- Prettier 2.0 --}}
<ul class="abc
       def">
</ul>
```

#### Preserve whitespace control characters ([#7575](https://github.com/prettier/prettier/pull/7575) by [@mahirshah](https://github.com/mahirshah))

<!-- prettier-ignore -->
```hbs
{{!-- Input --}}
{{~#if bar}}
  if1
{{~else~}}
  else
{{~/if~}}

{{!-- Prettier 1.19 --}}
{{#if bar}}
  if1
{{else}}
  else
{{/if}}

{{!-- Prettier 2.0 --}}
{{~#if bar}}
  if1
{{~else~}}
  else
{{~/if~}}
```

### GraphQL

#### Improve detection of separator between interfaces ([#7305](https://github.com/prettier/prettier/pull/7305) by [@fisker](https://github.com/fisker))

Even though using a comma to separate multiple implemented interfaces is deprecated syntax, in order to support legacy use cases, Prettier keeps the original separator and doesn't wilfully replace commas with ampersands. Previously, however, this logic contained a bug, so the wrong separator could end up in the output. This is fixed now.

<!-- prettier-ignore -->
```graphql
# Input
type Type1 implements A, B
# { &  <-- Removing this comment changes the separator in 1.19
{a: a}

type Type2 implements A, B & C{a: a}

# Prettier 1.19
type Type1 implements A & B {
  # { &  <-- Removing this comment changes the separator in 1.19
  a: a
}

type Type2 implements A & B & C {
  a: a
}

# Prettier 2.0
type Type1 implements A, B {
  # { &  <-- Removing this comment changes the separator in 1.19
  a: a
}

type Type2 implements A, B & C {
  a: a
}
```

### Markdown

#### Handle zero-based lists correctly ([#6852](https://github.com/prettier/prettier/pull/6852) by [@evilebottnawi](https://github.com/evilebottnawi))

<!-- prettier-ignore -->
```md
<!-- Input -->
0. List
1. List
2. List

<!-- Prettier 1.19 -->
0. List
1. List
1. List

<!-- Prettier 2.0 -->
0. List
1. List
2. List
```

#### Fix HTML formatting broken if the beginning tag starts after a list item ([#7181](https://github.com/prettier/prettier/pull/7181) and [#7220](https://github.com/prettier/prettier/pull/7220) by [@sasurau4](https://github.com/sasurau4))

Previously, when Prettier formatted an HTML tag placed just after a list item, it would insert indent and break the relationship of open and close tag. Now, Prettier no longer changes anything.

<!-- prettier-ignore -->
```md
<!-- Input -->
- A list item.
<details><summary>Summary</summary>
<p>

- A list item.

</p>
</details>

- A list item
  <blockquote>asdf</blockquote>

<!-- Prettier 1.19 -->
- A list item.

  <details><summary>Summary</summary>
  <p>

- A list item.

</p>
</details>

- A list item
  <blockquote>asdf</blockquote>

<!-- Prettier 2.0 -->
- A list item.
<details><summary>Summary</summary>
<p>

- A list item.

</p>
</details>

- A list item
  <blockquote>asdf</blockquote>
```

#### Fix formatting of multi-line footnote ([#7203](https://github.com/prettier/prettier/pull/7203) by [@sasurau4](https://github.com/sasurau4))

<!-- prettier-ignore -->
```md
<!-- Input -->
Here's a statement[^footnote].

[^footnote]:
    Here's a multi-line footnote walking back the above statement, and showing
    how it's all totally bollocks.

<!-- Prettier 1.19 -->
Here's a statement[^footnote].

[^footnote]:

  Here's a multi-line footnote walking back the above statement, and showing
  how it's all totally bollocks.

<!-- Prettier 2.0 -->
Here's a statement[^footnote].

[^footnote]:
  Here's a multi-line footnote walking back the above statement, and showing
  how it's all totally bollocks.
```

### MDX

#### Add support for JSX fragments ([#6398](https://github.com/prettier/prettier/pull/6398) by [@JounQin](https://github.com/JounQin))

<!-- prettier-ignore -->
```md
<!-- Input -->
<>
    test   <World        />   test
</>       123

<!-- Prettier 1.19 -->
<>
test <World        /> test
</> 123

<!-- Prettier 2.0 -->
<>
  test <World /> test
</> 123
```

#### Fix JSX parsing bugs introduced in Prettier 1.19 ([#6949](https://github.com/prettier/prettier/pull/6949) by [@Tigge](https://github.com/Tigge) & [@thorn0](https://github.com/thorn0))

MDX parsing for JSX failed when encountering JSX elements that where not
parsable as HTML, such as `<Tag value={{a: 'b'}}>test</Tag>`

<!-- prettier-ignore -->
```md
<!-- Input -->

<Tag  value={ {a :  'b'   } }>test</ Tag>

<Foo bg={ 'red' } >
   <div  style={{   display:   'block'}   }>
      <Bar    >hi    </Bar>
       {  hello       }
         {     /* another comment */}
            </div>
</Foo>

<!-- Prettier 1.19 -->

SyntaxError: Unexpected closing tag "Tag". It may happen when the tag has already been closed by another tag. For more info see https://www.w3.org/TR/html5/syntax.html#closing-elements-that-have-implied-end-tags (1:35)
> 1 | <Tag  value={ {a :  'b'   } }>test</ Tag>

<!-- Prettier 2.0 -->

<Tag value={{ a: "b" }}>test</Tag>

<Foo bg={"red"}>
  <div style={{ display: "block" }}>
    <Bar>hi </Bar>
    {hello}
    {/* another comment */}
  </div>
</Foo>

```

### CLI

#### Support file extensions `.cjs` and `.yaml.sed` ([#7210](https://github.com/prettier/prettier/pull/7210) by [@sosukesuzuki](https://github.com/sosukesuzuki))

```sh
# Prettier 1.19
$ prettier test.cjs
test.cjs[error] No parser could be inferred for file: test.cjs

# Prettier 2.0
$ prettier test.cjs
"use strict";
console.log("Hello, World!");
```

#### Respect `--ignore-path` when prettier executes from a subdirectory ([#7588](https://github.com/prettier/prettier/pull/7588) by [@heylookltsme](https://github.com/heylookltsme))

Changes the filename used when filtering ignored files to be relative to the
`--ignore-path`, if present, rather than the current working directory.

#### Remove `--stdin` ([#7668](https://github.com/prettier/prettier/pull/7668) by [@thorn0](https://github.com/thorn0))

This CLI flag, never properly documented, was supposed to make Prettier CLI read input from stdin, but Prettier CLI does so anyway when not given any file paths or glob patterns. So the flag was redundant. Now that it's been removed, if you use this flag in your commands, you'll see a warning: "Ignored unknown option". This warning is just information. It doesn't prevent the command from doing what it should do and doesn't affect the exit code.
