package io

import (
	"io"

	"git.sr.ht/~charles/rq/util"
	"github.com/alecthomas/chroma/v2/quick"
	"gopkg.in/yaml.v3"
)

func init() {
	registerOutputHandler("yaml", func() OutputHandler { return &YAMLOutputHandler{} })
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &YAMLOutputHandler{}

// YAMLOutputHandler handles parsing YAML data.
//
// YAMLOutputHandler does not support output:pretty or output:indent, because go-yaml
// does not support modifying the indent behavior.
type YAMLOutputHandler struct {
	style       string
	colorize    bool
	initialized bool
}

func (y *YAMLOutputHandler) init() {
	if y.initialized {
		return
	}

	y.colorize = false
	y.style = "native"
	y.initialized = true
}

// Name implements OutputHandler.Name().
func (y *YAMLOutputHandler) Name() string {
	return "yaml"
}

// SetOption implements OutputHandler.SetOption().
func (y *YAMLOutputHandler) SetOption(name string, value string) error {
	y.init()

	switch {
	case name == "output.colorize":
		y.colorize = util.StringToValue(value).(bool)
	case name == "output.style":
		y.style = value
	}

	return nil
}

// Format implements OutputHandler.Format()
func (y *YAMLOutputHandler) Format(writer io.Writer, data interface{}) error {
	var err error
	var yamlBytes []byte

	y.init()

	yamlBytes, err = yaml.Marshal(data)
	if err != nil {
		return err
	}

	if y.colorize {
		err := quick.Highlight(writer, string(yamlBytes), "yaml", "terminal", y.style)
		if err != nil {
			return nil
		}
	} else {
		_, err := writer.Write(yamlBytes)
		if err != nil {
			return nil
		}
	}

	return nil
}
