package structr

import (
	"cmp"
	"slices"
	"testing"

	"codeberg.org/gruf/go-byteutil"
	"codeberg.org/gruf/go-kv/format"
)

func TestTimeline(t *testing.T) {
	t.Run("structA.Field1", func(t *testing.T) {
		testTimeline(t, testStructA, "Field1", func(sa *structA) (string, bool) {
			return sa.Field1, (sa.Field1 != "")
		})
	})

	t.Run("structA.Field2", func(t *testing.T) {
		testTimeline(t, testStructA, "Field2", func(sa *structA) (int, bool) {
			return sa.Field2, (sa.Field2 != 0)
		})
	})

	t.Run("structA.Field3", func(t *testing.T) {
		testTimeline(t, testStructA, "Field3", func(sa *structA) (float32, bool) {
			return sa.Field3, (sa.Field3 != 0)
		})
	})

	t.Run("structB.Field1", func(t *testing.T) {
		testTimeline(t, testStructB, "Field1", func(sb *structB) (string, bool) {
			if sb.Field1 == nil {
				return "", false
			}
			return *sb.Field1, (*sb.Field1 != "")
		})
	})

	t.Run("structB.Field2", func(t *testing.T) {
		testTimeline(t, testStructB, "Field2", func(sb *structB) (int, bool) {
			if sb.Field2 == nil {
				return 0, false
			}
			return *sb.Field2, (*sb.Field2 != 0)
		})
	})

	t.Run("structB.Field3", func(t *testing.T) {
		testTimeline(t, testStructB, "Field3", func(sb *structB) (float64, bool) {
			if sb.Field3 == nil {
				return 0, false
			}
			return *sb.Field3, (*sb.Field3 != 0)
		})
	})

	t.Run("structC.Field1", func(t *testing.T) {
		testTimeline(t, testStructC, "Field1", func(sc *structC) (string, bool) {
			return sc.Field1, (sc.Field1 != "")
		})
	})

	t.Run("structC.Field3.Field1", func(t *testing.T) {
		testTimeline(t, testStructC, "Field3.Field1", func(sc *structC) (string, bool) {
			if sc.Field3 == nil {
				return "", false
			}
			return sc.Field3.Field1, (sc.Field3.Field1 != "")
		})
	})

	t.Run("structC.Field11.Field2", func(t *testing.T) {
		testTimeline(t, testStructC, "Field11.Field2", func(sc *structC) (string, bool) {
			return sc.Field11.Field2, (sc.Field11.Field2 != "")
		})
	})
}

func testTimeline[T any, PK cmp.Ordered](
	t *testing.T,
	test test[T],
	pkey string,
	get_pkey func(*T) (PK, bool),
) {
	var ti Timeline[*T, PK]

	// Create invalidate function hook
	// to track invalidated value ptrs.
	invalidated := make(map[string]bool)
	invalidateFn := func(value *T) {
		var buf byteutil.Buffer
		format.Appendf(&buf, "{:?}", value)
		invalidated[buf.String()] = true
	}
	wasInvalidated := func(value *T) bool {
		var buf byteutil.Buffer
		format.Appendf(&buf, "{:?}", value)
		return invalidated[buf.String()]
	}

	// Use our own data-backing for slice.
	test.indices = slices.Clone(test.indices)

	// Specifically delete the index config of field we're using as primary key.
	test.indices = slices.DeleteFunc(test.indices, func(idx IndexConfig) bool {
		return idx.Fields == pkey
	})

	// Initialize the timeline cache.
	ti.Init(TimelineConfig[*T, PK]{
		PKey: IndexConfig{
			Fields:    pkey,
			AllowZero: true,
			Multiple:  true,
		},
		Indices:    test.indices,
		Copy:       test.copyfn,
		Invalidate: invalidateFn,
	})

	// Check that fake indices cause panic
	for _, index := range test.indices {
		fake := index.Fields + "!"
		catchpanic(t, func() {
			ti.Index(fake)
		}, "unknown index: "+fake)
	}

	// Check that wrong
	// index causes panic
	catchpanic(t, func() {
		wrong := new(Index)
		ti.Invalidate(wrong)
	}, "invalid index for timeline")

	// Insert all values into timeline.
	t.Logf("Insert: %v", test.values)
	ti.Insert(test.values...)

	// Ensure timeline fully populated.
	if ti.Len() != len(test.values) {
		t.Fatal("timeline not fully populated after insert")
	}

	// Extract primary keys for all test values.
	pkeys := get_pkeys(test.values, get_pkey)
	slices.Sort(pkeys)
	min := pkeys[0]
	max := pkeys[len(pkeys)-1]
	length := len(pkeys)

	var check []*T

	// Select values over total range of primary keys,
	// in ascending order (i.e. from top to bottom).
	check = ti.Select(&min, nil, &length, Asc)
	t.Logf("select(asc): %+v", check)

	// Check whether returned values in same order as primary keys.
	if !slices.EqualFunc(pkeys, check, func(pkey PK, value *T) bool {
		pkey2, _ := get_pkey(value)
		return pkey2 == pkey
	}) {
		t.Fatalf("selected ASC values in unexpected order / length\n\npkeys=%v\n\ncheck=%v\n\n", pkeys, check)
	}

	// Reverse slice before selecting
	// values in opposite direction.
	slices.Reverse(pkeys)

	// Select values over total range of primary keys,
	// in descending order (i.e. from bottom to top).
	check = ti.Select(nil, &max, &length, Desc)
	t.Logf("select(desc): %+v", check)

	// Check whether returned values in same order as primary keys.
	if !slices.EqualFunc(pkeys, check, func(pkey PK, value *T) bool {
		pkey2, _ := get_pkey(value)
		return pkey2 == pkey
	}) {
		t.Fatalf("selected DESC values in unexpected order / length\n\npkeys=%v\n\ncheck=%v\n\n", pkeys, check)
	}

	// Range over all the test indices.
	for _, index := range test.indices {
		var keys []Key
		var expect []*T

		// Get associated structr index.
		idx := ti.Index(index.Fields)

		for _, value := range test.values {
			// Get struct key parts for value.
			parts, _ := indexkey(idx, value)

			// Generate key from parts.
			key := idx.Key(parts...)

			// Only deal with zero value keys if
			// this index permits indexing them.
			if key.Zero() && !index.AllowZero {
				continue
			}

			// Check whether this is a duplicate key.
			dupe := slices.ContainsFunc(keys, key.Equal)
			if !dupe {

				// Only append to keys if
				// this is a unique key.
				keys = append(keys, key)
			}

			if !dupe || index.Multiple {
				// Only append to values if
				// unique or allows multiple.
				expect = append(expect, value)
			}
		}

		var ranged []*T

		// Range the values for index keys in timeline.
		t.Logf("RangeKeys: %s %v", index.Fields, keys)
		ti.RangeKeys(idx, keys...)(func(value *T) bool {
			ranged = append(ranged, value)
			return true
		})

		// Check for expect no. values.
		if len(expect) != len(ranged) {
			t.Fatalf("unexpected length of ranged values: expect=%d ranged=%d", len(expect), len(ranged))
		}

		// Check values expected are there.
		for _, expect := range expect {
			if !slices.ContainsFunc(ranged, func(value *T) bool {
				return test.equalfn(expect, value)
			}) {
				t.Fatal("missing values in range")
			}
		}

		var returned bool

		// Range the values for index keys, but test yield return bool.
		t.Logf("RangeKeys: %s %v (return early)", index.Fields, keys)
		ti.RangeKeys(idx, keys...)(func(_ *T) bool {
			if returned {
				t.Fatal("range kept iterating after false return")
			}
			returned = true
			return false
		})
	}

	// Reverse the primary keys slice
	// to get them back in ASC order.
	slices.Reverse(pkeys)

	// Clear slice before
	// any reuse below.
	clear(check)
	check = check[:0]

	// Range all values in ASC order,
	// appending values to check slice.
	ti.Range(Asc)(func(value *T) bool {
		check = append(check, value)
		return true
	})
	t.Logf("Range(Asc): %#v", check)

	// Check whether returned values in same order as primary keys.
	if !slices.EqualFunc(pkeys, check, func(pkey PK, value *T) bool {
		pkey2, _ := get_pkey(value)
		return pkey2 == pkey
	}) {
		t.Fatalf("ranged ASC values in unexpected order / length\n\npkeys=%v\n\ncheck=%v\n\n", pkeys, check)
	}

	// Reverse the primary keys slice
	// to get them back in DESC order.
	slices.Reverse(pkeys)

	// Clear slice before
	// any reuse below.
	clear(check)
	check = check[:0]

	// Range all values in DESC order,
	// appending values to check slice.
	ti.Range(Desc)(func(value *T) bool {
		check = append(check, value)
		return true
	})
	t.Logf("Range(Desc): %#v", check)

	// Check whether returned values in same order as primary keys.
	if !slices.EqualFunc(pkeys, check, func(pkey PK, value *T) bool {
		pkey2, _ := get_pkey(value)
		return pkey2 == pkey
	}) {
		t.Fatalf("ranged DESC values in unexpected order / length\n\npkeys=%v\n\ncheck=%v\n\n", pkeys, check)
	}

	// Before doing invalidation
	// hook check, clear the map.
	clear(invalidated)

	// Invalidate each of these values from
	// timeline. It's easier to just iterate
	// through all the values for all indices
	// instead of getting particular about which
	// value is cached in which particular index.
	for _, index := range test.indices {
		var keys []Key

		// Get associated structr index.
		idx := ti.Index(index.Fields)

		for _, value := range test.values {
			// get struct key parts for value.
			parts, _ := indexkey(idx, value)

			// generate key from parts.
			key := idx.Key(parts...)

			if !index.AllowZero && key.Zero() {
				// Key parts contain a zero value and this
				// index does not allow that. Skip lookup.
				continue
			}

			// add index key to keys.
			keys = append(keys, key)
		}

		// Invalidate all keys in index.
		t.Logf("Invalidate: %s %v", index.Fields, keys)
		ti.Invalidate(idx, keys...)
	}

	// Ensure timeline empty
	// after invalidation.
	if ti.Len() != 0 {
		t.Fatal("timeline not empty after invalidation")
	}

	// Ensure all values were invalidated
	// as expected via the callback function.
	for _, value := range test.values {
		if !wasInvalidated(value) {
			t.Fatalf("expected value was not invalidated: %+v", value)
		}
	}

	// Reinsert all values into timeline.
	t.Logf("Insert: %v", test.values)
	ti.Insert(test.values...)

	// Ensure timeline fully populated.
	if ti.Len() != len(test.values) {
		t.Fatal("timeline not fully populated after insert")
	}

	// At this point, the values in `check`
	// should still be a collection of all
	// values in DESCENDING order. Use this
	// to check against range after trim.
	expect := slices.Clone(check)

	// Simulate a trim on expected slice from
	// TOP of timeline by remove first item.
	expect = expect[1:]

	// Trim 1 value from TOP of timeline.
	t.Logf("Trim: %d Desc", len(test.values)-1)
	ti.Trim(len(expect), Desc)

	// Ensure trim reduced length.
	if ti.Len() != len(expect) {
		t.Fatal("unexpected timeline length after trim")
	}

	// Clear slice before
	// any reuse below.
	clear(check)
	check = check[:0]

	// Range all values in DESC order,
	// appending values to check slice.
	ti.Range(Desc)(func(value *T) bool {
		check = append(check, value)
		return true
	})
	t.Logf("Range(Desc): %#v", check)

	// Check whether returned values in expected order.
	if !slices.EqualFunc(expect, check, test.equalfn) {
		t.Fatalf("ranged DESC values after trim in unexpected order / length\n\nexpect=%v\n\ncheck=%v\n\n", expect, check)
	}

	// Simulate a trim on expected slice from
	// BOTTOM of timeline by remove last item.
	expect = expect[:len(expect)-1]

	// Trim 1 value from BOTTOM of timeline.
	t.Logf("Trim: %d Asc", len(test.values)-1)
	ti.Trim(len(expect), Asc)

	// Ensure trim reduced length.
	if ti.Len() != len(expect) {
		t.Fatal("unexpected timeline length after trim")
	}

	// Clear slice before
	// any reuse below.
	clear(check)
	check = check[:0]

	// Range all values in DESC order,
	// appending values to check slice.
	ti.Range(Desc)(func(value *T) bool {
		check = append(check, value)
		return true
	})
	t.Logf("Range(Desc): %#v", check)

	// Check whether returned values in expected order.
	if !slices.EqualFunc(expect, check, test.equalfn) {
		t.Fatalf("ranged DESC values after trim in unexpected order / length\n\nexpect=%v\n\ncheck=%v\n\n", expect, check)
	}

	// Finally, clear timeline
	// i.e. call .Trim(0, ...).
	ti.Clear()

	// Ensure timeline
	// empty after clear.
	if ti.Len() != 0 {
		t.Fatal("timeline not empty after clear")
	}
}

func get_pkeys[T1, T2 any](v1 []T1, get func(T1) (T2, bool)) []T2 {
	if get == nil {
		panic("nil get fn")
	}
	v2 := make([]T2, 0, len(v1))
	for _, v := range v1 {
		vv, _ := get(v)
		v2 = append(v2, vv)
	}
	return v2
}
