package s3_test

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"io"
	"os"
	"strings"
	"testing"

	"codeberg.org/gruf/go-storage"
	"codeberg.org/gruf/go-storage/s3"
	"github.com/minio/minio-go/v7"
	"github.com/minio/minio-go/v7/pkg/credentials"
)

func TestS3Storage(t *testing.T) {
	// Load test S3 instance info
	addr := os.Getenv("MINIO_ADDR")
	bucket := os.Getenv("MINIO_BUCKET")

	if addr == "" && bucket == "" {
		t.Skip("skipping S3Storage tests")
		return
	}

	// Attempt to open connection to S3 storage
	st, err := s3.Open(addr, bucket, &s3.Config{
		CoreOpts: minio.Options{
			Creds: credentials.New(&credentials.EnvMinio{}),
		},
	})
	if err != nil {
		t.Fatalf("Failed opening storage: %v", err)
	}

	// Run the storage tests
	testStorage(t, st)
}

func TestS3StorageChunkedWrite(t *testing.T) {
	// Load test S3 instance info
	addr := os.Getenv("MINIO_ADDR")
	bucket := os.Getenv("MINIO_BUCKET")

	if addr == "" && bucket == "" {
		t.Skip("skipping S3Storage tests")
		return
	}

	// Attempt to open connection to S3 storage
	st, err := s3.Open(addr, bucket, &s3.Config{
		CoreOpts: minio.Options{
			Creds: credentials.New(&credentials.EnvMinio{}),
		},
	})
	if err != nil {
		t.Fatalf("Failed opening storage: %v", err)
	}

	data := []byte("hello world\n1\n2\n3\n4\nfoobar\nbarfoo")

	// Stream this data into store (no ReaderSize, will be chunked)
	if _, err := st.WriteStream(context.TODO(), "data", bytes.NewReader(data)); err != nil {
		t.Fatalf("Failed chunked writing of data: %v", err)
	}

	// Read the written data back from store
	check, err := st.ReadBytes(context.TODO(), "data")
	if err != nil {
		t.Fatalf("Failed reading stored data: %v", err)
	}

	// Check that data is as expected
	if string(data) != string(check) {
		t.Fatal("Failed ensuring stored data matches input")
	}
}

func dataf(s string, a ...interface{}) []byte {
	return []byte(fmt.Sprintf(s, a...))
}

func testStorage(t *testing.T, st storage.Storage) {
	expected := map[string][]byte{
		"test1": dataf("hello world!"),
		"test2": dataf("hello world!\nhelloworld!\nhelloworld!\nhewwoooo wooowddd\n"),
		"test3": dataf("i dunno some random data here %d", 64),
		"test4": dataf("aaaaaaaaaaaaaaaa asfj;asdflkjasd;kfnasdklf"),
	}

	t.Run("invalid key", func(t *testing.T) {
		testStorageInvalidKeys(t, st)
	})

	// Write bytes first to give something for later reads
	t.Run("Storage.WriteBytes()", func(t *testing.T) {
		testStorageWriteBytes(t, st, "test1", expected["test1"])
		testStorageWriteBytes(t, st, "test2", expected["test2"])
	})
	t.Run("Storage.WriteStream()", func(t *testing.T) {
		testStorageWriteStream(t, st, "test3", expected["test3"])
		testStorageWriteStream(t, st, "test4", expected["test4"])
	})

	// Now attempt to read from previous tests
	t.Run("Storage.ReadBytes()", func(t *testing.T) {
		testStorageReadBytes(t, st, "test1", expected["test1"])
		testStorageReadBytes(t, st, "test2", expected["test2"])
	})
	t.Run("Storage.ReadStream()", func(t *testing.T) {
		testStorageReadStream(t, st, "test3", expected["test3"])
		testStorageReadStream(t, st, "test4", expected["test4"])
	})

	// Check that files from previous tests exist
	t.Run("Storage.Stat()", func(t *testing.T) {
		for key := range expected {
			testStorageStat(t, st, key)
		}
	})

	// Attempt to remove 1 of files from previous test
	t.Run("Storage.Remove()", func(t *testing.T) {
		testStorageRemove(t, st, "test1")
		delete(expected, "test1")
	})

	// Walk keys and check expected
	t.Run("Storage.WalkKeys()", func(t *testing.T) {
		expect := make([]string, 0, len(expected))
		for key := range expected {
			expect = append(expect, key)
		}
		testStorageWalkKeys(t, st, expect)
	})

	// We run clean last so there is something to clean
	t.Run("Storage.Clean()", func(t *testing.T) { testStorageClean(t, st) })
}

func testStorageInvalidKeys(t *testing.T, st storage.Storage) {
	var err error

	for _, key := range []string{
		// possible invalid keys
		"",
		strings.Repeat(" ", 1025),
	} {
		_, err = st.ReadStream(context.Background(), key)
		if !errors.Is(err, storage.ErrInvalidKey) {
			t.Fatalf("invalid key did not return error: %v", err)
		}

		_, err = st.WriteStream(context.Background(), key, bytes.NewReader(nil))
		if !errors.Is(err, storage.ErrInvalidKey) {
			t.Fatalf("invalid key did not return error: %v", err)
		}

		_, err = st.Stat(context.Background(), key)
		if !errors.Is(err, storage.ErrInvalidKey) {
			t.Fatalf("invalid key did not return error: %v", err)
		}

		err = st.Remove(context.Background(), key)
		if !errors.Is(err, storage.ErrInvalidKey) {
			t.Fatalf("invalid key did not return error: %v", err)
		}
	}
}

func testStorageClean(t *testing.T, st storage.Storage) {
	err := st.Clean(context.TODO())
	if err != nil {
		t.Fatalf("Error cleaning storage: %v", err)
	}
}

func testStorageReadBytes(t *testing.T, st storage.Storage, key string, value []byte) {
	b, err := st.ReadBytes(context.TODO(), key)
	if err != nil {
		t.Fatalf("Error reading from storage: %v", err)
	}
	if string(b) != string(value) {
		t.Fatalf("Error reading expected value '%s' from storage stream '%s'", value, b)
	}
	if _, err := st.ReadBytes(context.TODO(), key+"."); !errors.Is(err, storage.ErrNotFound) {
		t.Fatalf("Unexpected error reading non-existent file from storage: %v", err)
	}
}

func testStorageReadStream(t *testing.T, st storage.Storage, key string, value []byte) {
	r, err := st.ReadStream(context.TODO(), key)
	if err != nil {
		t.Fatalf("Error reading from storage: %v", err)
	}
	defer r.Close()
	b, err := io.ReadAll(r)
	if err != nil {
		t.Fatalf("Error reading from storage stream: %v", err)
	}
	if string(b) != string(value) {
		t.Fatalf("Error reading expected value '%s' from storage stream '%s'", value, b)
	}
	if _, err := st.ReadStream(context.TODO(), key+"."); !errors.Is(err, storage.ErrNotFound) {
		t.Fatalf("Unexpected error reading non-existent file from storage: %v", err)
	}
}

func testStorageWriteBytes(t *testing.T, st storage.Storage, key string, value []byte) {
	n, err := st.WriteBytes(context.TODO(), key, value)
	if err != nil {
		t.Fatalf("Error writing to storage: %v", err)
	} else if n != len(value) {
		t.Fatalf("Error writing to storage: did not write expected no. bytes")
	}
}

func testStorageWriteStream(t *testing.T, st storage.Storage, key string, value []byte) {
	n, err := st.WriteStream(context.TODO(), key, bytes.NewReader(value))
	if err != nil {
		t.Fatalf("Error writing to storage: %v", err)
	} else if int(n) != len(value) {
		t.Fatalf("Error writing to storage: did not write expected no. bytes")
	}

	n, err = st.WriteStream(context.TODO(), key, &struct{ io.Reader }{bytes.NewReader(value)})
	if err != nil {
		t.Fatalf("Error writing to storage: %v", err)
	} else if int(n) != len(value) {
		t.Fatalf("Error writing to storage: did not write expected no. bytes")
	}
}

func testStorageStat(t *testing.T, st storage.Storage, key string) {
	stat, err := st.Stat(context.TODO(), key)
	if err != nil {
		t.Fatalf("Error stat'ing storage: %v", err)
	}
	if stat == nil {
		t.Fatalf("Error stat'ing expected file in storage")
	}
}

func testStorageRemove(t *testing.T, st storage.Storage, key string) {
	if err := st.Remove(context.TODO(), key); err != nil {
		t.Fatalf("Error removing from storage: %v", err)
	}
	if err := st.Remove(context.TODO(), key+"."); !errors.Is(err, storage.ErrNotFound) {
		t.Fatalf("Unexpected error removing non-existent file from storage: %#v", err)
	}
}

func testStorageWalkKeys(t *testing.T, st storage.Storage, expect []string) {
	remain := make([]string, len(expect))
	copy(remain, expect)

	err := st.WalkKeys(context.TODO(), storage.WalkKeysOpts{
		Step: func(entry storage.Entry) error {
			key := entry.Key

			var i int
			for i = 0; i < len(remain); i++ {
				if key == remain[i] {
					break
				}
			}

			if i == len(remain) {
				return fmt.Errorf("Error finding unexpected key in storage '%s'", key)
			} else {
				remain = append(remain[:i], remain[i+1:]...)
			}

			return nil
		},
	})
	if err != nil {
		t.Fatalf("Error walking keys in storage: %v", err)
	}

	if len(remain) > 0 {
		t.Fatalf("Error finding expected keys in storage, '%v' not found", remain)
	}
}
