/*
 * Copyright (c) 2015 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/param.h>

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif

#define STATUS_DELAY	5
 
static int
get_abricks(abricksp, abricks_countp)
	char ***abricksp;
	size_t *abricks_countp;
{
	char myname[MAXHOSTNAMELEN] = "";
	int order = 0;
	char **abricks = NULL;
	size_t abricks_count = 0;
	FILE *info;
	char line[1024];
	int i;
	char *cp;

	if (gethostname(myname, MAXHOSTNAMELEN) == 0) {
		if ((cp = strchr(myname, (int)'.')) != NULL)
			*cp = '\0';
	}

	if ((info = popen("gluster volume info", "r")) == NULL) {
		printf("CRITICAL: gluster volume info failed\n");
		return STATE_CRITICAL;
	}

	while (fgets(line, sizeof(line), info) != NULL) {
		if (strncmp(line, "Brick", 5) != 0)
			continue;

		for (i = 5; isdigit((int)line[i]); i++);

		if (line[i++] != ':' && line[i++] != ' ')
			continue;

		while (line[++i] == ' ');

		abricks_count++;
		abricks = realloc(abricks, 
				  sizeof(*abricks) * abricks_count);
		if (abricks == NULL) {
			printf("UNKNOWN: out of memory\n");
			return STATE_UNKNOWN;
		}

		abricks[abricks_count - 1] = strdup(line + i);
		if (abricks[abricks_count - 1] == NULL) {
			printf("UNKNOWN: strdup failed\n");
			return STATE_UNKNOWN;
		}

		if ((cp = strchr(abricks[abricks_count - 1], '\n')) != NULL)
			*cp = '\0';

		/* Find this peer frist occurence in the list */
		if (order == 0 &&
		    strncmp(abricks[abricks_count - 1],
			    myname, strlen(myname)) == 0)
			order = abricks_count; 
			
	}
	pclose(info);

	*abricksp = abricks;
	*abricks_countp = abricks_count;

	/*
	 * Sleep for a peer-specific duration so that all peers
	 * running this test do not call gluster volume status 
	 * at the same time: if one fails to get a lock it wrongly
	 * reports offline bricks as of glusterfs 3.7.4.
	 */
	sleep(order * STATUS_DELAY);

	return STATE_OK;
}

static int
get_obricks(obricksp, obricks_countp)
	char ***obricksp;
	size_t *obricks_countp;
{
	char **obricks = NULL;
	size_t obricks_count = 0;
	FILE *status;
	char line[1024];
	int i;
	char *cp;

	if ((status = popen("gluster volume status", "r")) == NULL) {
		printf("CRITICAL: gluster volume status failed\n");
		return STATE_CRITICAL;
	}

	while (fgets(line, sizeof(line), status) != NULL) {
		int online = 0;
		char *saved;

		if (strncmp(line, "Brick", 5) != 0)
			continue;

		for (i = 5; line[i] == ' '; i++);

		saved = strdup(line + i);
		if (saved == NULL) {
			printf("UNKNOWN: strdup failed\n");
			return STATE_UNKNOWN;
		}


		for ((cp = strtok(saved, " ")), i = 0;
		     cp != NULL;
		     (cp = strtok(NULL, " "), i++)) {
			if ((i == 3) && (*cp == 'Y')) {
				online = 1;
				break;
			}
		}

		if (!online) {
			free(saved);
			continue;
		}

		obricks_count++;
		obricks = realloc(obricks, 
				  sizeof(*obricks) * obricks_count);
		if (obricks == NULL) {
			printf("UNKNOWN: out of memory\n");
			return STATE_UNKNOWN;
		}


		obricks[obricks_count - 1] = saved;
	}
	pclose(status);

	*obricksp = obricks;
	*obricks_countp = obricks_count;
	return STATE_OK;
}

int
main(void)
{
	int i, j, status;
	char **abricks = NULL;
	size_t abricks_count = 0;
	char **obricks = NULL;
	size_t obricks_count = 0;

	if ((status = get_abricks(&abricks, &abricks_count)) != STATE_OK)
		return status;

	if ((status = get_obricks(&obricks, &obricks_count)) != STATE_OK)
		return status;

	if (abricks_count == obricks_count) {
		printf("OK: %d bricks online\n", abricks_count);
		return STATE_OK;
	}

	printf("CRITICAL: %d offline bricks ", abricks_count - obricks_count);

	for (i = 0; i < abricks_count; i++) {
		int online = 0;
		for (j = 0; j < obricks_count; j++) {
			if (strcmp(abricks[i], obricks[j]) == 0) {
				online = 1; 
				break;
			}
		}

		if (!online)
			printf("%s ", abricks[i]);
	}
	printf("\n");

	return STATE_CRITICAL;
}
