/*
 * Copyright (c) 2013-2022 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define _BSD_SOURCE
#define _XOPEN_SOURCE 500

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <signal.h>
#include <getopt.h>
#include <err.h>
#include <errno.h>
#include <sysexits.h>
#include <fnmatch.h>
#include <time.h>
#include <paths.h>
#include <sys/stat.h>
#include <sys/param.h>

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif

#define DOT_PID ".pid"

int
check_dir(dirname, soft_fail, running, running_len, dead, dead_len, state,
	  ex, exlen)
	char *dirname;
	int soft_fail;
	char *running;
	size_t running_len;
	char *dead;
	size_t dead_len;
	int state;
	char **ex;
	size_t exlen;
{
	DIR *dirp;
	struct dirent *dp;
	char tmp[4096] = "";

	if ((dirp = opendir(dirname)) == NULL) {
		if (soft_fail) {
			return state;
		} else {
			printf("CRITICAL: failed to open %s: %s\n", 
		       	dirname, strerror(errno));
			exit(STATE_CRITICAL);
		}
	}

	while ((dp = readdir(dirp)) != NULL) {
		struct stat st;
		char *suffix;
		FILE *fp;
		pid_t pid;
		int i;
		int skip = 0;
	
#ifdef __NetBSD__
		if (dp->d_namlen == 0)
			continue;
#endif
		if (strcmp(dp->d_name, ".") == 0 ||
		    strcmp(dp->d_name, "..") == 0)
			continue;

		for (i = 0; i < exlen; i++) {
			if (fnmatch(ex[i], dp->d_name, 0) == 0) {
				skip = 1;
				break;
			}
		}
		if (skip)
			continue;

		snprintf(tmp, sizeof(tmp), "%s/%s", dirname, dp->d_name);
		tmp[dead_len - 1] = '\0';

		if (lstat(tmp, &st) != 0) {
			printf("UNKNOWN: cannot stat %s: %s\n",
			       tmp, strerror(errno));
			exit(STATE_UNKNOWN);
		}

		if (S_ISDIR(st.st_mode)) {
			state = check_dir(tmp, soft_fail,
					  running, running_len,
					  dead, dead_len,
					  state,
					  ex, exlen);
			continue;
		}

		if (!S_ISREG(st.st_mode)) {
			continue;
		}

		if ((suffix = strrchr(dp->d_name, (int)'.')) == NULL)
			continue;

		if (strcmp(suffix, DOT_PID) != 0)
			continue;

		if ((fp = fopen(tmp, "r")) == NULL) {
			if (soft_fail) {
				continue;
			} else {
				printf("UNKNOWN: cannot open %s/%s: %s\n",
				       dirname, dp->d_name, strerror(errno));
				exit(STATE_UNKNOWN);
			}
		}

		*suffix = '\0';
			
		if (fscanf(fp, "%d", &pid) != 1) {
			printf("UNKNOWN: unexpected data from %s%s.pid\n",
			       dirname, dp->d_name);
			exit(STATE_UNKNOWN);
		}

		(void)fclose(fp);

		/* ignore EPERM: proccess exists */
		if (kill(pid, 0) != 0 && errno != EPERM) {
			strncpy(tmp, dead, dead_len);
			tmp[dead_len - 1] = '\0';
			snprintf(dead, dead_len, "%s%s ",
				 tmp, dp->d_name);
			state = STATE_CRITICAL;
		} else {
			strncpy(tmp, running, running_len);
			tmp[running_len - 1] = '\0';
			snprintf(running, running_len, "%s%s ",
				 tmp, dp->d_name);
		}
	}

	(void)closedir(dirp);

	return state;
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	char *progname = argv[0];
	char *dirname = _PATH_VARRUN;
	char **ex = NULL;
	size_t exlen = 0;
	int soft_fail = 0;
	char running[4096] = "";
	char dead[4096] = "";
	int state = STATE_OK;
	int i, ch;

	while ((ch = getopt(argc, argv, "d:sx:")) != -1) {
		switch (ch) {
		case 'd':
			dirname = optarg;
			break;
		case 's':
			soft_fail = 1;
			break;
		case 'x':
			exlen++;
			ex = realloc(ex, exlen * sizeof(*ex));
			if (ex == NULL) {
				printf("CRITICAL: realloc failed: %s\n", 
		       			strerror(errno));
				exit(STATE_CRITICAL);
			}
			ex[exlen - 1] = optarg;
			break;
		default:
			printf("usage: %s [-d dirname] [-x exclude_pattern] "
			       "[service1 ...]\n", progname);
			exit(EX_USAGE);
		}
	}

	/* Skip program name */
	if (optind == 0)
		optind = 1;

	argv += optind;
	argc -= optind;

	if (chdir(dirname) != 0) {
		printf("CRITICAL: failed to chdir to %s: %s\n", 
		       dirname, strerror(errno));
		exit(STATE_CRITICAL);
	}
		
	for (i = 0; i < argc; i++) {
		char name[MAXPATHLEN + 1];

		(void)snprintf(name, MAXPATHLEN, "%s%s", argv[i], DOT_PID);
		if (access(name, F_OK) != 0) {
			snprintf(dead, sizeof(dead), "%s%s ",
				 dead, argv[i]);
			state = STATE_CRITICAL;
		}
	}

	state = check_dir(dirname, soft_fail,
			  running, sizeof(running),
			  dead, sizeof(dead),
			  state,
			  ex, exlen);

	if (state == STATE_OK)
		printf("OK: %s\n", running);
	else
		printf("CRITICAL: %s\n", dead);
	
	return state;
}
