/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: x86desc.h,v 1.1.1.1 2003/02/16 14:31:12 bad Exp $
 * **********************************************************/

#ifndef _X86_DESC_H_
#define _X86_DESC_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_MONITOR
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#include "includeCheck.h"

/*
 * Segment Descriptors.
 */

typedef struct Descriptor {
   unsigned   limit_lo  : 16;
   unsigned   base_lo   : 16;
   unsigned   base_mid  : 8;
   unsigned   type      : 4;
   unsigned   S         : 1;
   unsigned   DPL       : 2;
   unsigned   present   : 1;
   unsigned   limit_hi  : 4;
   unsigned   AVL       : 1;
   unsigned   unused    : 1;
   unsigned   DB        : 1;
   unsigned   gran      : 1;
   unsigned   base_hi   : 8;
} Descriptor;

typedef union {
   Descriptor desc;
   uint32     word[2];
} DescriptorUnion;

/* 
 * Descriptors store a 32bit segment base in 3 parts (low,
 * mid, high) and the 20bit limit in 2 parts (low, high).  The
 * following macros extract these components from the original
 * base and limit.
 */

#define BASE_LO(_dw)              ((_dw) & 0xffff)
#define BASE_MID(_dw)             (((_dw) >> 16) & 0xff)
#define BASE_HI(_dw)              (((_dw) >> 24) & 0xff)
#define LIMIT_LO(_dw)             ((_dw) & 0xffff)
#define LIMIT_HI(_dw)             (((_dw) >> 16) & 0xf)


/*
 * Accessor functions for descriptors.  
 * 
 * Note: The fields of a descriptor should always be accessed with the
 * following functions.  ANSI C specifies that any expression
 * involving integer types smaller than an int have all the variables
 * automatically promoted to a *signed* int.  This means that
 * expressions that use the fields of a descriptor directly will
 * treat them as signed quantities.  This could have unwanted effects,
 * e.g. base and limit added together could result in a negative
 * quantity.  The functions that read the bitfields always return an
 * unsigned integer so as to avoid any potential signed/unsigned
 * problems.  Functions to write the bitfields are also provided for
 * consistency.  
 */

static INLINE uint32 Desc_Type(const Descriptor *d)     { return d->type; }
static INLINE uint32 Desc_S(const Descriptor *d)        { return d->S; }
static INLINE uint32 Desc_DPL(const Descriptor *d)      { return d->DPL; }
static INLINE uint32 Desc_Present(const Descriptor *d)  { return d->present; }
static INLINE uint32 Desc_AVL(const Descriptor *d)      { return d->AVL; }
static INLINE uint32 Desc_DB(const Descriptor *d)       { return d->DB; }
static INLINE uint32 Desc_Gran(const Descriptor *d)     { return d->gran; }

static INLINE LA
Desc_GetBase(const Descriptor *d)
{
   return (d->base_hi << 24) | (d->base_mid << 16) | d->base_lo;
}

static INLINE VA
Desc_GetLimit(const Descriptor *d)
{
   return (d->limit_hi << 16) | d->limit_lo;
}

static INLINE void Desc_SetType(Descriptor *d, uint32 val)     { d->type     = val; }
static INLINE void Desc_SetS(Descriptor *d, uint32 val)        { d->S        = val; }
static INLINE void Desc_SetDPL(Descriptor *d, uint32 val)      { d->DPL      = val; }
static INLINE void Desc_SetPresent(Descriptor *d, uint32 val)  { d->present  = val; }
static INLINE void Desc_SetDB(Descriptor *d, uint32 val)       { d->DB       = val; }
static INLINE void Desc_SetGran(Descriptor *d, uint32 val)     { d->gran     = val; }

static INLINE void
Desc_SetBase(Descriptor *d, LA newBase)
{
   d->base_hi  = BASE_HI(newBase);
   d->base_mid = BASE_MID(newBase);
   d->base_lo  = BASE_LO(newBase);
#ifdef ASSERT
   ASSERT(Desc_GetBase(d) == newBase);
#endif
}

static INLINE void
Desc_SetLimit(Descriptor *d, VA newLimit)
{
   d->limit_lo = LIMIT_LO(newLimit);
   d->limit_hi = LIMIT_HI(newLimit);
#ifdef ASSERT
   ASSERT(Desc_GetLimit(d) == newLimit);
#endif
}

/* 
 * NOTE: the implementation of Desc_SetDescriptor() assumes a little-endian
 * byte order.  
 */
static INLINE void 
Desc_SetDescriptor(Descriptor *d, LA base, VA limit, uint32 type, 
                   uint32 S, uint32 DPL, uint32 present, uint32 DB, 
                   uint32 gran)
{
   DescriptorUnion *desc = (DescriptorUnion *) d;

   desc->word[0] = BASE_LO(base)   << 16 | LIMIT_LO(limit);
   desc->word[1] = BASE_HI(base)   << 24 | 
                   gran            << 23 | 
                   DB              << 22 | 
                   LIMIT_HI(limit) << 16 | 
                   present         << 15 | 
                   DPL             << 13 |
                   S               << 12 | 
                   type            << 8  | BASE_MID(base);

   /* 
    * Assert that all the fields were properly filled in.
    */

#ifdef ASSERT
   ASSERT(Desc_GetBase(d) == base);
   ASSERT(Desc_GetLimit(d) == limit);
   ASSERT(Desc_Type(d) == type);
   ASSERT(Desc_S(d) == S);
   ASSERT(Desc_DPL(d) == DPL);
   ASSERT(Desc_Present(d) == present);
   ASSERT(Desc_DB(d) == DB);
   ASSERT(Desc_Gran(d) == gran);
   ASSERT(d->AVL == 0 && d->unused == 0);
#endif
}


/*
 * Accessor functions that operate directly on a descriptor.  These
 * are included only for backwards compatibility with existing macros.  
 */

static INLINE uint32 DESC_TYPE(Descriptor d)       { return d.type; }
static INLINE uint32 DESC_S(Descriptor d)          { return d.S; }
static INLINE uint32 DESC_DPL(Descriptor d)        { return d.DPL; }
static INLINE uint32 DESC_PRESENT(Descriptor d)    { return d.present; }

#define DT_CODE(_d)               ( DESC_S(_d) && ((DESC_TYPE(_d) & 0x8) == 0x8))
#define DT_CONFORMING_CODE(_d)    ( DESC_S(_d) && ((DESC_TYPE(_d) & 0xc) == 0xc))
#define DT_NONCONFORMING_CODE(_d) ( DESC_S(_d) && ((DESC_TYPE(_d) & 0xc) == 0x8))
#define DT_READABLE_CODE(_d)      ( DESC_S(_d) && ((DESC_TYPE(_d) & 0xa) == 0xa))
#define DT_DATA(_d)               ( DESC_S(_d) && ((DESC_TYPE(_d) & 0x8) == 0x0))
#define DT_WRITEABLE_DATA(_d)     ( DESC_S(_d) && ((DESC_TYPE(_d) & 0xa) == 0x2))
#define DT_EXPAND_DOWN(_d)        ( DESC_S(_d) && ((DESC_TYPE(_d) & 0xc) == 0x4))
#define DT_CALL_GATE(_d)          (!DESC_S(_d) && ((DESC_TYPE(_d) & 0x7) == 0x4))
#define DT_LDT(_d)                (!DESC_S(_d) && ((DESC_TYPE(_d) & 0xf) == 0x2))
#define DT_TASK_GATE(_d)          (!DESC_S(_d) && ((DESC_TYPE(_d) & 0xf) == 0x5))
#define DT_TSS(_d)                (!DESC_S(_d) && ((DESC_TYPE(_d) & 0x5) == 0x1))
#define DT_AVAIL_TSS(_d)          (!DESC_S(_d) && ((DESC_TYPE(_d) & 0x7) == 0x1))

#define DT_ACCESS                 0x2
#define DT_32BIT                  0x8
#define DT_TSS_BUSY               0x2

#define DATA_DESC                 0x2
#define CODE_DESC                 0xa
#define TASK_DESC                 0x9  // TSS available
#define TASK_DESC_BUSY            0xb  // TSS busy


/*
 *----------------------------------------------------------------------
 *
 * Desc_ExpandedLimit --
 *
 *      Return the limit in bytes of the descriptor.
 *
 * Results:
 *      The limit in bytes of the descriptor.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE uint32
Desc_ExpandedLimit(const Descriptor *d)
{
   return Desc_Gran(d) ? (Desc_GetLimit(d) << 12) | 0xfff : Desc_GetLimit(d);
}


/*
 *----------------------------------------------------------------------
 *
 * Desc_InBounds and Desc_LegalAccess --
 *
 *      Check whether the access is within the bounds of the segment.
 *
 * Results:
 *      TRUE is the access is within the bounds of the segment, 
 *      FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE Bool
Desc_InBounds(VA vaddr, VA limit, uint32 size, Bool expandDown, uint32 DB)
{
   if (expandDown) {
      VA end = DB ? 0xffffffff : 0xffff; 
      return  ((vaddr >= limit) && (vaddr <= end) && ((size - 1) <= (end - vaddr)));
   } else {
      return  ((vaddr <= limit) && ((size - 1) <= (limit - vaddr)));
   }
}

static INLINE Bool
Desc_LegalAccess(const Descriptor *d, VA vaddr, uint32 size)
{
   return Desc_InBounds(vaddr, Desc_ExpandedLimit(d), size, 
                        DT_EXPAND_DOWN(*d), Desc_DB(d));
}


/*
 * Call Gates.
 */

typedef struct Gate {
   unsigned   offset_lo : 16;
   unsigned   segment   : 16;
   unsigned   params    : 5;
   unsigned   unused    : 3;
   unsigned   type      : 5;
   unsigned   DPL       : 2;
   unsigned   present   : 1;
   unsigned   offset_hi : 16;
} Gate;


#define GATE_OFFSET(_gate)       (((_gate).offset_hi << 16) | (_gate).offset_lo)

#define GATE_OFFSET_LO(_dw)      (((int)(_dw)) & 0xffff)
#define GATE_OFFSET_HI(_dw)      ((((int)(_dw)) >> 16) & 0xffff)

#define TASK_GATE                0x05
#define INTER_GATE               0x0e
#define TRAP_GATE                0x0f

#define GT_TASK(_gate)           (((_gate).type & 0x1f) == 0x05)
#define GT_INTR(_gate)           (((_gate).type & 0x17) == 0x06)
#define GT_TRAP(_gate)           (((_gate).type & 0x17) == 0x07)
#define GT_32BIT                 0x08
#define GT_32BIT_INTR            0xe
#define GT_32BIT_TRAP            0xf

/*
 * Descriptor Table Registers.
 */

#ifdef __GNUC__

/* 
 * Need to pack the DTR struct so the offset starts right after the
 * limit.  
 */
typedef struct DTR {
   uint16 limit;
   uint32 offset __attribute__ ((packed));
} DTR;

#elif _MSC_VER

#pragma pack(push)
#pragma pack(2)

typedef struct {
   uint16 limit;
   uint32 offset;
} DTR;

#pragma pack(pop)

#else
# error
#endif

typedef union {
   DTR dtr;
   uint32 word[2];
} DTRWords;


#endif //_X86_DESC_H_
