/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: x86.h,v 1.9 2003/06/30 17:53:08 jdolecek Exp $
 * **********************************************************/

#ifndef _X86_H_
#define _X86_H_

#ifdef __NetBSD__
#include <sys/param.h>		/* for __NetBSD_Version__ */
#include <i386/specialreg.h>	/* for MSR_* */

# if __NetBSD_Version__ > 105009900
#  include <uvm/uvm_extern.h>
# else
#  include <vm/vm.h>
# endif

#endif

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_MONITOR
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#include "includeCheck.h"

#include "vm_types.h"
#include "x86apic.h"
#include "x86desc.h"

#define SIZE_8BIT   1
#define SIZE_16BIT  2
#define SIZE_24BIT  3
#define SIZE_32BIT  4
#define SIZE_48BIT  6
#define SIZE_64BIT  8
#define SIZE_80BIT  10
#define SIZE_128BIT 16
#define SIZE_M512   512

/*
 *----------------------------------------------------------------------
 *
 * OpsizeValue --
 *
 *	Called by code that can handle 32, 16, or 8 bits values to return
 *	only the defined bits for the current size.
 *
 * Results:
 *      Value of the specified bits, zero-extended out to 32 bits.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

#if defined(LOWORD) && defined(LOBYTE) // Windows doesn't have LOWORD here -- edward
static INLINE uint32
OpsizeValue(uint32 value,	// IN: The 32 bit value to be sized.
	    int opSize)		// IN: the size 
{
#ifdef ASSERT
   ASSERT(opSize == SIZE_8BIT || opSize == SIZE_16BIT || opSize == SIZE_32BIT);
#endif

   return opSize == SIZE_8BIT  ? LOBYTE(value) : 
         (opSize == SIZE_16BIT ? LOWORD(value) : value);
}
#endif

#define NUM_IDT_VECTORS 256


/*
 * simple structures are defined but shift and masks where more
 * complicated structures use structs and bitmasks
 */


/*
 *   control registers
 */

#define CR0_PE         0x00000001
#define CR0_MP         0x00000002
#define CR0_EM         0x00000004
#define CR0_TS         0x00000008
#define CR0_ET         0x00000010
#define CR0_NE         0x00000020
#define CR0_WP         0x00010000
#define CR0_AM         0x00040000
#define CR0_NW         0x20000000
#define CR0_CD         0x40000000
#define CR0_PG         0x80000000
#define CR0_RESERVED   0x1ffaffc0
#define CR0_NOT_IMP    0x00000000

#define CR3_PWT        0x00000008
#define CR3_PCD        0x00000010
#define CR3_PDB_SHIFT  12
#define CR3_PDB_MASK   0xfffff000
#define CR3_RESERVED   0x00000fe7
#define CR3_NOT_IMP    0x00000018

#define CR4_VME        0x00000001
#define CR4_PVI        0x00000002
#define CR4_TSD        0x00000004
#define CR4_DE         0x00000008
#define CR4_PSE        0x00000010
#define CR4_PAE        0x00000020
#define CR4_MCE        0x00000040
#define CR4_PGE        0x00000080
#define CR4_PCE        0x00000100

#define CR4_OSFXSR     0x00000200 /* CPU/OS supports SIMD insts */
#define CR4_OSXMMEXCPT 0x00000400 /* #XF exception enable PIII only */

#define CR4_RESERVED   0xfffff800
#define CR4_NOT_IMP    0x00000006

#define CR4_RESERVED_WO_KNI   (CR4_RESERVED|CR4_OSXMMEXCPT)
#define CR4_NOT_IMP_WO_KNI    (CR4_NOT_IMP|CR4_OSXMMEXCPT)

#define EFLAGS_CF      0x00000001
#define EFLAGS_SET     0x00000002
#define EFLAGS_PF      0x00000004
#define EFLAGS_AF      0x00000010
#define EFLAGS_ZF      0x00000040
#define EFLAGS_SF      0x00000080
#define EFLAGS_TF      0x00000100
#define EFLAGS_IF      0x00000200
#define EFLAGS_DF      0x00000400
#define EFLAGS_OF      0x00000800
#define EFLAGS_IOPL    0x00003000
#define EFLAGS_NT      0x00004000
#define EFLAGS_RF      0x00010000
#define EFLAGS_VM      0x00020000
#define EFLAGS_AC      0x00040000
#define EFLAGS_VIF     0x00080000
#define EFLAGS_VIP     0x00100000
#define EFLAGS_ID      0x00200000

#define EFLAGS_ALL       (0x003f7fd5)

#define EFLAGS_REAL_32   (EFLAGS_ALL & ~(EFLAGS_VIP|EFLAGS_VIF|EFLAGS_VM))
#define EFLAGS_V8086_32  (EFLAGS_ALL & ~(EFLAGS_VIP|EFLAGS_VIF|EFLAGS_VM|EFLAGS_IOPL))
#define EFLAGS_ALL_16    LOWORD(EFLAGS_ALL)
#define EFLAGS_REAL_16   LOWORD(EFLAGS_REAL_32)
#define EFLAGS_V8086_16  LOWORD(EFLAGS_V8086_32)
#define EFLAGS_CC        0x00000cd5

#define EFLAGS_CLEAR_ON_EXC     (EFLAGS_TF|EFLAGS_VM|EFLAGS_RF|EFLAGS_NT)

#define EFLAGS_IOPL_SHIFT 12
#define EFLAGS_IOPL_MASK  0x3000


#define DR6_B0         0x00000001
#define DR6_B1         0x00000002
#define DR6_B2         0x00000004
#define DR6_B3         0x00000008
#define DR6_BD         0x00002000
#define DR6_BS         0x00004000
#define DR6_BT         0x00008000
#define DR6_ONES       0xffff0ff0
#define DR6_RESERVED_MASK 0xffff1ff0

#define DR7_L0         0x00000001
#define DR7_G0         0x00000002
#define DR7_L1         0x00000004
#define DR7_G1         0x00000008
#define DR7_L2         0x00000010
#define DR7_G2         0x00000020
#define DR7_L3         0x00000040
#define DR7_G3         0x00000080
#define DR7_LE         0x00000100
#define DR7_GE         0x00000200
#define DR7_GD         0x00002000

#define DR7_RW(_r,_n)  (((_r) >> (16+(_n)*4)) & 0x3)
#define DR7_L(_r,_n)   (((_r) >> ((_n)*2)) & 1)
#define DR7_G(_r,_n)   (((_r) >> (1 + (_n)*2)) & 1)
#define DR7_LEN(_r,_n) (((_r) >> (18+(_n)*4)) & 0x3)

#define DR7_RW_INST    0x0
#define DR7_RW_WRITES  0x1
#define DR7_RW_IO      0x2
#define DR7_RW_ACCESS  0x3

#define DR7_ONES       0x00000400



/* http://www.sandpile.org/ia32/cpuid.htm */

/* Common CPUID feature bits */
#define CPUID_FEATURE_COMMON_FPU    0x00000001
#define CPUID_FEATURE_COMMON_VME    0x00000002
#define CPUID_FEATURE_COMMON_DBGE   0x00000004
#define CPUID_FEATURE_COMMON_PGSZE  0X00000008
#define CPUID_FEATURE_COMMON_TSC    0x00000010
#define CPUID_FEATURE_COMMON_MSR    0x00000020
#define CPUID_FEATURE_COMMON_PAE    0x00000040
#define CPUID_FEATURE_COMMON_MCK    0x00000080
#define CPUID_FEATURE_COMMON_CPMX   0x00000100
#define CPUID_FEATURE_COMMON_APIC   0x00000200
#define CPUID_FEATURE_COMMON_RSVD10 0x00000400
#define CPUID_FEATURE_COMMON_SEP    0x00000800
#define CPUID_FEATURE_COMMON_MTTR   0x00001000
#define CPUID_FEATURE_COMMON_PGE    0x00002000 
#define CPUID_FEATURE_COMMON_MCA    0x00004000
#define CPUID_FEATURE_COMMON_CMOV   0x00008000 
#define CPUID_FEATURE_COMMON_PAT    0x00010000
#define CPUID_FEATURE_COMMON_36PG   0x00020000
/* feature bits 18-22 */
#define CPUID_FEATURE_COMMON_MMX    0x00800000 
#define CPUID_FEATURE_COMMON_FXSAVE 0X01000000
#define CPUID_FEATURE_COMMON_XMM    0x02000000
/* feature bits 26-31 */

/* Intel CPUID feature bits */
#define CPUID_FEATURE_INTEL_PSN    0x00040000
#define CPUID_FEATURE_INTEL_CLFL   0x00080000
#define CPUID_FEATURE_INTEL_RSVD20 0x00100000
#define CPUID_FEATURE_INTEL_DTES   0x00200000
#define CPUID_FEATURE_INTEL_ACPI   0x00400000
/* CPUID_FEATURE_COMMON_MMX */
/* CPUID_FEATURE_COMMON_FXSAVE */
/* CPUID_FEATURE_COMMON_XMM    */
#define CPUID_FEATURE_INTEL_XMM2   0x04000000
#define CPUID_FEATURE_INTEL_SS     0x08000000 
#define CPUID_FEATURE_INTEL_RSVD28 0x10000000  // Pentium 4 sets it to 1!
#define CPUID_FEATURE_INTEL_TM     0x20000000 
#define CPUID_FEATURE_INTEL_RSVD30 0x40000000 
#define CPUID_FEATURE_INTEL_RSVD31 0x80000000 

#define CPUID_FEATURE_INTEL_RSVD   0xc0100400 

/* AMD CPUID feature bits */
#define CPUID_FEATURE_AMD_RSVD18 0X00040000
#define CPUID_FEATURE_AMD_RSVD19 0X00080000
#define CPUID_FEATURE_AMD_RSVD20 0X00100000
#define CPUID_FEATURE_AMD_RSVD21 0X00200000
#define CPUID_FEATURE_AMD_MMX1   0X00400000
/* CPUID_FEATURE_COMMON_MMX */
/* CPUID_FEATURE_COMMON_FXSAVE */
/* CPUID_FEATURE_COMMON_XMM    */
#define CPUID_FEATURE_AMD_RSVD26 0X04000000
#define CPUID_FEATURE_AMD_RSVD27 0X08000000
#define CPUID_FEATURE_AMD_RSVD28 0X10000000
#define CPUID_FEATURE_AMD_EXTENDED_LM     0X20000000
#define CPUID_FEATURE_AMD_EXTENDED_3DNOW1 0X40000000
#define CPUID_FEATURE_AMD_EXTENDED_3DNOW  0X80000000

#define CPUID_FEATURE_AMD_RSVD   0x1c3c0400 

#define CPUID_STEPPING(_eax) ((_eax) & 0xf)

#define CPUID_FAMILY(_eax)   (((_eax) >> 8) & 0xf)

/* Intel CPU Family */
#define CPUID_FAMILY_486     4
#define CPUID_FAMILY_P5      5
#define CPUID_FAMILY_P6      6
#define CPUID_FAMILY_EXTENDED 15

/* AMD CPU Family */
#define CPUID_FAMILY_5x86    4
#define CPUID_FAMILY_K5      5
#define CPUID_FAMILY_K6      5
#define CPUID_FAMILY_K7      6

#define CPUID_EXTENDED_FAMILY(_eax)   (((_eax) >> 20) & 0xff)

#define CPUID_EXTENDED_FAMILY_PENTIUM4	0

#define CPUID_FAMILY_IS_486(_eax) (CPUID_FAMILY(_eax) == CPUID_FAMILY_486)
#define CPUID_FAMILY_IS_P5(_eax)  (CPUID_FAMILY(_eax) == CPUID_FAMILY_P5)
#define CPUID_FAMILY_IS_P6(_eax)  (CPUID_FAMILY(_eax) == CPUID_FAMILY_P6)
#define CPUID_FAMILY_IS_PENTIUM4(_eax)  \
           (CPUID_FAMILY(_eax) == CPUID_FAMILY_EXTENDED && \
            CPUID_EXTENDED_FAMILY(_eax) == CPUID_EXTENDED_FAMILY_PENTIUM4)


#define CPUID_MODEL(_eax)    (((_eax) >> 4) & 0xf)

#define CPUID_MODEL_PPRO       1
#define CPUID_MODEL_PII_3      3
#define CPUID_MODEL_PII_5      5
#define CPUID_MODEL_CELERON_6  6
#define CPUID_MODEL_PIII_7     7
#define CPUID_MODEL_PIII_8     8
#define CPUID_MODEL_PIII_A     10
#define CPUID_MODEL_EXTENDED   15

#define CPUID_EXTENDED_MODEL(_eax)    (((_eax) >> 16) & 0xf)

#define MSR_L2CFG            0x0000011e
#if !defined(__NetBSD__) || !defined(MSR_TSC)
#define MSR_TSC              0x00000010
#define MSR_PERFCTR0         0x000000c1
#define MSR_PERFCTR1         0x000000c2
#define MSR_MTRR_CAP         0x000000fe
#define MSR_EVNTSEL0         0x00000186
#define MSR_EVNTSEL1         0x00000187
#endif
#if !defined(__NetBSD__) || !defined(MSR_SYSENTER_CS)
#define MSR_SYSENTER_CS      0x00000174
#define MSR_SYSENTER_ESP     0x00000175
#define MSR_SYSENTER_EIP     0x00000176
#endif

typedef uint32 CReg;


/*
 *   segment selectors
 */

typedef uint16 Selector;

#define SELECTOR_GDT             0
#define SELECTOR_LDT             1
#define SELECTOR_RPL_SHIFT       0
#define SELECTOR_RPL_MASK        0x03
#define SELECTOR_TI_SHIFT        2
#define SELECTOR_TI_MASK         0x4
#define SELECTOR_INDEX_SHIFT     3
#define SELECTOR_INDEX_MASK      0xfff8

#define SELECTOR_RPL(_sel)       (((Selector)(_sel)) & SELECTOR_RPL_MASK)
#define SELECTOR_TABLE(_sel)     ((((Selector)(_sel)) & SELECTOR_TI_MASK) >> SELECTOR_TI_SHIFT)
#define SELECTOR_INDEX(_sel)     (((Selector)(_sel)) >> SELECTOR_INDEX_SHIFT)
#define NULL_SELECTOR(_sel)      (!(((Selector)(_sel)) & ~SELECTOR_RPL_MASK))
#define SELECTOR_CLEAR_RPL(_sel) ((_sel) & ~SELECTOR_RPL_MASK)

/*
 *   tasks
 */

#ifdef __GNUC__
struct Task {
#else
typedef struct Task {
#endif
   uint16     prevTask,  __prevTasku;
   uint32     esp0;
   uint16     ss0,  __ss0u;
   uint32     esp1;
   uint16     ss1,  __ss1u;
   uint32     esp2;
   uint16     ss2,  __ss2u;
   uint32     cr3;
   uint32     eip;
   uint32     eflags;
   uint32     eax;
   uint32     ecx;
   uint32     edx;
   uint32     ebx;
   uint32     esp;
   uint32     ebp;
   uint32     esi;
   uint32     edi;
   uint16     es,  __esu;
   uint16     cs,  __csu;
   uint16     ss,  __ssu;
   uint16     ds,  __dsu;
   uint16     fs,  __fsu;
   uint16     gs,  __gsu;
   uint16     ldt,  __ldtu;
   uint16     trap;
   uint16     IOMapBase;
#ifdef __GNUC__
}  __attribute__ ((packed));

typedef struct Task Task;
#else
} Task;
#endif


typedef struct {
   uint16     prevTask;
   uint16     sp0;
   uint16     ss0;
   uint16     sp1;
   uint16     ss1;
   uint16     sp2;
   uint16     ss2;
   uint16     ip;
   uint16     flags;
   uint16     ax;
   uint16     cx;
   uint16     dx;
   uint16     bx;
   uint16     sp;
   uint16     bp;
   uint16     si;
   uint16     di;
   uint16     es;
   uint16     cs;
   uint16     ss;
   uint16     ds;
   uint16     fs;
   uint16     gs;
   uint16     ldt;
} Task16;


/*
 *   far pointers
 */

typedef struct {
   uint32 va;
   Selector seg;
} FarPtr;

typedef struct FarPtr16 {
   uint16   offset;
   uint16   selector;
} FarPtr16;

typedef struct FarPtr32 {
   uint32   offset;
   uint16   selector;
} FarPtr32;


/*
 * General-purpose registers and segment registers 
 */

typedef enum RegisterName {
   REG_NULL = -1, // -1 so that NUM_REGS doesn't include REG_NULL
   REG_EAX,
   REG_ECX,
   REG_EDX,
   REG_EBX,
   REG_ESP,
   REG_EBP,
   REG_ESI,
   REG_EDI,
   NUM_REGS
} RegisterName;


/* Return TRUE iff "reg" is a valid register number. */
static INLINE Bool 
x86_IsGoodRegister(RegisterName regName) 
{ 
   return (REG_NULL < regName && regName < NUM_REGS); 
}

typedef enum SegmentName {
   SEG_NULL = -1, // -1 so that NUM_SEGS doesn't include SEG_NULL
   SEG_ES,
   SEG_CS,
   SEG_SS,
   SEG_DS,
   SEG_FS,
   SEG_GS,
   SEG_LDTR,
   SEG_TR,
   NUM_SEGS
} SegmentName;

#define NUM_NORMAL_SEGS  6

static INLINE Bool 
x86_IsGoodSegmentName(SegmentName segName)
{ 
   return SEG_NULL < segName && segName < NUM_SEGS; 
}

/* Return FALSE iff "segName" refers to the special SEG_LDTR or SEG_TR. */
static INLINE Bool 
x86_IsNormalSegmentName(SegmentName segName)
{
#ifdef ASSERT
   ASSERT(x86_IsGoodSegmentName(segName));
#endif
   return segName < NUM_NORMAL_SEGS;
}

  
/*
 * Exception frame.
 */
typedef struct ExcFrame {
   /* pushed by sw */
   uint16      es, __esu;
   uint16      csCopy, __csCopyu; /* Copy from HW pushed value - not restored */
   uint16      ssCopy, __ssCopyu; /* Copy from HW pushed value - not restored */
   uint16      ds, __dsu;
   uint16      fs, __fsu;
   uint16      gs, __gsu;

   Reg         eax;
   Reg         ecx;
   Reg         edx;
   Reg         ebx;
   Reg         espCopy; /* Copy from HW pushed - not restored */
   Reg         ebp;
   Reg         esi;
   Reg         edi;

   /* pushed by SW or HW depending on the interrupt */
   Reg         errorCode;
   
   /* pushed by HW */
   Reg         eip;
   uint16      cs, __csu;
   Reg         eflags;
   
   /* pushed by HW on protection level changes */
   Reg         esp;
   uint16      ss, __ssu;

   /* pushed by HW coming from virtual 8086 mode */
   uint16      v8086_es, __v8086_esu;
   uint16      v8086_ds, __v8086_dsu;
   uint16      v8086_fs, __v8086_fsu;
   uint16      v8086_gs, __v8086_gsu;
   
} ExcFrame;

/*
 *   page tables
 */

#define PTE_P           0x00000001
#define PTE_RW          0x00000002
#define PTE_US          0x00000004
#define PTE_PWT         0x00000008
#define PTE_PCD         0x00000010
#define PTE_A           0x00000020
#define PTE_D           0x00000040
#define PTE_PS          0x00000080
#define PTE_G           0x00000100

#define PTE_FLAGS       (PTE_P|PTE_RW|PTE_US|PTE_G)

#define PTE_PAGE_TABLE  (PTE_P|PTE_RW|PTE_US|PTE_A|PTE_D) 

#define PTE_PRESENT(_pte)   (((_pte) & PTE_P) != 0)
#define PTE_WRITEABLE(_pte) (((_pte) & PTE_RW) != 0)
#define PTE_DIRTY(_pte)     (((_pte) & PTE_D) != 0)
#define PTE_USER(_pte)      (((_pte) & PTE_US) != 0)
#define PTE_GLOBAL(_pte)    (((_pte) & PTE_G) != 0)

#define PTE_AVAIL_MASK       0xe00
#define PTE_AVAIL_SHIFT      9
#define PTE_PFN_MASK         0xfffff000
#define PAE_PTE_PFN_MASK     0xffffff000LL
#define PTE_PFN_SHIFT        12
#define PAE_PTE_PFN_SHIFT    12
#define PTE_2_PFN(_pte)      (((_pte) & PTE_PFN_MASK) >> PTE_PFN_SHIFT)
#define PAE_PTE_2_PFN(_pte)  (((_pte) & PAE_PTE_PFN_MASK) >> PAE_PTE_PFN_SHIFT)

#define PDE_PFN_MASK         0xffc00000
#define PAE_PDE_PFN_MASK     0xfffe00000LL
#define PDE_PFN_SHIFT        PTE_PFN_SHIFT
#define PAE_PDE_PFN_SHIFT    PAE_PTE_PFN_SHIFT

/* Use these only for 4M or 2M pde's, not the 4K ones. */
#define PDE_2_PFN(_pte)      (((_pte) & PDE_PFN_MASK) >> PTE_PFN_SHIFT)
#define PAE_PDE_2_PFN(_pde)  (((_pde) & PAE_PDE_PFN_MASK) >> PAE_PDE_PFN_SHIFT)

#define PDPTE_PFN_MASK       0xffffff000LL
#define PDPTE_PFN_SHIFT      12
#define PDPTE_2_PFN(_p)      (((_p) & PDPTE_PFN_MASK) >> PDPTE_PFN_SHIFT)

#define PDES_PER_PMAP      (PAGE_SIZE/sizeof(PDE))
#define PTES_PER_PDIR      (PAGE_SIZE/sizeof(PTE))

#define PAE_PGOFF_MASK        0x1ff
#define PAE_PDOFF_MASK        0x1ff
#define PAE_RANGE_MASK        0x7ff

#define PFN_2_PDOFF(_a)       ((_a) >> 10)
#define PAE_PFN_2_PDOFF(_a)   (((_a) >> 9) & PAE_PDOFF_MASK)
#define PFN_2_PGOFF(_a)       ((_a) & 0x3ff)
#define PAE_PFN_2_PGOFF(_a)   ((_a) & PAE_PGOFF_MASK)

#define PA_2_PDOFF(_a)   ((_a) >> (10+PAGE_SHIFT))
#define PA_2_PGOFF(_a)   ((PA_2_PFN(_a)) & 0x3ff)

#define PDOFF_2_LA(_p)            ((_p) << 22)
#define LA_2_PDOFF(_a)            ((_a) >> 22)
#define PAE_LA_2_PDOFF(_a)        (((_a) >> 21) & PAE_PDOFF_MASK)
#define LA_2_PGOFF(_a)            (PFN_2_PGOFF(VA_2_VPN(_a)))
#define LA_2_RANGE(_a)            LA_2_PDOFF((_a))
#define PAE_LA_2_RANGE(_a)        (((_a) >> 21) & PAE_RANGE_MASK)
#define PAE_LA_2_PDPTOFF(_a)      (((_a) >> 30) & 0x3)

/* This following macro only work for flat model were VA==LA */
#define VA_2_PDOFF(_a)    ((_a) >> 22)
#define VA_2_PGOFF(_a)    (PFN_2_PGOFF(VA_2_VPN(_a)))

#define PTE_2_PA(_a)      (PPN_2_PA(PTE_2_PFN(_a)))
#define PDE_2_PA(_a)      (PPN_2_PA(PDE_2_PFN(_a)))

/*
 *   exception error codes
 */

#define EXC_NONE         -1
#define EXC_DE            0
#define EXC_DB            1
#define EXC_NMI           2
#define EXC_BP            3
#define EXC_OF            4
#define EXC_BR            5
#define EXC_UD            6
#define EXC_NM            7
#define EXC_DF            8
#define EXC_TS           10
#define EXC_NP           11
#define EXC_SS           12
#define EXC_GP           13
#define EXC_PF           14
#define EXC_MF           16
#define EXC_AC           17
#define EXC_MC           18
#define EXC_XF           19  // SIMD exception

#define PF_P            0x1
#define PF_RW           0x2
#define PF_US           0x4
#define PF_RSVD         0x8


/*
 *  instruction decoding
 */

#define MODRM_MOD(_m)  (((_m) >> 6) & 0x3)
#define MODRM_REG(_m)  (((_m) >> 3) & 0x7)
#define MODRM_RM(_m)   (((_m) >> 0) & 0x7)

/*
 * Debug registers
 */

#define DEBUG_STATUS_B0   (1<<0)
#define DEBUG_STATUS_B1   (1<<1)
#define DEBUG_STATUS_B2   (1<<2)
#define DEBUG_STATUS_B3   (1<<3)
#define DEBUG_STATUS_DB   (1<<13)
#define DEBUG_STATUS_BS   (1<<14)
#define DEBUG_STATUS_BT   (1<<15)

typedef struct DebugControlRegister {

   int l0:1;
   int g0:1;
   int l1:1;
   int g1:1;
   int l2:1;
   int g2:1;
   int l3:1;
   int g3:1;
   
   int le:1;
   int ge:1;
   int oo1:3;
   
   int gd:1;
   int oo:2;
   
   int rw0:2;
   int len0:2;
   int rw1:2;
   int len1:2;
   int rw2:2;
   int len2:2;
   int rw3:2;
   int len3:2;
   
} DebugControlRegister;

  

                 
#define MAKE_SELECTOR(_index, _ti, _RPL)                     \
( (((_index) << SELECTOR_INDEX_SHIFT) & SELECTOR_INDEX_MASK) \
  | (((_ti)  << SELECTOR_TI_SHIFT)    & SELECTOR_TI_MASK)    \
  | (((_RPL) << SELECTOR_RPL_SHIFT)   & SELECTOR_RPL_MASK) )


#define MAKE_PTE(_mpfn, _avail, _flags) \
                        (((_mpfn) << PTE_PFN_SHIFT) \
                         | (((_avail) << PTE_AVAIL_SHIFT) & PTE_AVAIL_MASK) \
                         | (_flags))


/*
 * SMM State Save Map 
 * See the Intel processor manual for documentation.
 * The extract layout of the structure is guesswork.  It should really
 * matter also as the Intel documented fields are at the right offset. 
 */

/*
 * SMMSegState - Store the "hidden" state of a segment descriptor in 
 * the SMM save state map. NOTE: Format is somewhat defined by the 
 * Intel hardware.
 */
typedef struct SMMSegState {
   uint32 limit;   /* Limit from descriptor */
   uint32 base;    /* Base from descriptor */
   uint32 rights;  /* Rest of the descriptor */
} SMMSegState;

/*
 * Format the the SMM save state map.
 * NOTE: This too is defined by the Intel hardware.
 */

typedef struct SMMStateSaveMap {
   union {
      uint32 selState;       // We steal a enough of this (hopefully) unused
                             // space to stash away the selector state. 
      uint8  reserved0[248];
   } u;
   uint32 smbase;
   uint32 revId; 
   uint16 ioInstRestart;
   uint16 haltRestart;
   uint8  reserved01[12];
   uint32 ioInstEIP; 
   uint8  reserved1[18];
   uint32 cr4reg;
   uint8  reserved2[4];
   SMMSegState hiddenSegPart[7];
   SMMSegState gdtr;
   SMMSegState idtr;
   SMMSegState hiddenTRpart;
   uint32 segRegs[8];
   Reg    dr7reg;
   Reg    dr6reg;
   Reg    gpRegs[8];
   Reg    eip;
   Reg    eflags;
   Reg    cr3reg;
   Reg    cr0reg;
} SMMStateSaveMap;

#define SMM_SAVEMAP_SIZE   512
#define SMRAM_DEFAULT_BASE 0x30000
#define SMRAM_ENTRY_POINT 0x8000
#define SMRAM_SAVE_STATE_OFFSET 0xfe00

#define SMM_RIGHTS_BITS   0x00f0ff00

/*
 * FPU FSAVE/FXSAVE area format. 
 *
 * fsave_hard_struct - Memory format of the FSAVE/FNSAVE instruction. 108 bytes
 * fxsave_hard_struct - Memory format of the FXSAVE instruction. 512 bytes
 *                    Note: fxsave_hard_struct must be 128bit aligned. 
 *
 * Taken from linux headers
 */

struct fsave_hard_struct {
   long   cwd;
   long   swd;
   long   twd;
   long   fip;
   long   fcs;
   long   foo;
   long   fos;
   long   st_space[20];   /* 8*10 bytes for each FP-reg = 80 bytes */
};

struct fxsave_hard_struct {
   unsigned short fxcwd;
   unsigned short fxswd;
   unsigned short fxtwd;
   unsigned short fxfopcode;
   long     fxfip;
   short    fxfcs;
   short    __reserved_00;
   long     fxfoo;
   short    fxfos;
   short    __reserved_01;
   long     mxcsr;
   long     __reserved_02;
   long     st_space[32];     /* 8*16 bytes for each FP/MMX-reg = 128 bytes */
   long     xmm_space[32];    /* 8*16 bytes for each XMM-reg = 128 bytes */
   long     __reserved_03 [14*4]; /* 14 16byte lines for remainder */
};


#endif /* _X86_H_ */

