/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/
#ifdef VMX86_DEVEL
char rcsId_vmx86[] = "$Id: vmx86.c,v 1.2 2003/02/16 01:29:50 bad Exp $";
#else
#define FILECODE "F(301)"
#endif 

/*
 * vmx86.c --
 *
 *     Platform independent routines for creating/destroying/running
 *     virtual machine monitors.  
 *      
 */


#ifdef linux
# include "driver-config.h" /* Versioned symbols from linux/string.h */
# include <linux/string.h> /* memset() in the kernel */
#elif __FreeBSD__
# include <sys/param.h>
# include <sys/systm.h>
# include <string.h>
#elif defined(__NetBSD__)
# include <sys/param.h>
# include <sys/systm.h>
# include <sys/lock.h>
# if __NetBSD_Version__ > 105009900
#  include <uvm/uvm_extern.h>
# else
#  include <vm/vm.h>
# endif
#elif defined(WINNT_DDK)
# include <string.h>
#else
# error "Unknown platform"
#endif

#include "x86.h"
#include "vm_types.h"
#include "vm_assert.h"
#include "modulecall.h"
#include "vmx86.h"
#include "hostif.h"
#include "task.h"
#include "initblock.h"
#include "vm_asm.h"
#include "vm_time.h"
#include "vtrace.h"
#include "memmgmt.h"

/*
 * Keep track of the virtual machines that have been 
 * created using the following structures. 
 *
 */ 


static VMDriver *vmDriverList = NULL;

static int32 lockedPageLimit;
static int32 numLockedPages;
static int32 vmCount;
static int32 vmUID;

static void SetStartTime(VMDriver *vm, InitBlock *);
static void FreeAllResources(VMDriver *vm);

static uint32 unlockedPages;
static uint32 dirtyUnlockedPages;

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_Init --
 *
 *      Initialized a driver structure for a virtual machine.
 *
 * Results: 
 &      VMDriver structure or  NULL on error
 *
 * Side effects:
 *       Memory allocated.
 *
 *----------------------------------------------------------------------
 */


VMDriver *
Vmx86_Init(void *uniqHandle,    /* Unique id for VM to be created */
	   void *processId)     /* Process creating VM. */
{
   VMDriver *vm;
   Bool smpMachine = HostIF_IsSystemSMP();


   /*
   // XXXX for NT:
   // The code within a critical region guarded by an spin lock must 
   // neither be pageable nor make any references to pageable data. 
   */
   vm = (VMDriver *) HostIF_AllocKernelMem(sizeof(struct VMDriver), TRUE);
   if (!vm) {
      return NULL;
   }
   memset(vm,0,sizeof(*vm));
#ifdef linux
   vm->logFD = -1;
#endif
   
   HostIF_GlobalVMLock(0);
   
   vmCount++;

   if (vmDriverList == NULL) {
      vmDriverList = vm;
   } else {
      VMDriver *lastVM = vmDriverList;
      while (lastVM->nextDriver) {
	 ASSERT (lastVM->uniqHandle != uniqHandle);
	 lastVM = lastVM->nextDriver;
      }
      lastVM->nextDriver = vm;
   }
   vm->uniqHandle = uniqHandle;
   vm->processID = processId;
   vm->vmUID = ++vmUID;
   vm->smpMachine = smpMachine;
  
   /* HostIF_APIC_Base(vm, TRUE); */

   HostIF_GlobalVMUnLock(0);
   
   if (HostIF_Init(vm)) {
      Vmx86_DestroyVM(vm);
      return NULL;
   }
      
   return vm;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_CreateVM --
 *
 *      Creation of the VM. Expects all initial arguments
 *      to be part of the InitBlock structure.
 *
 * Results: 
 &     non-zero on error, zero on success;
 *
 * Side effects:
 *      many
 *
 *----------------------------------------------------------------------
 */

int 
Vmx86_CreateVM(VMDriver *vm, 
          InitBlock *initParams) /* Initial params from the VM */ 
{
   int retval;

   if (initParams->magicNumber != INIT_BLOCK_MAGIC) {
     Warning("Bad magic number for init block 0x%x\n", initParams->magicNumber);
     return 1;
   }
   vm->logFD = initParams->logFD;

   HostIF_InitFP(vm);
     
   /*
    * Initialize the driver's part of the cross-over page used to 
    * talk to the monitor
    */
   
   retval = Task_InitCrosspage(vm,initParams);
   if (retval) {
      Warning("Task crosspage init died with retval=%d\n", retval);
      goto error;
   }

   unlockedPages = dirtyUnlockedPages = 0;

   SetStartTime(vm, initParams);
   return 0;

 error:
   FreeAllResources(vm);
   return 1;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_LateInit --
 *
 *      Do late initialization of the driver.
 *	This should be called after Vmx86_CreateVM and
 *	after all the user-level device initialization.
 *
 * Results: 
 *      none
 *
 * Side effects:
 *      VTrace_Init is called
 *
 *----------------------------------------------------------------------
 */

void
Vmx86_LateInit(VMDriver *vm)
{
#ifdef SUPPORT_VTRACE
   VTrace_Init(vm);
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_RunVM  --
 *
 *      Return to the monitor. 
 *      Main interaction between the module and the monitor.
 *      Leaves the loop only to go back to user mode.
 *      Every interation of the loop switches back to the monitor
 *
 * Results:
 *      Returns the MODULECALL that forced it to leave the loop.
 *      back to the IOCTL handler of the module device driver
 *
 * Side effects:
 *      Dispatches the messages, everything can change
 *
 *----------------------------------------------------------------------
 */
#ifdef __FreeBSD__
#include <sys/param.h>
#include <sys/systm.h>
#include <vm/vm.h>
#include <vm/vm_param.h>
#include <sys/lock.h>
#include <vm/pmap.h>
#endif 
int 
Vmx86_RunVM(VMDriver *vm)
{
   ModuleCallType op;   
   int retval;

   ASSERT(vm && vm->crosspage);


   retval = MODULECALL_USERRETURN;

   while(1) {

      vm->crosspage->retval = retval;

      /*
       * Task_Switch changes the world to the monitor.
       * The monitor is waiting in the BackToHost routine
       */ 
      Task_Switch(vm);

      op = vm->crosspage->moduleCallType; 

      retval = MODULECALL_USER_START;
      
      if (op >  MODULECALL_USER_START&& op < MODULECALL_USER_END) {
         return op;
      } 
         
      
      switch (op) {
      case MODULECALL_NONE : 
         break;
         

      case MODULECALL_INTR : {
	 /*
	  * Note that as of SMP support, irq is the actual Interrupt number
	  * rather than the IRQ, b/c the IO APIC redirects IRQs every which
	  * way. --Jeremy.
	  */
#ifdef notdef
	 if (vm->crosspage->args[1] == USERCALL_NONE) {
	    /* nothing to be done if MODULECALL_USERRETURN */
	    retval = MODULECALL_USERRETURN;
	    break;
	 } 
#endif
	 return vm->crosspage->args[1];
      }
      case MODULECALL_ISMPNLOCKED : {
         MPN mpn = vm->crosspage->args[0];
         retval = HostIF_IsMPNLocked(vm, mpn);
         break;
      }

      case MODULECALL_LOCKPAGE: {
         char dummy;
         void *addr = (void *)vm->crosspage->args[0];
#ifdef linux
	 HostIF_LockKernel();
#endif
         retval = Vmx86_LockPage(vm, addr, FALSE);
         if (retval != PAGE_LOCK_FAILED && retval != PAGE_LOCK_LIMIT_EXCEEDED) {
            if (HostIF_CopyFromUser(&dummy, addr, 1) == 0) {
               HostIF_CopyToUser(addr, &dummy, 1);
            } else {
               (void) Vmx86_UnlockPage(vm, addr, FALSE);
               retval = 0;
            }
         }
#ifdef linux
	 HostIF_UnLockKernel();
#endif
         break;
      }

      case MODULECALL_UNLOCKPAGE: {
         char dummy;
         void *addr = (void *)vm->crosspage->args[0];
         int dirty = vm->crosspage->args[1];
#ifdef linux
	 HostIF_LockKernel();
#endif

         retval = FALSE;
         unlockedPages++;
         if (dirty) {
            dirtyUnlockedPages++;
            if (HostIF_CopyFromUser(&dummy, addr, 1) == 0) {
               if (HostIF_CopyToUser(addr, &dummy, 1) == 0) {
                  if (!Vmx86_UnlockPage(vm, addr, FALSE)) {
                     retval = TRUE;
                  }
               }
            }
         } else {
            if (!Vmx86_UnlockPage(vm, addr, FALSE)) {
               retval = TRUE;
            }
         }
#ifdef linux
	 HostIF_UnLockKernel();
#endif
         break;
      }

      default: 
         Panic("ModuleCallLoop %d not supported \n",op);
      }
   }

   ASSERT(op >= MODULECALL_USER_START && op < MODULECALL_USER_END);
   return op;
}

   
/*
 *----------------------------------------------------------------------
 *
 * Vmx86_DestroyVM  --
 *
 *      Cleanup, called on close of /dev/vmx86, including
 *      after an exit/panic/segv/...
 * 
 * Results:
 *      
 *      zero if successful
 *
 * Side effects:
 *
 *      Release resources (those that are left). 
 *
 *----------------------------------------------------------------------
 */
int
Vmx86_DestroyVM(VMDriver *vm)
{
   int found = FALSE;

   Log("Vmx86_DestroyVM: unlocked pages: %d, unlocked dirty pages: %d\n",
           unlockedPages, dirtyUnlockedPages);

   /*
    *   Deletion of the virtual machine from list
    */
   HostIF_GlobalVMLock(1);

   ASSERT(vmDriverList);
   if (vmDriverList == vm) {
      vmDriverList = vm->nextDriver;
      found = TRUE;
   } else {
      VMDriver *lastVM = vmDriverList;
      while (lastVM->nextDriver) {
         if (lastVM->nextDriver == vm) {
            lastVM->nextDriver = vm->nextDriver;
            found = TRUE;
            break;
         }
         lastVM = lastVM->nextDriver;
      }
   }

   vmCount--;

   HostIF_GlobalVMUnLock(1);
  

   if (!found) {
      Panic("Corrupted DriverList in Vmx86_Destroy");
   }

   FreeAllResources(vm);


   return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * SetStartTime --
 *
 *      Initial the data structures that track the starting time of the
 *      virtual machine. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      none  
 *
 *
 *----------------------------------------------------------------------
 */

static void
SetStartTime(VMDriver *vm,            /* IN */
             InitBlock *initParams)   /* IN-OUT */
{
   VmTimeStart  *st = &initParams->st;
   uint32 flags;
   uint32 mhzEstimate;
  
   mhzEstimate = HostIF_MhzRating(vm);

   SAVE_FLAGS(flags); 
   CLEAR_INTERRUPTS();

   st->time = HostIF_ReadTime(); 
   st->count = GET_TSC();
   st->mhzEstimate = mhzEstimate;

   RESTORE_FLAGS(flags);
   
   
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx65_CurrentVM --
 *
 *      Return the VMDriver structure associated with
 *      the handle
 *      
 *
 * Results:
 *      VMDriver * structure.NULL if not available
 *
 * 
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
VMDriver *
Vmx86_CurrentVM(void *uniqHandle)
{
  VMDriver *vm;

  HostIF_GlobalVMLock(2);

  vm = vmDriverList;

  while (vm) { 
     if (vm->uniqHandle == uniqHandle) {
        break;
     }
     vm = vm->nextDriver;
  }

  HostIF_GlobalVMUnLock(2);

  return vm;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx65_GetVMForProcess  --
 *
 *      Return the VMDriver structure associated with
 *      a process
 *      
 *
 * Results:
 *      VMDriver * structure.NULL if not available
 *
 * 
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

VMDriver *
Vmx86_GetVMforProcess(void *processId)
{
  VMDriver *vm;

  HostIF_GlobalVMLock(3);
  
  vm = vmDriverList;

  while (vm) { 
    if (vm->processID == processId) {
       break;
    }
    vm = vm->nextDriver;
  }
  
  HostIF_GlobalVMUnLock(3);

  return vm;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetNumVMs  --
 *
 *      Return the number of VMs.
 *      
 * Results:
 *      The number of VMs.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
int32
Vmx86_GetNumVMs()
{
   return vmCount;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetLockedPagesLimit -- 
 *
 *      Set the hard limit for the number of locked pages.
 *      The limit can be set only when one VM is running or if it not set.
 *      
 * Results:
 *      Returns TRUE on success and FALSE on failure to set the limit
 * 
 * Side effects:
 *      Hard limit changed.
 *
 *----------------------------------------------------------------------
 */
Bool
Vmx86_SetLockedPagesLimit(int32 limit)
{
   int retval = FALSE;

   HostIF_GlobalVMLock(4);
   if (lockedPageLimit == 0 || (lockedPageLimit != 0 && vmCount == 1)) {
      lockedPageLimit = limit;
      retval = TRUE;
   }
   HostIF_GlobalVMUnLock(4);

   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_ReduceLockedPagesLimit -- 
 *
 *      Reduce the hard limit for the number of locked pages.
 *      The limit can be reduced at any time.
 *      
 * Results:
 *      Returns TRUE on success and FALSE on failure to set the limit
 * 
 * Side effects:
 *      Hard limit changed.
 *
 *----------------------------------------------------------------------
 */
int32
Vmx86_ReduceLockedPagesLimit(int32 limit)
{
   HostIF_GlobalVMLock(4);
   if (lockedPageLimit > limit) {
      lockedPageLimit = limit;
   }
   HostIF_GlobalVMUnLock(4);

   return lockedPageLimit;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetLockedPagesLimit -- 
 *
 *      Set the hard limit for the number of locked pages.
 *      
 * Results:
 *      None
 * 
 * Side effects:
 *      Hard limit changed.
 *
 *----------------------------------------------------------------------
 */
int32
Vmx86_GetLockedPagesLimit()
{
   return lockedPageLimit;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_LockPage -- 
 *
 *      Lock a page.
 *      
 * Results:
 *      None
 * 
 * Side effects:
 *      Number of global and per-VM locked pages increased.
 *
 *----------------------------------------------------------------------
 */
int
Vmx86_LockPage(VMDriver *vm, void * addr, Bool userPage)
{
   if (lockedPageLimit != 0 && numLockedPages >= lockedPageLimit) {
      return PAGE_LOCK_LIMIT_EXCEEDED;
   } else {
      int retval = (int)HostIF_LockPage(vm,addr);
      if (retval != 0) {
         HostIF_GlobalVMLock(5);

         numLockedPages++;

         HostIF_GlobalVMUnLock(5);

         vm->memMgmtInfo.currentLocked++;

         if (userPage) {
            vm->stats.userLockedPages++;
            if (vm->stats.userLockedPages > vm->stats.maxUserLockedPages) {
               vm->stats.maxUserLockedPages = vm->stats.userLockedPages;
            }
         } else {
            vm->stats.monitorLockedPages++;
            if (vm->stats.monitorLockedPages > vm->stats.maxMonitorLockedPages) {
               vm->stats.maxMonitorLockedPages = vm->stats.monitorLockedPages;
            }
         }
         return retval;
      } else  {
         return PAGE_LOCK_FAILED;
      }
   }
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_UnlockPage -- 
 *
 *      Unlock a page.
 *      
 * Results:
 *      None
 * 
 * Side effects:
 *      Number of global and per-VM locked pages decreased.
 *
 *----------------------------------------------------------------------
 */
int
Vmx86_UnlockPage(VMDriver *vm, void * addr, Bool userPage)
{
   int retVal = (int)HostIF_UnlockPage(vm, addr);
   if (retVal == 0) {
      HostIF_GlobalVMLock(6);

      numLockedPages--;

      HostIF_GlobalVMUnLock(6);

      vm->memMgmtInfo.currentLocked--;

      if (userPage) {
         vm->stats.userLockedPages--;
      } else {
         vm->stats.monitorLockedPages--;
      }
   }

   return retVal;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetMemInfo -- 
 *
 *      Return the info about all VMs.
 *      
 * Results:
 *      None.
 * 
 * Side effects:
 *      GetSetMemInfoOutArgs is filled in.
 *
 *----------------------------------------------------------------------
 */
Bool
Vmx86_GetMemInfo(VMDriver *curVM,
                 int32 curVMOnly,
                 VMGetSetMemInfoOutArgs *outArgs,
                 int outArgsLength)
{
   VMDriver *vm;
   int i;
   int outSize;
   int wantedVMs;

   HostIF_GlobalVMLock(7);

   if (curVMOnly) {
      wantedVMs = 1;
   } else {
      wantedVMs = vmCount;
   }

   outSize = (wantedVMs - 1) * sizeof(VMMemMgmtInfo) + sizeof(VMGetSetMemInfoOutArgs);
   if (outSize > outArgsLength) {
      HostIF_GlobalVMUnLock(7);
      return FALSE;
   }

   outArgs->numVMs = wantedVMs;
   outArgs->numLockedPages = numLockedPages;
   outArgs->maxLockedPages = lockedPageLimit;
   outArgs->callerIndex = -1;

   if (wantedVMs == 1) {
      outArgs->memMgmtInfo[0] = curVM->memMgmtInfo;
      outArgs->callerIndex = 0;
   } else {
      vm = vmDriverList;
      i = 0;
      outArgs->callerIndex = -1;
      while (vm != NULL && i < vmCount) {
	 if (vm == curVM) {
	    outArgs->callerIndex = i;
	 }
	 outArgs->memMgmtInfo[i] = vm->memMgmtInfo;
	 i++;	 

	 vm = vm->nextDriver;
      }
   }

   HostIF_GlobalVMUnLock(7);

   return TRUE;
}

#define INFO_INCREMENT 16
static VMMemMgmtInfo info[INFO_INCREMENT];

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetMemInfoCopy -- 
 *
 *      Return the info about all VMs by copying the data out to
 *      the user.  outArgs->numVMs indicates how much space has been
 *      allocated for VMs.
 *      
 * Results:
 *      None.
 * 
 * Side effects:
 *      GetSetMemInfoOutArgs is filled in.
 *
 *----------------------------------------------------------------------
 */
Bool
Vmx86_GetMemInfoCopy(VMDriver *curVM, VMGetSetMemInfoOutArgs *outArgs)
{
   VMDriver *vm;
   int vmIndex, outIndex;
   int err;
   Bool status = FALSE;
   VMGetSetMemInfoOutArgs hdr;   

   err = HostIF_CopyFromUser(&hdr, outArgs, sizeof(VMGetSetMemInfoOutArgs));
   if (err != 0) {
      return FALSE;
   }

   HostIF_GlobalVMLock(8);

   hdr.numLockedPages = numLockedPages;
   hdr.maxLockedPages = lockedPageLimit;

   if (hdr.numVMs == 1) {
      hdr.memMgmtInfo[0] = curVM->memMgmtInfo;      
      err = HostIF_CopyToUser(outArgs, &hdr, sizeof(VMGetSetMemInfoOutArgs));
      if (err == 0) {
         status = TRUE;
      }
      goto copyReturn;
   }

   hdr.numVMs = vmCount;

   vm = vmDriverList;
   vmIndex = 0;
   outIndex = 0;
   while (vm != NULL && vmIndex < vmCount) {
      if (vm == curVM) {
	 hdr.callerIndex = vmIndex;
      }
      info[vmIndex % INFO_INCREMENT] = vm->memMgmtInfo;
      vmIndex++;
      if (vmIndex != 0 && vmIndex % INFO_INCREMENT == 0) {
	 err = HostIF_CopyToUser(&(outArgs->memMgmtInfo[outIndex]), info,
				 INFO_INCREMENT * sizeof(VMMemMgmtInfo));
	 if (err != 0) {
	    goto copyReturn;
	 }
	 outIndex += INFO_INCREMENT;
      }

      vm = vm->nextDriver;	 
   }
   if (vmIndex % INFO_INCREMENT != 0) {
      err = HostIF_CopyToUser(&(outArgs->memMgmtInfo[outIndex]), info,
			      (vmIndex % INFO_INCREMENT) * sizeof(VMMemMgmtInfo));
      if (err != 0) {
	 goto copyReturn;
      }
   }      
   err = HostIF_CopyToUser(outArgs, &hdr, sizeof(VMGetSetMemInfoOutArgs) -
			   sizeof(VMMemMgmtInfo));
   if (err != 0) {
      goto copyReturn;
   }      

   status = TRUE;
  
copyReturn:

   HostIF_GlobalVMUnLock(8);

   return status;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetMemInfo -- 
 *
 *      Set the memory management information about this VM.
 *      
 * Results:
 *      None.
 * 
 * Side effects:
 *      GetSetMemInfoOutArgs is filled in.
 *
 *----------------------------------------------------------------------
 */
void
Vmx86_SetMemInfo(VMDriver *curVM,
                 VMGetSetMemInfoInArgs *inArgs)
{
   HostIF_GlobalVMLock(9);

   if (inArgs->info.pageFaultRate >= 0) {
      curVM->memMgmtInfo.pageFaultRate = inArgs->info.pageFaultRate;
   }
   if (inArgs->info.lockFailRate >= 0) {
      curVM->memMgmtInfo.lockFailRate = inArgs->info.lockFailRate;
   }
   if (inArgs->info.maxLocked >= 0) {
      curVM->memMgmtInfo.maxLocked = inArgs->info.maxLocked;
   }
   if (inArgs->info.highWaterMark >= 0) {
      curVM->memMgmtInfo.highWaterMark = inArgs->info.highWaterMark;
   }
   if (inArgs->info.waitingForMemoryTime >= 0) {
      curVM->memMgmtInfo.waitingForMemoryTime = inArgs->info.waitingForMemoryTime;
   }
   if (inArgs->info.poweringOn >= 0) {
      curVM->memMgmtInfo.poweringOn = inArgs->info.poweringOn;
   }
   if (inArgs->info.hasFocus >= 0) {
      curVM->memMgmtInfo.hasFocus = inArgs->info.hasFocus;
   }

   HostIF_GlobalVMUnLock(9);
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetVMList 
 *
 *     Return the ids for all currently running VMs.
 *      
 *
 * Results:
 *      TRUE if the caller gave us enough space for the result.
 * 
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
Bool 
Vmx86_GetVMList(VMGetVMListResult *list, uint32 length)
{
   VMDriver *vm;
   int i = 0;

   HostIF_GlobalVMLock(14);
  
   if (vmCount * sizeof(uint32) + sizeof(VMGetVMListResult) > length) {
      HostIF_GlobalVMUnLock(14);
      return FALSE;
   }

   list->vmCount = vmCount;

   vm = vmDriverList;

   while (vm) { 
      list->vmUIDs[i++] = vm->vmUID;
      vm = vm->nextDriver;
   }
  
   HostIF_GlobalVMUnLock(14);

   return TRUE;
}

#define VM_LIST_INCREMENT 32
static uint32 vmList[VM_LIST_INCREMENT];

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetVMListCopy
 *
 *     Return the ids for all currently running VMs by copying out each
 *     
 *      
 *
 * Results:
 *      TRUE if the caller gave us enough space for the result.
 * 
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
Bool 
Vmx86_GetVMListCopy(VMGetVMListResult *list)
{
   VMDriver *vm;
   int vmIndex, outIndex;
   int err;
   Bool status = FALSE;
   VMGetVMListResult hdr;

   err = HostIF_CopyFromUser(&hdr, list, sizeof(VMGetVMListResult));
   if (err != 0) {
      return FALSE;
   }

   HostIF_GlobalVMLock(15);

   if (vmCount > hdr.vmCount) {
      goto copyReturn;
   }

   hdr.vmCount = vmCount;

   err = HostIF_CopyToUser(&(list->vmCount), &(hdr.vmCount), sizeof(hdr.vmCount));
   if (err != 0) {
      goto copyReturn;
   }

   vm = vmDriverList;
   vmIndex = 0;
   outIndex = 0;
   while (vm != NULL && vmIndex < vmCount) {
      vmList[vmIndex % VM_LIST_INCREMENT] = vm->vmUID;
      vm = vm->nextDriver;
      vmIndex++;
      if (vmIndex != 0 && vmIndex % VM_LIST_INCREMENT == 0) {
         err = HostIF_CopyToUser(&(list->vmUIDs[outIndex]), vmList,
                                 VM_LIST_INCREMENT * sizeof(uint32));
         if (err != 0) {
            goto copyReturn;
         }
         outIndex += VM_LIST_INCREMENT;
      }
   }

   if (vmIndex % VM_LIST_INCREMENT != 0) {
      err = HostIF_CopyToUser(&(list->vmUIDs[outIndex]), vmList,
                              (vmIndex % INFO_INCREMENT) * sizeof(uint32));
      if (err != 0) {
         goto copyReturn;
      }
   }

   status = TRUE;
  
copyReturn:

   HostIF_GlobalVMUnLock(15);

   return status;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetVMInfo
 *
 *     Set the static information for this VM such as its name and port.  
 *
 * Results:
 *     None.
 * 
 * Side effects:
 *     The VMDriver structure is updated with this info.
 *
 *----------------------------------------------------------------------
 */
void 
Vmx86_SetVMInfo(VMDriver *vm, VMSetVMInfoArgs *info)
{
   memcpy(vm->name, info->name, VM_NAME_LENGTH);
   memcpy(vm->userName, info->userName, VM_USER_NAME_LENGTH);
   vm->port = info->port;
   vm->state = info->state;
}

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetVMInfo
 *
 *     Get the static information for this VM such as its name and port.  
 *
 * Results:
 *     TRUE if a VM with this uid was found, false otherwise.
 * 
 * Side effects:
 *     None.
 *
 *----------------------------------------------------------------------
 */
Bool Vmx86_GetVMInfo(uint32 vmUID, VMGetVMInfoResult *info)
{
   Bool status = FALSE;
   VMDriver *vm;

   HostIF_GlobalVMLock(16);

   vm = vmDriverList;

   while (vm) { 
      if (vm->vmUID == vmUID) {
         memcpy(info->name, vm->name, VM_NAME_LENGTH);
         memcpy(info->userName, vm->userName, VM_USER_NAME_LENGTH);
         info->port = vm->port;
         info->vmUID = vm->vmUID;
	 info->state = vm->state;
         status = TRUE;
         break;
      }
      vm = vm->nextDriver;
   }

   HostIF_GlobalVMUnLock(16);

   return status;
}

/*
 *----------------------------------------------------------------------
 *
 * FreeAllResources 
 *
 *     Free all the resources allocated for a vm.
 *      
 *
 * Results:
 *      None
 * 
 * Side effects:
 *      Memory freed.
 *
 *----------------------------------------------------------------------
 */


static void
FreeAllResources(VMDriver *vm)
{
   if (vm) { 
      HostIF_FreeAllResources(vm);

      HostIF_GlobalVMLock(10);
      numLockedPages -= vm->memMgmtInfo.currentLocked;
      HostIF_GlobalVMUnLock(10);

      HostIF_FreeKernelMem(vm);
   }
#ifdef SUPPORT_VTRACE
   VTrace_FreeAllResources();
#endif
}
